/*
 * Copyright (c) 2020, Anton Staaf.  All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be
 * found in the LICENSE file.
 */

#include "ifs.h"

static float get_channel(std::uint16_t color, std::uint16_t mask)
{
    return float(color & mask) / float(mask);
}

static constexpr float bias(std::uint16_t mask)
{
    return float(mask & -mask) / 2.0f;
}

static std::uint16_t put_channel(float channel, std::uint16_t mask)
{
    return std::uint16_t(channel * mask + bias(mask)) & mask;
}

extern "C" void csub(std::int64_t * framebuffer,
                     std::int64_t * resolution,
                     std::int64_t * first_,
                     std::int64_t * last_,
                     State * state,
                     DisplayTransform * display,
                     std::int64_t * iterations,
                     double * blend)
{
    auto fb    = reinterpret_cast<std::uint16_t *>(*framebuffer);
    auto first = reinterpret_cast<Transform *>(*first_);
    auto last  = reinterpret_cast<Transform *>(*last_);

    auto x_res = int(resolution[0]);
    auto y_res = int(resolution[1]);
    auto count = int(*iterations);

    Math::Vector2f offset(display->offset);
    float          scale_x(static_cast<float>(display->scale_x));
    float          scale_y(static_cast<float>(display->scale_y));
    Math::Vector2f position(state->position);
    Color::Color3f color(state->color);

    for (int i = 0; i < count; ++i)
    {
        float p = state->random.next_float();

        for (Transform * transform = first; transform <= last; transform++)
        {
            p -= float(transform->probability);

            if (p <= 0)
            {
                auto matrix      = Math::Matrix22f(transform->matrix);
                auto translate   = Math::Vector2f(transform->translate);
                auto target      = Color::Color3f(transform->color);
                auto lerp_factor = float(transform->lerp_factor);

                position = matrix * position + translate;
                color    = lerp(target, color, lerp_factor);
                break;
            }
        }

        int x = int(position.x * scale_x + offset.x);
        int y = int(position.y * scale_y + offset.y);

        if (unsigned(x) < unsigned(x_res) &&
            unsigned(y) < unsigned(y_res))
        {
            constexpr std::uint16_t r_mask = 0b11111'000000'00000;
            constexpr std::uint16_t g_mask = 0b00000'111111'00000;
            constexpr std::uint16_t b_mask = 0b00000'000000'11111;

            int index = y * x_res + x;

            Color::Color3f a(get_channel(fb[index], r_mask),
                             get_channel(fb[index], g_mask),
                             get_channel(fb[index], b_mask));
            Color::Color3f c(lerp(a, color, float(*blend)));

            fb[index] = std::uint16_t(put_channel(c.r, r_mask) |
                                      put_channel(c.g, g_mask) |
                                      put_channel(c.b, b_mask));
        }
    }

    state->position = Math::Vector2d(position);
    state->color    = Color::Color3d(color);
}
