'
' Copyright (c) 2020, Anton Staaf.  All rights reserved.
' Use of this source code is governed by a BSD-style license that can be
' found in the LICENSE file.
'

'------------------------------------------------------------------------------
'
' An affine 2D transformation can be encoded in a 3x3 affine transformation
' matrix.  The last row will always be [0,0,1] so we don't have to store it.
' Instead, the six coefficients of the transformation are stored in an array
' of floats.  Since we lack array slices the array of transformations that is
' passed in to each of these functions is actually a 2D array, with the second
' dimension used to select from multiple transformations.  The alternative is
' to make a copy to and from a different array to do any operations on a
' transform.
'
' Normal matrix multiplication is used to compose transformations.
'
' [a0, a2, a4]   [b0, b2, b4]   [a0b0 + a2b1, a0b2 + a2b3, a0b4 + a2b5 + a4]
' [a1, a3, a5] * [b1, b3, b5] = [a1b0 + a3b1, a1b2 + a3b3, a1b4 + a3b5 + a5]
' [ 0,  0,  1]   [ 0,  0,  1]   [          0,           0,                1]
'
' Affine transformation matricies are applied to vectors by augmenting the
' vector into 3D as [x,y,1], and normal matrix/vector multiplication is then
' used to apply the transformation.
'
'------------------------------------------------------------------------------
' Apply the transform to the point x,y to compute either the new x or new y
' value.
'
function transform.x(transforms() as float, i as integer, x as float, y as float) as float
    transform.x = transforms(0,i) * x + transforms(2,i) * y + transforms(4,i)
end function

function transform.y(transforms() as float, i as integer, x as float, y as float) as float
    transform.y = transforms(1,i) * x + transforms(3,i) * y + transforms(5,i)
end function

sub transform.apply transforms() as float, i as integer, x as float, y as float
    local float t

    t = transforms(0,i) * x + transforms(2,i) * y + transforms(4,i)
    y = transforms(1,i) * x + transforms(3,i) * y + transforms(5,i)
    x = t
end sub

'------------------------------------------------------------------------------
' Compute the determinant of the affine transformation.  The determinant will
' indicate the overall area scaling done by the transform.  Because of the
' special shape of the augmented transformation matrix (bottom row of [0,0,1])
' we only have to account for the upper left 2x2 portion of the matrix.
'
function transform.determinant(transforms() as float, i as integer) as float
    transform.determinant = transforms(0,i) * transforms(3,i) - transforms(1,i) * transforms(2,i)
end function

'------------------------------------------------------------------------------
' Translate the transform by a given x,y amount
'
' [1, 0, x]   [b0, b2, b4]   [b0, b2, b4 + x]
' [0, 1, y] * [b1, b3, b5] = [b1, b3, b5 + y]
' [0, 0, 1]   [ 0,  0,  1]   [ 0,  0,      1]
'
sub transform.translate transforms() as float, i as integer, x as float, y as float
    transforms(4,i) = transforms(4,i) + x
    transforms(5,i) = transforms(5,i) + y
end sub

'------------------------------------------------------------------------------
' Scale the transform by the given x,y amount
'
' [ s,  0,  0]   [b0, b2, b4]   [sb0, sb2, sb4]
' [ 0,  t,  0] * [b1, b3, b5] = [tb1, tb3, tb5]
' [ 0,  0,  1]   [ 0,  0,  1]   [  0,   0,   1]
'
sub transform.scale transforms() as float, i as integer, s as float, t as float
    transforms(0,i) = s * transforms(0,i)
    transforms(1,i) = t * transforms(1,i)
    transforms(2,i) = s * transforms(2,i)
    transforms(3,i) = t * transforms(3,i)
    transforms(4,i) = s * transforms(4,i)
    transforms(5,i) = t * transforms(5,i)
end sub

'------------------------------------------------------------------------------
' Rotate the transform by theta radians.  The new rotation is applied to the
' left of the transform matrix so it wll happen after any existing scale,
' shear, or rotation.
'
' c = cos(theta)
' s = sin(theta)
'
' [1, t, 0]   [b0, b2, b4]   [b0 + tb1, b2 + tb3, b4 + tb5]
' [0, 1, 0] * [b1, b3, b5] = [      b1,       b3,       b5]
' [0, 0, 1]   [ 0,  0,  1]   [       0,        0,        1]
'
sub transform.shear transforms() as float, i as integer, t as float
    transforms(0,i) = transforms(0,i) + t * transforms(1,i)
    transforms(2,i) = transforms(2,i) + t * transforms(3,i)
    transforms(4,i) = transforms(4,i) + t * transforms(5,i)
end sub

'------------------------------------------------------------------------------
' Rotate the transform by theta radians.  The new rotation is applied to the
' left of the transform matrix so it wll happen after any existing scale,
' shear, or rotation.
'
' c = cos(theta)
' s = sin(theta)
'
' [c, -s, 0]   [b0, b2, b4]   [cb0 - sb1, cb2 - sb3, cb4 - sb5]
' [s,  c, 0] * [b1, b3, b5] = [sb0 + cb1, sb2 + cb3, sb4 + cb5]
' [0,  0, 1]   [ 0,  0,  1]   [        0,         0,         1]
'
sub transform.rotate transforms() as float, i as integer, theta as float
    local integer j
    local float   s = sin(theta)
    local float   c = cos(theta)
    local float   t(5)

    t(0) = c * transforms(0,i) - s * transforms(1,i)
    t(1) = s * transforms(0,i) + c * transforms(1,i)
    t(2) = c * transforms(2,i) - s * transforms(3,i)
    t(3) = s * transforms(2,i) + c * transforms(3,i)
    t(4) = c * transforms(4,i) - s * transforms(5,i)
    t(5) = s * transforms(4,i) + c * transforms(5,i)

    for j = 0 to 5
        transforms(j,i) = t(j)
    next
end sub

'------------------------------------------------------------------------------
' Compute the derivative of the largest singular value of the transformation
' (ignoring the translation portion, so just the upper left 2x2 portion) with
' respect to the transformation matrix.  This gives a gradient that can be used
' to increase or decrease the largest singular value.  The gradient is returned
' in gradient(), a 4 entry array, and the largest singular value is returned in
' sigma.
'
' The singular values of a matrix represent how the matrix stretches the values
' that are transformed by it.  If the largest singular value is greater than
' one the transformation will expand the space it is applied to.  If it is less
' than one it will contract the space.
'
sub transform.gradient transforms() as float, index as integer, gradient() as float, sigma as float
    local float a   = transforms(0,index)
    local float b   = transforms(2,index)
    local float c   = transforms(1,index)
    local float d   = transforms(3,indeX)
    local float det = a * d - b * c
    local float q   = (((a-d)*(a-d)+(b+c)*(b+c))*((a+d)*(a+d)+(b-c)*(b-c)))
    local float f   = a * a + b * b + c * c + d * d
    local float sq  = sqr(q)

    sigma = sqr((f + sq) / 2)

    if sq = 0 then
        gradient(0) = a / (2 * sigma)
        gradient(2) = b / (2 * sigma)
        gradient(1) = c / (2 * sigma)
        gradient(3) = d / (2 * sigma)
    else
        gradient(0) = ((a * f - 2 * det * d) / sq + a) / (2 * sigma)
        gradient(2) = ((b * f + 2 * det * c) / sq + b) / (2 * sigma)
        gradient(1) = ((c * f + 2 * det * b) / sq + c) / (2 * sigma)
        gradient(3) = ((d * f - 2 * det * a) / sq + d) / (2 * sigma)
    end if
end sub
