' ***********************************************************************
' *      F A S T E R L I F E - an MMBasic implementation for CMM2       *
' *    With a hefty nod to Mike Abrash / Computer Graphics Black Book   *
' *           Steve Johnson - "NPHighview"  -   November 2020           *
' ***********************************************************************
'
' ********************************** Declarations ***********************

OPTION BASE 0                       ' Required for modulo arithmetic,
OPTION Y_AXIS UP                    ' and to keep me
OPTION EXPLICIT                     ' from going nuts :-)

' number of rows and columns.  No need to make these powers of 2.
CONST       xext = 400              ' 512 is the largest value possible
CONST       yext = 200              ' 256 is the largest value possible

CONST       LifeProb = 0.1          ' Set here; used for random life subroutine

CONST       ScreenMode = 8          ' 640x480 will accommodate all cell sizes that we can fit
CONST       ColorDepth = 8          ' Will not work at color depth 12.

CONST       fcolor = &H333333       ' field / background color for field
CONST       bcolor = &HFFFFFF       ' border color
CONST       dcolor = &H000000       ' color for "dead" cell
CONST       acolor = &HFFFF00       ' color for "alive" cell
CONST       ecolor = &H00FFFF       ' color for "previously alive" cell

DIM STRING  key$, keymod$, filename$' For keycapture and for screen print

DIM INTEGER xoff                    ' Offset to left edge of playing field.
DIM INTEGER yoff                    ' Offset to bottom edge of playing field.
DIM INTEGER wrapx = 0               ' Check for x border excursion
DIM INTEGER wrapy = 0               ' Check for y border excursion
DIM INTEGER gap = 2                 ' Safety gap for border excursions

DIM INTEGER x, y                    ' working coordinates within playing field
DIM INTEGER minx, maxx, miny, maxy  ' Use these to restrict evaluation area when possible
DIM INTEGER box_min_x = xext-1
DIM INTEGER box_max_x = 0           ' Use these during setup for next evaluation
DIM INTEGER box_min_y = yext-1
DIM INTEGER box_max_y = 0           ' Use these during setup for next evaluation
DIM INTEGER z, NotZ                 ' z (this) and NotZ (next) generation index.
DIM INTEGER xm1, xp1, ym1, yp1      ' That's x-1, x+1, y-1, and y+1.  Calculated/wrapped on the fly.
DIM INTEGER N                       ' # of neighbors
DIM INTEGER life(xext, yext, 1)     ' Storage for current & next life generations.
DIM INTEGER ccol(1)=(ecolor,acolor) ' Alternatng cell colors. Don't take () for size, rather max index.
DIM INTEGER change, count, elapsed  ' execution statistics
DIM FLOAT   rate

' ********************************  Main Program  *********************************

IF xext*yext > (512*256) THEN
  PAGE WRITE 0 : PRINT "Playing Field is too big; please reduce xext and/or yext. Press any key"
  key$ = INKEY$ : DO WHILE key$ = "" : key$ = INKEY$ : LOOP
  CLS : END
END IF

MODE ScreenMode,ColorDepth
PAGE WRITE 1 : CLS : PAGE WRITE 0 : CLS

xoff = (MM.HRes/2) - ((xext+6)/2)   ' Center the field on the screen
yoff = (MM.VRes/2) - ((yext+6)/2)   ' Regardless of mode & extent

MATH SET 0, Life()                  ' use mmBasic-specific syntax to clear all life

z = 0                               ' Use Gen. 0 for initialization.  Toggle later.

InteractiveInitialize               ' Position cursor and pick the objects to place.

PAGE WRITE 0 : TEXT MM.Hres/2, yoff/2, "   Working...   ", "CB", 1, 1, RGB(yellow), RGB(black)
PAGE WRITE 0 : TEXT MM.Hres/2, yoff/2-48, "[p] print [q] quit", "CB", 1, 1, RGB(&HA0, &HA0, &HA0), RGB(black)

count = 0 : elapsed = 0

DO
  TIMER = 0
  change = 0                            ' Keep track of the number of cells that change state each generation
  CountNeighbors                        ' Calculate & plot the next generation
  elapsed = elapsed + TIMER
  z = 1-z                               ' What's the index value for the next generation
  count = count + 1                     ' maintain count of generations
  rate = elapsed/count                  ' Calculate mSec per generation

  PAGE WRITE 0 : TEXT MM.Hres/2, yoff/2,    "  Generation " + STR$(count) + ": "+STR$(change) + " changes in "+FORMAT$(TIMER/1000, "%-.2f")+" Sec  ", "CB", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT MM.Hres/2, yoff/2-16, "  Average of " + FORMAT$(rate/1000, "%-.2f") + " Sec/gen  ", "CB", 1, 1, RGB(yellow), RGB(black)

  key$ = UCASE$(INKEY$)

  SELECT CASE key$
    CASE "Q"  : CLS : END
    CASE "P"    ' "Print" the screen to a bitmap file, showing extent, generations, and rate
      filename = "Life"++STR$(xext)+"x"+STR$(yext)+"x"+STR$(count)+".bmp"
      SAVE IMAGE Filename$
  END SELECT

LOOP

CLS : END

' ********************************  Subroutines  *********************************

SUB SetLife(x,y,DorA)               ' This draws on both Page 0 and 1 to initially display quickly,
                                    ' and also persist through the CountNeighbors process
  life(x,y,z) = DorA                ' Record the fact that there's a cell dead or alive here

  SELECT CASE DorA                  ' Is the cell dead (0) or alive (1)?
    CASE 1 :                        ' If alive, draw it, and then expand the evaluation box to accommodate it
      PAGE WRITE 0 : PIXEL x+xoff, y+yoff, ccol(1-z)
      PAGE WRITE 1 : PIXEL x+xoff, y+yoff, ccol(1-z)
      box_max_x = MAX(box_max_x, x)
      box_min_x = MIN(box_min_x, x)
      box_max_y = MAX(box_max_y, y)
      box_min_y = MIN(box_min_y, y)
      change = change + 1
    CASE ELSE :
      PAGE WRITE 0 : PIXEL x+xoff, y+yoff, fcolor
      PAGE WRITE 1 : PIXEL x+xoff, y+yoff, fcolor
  END SELECT
END SUB


SUB CopyLife                          ' This copies the display field area from Page 1 to the center of Page 0'
  PAGE WRITE 0 : BLIT xoff, yoff, xoff, yoff, xext, yext, 1 : PAGE WRITE 1
END SUB


SUB DisplayBorder(border)             ' Displays border and empty field using global offsets and extents
  PAGE WRITE 0 : BOX xoff-border-2, yoff-border-3, xext+2*(2+border), yext+2*(3+border), border, bcolor
  PAGE WRITE 0 : BOX xoff, yoff, xext, yext, 1, fcolor, fcolor
  PAGE WRITE 1 : BOX xoff-border-2, yoff-border-3, xext+2*(2+border), yext+2*(3+border), border, bcolor
  PAGE WRITE 1 : BOX xoff, yoff, xext, yext, 1, fcolor, fcolor
END SUB


SUB DisplayField                      ' Displays the field as pixels
  PAGE WRITE 1                        ' This is in the lower left corner of Page 1
  FOR y = miny To maxy                 ' Then copied into the border on Page 0
    FOR x = minx To maxx
      SELECT CASE life(x, y, z)
        CASE    1 : PIXEL x+xoff, y+yoff, ccol(1-z)
        CASE ELSE : PIXEL x+xoff, y+yoff, fcolor
      END SELECT
    NEXT x
  NEXT y
  CopyLife
END SUB


SUB CountNeighbors                    ' For a given plane, count all neighbors, set and draw the next generation
  LOCAL INTEGER xm1, xp1, ym1, yp1

  box_min_x = xext-1 : box_max_x = 0  ' prep for establishing evaluation bounding box
  box_min_y = yext-1 : box_max_y = 0

  NotZ = 1-z                          ' Whatever current index, next index is given by 1-z

  FOR x = minx TO maxx                ' Iterate only over region previously found to have cells in it

    xm1 = (x+xext-1) MOD xext
    xp1 = (x+1)      MOD xext         ' calculate array indices for neighbors even if around the border

    PAGE WRITE 0 : PIXEL x+xoff, yoff-10, ccol(z) : PAGE WRITE 1        ' show progress bar.  No impact on timing.

    FOR y = miny TO maxy              ' Iterate only over region previously found to have cells in it

      ym1 = (y+yext-1) MOD yext
      yp1 = (y+1)      MOD yext       ' calculate array indices for neighbors, even if around the border

      N = life(xm1,ym1,z) + life(xm1,y,z) + life(xm1,yp1,z) + life(x,ym1,z) + life(x,yp1,z) + life(xp1,ym1,z) + life(xp1,y,z) + life(xp1,yp1,z)

      SELECT CASE Life(x,y,z)         ' OK, figure out what to do.  First, am I alive or dead.
        CASE 0                        ' Currently dead.  Only a 3 can revive
          SELECT CASE N               ' Draw only if cells change.
            CASE 3
              life(x,y,NotZ) = 1
              PIXEL x+xoff, y+yoff, ccol(z)
              change = change + 1
              box_max_x = MAX(box_max_x, x) : box_min_x = MIN(box_min_x, x)
              box_max_y = MAX(box_max_y, y) : box_min_y = MIN(box_min_y, y)
            CASE ELSE : life(x,y,NotZ) = 0
          END SELECT

        CASE ELSE                     ' Currently alive.  N <> 2 or 3 kills
          SELECT CASE N               ' Draw only if cells change.
            CASE 2, 3
              life(x,y,NotZ) = 1
              box_max_x = MAX(box_max_x, x) : box_min_x = MIN(box_min_x, x)
              box_max_y = MAX(box_max_y, y) : box_min_y = MIN(box_min_y, y)

            CASE ELSE
              life(x,y,NotZ) = 0
              PIXEL x+xoff, y+yoff, dcolor
              change = change + 1                                       ' Count as a change
          END SELECT

      END SELECT

    NEXT y

  NEXT x

  IF change > 0 THEN
    minx = box_min_x - gap : IF minx <= 0        THEN wrapx = 1
    maxx = box_max_x + gap : IF maxx >= (xext-1) THEN wrapx = 1
    miny = box_min_y - gap : IF miny <= 0        THEN wrapy = 1
    maxy = box_max_y + gap : IF maxy >= (yext-1) THEN wrapy = 1
    IF wrapx = 1 THEN minx = 0 : maxx = xext-1 : wrapx = 0
    IF wrapy = 1 THEN miny = 0 : maxy = yext-1 : wrapy = 0
  ENDIF

  CopyLife       ' BLIT the field from page 1 to 0

END SUB


SUB InteractiveInitialize           ' Provide the user the opportunity to set up the playing DisplayField
  LOCAL INTEGER xinc, yinc          ' Movement increments.  1 if no keyboard modifiers, 10 if a shift, ctrl, or alt is pressed
  LOCAL STRING keymod$              ' place for keyboard modifiers, if pressed

  box_min_x = xext-1 : box_max_x = 0  ' prep for establishing evaluation bounding box
  box_min_y = yext-1 : box_max_y = 0

  ' Display the title and playing field dimensions

  DisplayBorder(3)
  PAGE WRITE 0 : TEXT MM.Hres/2, MM.Vres-yoff/2,    "Conway's Game of Life",  "CT", 4, 1, RGB(white), RGB(black)
  PAGE WRITE 0 : TEXT MM.Hres/2, MM.Vres-yoff/2-20, STR$(xext)+"x"+STR$(yext), "CT", 4, 1, RGB(white), RGB(black)

  ' Let the user know what options are available for initializing the playing field

  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 10, CHR$(144)+CHR$(145)+": Move Cursor    [Ctrl]"+CHR$(144)+CHR$(145)+": Move Cursor Faster", "CT", 1, 1, RGB(cyan), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 24, " .: Single Cell                                ", "CT", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 38, " H: Heavyweight Ship      D: Pentadecathlon    ", "CT", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 52, " M: Mid-weight  Ship      P: rPentomino        ", "CT", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 66, " L: Lightweight Ship      G: Glider Gun        ", "CT", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 80, " R: Random / Continue     B: Blinker           ", "CT", 1, 1, RGB(yellow), RGB(black)
  PAGE WRITE 0 : TEXT xoff+xext/2, yoff - 94, " C: Continue              Q: Quit              ", "CT", 1, 1, RGB(cyan),   RGB(black)

  SPRITE LOAD "nice_cursor.spr",1   ' load the mouse cursor as sprite number 1
  x=xext/2 : y=yext/2               ' Put the point of the cursor in the middle of the field
  PAGE WRITE 1 : SPRITE show safe 1, x+xoff, y+yoff, 3   ' show the cursor

  CopyLife                          ' Blit the playing field from Page 1 to Page 0.  Necessary??

  change = 0

  DO : key$=UCASE$(inkey$) : LOOP UNTIL key$ <> ""

  DO
    PAGE WRITE 1 : SPRITE HIDE SAFE 1      ' prepare to move the Cursor

    SELECT CASE key$                       ' Cursor movement, slow or fast
      CASE CHR$(128)
        IF KEYDOWN(7) > 0 THEN yinc = 10 ELSE yinc = 1
        y = MIN(yext-1, MAX(0, y+yinc))    ' don't go too far off screen
      CASE CHR$(129)
        IF KEYDOWN(7) > 0 THEN yinc = 10 ELSE yinc = 1
        y = MIN(yext-1, MAX(0, y-yinc))
      CASE CHR$(130)
        IF KEYDOWN(7) > 0 THEN xinc = 10 ELSE xinc = 1
        x = MIN(xext-1, MAX(0, x-xinc))
      CASE CHR$(131)
        IF KEYDOWN(7) > 0 THEN xinc = 10 ELSE xinc = 1
        x = MIN(xext-1, MAX(0, x+xinc))
      CASE "."       : SetLife(x,y,1)      ' just activate the single cell here
      CASE "R"       : RandomLife(LifeProb) : EXIT DO
      CASE "L"       : PlaceLWSS(x, y)     ' Lightweight Spaceship
      CASE "M"       : PlaceMWSS(x, y)     ' Medium weight Spaceship
      CASE "H"       : PlaceHWSS(x, y)     ' Heavyweight Spaceship
      CASE "B"       : PlaceBlinker(x, y, CINT(rnd))
      CASE "G"       : PlaceGG(x, y)       ' Glider Gun
      CASE "P"       : PlaceRPentomino(x, y)
      CASE "D"       : PlacePentadecathlon(x, y)
      CASE "C"       : EXIT DO
      CASE "Q"       : END
    END SELECT

    IF x<> sprite(x,1) OR y <> sprite(y,1) THEN
      PAGE WRITE 1 : SPRITE SHOW SAFE 1, x+xoff, y+yoff, 3
      CopyLife
    END IF

    key$ = UCASE$(inkey$)

  LOOP

  ' Interactive setup is complete.  Clear the prompts and prepare to go!
  PAGE WRITE 0 : BOX 0, 0, MM.Hres, yoff - 10, 1, RGB(black), RGB(black)
  PAGE WRITE 1 : BOX 0, 0, MM.Hres, yoff - 10, 1, RGB(black), RGB(black)

  PAGE WRITE 1 : SPRITE HIDE SAFE 1

  ' Now establish the evaluation bounding box for the first display

  IF change > 0 THEN
    minx = box_min_x - gap : IF minx <= 0        THEN wrapx = 1
    maxx = box_max_x + gap : IF maxx >= (xext-1) THEN wrapx = 1
    miny = box_min_y - gap : IF miny <= 0        THEN wrapy = 1
    maxy = box_max_y + gap : IF maxy >= (yext-1) THEN wrapy = 1
    IF wrapx = 1 THEN minx = 0 : maxx = xext-1 : wrapx = 0
    IF wrapy = 1 THEN miny = 0 : maxy = yext-1 : wrapy = 0
  ENDIF

  CopyLife       ' BLIT the field from page 1 to 0

END SUB

' ***************************** Various Life Patterns *************************************

SUB RandomLife(prob)                ' Generate life randomly with specified probability
  FOR y = 0 TO yext-1
    FOR x = 0 TO xext-1
      IF prob >= Rnd() THEN SetLife(x,y,1)
    NEXT x
  NEXT y
END SUB


SUB PlaceLWSS(x, y)                 ' Place a Light Weight Space Ship
  LOCAL INTEGER xp1, xp2, xp3, xp4, yp1, yp2, yp3
  x = x MOD xext : xp1 = (x+1) MOD xext : xp2 = (x+2) MOD xext : xp3 = (x+3) MOD xext : xp4 = (x+4) MOD xext
  y = y MOD yext : yp1 = (y+1) MOD yext : yp2 = (y+2) MOD yext : yp3 = (y+3) MOD yext

  SetLife(xp1, y,   1) : SetLife(xp2, y,   1) : SetLife(xp3, y,   1) : SetLife(xp4, y, 1) : SetLife(x,   yp1, 1)
  SetLife(xp4, yp1, 1) : SetLife(xp4, yp2, 1) : SetLife(x,   yp3, 1) : SetLife(xp3, yp3, 1)
END SUB


SUB PlaceMWSS(x, y)                 ' Place a Middle Weight Space Ship
  LOCAL INTEGER xp1, xp2, xp3, xp4, xp5, yp1, yp2, yp3, yp4
  x=x MOD xext : xp1=(x+1) MOD xext : xp2=(x+2) MOD xext : xp3=(x+3) MOD xext : xp4=(x+4) MOD xext : xp5=(x+5) MOD xext
  y=y MOD yext : yp1=(y+1) MOD yext : yp2=(y+2) MOD yext : yp3=(y+3) MOD yext : yp4=(y+4) MOD yext

  SetLife(xp2, y,   1) : SetLife(x,   yp1, 1) : SetLife(xp4, yp1, 1) : SetLife(xp5, yp2, 1) : SetLife(x,   yp3, 1)
  SetLife(xp5, yp3, 1) : SetLife(xp1, yp4, 1) : SetLife(xp2, yp4, 1) : SetLife(xp3, yp4, 1) : SetLife(xp4, yp4, 1)
  SetLife(xp5, yp4, 1)
END SUB


SUB PlaceHWSS(x, y)                 ' Place a Heavyweight Space Ship
  LOCAL INTEGER xp1, xp2, xp3, xp4, xp5, xp6, yp1, yp2, yp3, yp4
  x=x MOD xext : xp1=(x+1) MOD xext : xp2=(x+2) MOD xext : xp3=(x+3) MOD xext : xp4=(x+4) MOD xext : xp5=(x+5) MOD xext
  xp6 = (x+6) MOD xext : y=y MOD yext : yp1=(y+1) MOD yext : yp2=(y+2) MOD yext : yp3=(y+3) MOD yext : yp4=(y+4) MOD yext

  SetLife(xp2, y,   1) : SetLife(xp3, y,   1) : SetLife(x,   yp1, 1) : SetLife(xp5, yp1, 1) : SetLife(xp6, yp2, 1)
  SetLife(x,   yp3, 1) : SetLife(xp6, yp3, 1) : SetLife(xp1, yp4, 1) : SetLife(xp2, yp4, 1) : SetLife(xp3, yp4, 1)
  SetLife(xp4, yp4, 1) : SetLife(xp5, yp4, 1) : SetLife(xp6, yp4, 1)
END SUB


SUB PlaceGG(x, y)  ' Place a glider gun at specified coordinates, randomly facing U/D/L/R
  LOCAL INTEGER xm1, xp1, ym1, yp1
  x = x MOD xext : xm1 = (x+xext-1) MOD xext : xp1 = (x+1) MOD xext
  y = y MOD yext : ym1 = (y+yext-1) MOD yext : yp1 = (y+1) MOD yext

  SELECT CASE INT(4*rnd)
    CASE 0 : SetLife(xm1, y,   1) : SetLife(x,   ym1, 1) : SetLife(x, y,   1) : SetLife(xp1, ym1, 1) : SetLife(xp1, yp1, 1)
    CASE 1 : SetLife(xm1, ym1, 1) : SetLife(xm1, yp1, 1) : SetLife(x, y,   1) : SetLife(x,   yp1, 1) : SetLife(xp1, y,   1)
    CASE 2 : SetLife(xm1, yp1, 1) : SetLife(x,   y,   1) : SetLife(x, ym1, 1) : SetLife(xp1, yp1, 1) : SetLife(xp1, y,   1)
    CASE 3 : SetLife(xm1, ym1, 1) : SetLife(xm1, y,   1) : SetLife(x, y,   1) : SetLife(x,   yp1, 1) : SetLife(xp1, ym1, 1)
  END SELECT
END SUB


SUB PlaceRPentomino(x, y)       ' Place an "R" Pentomino Oscillator
  LOCAL INTEGER xp1, xp2, yp1, yp2
  x = x MOD xext : xp1 = (x+1) MOD xext : xp2 = (x+2) MOD xext
  y = y MOD yext : yp1 = (y+1) MOD yext : yp2 = (y+2) MOD yext

  SetLife(xp1, y,   1) : SetLife(x,   yp1, 1) : SetLife(xp1, yp1, 1) : SetLife(xp1, yp2, 1) : SetLife(xp2, yp2, 1)
END SUB


SUB PlacePentadecathlon(x, y)     ' Place a Pentadecathlon Oscillator
  LOCAL INTEGER xp1, xp2, yp1, yp2, yp3, yp4, yp5, yp6, yp7, yp8, yp9, yp10, yp11
  x = x MOD xext : xp1 = (x+1) MOD xext : xp2 = (x+2) MOD xext
  y = y MOD yext : yp1 = (y+1) MOD yext : yp2 = (y+2) MOD yext : yp3 = (y+3) MOD yext : yp4 = (y+4) MOD yext : yp5 = (y+5) MOD yext
  yp6 = (y+6) MOD yext : yp7 = (y+7) MOD yext : yp8 = (y+8) MOD yext : yp9 = (y+9) MOD yext : yp10 = (y+10) MOD yext : yp11 = (y+11) MOD yext

  SetLife(x, y,   1) : SetLife(xp1, y,   1) : SetLife(xp2, y,   1) ' bottom row
  SetLife(x, yp3, 1) : SetLife(xp1, yp3, 1) : SetLife(xp2, yp3, 1)
  SetLife(x, yp5, 1) : SetLife(xp1, yp5, 1) : SetLife(xp2, yp5, 1)
  SetLife(x, yp6, 1) : SetLife(xp1, yp6, 1) : SetLife(xp2, yp6, 1)
  SetLife(x, yp8, 1) : SetLife(xp1, yp8, 1) : SetLife(xp2, yp8, 1)
  SetLife(x, yp11,1) : SetLife(xp1, yp11,1) : SetLife(xp2, yp11,1)

  SetLife(xp1, yp1, 1) : SetLife(xp1, yp2, 1)
  SetLife(xp1, yp2, 1) : SetLife(xp1, yp2, 1)
  SetLife(xp1, yp9, 1) : SetLife(xp1, yp10,1)
  SetLife(xp1, yp10,1) : SetLife(xp1, yp2, 1)
END SUB


Sub PlaceBlinker(x, y, Dir)         ' Place an oscillator at the specified coordinates
  LOCAL INTEGER xm1, xp1, ym1, yp1
  x = x MOD xext : xm1 = (x+xext-1) MOD xext : xp1 = (x+1) MOD xext
  y = y MOD yext : ym1 = (y+yext-1) MOD yext : yp1 = (y+1) MOD yext

  IF Dir = 0 THEN
    SetLife(xm1, y, 1) : SetLife(x,   y, 1) : SetLife(xp1, y, 1)
  ELSE
    SetLife(x, ym1, 1) : SetLife(x, y,   1) : SetLife(x, yp1, 1)
  ENDIF
END SUB
