' Transpiled on 07-01-2024 18:05:29
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Option Base 0
Option Default None
Option Explicit On
' src/splib/system.inc ++++
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' Preprocessor value GAMEMITE defined
Const sys.VERSION = 102201
Const sys.NO_DATA$ = Chr$(&h7F)
Const sys.CRLF$ = Chr$(13) + Chr$(10)
Const sys.FIRMWARE = Int(1000000 * MM.Info(Version))
Const sys.SUCCESS = 0
Const sys.FAILURE = -1
Dim sys.break_flag%
Dim sys.err$

Sub sys.override_break(callback$)
 sys.break_flag% = 0
 Option Break 4
 If Len(callback$) Then
  Execute "On Key 3, " + callback$ + "()"
 Else
  On Key 3, sys.break_handler()
 EndIf
End Sub

Sub sys.break_handler()
 Inc sys.break_flag%
 If sys.break_flag% > 1 Then
  sys.restore_break()
  End
 EndIf
End Sub

Sub sys.restore_break()
 sys.break_flag% = 0
 On Key 3, 0
 Option Break 3
End Sub

Function sys.error%(code%, msg$)
 If Not code% Then Error "Invalid error code"
 sys.error% = code%
 sys.err$ = msg$
End Function

' ---- src/splib/system.inc
' src/splib/ctrl.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const ctrl.R      = &h01
Const ctrl.START  = &h02
Const ctrl.HOME   = &h04
Const ctrl.SELECT = &h08
Const ctrl.L      = &h10
Const ctrl.DOWN   = &h20
Const ctrl.RIGHT  = &h40
Const ctrl.UP     = &h80
Const ctrl.LEFT   = &h100
Const ctrl.ZR     = &h200
Const ctrl.X      = &h400
Const ctrl.A      = &h800
Const ctrl.Y      = &h1000
Const ctrl.B      = &h2000
Const ctrl.ZL     = &h4000
Const ctrl.OPEN  = -1
Const ctrl.CLOSE = -2
Const ctrl.SOFT_CLOSE = -3
Const ctrl.PULSE = 0.001
Const ctrl.UI_DELAY = 200
Dim ctrl.open_drivers$
Dim ctrl.key_type%
Dim ctrl.key_map%(31 + MM.Info(Option Base))

Sub ctrl.init_keys(use_inkey%, period%, nbr%)
 ctrl.term_keys()
 ctrl.key_type% = 0
 On Key ctrl.on_key()
End Sub

Sub ctrl.on_key()
 Poke Var ctrl.key_map%(), Asc(LCase$(Inkey$)), 1
End Sub

Sub ctrl.term()
 ctrl.term_keys()
 On Error Ignore
 Do While Len(ctrl.open_drivers$)
  Call Field$(ctrl.open_drivers$, 1), ctrl.CLOSE
 Loop
 On Error Abort
End Sub

Sub ctrl.term_keys()
 On Key 0
 Memory Set Peek(VarAddr ctrl.key_map%()), 0, 256
 Do While Inkey$ <> "" : Loop
End Sub

Function ctrl.keydown%(i%)
 ctrl.keydown% = Peek(Var ctrl.key_map%(), i%)
 If ctrl.key_type% = 0 Then Poke Var ctrl.key_map%(), i%, 0
End Function

Sub ctrl.wait_until_idle(d1$, d2$, d3$, d4$, d5$)
 Local k%
 Do
  Call d1$, k%
  If Not k% Then If Len(d2$) Then Call d2$, k%
  If Not k% Then If Len(d3$) Then Call d3$, k%
  If Not k% Then If Len(d4$) Then Call d4$, k%
  If Not k% Then If Len(d5$) Then Call d5$, k%
  If Not k% Then Exit Do
  Pause 5
 Loop
End Sub

Sub keys_cursor(x%)
 If x% < 0 Then Exit Sub
 x% =    ctrl.keydown%(32)  * ctrl.A
 Inc x%, ctrl.keydown%(128) * ctrl.UP
 Inc x%, ctrl.keydown%(129) * ctrl.DOWN
 Inc x%, ctrl.keydown%(130) * ctrl.LEFT
 Inc x%, ctrl.keydown%(131) * ctrl.RIGHT
End Sub

Sub keys_cursor_ext(x%)
 If x% < 0 Then Exit Sub
 x% =    ctrl.keydown%(32)  * ctrl.A
 Inc x%, ctrl.keydown%(98)  * ctrl.B
 Inc x%, (ctrl.keydown%(101) Or ctrl.keydown%(113)) * ctrl.SELECT
 Inc x%, ctrl.keydown%(115) * ctrl.START
 Inc x%, ctrl.keydown%(128) * ctrl.UP
 Inc x%, ctrl.keydown%(129) * ctrl.DOWN
 Inc x%, ctrl.keydown%(130) * ctrl.LEFT
 Inc x%, ctrl.keydown%(131) * ctrl.RIGHT
End Sub

Sub ctrl.open_driver(d$)
 Cat ctrl.open_drivers$, d$ + ","
End Sub

Sub ctrl.close_driver(d$)
 Local idx% = Instr(ctrl.open_drivers$, d$)
 Select Case idx%
  Case 0
  Case 1
   ctrl.open_drivers$ = Mid$(ctrl.open_drivers$, Len(d$) + 2)
  Case Else
   ctrl.open_drivers$ = Mid$(ctrl.open_drivers$, 1, idx% - 1) + Mid$(ctrl.open_drivers$, idx% + Len(d$) + 1)
 End Select
End Sub

Sub ctrl.gamemite(x%)
 Select Case x%
  Case >= 0
   x% = Port(GP12,2,GP11,2,GP8,1,GP8,1,GP11,1,GP10,1,GP9,1,GP13,3,GP13,3)
   x% = (x% Xor &h7FFF) And &h29EA
   Exit Sub
  Case ctrl.OPEN
   Local i%
   For i% = 8 To 15 : SetPin MM.Info(PinNo "GP" + Str$(i%)), Din, PullUp : Next
   ctrl.open_driver("ctrl.gamemite")
  Case ctrl.CLOSE, ctrl.SOFT_CLOSE
   Local i%
   For i% = 8 To 15 : SetPin MM.Info(PinNo "GP" + Str$(i%)), Off : Next
   ctrl.close_driver("ctrl.gamemite")
 End Select
End Sub

' ---- src/splib/ctrl.inc
' src/splib/sound.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const sound.FX_FLAG% = &b01
Const sound.MUSIC_FLAG% = &b10
Dim sound.F!(127)
Dim sound.FX_BLART%(2)
Dim sound.FX_SELECT%(2)
Dim sound.FX_DIE%(4)
Dim sound.FX_WIPE%(4)
Dim sound.FX_READY_STEADY_GO%(13)
Dim sound.enabled%
Dim sound.fx_int$
Dim sound.fx_ptr%
Dim sound.fx_start_ptr%
Dim sound.music_done_cb$
Dim sound.music_int$
Dim sound.music_ptr%
Dim sound.music_start_ptr%
Dim sound.music_tick% = 200
Dim sound.music_volume% = 15

Sub sound.init(fx_int$, music_int$)
 sound.fx_int$ = Choice(Len(fx_int$), fx_int$, "sound.fx_default_int")
 sound.music_int$ = Choice(Len(music_int$), music_int$, "sound.music_default_int")
 Const offset% = Choice("Game*Mite" = "Game*Mite", 12, 0)
 Local i%
 For i% = 0 To 127
  sound.F!(i%) = Max(440 * 2^((i% - 58 + offset%) / 12.0), 16.35)
 Next
 sound.load_data("sound.blart_data", sound.FX_BLART%())
 sound.load_data("sound.select_data", sound.FX_SELECT%())
 sound.load_data("sound.die_data", sound.FX_DIE%())
 sound.load_data("sound.wipe_data", sound.FX_WIPE%())
 sound.load_data("sound.ready_steady_go_data", sound.FX_READY_STEADY_GO%())
 sound.enable(sound.FX_FLAG% Or sound.MUSIC_FLAG%)
End Sub

Sub sound.enable(flags%)
 Local raw%
 If flags% And sound.MUSIC_FLAG% Then
  If Not (sound.enabled% And sound.MUSIC_FLAG%) Then
   Execute "SetTick " + Str$(sound.music_tick%) + ", " + sound.music_int$ + ", 1"
   sound.enabled% = sound.enabled% Or sound.MUSIC_FLAG%
  EndIf
 Else
  If sound.enabled% And sound.MUSIC_FLAG% Then
   sound.enabled% = sound.enabled% Xor sound.MUSIC_FLAG%
   SetTick 0, 0, 1
   If raw% Then
    PWM 2, sound.F!(0), 0
   Else
    Play Sound 1, B, O
    Play Sound 2, B, O
    Play Sound 3, B, O
   EndIf
   sound.music_ptr% = 0
  EndIf
 EndIf
 If flags% And sound.FX_FLAG% Then
  If Not (sound.enabled% And sound.FX_FLAG%) Then
   Execute "SetTick 40, " + sound.fx_int$ + ", 2"
   sound.enabled% = sound.enabled% Or sound.FX_FLAG%
  EndIf
 Else
  If sound.enabled% And sound.FX_FLAG% Then
   sound.enabled% = sound.enabled% Xor sound.FX_FLAG%
   SetTick 0, 0, 2
   If raw% Then
    PWM 3, sound.F!(0), 0
   Else
    Play Sound 4, B, O
   EndIf
   sound.fx_ptr% = 0
  EndIf
 EndIf
End Sub

Sub sound.term()
 sound.enable(&h00)
 Play Stop
End Sub

Sub sound.load_data(data_label$, notes%())
 Local i%, num_notes%, num_channels%
 Read Save
 Restore data_label$
 Read num_notes%, num_channels%
 Const max_notes% = 8 * (Bound(notes%(), 1) - Bound(notes%(), 0))
 If num_notes% > max_notes% Then
  Local err$ = "Too much DATA; "
  Error err$ + "expected " + Str$(max_notes%) + " bytes but found " + Str$(num_notes%)
 EndIf
 notes%(0) = num_channels% + 256 * num_notes%
 For i% = 1 To num_notes% \ 8 : Read notes%(i%) : Next
 Read Restore
End Sub

Sub sound.music_default_int()
 If Not sound.music_ptr% Then Exit Sub
 Local n% = PEEK(BYTE sound.music_ptr%)
 If n% < 255 Then
  Play Sound 1, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  n% = PEEK(BYTE sound.music_ptr% + 1)
  Play Sound 2, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  n% = PEEK(BYTE sound.music_ptr% + 2)
  Play Sound 3, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  Inc sound.music_ptr%, 3
  Exit Sub
 EndIf
 sound.music_ptr% = 0
 If Len(sound.music_done_cb$) Then Call sound.music_done_cb$
End Sub

Sub sound.play_fx(fx%(), block%)
 If Not (sound.enabled% And sound.FX_FLAG%) Then Exit Sub
 sound.fx_start_ptr% = Peek(VarAddr fx%())
 sound.fx_ptr% = sound.fx_start_ptr% + 8
 If block% Then Do While sound.fx_ptr% > 0 : Loop
End Sub

Sub sound.fx_default_int()
 If Not sound.fx_ptr% Then Exit Sub
 Local n% = PEEK(BYTE sound.fx_ptr%)
 If n% < 255 Then
  Play Sound 4, B, S, sound.F!(n%), (n% > 0) * 25
  Inc sound.fx_ptr%
 Else
  sound.fx_ptr% = 0
 EndIf
End Sub

sound.blart_data:
Data 16
Data 1
Data &hFFFFFF0036373C3D, &hFFFFFFFFFFFFFFFF
sound.select_data:
Data 16
Data 1
Data &hFFFFFFFF0048443C, &hFFFFFFFFFFFFFFFF
sound.die_data:
Data 32
Data 1
Data &h4748494A4B4C4D4E, &h3F40414243444546, &h0038393A3B3C3D3E, &hFFFFFFFFFFFFFFFF
sound.wipe_data:
Data 32
Data 1
Data &h3F3E3D3C3B3A3938, &h4746454443424140, &h004E4D4C4B4A4948, &hFFFFFFFFFFFFFFFF
sound.ready_steady_go_data:
Data 104
Data 1
Data &h3C3C3C3C3C3C3C3C, &h3C3C3C3C3C3C3C3C, &h0000000000000000, &h0000000000000000
Data &h3C3C3C3C3C3C3C3C, &h3C3C3C3C3C3C3C3C, &h0000000000000000, &h0000000000000000
Data &h4848484848484848, &h4848484848484848, &h4848484848484848, &h0000000048484848
Data &hFFFFFFFFFFFFFFFF

Function str.rpad$(s$, x%)
 str.rpad$ = s$
 If Len(s$) < x% Then str.rpad$ = s$ + Space$(x% - Len(s$))
End Function

Function str.decode$(s$)
 Local ad%, ch%, prev%, state%, t$
 For ad% = Peek(VarAddr s$) + 1 To Peek(VarAddr s$) + Len(s$)
  ch% = PEEK(BYTE ad%)
  Select Case ch%
   Case &h5C
    state% = 1
   Case Else
    Select Case state%
     Case 0
      Cat t$, Chr$(ch%)
     Case 1
      Select Case ch%
       Case &h22 : ch% = &h22
       Case &h27 : ch% = &h27
       Case &h30 : ch% = &h00
       Case &h3F : ch% = &h3F
       Case &h5C : ch% = &h5C
       Case &h61 : ch% = &h07
       Case &h62 : ch% = &h08
       Case &h65 : ch% = &h1B
       Case &h66 : ch% = &h0C
       Case &h6E : ch% = &h0A
       Case &h71 : ch% = &h22
       Case &h72 : ch% = &h0D
       Case &h74 : ch% = &h09
       Case &h76 : ch% = &h0B
       Case &h78 : state% = 2
       Case Else : Cat t$, "\"
      End Select
      If state% = 1 Then
       Cat t$, Chr$(ch%)
       state% = 0
      EndIf
     Case 2
      Select Case ch%
       Case &h30 To &h39, &h41 To &h46, &h61 To &h66
        prev% = ch%
        state% = 3
       Case Else
        Cat t$, "\x" + Chr$(ch%)
        state% = 0
      End Select
     Case 3
      Select Case ch%
       Case &h30 To &h39, &h41 To &h46, &h61 To &h66
        Cat t$, Chr$(Val("&h" + Chr$(prev%) + Chr$(ch%)))
       Case Else
        Cat t$, "\x" + Chr$(prev%) + Chr$(ch%)
      End Select
      state% = 0
    End Select
  End Select
 Next
 Select Case state%
  Case 1 : Cat t$, "\"
  Case 2 : Cat t$, "\x"
  Case 3 : Cat t$, "\x" + Chr$(prev%)
 End Select
 str.decode$ = t$
End Function

Function str.wwrap$(s$, p%, len%)
 Const slen% = Len(s$)
 Local ch%, q%, word$
 For q% = p% To slen% + 1
  ch% = Choice(q% > slen%, 0, Peek(Var s$, q%))
  Select Case ch%
   Case 0, &h0A, &h0D, &h20
    If Len(str.wwrap$) + Len(word$) > len% Then
     If Len(word$) > len% Then
      word$ = Left$(word$, len% - Len(str.wwrap$))
      Cat str.wwrap$, word$
      Inc p%, Len(word$)
     EndIf
     Exit For
    EndIf
    Cat str.wwrap$, word$
    p% = q% + 1
    Select Case ch%
     Case &h0D
      If Choice(p% > slen%, 0, Peek(Var s$, p%)) = &h0A Then Inc p%
      Exit For
     Case &h20
      If Len(str.wwrap$) = len% Then Exit For
      Cat str.wwrap$, " "
      word$ = ""
     Case Else
      Exit For
    End Select
   Case Else
    Cat word$, Chr$(ch%)
  End Select
 Next
 p% = Min(p%, slen% + 1)
End Function

' ---- src/splib/string.inc
' src/splib/file.inc ++++
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const file.SEPARATOR = "/"

Function file.exists%(f$, type$)
 Local f_$ = Choice(Len(f$), f$, "."), i%
 Local type_$ = "|" + Choice(Len(type$), LCase$(type$), "all") + "|"
 If f_$ = "."  Then f_$ = Cwd$
 If Len(f_$) = 2 Then
  If Mid$(f_$, 2, 1) = ":" Then Cat f_$, "/"
 EndIf
 For i% = 1 To Len(f_$)
  If Peek(Var f_$, i%) = 92 Then Poke Var f_$, i%, 47
 Next
 If Instr("|file|all|", type_$) Then Inc file.exists%, MM.Info(Exists File f_$)
 If Instr("|dir|all|", type_$) Then Inc file.exists%, MM.Info(Exists Dir f_$)
 file.exists% = file.exists% > 0
End Function

Function file.fnmatch%(pattern$, name$)
 Local p$ = UCase$(pattern$)
 Local n$ = UCase$(name$)
 Local c%, px% = 1, nx% = 1, nextPx% = 0, nextNx% = 0
 Do While px% <= Len(p$) Or nx% <= Len(n$)
  If px% <= Len(p$) Then
   c% = Peek(Var p$, px%)
   Select Case c%
    Case 42
     nextPx% = px%
     nextNx% = nx% + 1
     Inc px%
     GoTo file.fnmatch_cont
    Case 63
     If nx% <= Len(n$) Then
      Inc px%
      Inc nx%
      GoTo file.fnmatch_cont
     EndIf
    Case Else
     If nx% <= Len(n$) Then
      If c% = Peek(Var n$, nx%) Then
       Inc px%
       Inc nx%
       GoTo file.fnmatch_cont
      EndIf
     EndIf
   End Select
  EndIf
  If nextNx% > 0 Then
   If nextNx% <= Len(n$) + 1 Then
    px% = nextPx%
    nx% = nextNx%
    GoTo file.fnmatch_cont
   EndIf
  EndIf
  Exit Function
file.fnmatch_cont:
 Loop
 file.fnmatch% = 1
End Function

Function file.get_extension$(f$)
 Local i%
 For i% = Len(f$) To 1 Step -1
  Select Case Peek(Var f$, i%)
   Case 46
    file.get_extension$ = Mid$(f$, i%)
    Exit Function
   Case 47, 92
    Exit For
  End Select
 Next
End Function

Function file.get_files%(d$, pattern$, type$, hlout$())
 If Not file.is_directory%(d$) Then
  file.get_files% = sys.error%(sys.FAILURE, "Not a directory '" + d$ + "'")
  Exit Function
 EndIf
 Local f$
 Select Case LCase$(type$)
  Case "all"      : f$ = Dir$(d$ + "/*", All)
  Case "dir"      : f$ = Dir$(d$ + "/*", Dir)
  Case "file", "" : f$ = Dir$(d$ + "/*", File)
  Case Else
   file.get_files% = sys.error%(sys.FAILURE, "Invalid file type '" + type$ + "'")
   Exit Function
 End Select
 Local i% = MM.Info(Option Base)
 Do While f$ <> ""
  If file.fnmatch%(pattern$, f$) Then
   If i% <= Bound(hlout$(), 1) Then hlout$(i%) = f$:Inc i%
   Inc file.get_files%
  EndIf
  f$ = Dir$()
 Loop
 If file.get_files% > 0 Then
  Sort hlout$(), , &b10, MM.Info(Option Base), i% - MM.Info(Option Base)
 EndIf
End Function

Function file.get_name$(f$)
 Local i%
 For i% = Len(f$) To 1 Step -1
  If Instr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
 Next
 file.get_name$ = Mid$(f$, i% + 1)
End Function

Function file.get_parent$(f$)
 Select Case Len(f$)
  Case 1
   If f$ = "/" Or f$ = "\" Then Exit Function
  Case 2
   If Mid$(f$, 2, 1) = ":" Then Exit Function
  Case 3
   Select Case Right$(f$, 2)
    Case ":/", ":\"
     Exit Function
   End Select
 End Select
 Local i%
 For i% = Len(f$) To 1 Step -1
  If Instr("/\", Mid$(f$, i%, 1)) > 0 Then Exit For
 Next
 If i% > 0 Then file.get_parent$ = Left$(f$, i% - 1)
 If file.get_parent$ = "" Then
  If Instr("/\", Left$(f$, 1)) Then file.get_parent$ = Left$(f$, 1)
 EndIf
End Function

Function file.is_directory%(f$)
 file.is_directory% = file.exists%(f$, "dir")
End Function

' ---- src/splib/file.inc
' src/splib/txtwm.inc ++++
' Copyright (c) 2021-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
If MM.Info(Option Base) <> 0 Then Error "expects OPTION BASE 0"
Const twm.BLACK%   = 0
Const twm.RED%     = 1
Const twm.GREEN%   = 2
Const twm.YELLOW%  = 3
Const twm.BLUE%    = 4
Const twm.MAGENTA% = 5
Const twm.CYAN%    = 6
Const twm.WHITE%   = 7
Dim twm.max_num%
Dim twm.num%
Dim twm.fw%
Dim twm.fh%
Dim twm.last_at%
Dim twm.cursor_enabled%
Dim twm.cursor_locked%
Dim twm.id%
Dim twm.a%
Dim twm.b%
Dim twm.w%
Dim twm.h%
Dim twm.x%
Dim twm.y%
Dim twm.at%
Dim twm.pc%
Dim twm.pa%

Sub twm.init(max_num%, mem_sz%)
 If twm.max_num% > 0 Then Error "'txtwm' already initialised"
 If max_num% < 1 Or max_num% > 10 Then Error "Invalid 'max_num'; must be between 1 and 10: " + Str$(max_num%)
 If mem_sz% < 100 Then Error "Invalid 'mem_sz'; must be >= 100: " + Str$(mem_sz%)
 twm.max_num% = max_num%
 twm.id%      = -1
 twm.last_at% = -1
 twm.fw%      = MM.Info(FontWidth)
 twm.fh%      = MM.Info(FontHeight)
 Dim twm.data%((mem_sz% \ 8) + 1)
 Dim twm.ptr%(Choice(max_num% = 1, 1, max_num% - 1))
 twm.init_serial_attrs()
 twm.init_screen_attrs()
 twm.init_box_chars()
 twm.enable_cursor(0)
End Sub

Sub twm.init_serial_attrs()
 Local i%, vt$
 Dim twm.vt$(255) Length 20
 For i% = 0 To 255
  vt$ = Chr$(27) + "[0m"
  Cat vt$, Chr$(27) + "[" + Choice(i% And &b01000000, "1;3", "3") + Str$(i% And &b00000111) + "m"
  Cat vt$, Chr$(27) + "[4" + Str$((i% And &b00111000) >> 3) + "m"
  If i% And &b10000000 Then Cat vt$, Chr$(27) + "[7m"
  twm.vt$(i%) = vt$
 Next
End Sub

Sub twm.init_screen_attrs()
 Local bg%, fg%, i%
 Dim twm.fg%(255)
 Dim twm.bg%(255)
 For i% = 0 To 255
  Select Case i% And &b00000111
   Case twm.BLACK%   : fg% = RGB(Black)
   Case twm.RED%     : fg% = RGB(Red)
   Case twm.GREEN%   : fg% = RGB(Green)
   Case twm.YELLOW%  : fg% = RGB(Yellow)
   Case twm.BLUE%    : fg% = RGB(Blue)
   Case twm.MAGENTA% : fg% = RGB(Magenta)
   Case twm.CYAN%    : fg% = RGB(Cyan)
   Case twm.WHITE%   : fg% = RGB(White)
   Case Else Error
  End Select
  If i% And &b01000000 Then
   If fg% <> 0 Then fg% = fg% Or &h404040
  EndIf
  Select Case (i% And &b00111000) >> 3
   Case twm.BLACK%   : bg% = RGB(Black)
   Case twm.RED%     : bg% = RGB(Red)
   Case twm.GREEN%   : bg% = RGB(Green)
   Case twm.YELLOW%  : bg% = RGB(Yellow)
   Case twm.BLUE%    : bg% = RGB(Blue)
   Case twm.MAGENTA% : bg% = RGB(Magenta)
   Case twm.CYAN%    : bg% = RGB(Cyan)
   Case twm.WHITE%   : bg% = RGB(White)
   Case Else Error
  End Select
  If i% And &b10000000 Then
   twm.bg%(i%) = fg%
   twm.fg%(i%) = bg%
  Else
   twm.bg%(i%) = bg%
   twm.fg%(i%) = fg%
  EndIf
 Next
End Sub

Sub twm.init_box_chars()
 Dim twm.c2b%(255)
 twm.c2b%(&hB9) = &b1101
 twm.c2b%(&hBA) = &b0101
 twm.c2b%(&hBB) = &b1100
 twm.c2b%(&hBC) = &b1001
 twm.c2b%(&hC8) = &b0011
 twm.c2b%(&hC9) = &b0110
 twm.c2b%(&hCA) = &b1011
 twm.c2b%(&hCB) = &b1110
 twm.c2b%(&hCC) = &b0111
 twm.c2b%(&hCD) = &b1010
 twm.c2b%(&hCE) = &b1111
 Dim twm.b2c%(15)
 twm.b2c%(&b1101) = &hB9
 twm.b2c%(&b0101) = &hBA
 twm.b2c%(&b1100) = &hBB
 twm.b2c%(&b1001) = &hBC
 twm.b2c%(&b0011) = &hC8
 twm.b2c%(&b0110) = &hC9
 twm.b2c%(&b1011) = &hCA
 twm.b2c%(&b1110) = &hCB
 twm.b2c%(&b0111) = &hCC
 twm.b2c%(&b1010) = &hCD
 twm.b2c%(&b1111) = &hCE
End Sub

Function twm.new_win%(x%, y%, w%, h%)
 If twm.num% > Bound(twm.ptr%(), 1) Then Error "maximum number of windows reached: " + Str$(twm.num%)
 Local ptr%
 If twm.num% = 0 Then
  ptr% = Peek(VarAddr twm.data%())
 Else
  ptr% = twm.ptr%(twm.num% - 1)
  Inc ptr%, 7 + PEEK(BYTE ptr% + 2) * PEEK(BYTE ptr% + 3) * 2
 EndIf
 twm.ptr%(twm.num%) = ptr%
 Const reqd% = ptr% + 7 + w% * h% * 2 - twm.ptr%(0)
 Const alloc% = (Bound(twm.data%(), 1) - Bound(twm.data%(), 0)) * 8
 If reqd% > alloc% Then
  Error "out of txtwm memory: " + Str$(alloc%) + " bytes allocated, " + Str$(reqd%) + " required"
 EndIf
 Poke Byte ptr% + 0, x% + 1
 Poke Byte ptr% + 1, y% + 1
 Poke Byte ptr% + 2, w%
 Poke Byte ptr% + 3, h%
 Poke Byte ptr% + 4, 0
 Poke Byte ptr% + 5, 0
 Poke Byte ptr% + 6, twm.WHITE%
 Memory Set ptr% + 7, 32, w% * h%
 Memory Set ptr% + 7 + w% * h%, twm.WHITE%, w% * h%
 twm.new_win% = twm.num%
 Inc twm.num%
End Function

Sub twm.switch(id%)
 twm.last_at% = -1
 If twm.id% = id% Then Exit Sub
 twm.lock_vga_cursor(1)
 Local ptr%
 If twm.id% > -1 Then
  ptr% = twm.ptr%(twm.id%)
  Poke Byte ptr% + 4, twm.x%
  Poke Byte ptr% + 5, twm.y%
  Poke Byte ptr% + 6, twm.at%
 EndIf
 twm.id% = id%
 ptr%    = twm.ptr%(twm.id%)
 twm.a%  = PEEK(BYTE ptr% + 0)
 twm.b%  = PEEK(BYTE ptr% + 1)
 twm.w%  = PEEK(BYTE ptr% + 2)
 twm.h%  = PEEK(BYTE ptr% + 3)
 twm.x%  = PEEK(BYTE ptr% + 4)
 twm.y%  = PEEK(BYTE ptr% + 5)
 twm.at% = PEEK(BYTE ptr% + 6)
 twm.pc% = ptr% + 7
 twm.pa% = twm.pc% + twm.w% * twm.h%
 twm.lock_vga_cursor(0)
End Sub

Sub twm.foreground(col%)
 twm.at% = (twm.at% And &b11111000) Or col%
End Sub

Sub twm.inverse(z%)
 twm.at% = (twm.at% And &b01111111) Or (z% << 7)
End Sub

Sub twm.print_at(x%, y%, s$)
 twm.lock_vga_cursor(1)
 twm.x% = x%
 twm.y% = y%
 twm.print(s$)
 twm.lock_vga_cursor(0)
End Sub

Sub twm.putc(ch%)
 Local at% = twm.at%
 Local s$ = Chr$(ch%)
 Local of% = twm.y% * twm.w% + twm.x%
 Local ax% = twm.a% + twm.x%
 Local by% = twm.b% + twm.y%
 Poke Byte twm.pc% + of%, ch%
 Poke Byte twm.pa% + of%, at%
 If twm.last_at% <> at% Then Print vt$; : twm.last_at% = at%
 Print Chr$(27) "[" Str$(by%) ";" Str$(ax%) "H" s$;
 Text (ax%-1)*twm.fw%,(by%-1)*twm.fh%,s$,,,,twm.fg%(at%),twm.bg%(at%)
 If at% And &b01000000 Then Text (ax%-1)*twm.fw%+1,(by%-1)*twm.fh%,s$,,,,twm.fg%(at%),-1
End Sub

Sub twm.print(s$)
 Local is% = 1
 Local ls% = Len(s$)
 Local ps% = Peek(VarAddr s$)
 Local nc% = Min(twm.w% - twm.x%, ls%)
 Local of%
 Local at% = twm.at%
 Local fg% = twm.fg%(at%)
 Local bg% = twm.bg%(at%)
 Local ax%
 Local by%
 Local seg$
 Local vt$ = twm.vt$(at%)
 If nc% = 0 Then Exit Sub
 If twm.last_at% <> at% Then Print vt$; : twm.last_at% = at%
 twm.lock_vga_cursor(1)
 Do
  If twm.y% = twm.h% Then
   twm.scroll_up(1)
   Inc twm.y%, -1
   If twm.last_at% <> at% Then Print vt$ : twm.last_at% = at%
  EndIf
  of% = twm.y% * twm.w% + twm.x%
  Memory Copy ps% + is%, twm.pc% + of%, nc%
  Memory Set twm.pa% + of%, at%, nc%
  seg$ = Mid$(s$, is%, nc%)
  ax% = twm.a% + twm.x%
  by% = twm.b% + twm.y%
  Print Chr$(27) "[" Str$(by%) ";" Str$(ax%) "H" seg$;
  Text (ax% - 1) * twm.fw%, (by% - 1) * twm.fh%, seg$,,,, fg%, bg%
  If at% And &b01000000 Then Text (ax% - 1) * twm.fw% + 1, (by% - 1) * twm.fh%, seg$,,,, fg%, -1
  Inc is%, nc%
  Inc twm.x%, nc%
  If twm.x% = twm.w% Then
   twm.x% = 0
   Inc twm.y%
   nc% = Min(twm.w% - twm.x%, ls% - is% + 1)
  EndIf
 Loop While is% <= ls%
 twm.lock_vga_cursor(0)
End Sub

Sub twm.scroll_up(redraw%)
 twm.lock_vga_cursor(1)
 Local pa% = twm.pa%
 Local pc% = twm.pc%
 Local y%
 For y% = 1 To twm.h% - 1
  Memory Copy pa% + twm.w%, pa%, twm.w%
  Memory Copy pc% + twm.w%, pc%, twm.w%
  Inc pa%, twm.w%
  Inc pc%, twm.w%
 Next
 Memory Set pa%, twm.at%, twm.w%
 Memory Set pc%, 32, twm.w%
 If redraw% Then twm.redraw()
 twm.lock_vga_cursor(0)
End Sub

Sub twm.redraw()
 Local at%, ch$, x%, y%
 Local pa% = twm.pa%
 Local pc% = twm.pc%
 Local vx%, vy%
 vy% = (twm.b% - 1) * twm.fh%
 For y% = 0 To twm.h% - 1
  Print Chr$(27) "[" Str$(twm.b% + y%) ";" Str$(twm.a%) "H";
  vx% = (twm.a% - 1) * twm.fw%
  For x% = 0 To twm.w% - 1
   at% = PEEK(BYTE pa% + x%)
   ch$ = Chr$(PEEK(BYTE pc% + x%))
   If twm.last_at% <> at% Then Print twm.vt$(at%); : twm.last_at% = at%
   Print ch$;
   Text vx%, vy%, ch$,,,, twm.fg%(at%), twm.bg%(at%)
   If at% And &b01000000 Then Text vx%+1, vy%, ch$,,,, twm.fg%(at%), -1
   Inc vx%, twm.fw%
  Next
  Inc pa%, twm.w%
  Inc pc%, twm.w%
  Inc vy%, twm.fh%
 Next
End Sub

Sub twm.cls(x%, y%, w%, h%)
 twm.lock_vga_cursor(1)
 Local s$ = Space$(Choice(w%, w%, twm.w%)), yy%
 For yy% = y% To y% + Choice(h%, h%, twm.h%) - 1
  twm.x% = x% : twm.y% = yy%
  twm.print(s$)
 Next
 twm.lock_vga_cursor(0)
End Sub

Sub twm.box(x%, y%, w%, h%)
 Local ad%, i%, pc% = twm.pc%, s$
 twm.lock_vga_cursor(1)
 Poke Var s$, 0, w%
 ad% = pc% + twm.w% * y% + x%
 Poke Var s$, 1, twm.box_or%(&hC9, PEEK(BYTE ad%))
 For i% = 2 To w% - 1
  Poke Var s$, i%, twm.box_or%(&hCD, PEEK(BYTE ad% + i% - 1))
 Next
 Poke Var s$, w%, twm.box_or%(&hBB, PEEK(BYTE ad% + w% - 1))
 twm.print_at(x%, y%, s$)
 For twm.y% = y% + 1 To y% + h% - 2
  Inc ad%, twm.w%
  twm.x% = x%
  twm.putc(twm.box_or%(&hBA, PEEK(BYTE ad%)))
  twm.x% = x% +  w% - 1
  twm.putc(twm.box_or%(&hBA, PEEK(BYTE ad% + w% - 1)))
 Next
 Inc ad%, twm.w%
 Poke Var s$, 1, twm.box_or%(&hC8, PEEK(BYTE ad%))
 For i% = 2 To w% - 1
  Poke Var s$, i%, twm.box_or%(&hCD, PEEK(BYTE ad% + i% - 1))
 Next
 Poke Var s$, w%, twm.box_or%(&hBC, PEEK(BYTE ad% + w% - 1))
 twm.print_at(x%, y% + h% - 1, s$)
 twm.lock_vga_cursor(0)
End Sub

Sub twm.box1(x%, y%, w%, h%)
 Local ad%, i%, pc% = twm.pc%, s$
 twm.lock_vga_cursor(1)
 Poke Var s$, 0, w%
 ad% = pc% + twm.w% * y% + x%
 Poke Var s$, 1, twm.box_or%(&hDA, PEEK(BYTE ad%))
 For i% = 2 To w% - 1
  Poke Var s$, i%, twm.box_or%(&hC4, PEEK(BYTE ad% + i% - 1))
 Next
 Poke Var s$, w%, twm.box_or%(&hBF, PEEK(BYTE ad% + w% - 1))
 twm.print_at(x%, y%, s$)
 For twm.y% = y% + 1 To y% + h% - 2
  Inc ad%, twm.w%
  twm.x% = x%
  twm.putc(twm.box_or%(&hB3, PEEK(BYTE ad%)))
  twm.x% = x% +  w% - 1
  twm.putc(twm.box_or%(&hB3, PEEK(BYTE ad% + w% - 1)))
 Next
 Inc ad%, twm.w%
 Poke Var s$, 1, twm.box_or%(&hC0, PEEK(BYTE ad%))
 For i% = 2 To w% - 1
  Poke Var s$, i%, twm.box_or%(&hC4, PEEK(BYTE ad% + i% - 1))
 Next
 Poke Var s$, w%, twm.box_or%(&hD9, PEEK(BYTE ad% + w% - 1))
 twm.print_at(x%, y% + h% - 1, s$)
 twm.lock_vga_cursor(0)
End Sub

Function twm.box_or%(ch%, ex%)
 Local tmp% = twm.c2b%(ch%)
 twm.box_or% = Choice(tmp% = 0, ch%, twm.b2c%(tmp% Or twm.c2b%(ex%)))
End Function

Sub twm.enable_cursor(z%)
 Print Chr$(27) Choice(z%, "[?25h", "[?25l");
 If z% = twm.cursor_enabled% Then Exit Sub
 twm.cursor_enabled% = z%
 If z% Then
  SetTick 500, twm.update_vga_cursor, 4
 Else
  SetTick 0, twm.update_vga_cursor, 4
 EndIf
End Sub

Sub twm.update_vga_cursor()
 Static lit% = 0
 If twm.cursor_locked% Then Exit Sub
 If twm.x% >= twm.w% Or twm.y% >= twm.h% Then Exit Sub
 lit% = Not lit%
 Local of% = twm.y% * twm.w% + twm.x%
 Local ax% = twm.a% + twm.x%
 Local by% = twm.b% + twm.y%
 Local fg% = Choice(lit%, RGB(Black), RGB(White))
 Text (ax%-1)*twm.fw%,(by%-1)*twm.fh%,"_",,,,fg%,-1
 Text (ax%-1)*twm.fw%,(by%-1)*twm.fh% + 1,"_",,,,fg%,-1
End Sub

Sub twm.lock_vga_cursor(lock%)
 Inc twm.cursor_locked%, Choice(lock%, 1, -1))
 If twm.id% = -1 Then Exit Sub
 If twm.cursor_locked% < 0 Then Error "Unbalanced cursor lock/unlock."
 If Not lock% Then Exit Sub
 If twm.cursor_locked% > 1 Then Exit Sub
 If twm.x% >= twm.w% Or twm.y% >= twm.h% Then Exit Sub
 Local ax% = twm.a% + twm.x%
 Local by% = twm.b% + twm.y%
 Local of% = twm.y% * twm.w% + twm.x%
 Local ch% = PEEK(BYTE twm.pc% + of%)
 Local at% = PEEK(BYTE twm.pa% + of%)
 Local fg% = twm.fg%(at%)
 Text (ax%-1)*twm.fw%,(by%-1)*twm.fh%,Chr$(ch%),,,,fg%,twm.bg%(at%)
 If at% And &b01000000 Then Text (ax%-1)*twm.fw%+1,(by%-1)*twm.fh%,Chr$(ch%),,,,fg%,-1
End Sub

' ---- src/splib/txtwm.inc
' src/splib/menu.inc ++++
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Sub menu.init(ctrl$, callback$)
 Dim menu.OK_BTN$(1) Length 2 = ("OK", "")
 Dim YES_NO_BTNS$(1) Length 3 = ("Yes", "No")
 Dim menu.callback$ = callback$, menu.ctrl$ = ctrl$
 Dim menu.item_count%, menu.selection%
 Font 1
 Dim menu.height% = MM.VRES \ MM.Info(FontHeight)
 Dim menu.width% = MM.HRES \ MM.Info(FontWidth)
 Local sz%
 Inc sz%, 7 + 2 * menu.height% * menu.width%
 Inc sz%, 7 + 2 * (menu.height% - 10) * (menu.width% - 10)
 twm.init(2, sz%)
 Dim menu.win1% = twm.new_win%(0, 0, menu.width%, menu.height%)
 Dim menu.win2% = twm.new_win%(5, 5, menu.width% - 10, menu.height% - 10)
End Sub

Sub menu.term(msg$)
 sys.restore_break()
 twm.enable_cursor(1)
 FRAMEBUFFER Write N
 CLS
 If Len(msg$) Then Text 160, 110, msg$, CM
End Sub

Sub menu.on_break()
 menu.term("Exited due to Ctrl-C")
 End
End Sub

Sub menu.render(draw_box%)
 twm.switch(menu.win1%)
 twm.cls(1, 1, menu.width% - 2, menu.height% - 2)
 twm.foreground(twm.WHITE%)
 If draw_box% Then twm.box(0, 0, menu.width%, menu.height%)
 Local i%
 For i% = 0 To menu.item_count% - 1
  menu.render_item(i%)
 Next
 If Len(menu.callback$) Then Call menu.callback$, "render"
 FRAMEBUFFER Copy F , N
End Sub

Sub menu.render_item(idx%)
 twm.foreground(Choice(idx% = 0, twm.YELLOW%, twm.WHITE%))
 twm.inverse(idx% = menu.selection%)
 Local item$ = Left$(menu.items$(idx%), Instr(menu.items$(idx%), "|") - 1)
 Local x% = (menu.width% - Len(item$)) \ 2
 twm.print_at(x%, idx% + 2, item$)
 twm.inverse(0)
End Sub

Sub menu.main_loop()
 Local key%
 Do
  If sys.break_flag% Then menu.on_break()
  Call menu.ctrl$, key%
  If Not key% Then keys_cursor(key%)
  menu.process_key(key%)
 Loop
End Sub

Sub menu.process_key(key%)
 Local cmd$, new_sel%
 If Not key% Then Exit Sub
 Select Case key%
  Case ctrl.A, ctrl.B, ctrl.START, ctrl.SELECT, ctrl.LEFT, ctrl.RIGHT
   cmd$ = Field$(menu.items$(menu.selection%), 2, "|")
   Call cmd$, key%
  Case ctrl.DOWN, ctrl.UP
   new_sel% = menu.selection%
   Do
    Inc new_sel%, Choice(key% = ctrl.DOWN, 1, -1)
    If new_sel% < 0 Or new_sel% >= menu.item_count% Then
     new_sel% = menu.selection%
     menu.play_invalid_fx()
     Pause ctrl.UI_DELAY
     Exit Do
    EndIf
   Loop Until Len(Field$(menu.items$(new_sel%), 2, "|"))
   menu.select_item(new_sel%)
  Case Else
   menu.play_invalid_fx()
   Pause ctrl.UI_DELAY
 End Select
End Sub

Sub menu.select_item(new_sel%)
 If new_sel% = menu.selection% Then Exit Sub
 menu.play_valid_fx()
 Pause ctrl.UI_DELAY
 Local old_sel% = menu.selection%
 menu.selection% = new_sel%
 FRAMEBUFFER Write F
 menu.render_item(menu.selection%)
 menu.render_item(old_sel%)
 If Len(menu.callback$) Then Call menu.callback$, "selection_changed"
 FRAMEBUFFER Copy F , N
End Sub

Sub menu.play_valid_fx(block%)
 sound.play_fx(sound.FX_SELECT%(), block%)
End Sub

Sub menu.play_invalid_fx(block%)
 If MM.DEVICE$ = "MMB4L" Then
  Console Bell
 Else
  sound.play_fx(sound.FX_BLART%(), block%)
 EndIf
End Sub

Function menu.msgbox%(msg$, buttons$(), default%, frameCol%)
 menu.msgbox% = default%
 Local base% = MM.Info(Option Base), num% = Bound(buttons$(), 1) - base% + 1
 If buttons$(base% + 1) = "" Then num% = 1
 Local i%, key%, p%, released%, valid% = 1, word$, x%(num%)
 x%(base%) = 2
 For i% = base% + 1 To base% + num% - 1
  x%(i%) = x%(i% - 1) + Len(buttons$(i% - 1)) + 5
 Next
 twm.switch(menu.win2%)
 twm.cls()
 twm.foreground(Choice(frameCol%, frameCol%, twm.CYAN%))
 twm.box(0, 0, twm.w%, twm.h%)
 twm.foreground(twm.WHITE%)
 i% = 1 : p% = 1
 Do While p% <= Len(msg$)
  twm.print_at(2, i%, str.wwrap$(msg$, p%, twm.w% - 4))
  Inc i%
 Loop
 Do
  If sys.break_flag% Then menu.on_break()
  If valid% Then
   For i% = base% To base% + num% - 1
    menu.button(x%(i%), twm.h% - 4, buttons$(i%), menu.msgbox% = i%)
   Next
   FRAMEBUFFER Copy F , N
   valid% = 0
  EndIf
  Call menu.ctrl$, key%
  If Not key% Then keys_cursor(key%)
  If Not key% Then released% = 1 : Continue Do
  If Not released% Then key% = 0 : Continue Do
  valid% = 0
  Select Case key%
   Case ctrl.A, ctrl.SELECT
    key% = ctrl.SELECT
    valid% = 1
   Case ctrl.LEFT
    If menu.msgbox% = 1 Then menu.msgbox% = 0 : valid% = 1
   Case ctrl.RIGHT
    If menu.msgbox% = 0 Then menu.msgbox% = 1 : valid% = 1
  End Select
  If valid% Then menu.play_valid_fx() Else menu.play_invalid_fx()
  Pause ctrl.UI_DELAY
 Loop Until key% = ctrl.SELECT
End Function

Sub menu.button(x%, y%, txt$, selected%)
 twm.lock_vga_cursor(1)
 twm.box1(x%, y%, Len(txt$) + 4, 3)
 If selected% Then twm.inverse(1)
 twm.print_at(x% + 2, y% + 1, txt$)
 If selected% Then twm.inverse(0)
 twm.lock_vga_cursor(0)
End Sub

' ---- src/splib/menu.inc
' src/splib/gamemite.inc ++++
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Function gamemite.file$(f$)
 If Instr("A:/B:/", UCase$(Left$(f$, 3))) Then
  gamemite.file$ = f$
 Else
  Local f_$ = "A:/GameMite" + Choice(f$ = "", "", "/" + f$), x%
  x% = MM.Info(Exists File f_$)
  If Not x% Then
   f_$ = "B" + Mid$(f_$, 2)
   On Error Skip
   x% = MM.Info(Exists File f_$)
  EndIf
  If Not x% Then f_$ = "A" + Mid$(f_$, 2)
  gamemite.file$ = f_$
 EndIf
End Function

Sub gamemite.end(break%)
 FRAMEBUFFER Write N
 Colour RGB(White), RGB(Black)
 CLS
 sys.restore_break()
 On Error Skip : sound.term()
 On Error Skip : ctrl.term()
 On Error Skip
 twm.enable_cursor(1)
 If break% Then
  Const f$ = "", msg$ = "Exited due to Ctrl-C"
 Else
  Const f$ = gamemite.file$("menu.bas")
  Const x% = MM.Info(Exists File f$)
  Const msg$ = Choice(x%, "Loading menu ...", "Menu program not found!")
 EndIf
 Text 160, 110, msg$, CM
 If Len(f$) Then Run f$ Else End
End Sub

' ---- src/splib/gamemite.inc
Const MAX_FILES = 500
Const FILES_PER_PAGE = 12
If "Game*Mite" = "Game*Mite" Then
 Const num_drives% = 2
 Dim drives$(num_drives% - 1) = ("A:/", "B:/")
Else
 Const num_drives% = 1
 Dim drives$(1) = ("/", "/")
EndIf
Dim drive_idx% = 0
Dim file_list$(MAX_FILES + 1) Length 64
Dim menu.items$(15) Length 127
Dim num_files%
Dim cur_page%
Dim num_pages%
FRAMEBUFFER Create
FRAMEBUFFER Write F
main()
Error "Invalid state"

Sub main()
 Const ctrl$ = Choice("Game*Mite" = "Game*Mite", "ctrl.gamemite", "keys_cursor_ext")
 ctrl.init_keys()
 sys.override_break()
 Call ctrl$, ctrl.OPEN
 sound.init()
 menu.init(ctrl$, "menu_cb")
 update_files()
 update_menu_data()
 menu.render(1)
 menu.main_loop()
End Sub

Sub update_files()
 On Error Skip
 Local s$ = Dir$(drives$(drive_idx%))
 If MM.Errno Then num_files% = -1 : Exit Sub
 num_files% = file.get_files%(drives$(drive_idx%), "*", "all", file_list$())
 If (num_files% > MAX_FILES) Then
  file_list$(MAX_FILES) = "... and " + Str$(num_files% - MAX_FILES) + " more"
  num_files% = MAX_FILES + 1
 EndIf
 If Len(file.get_parent$(drives$(drive_idx%))) Then
  Const p% = Peek(VarAddr file_list$())
  Memory Copy p%, p% + 65, 65 * (MAX_FILES + 1)
  file_list$(0) = ".."
  Inc num_files%
 EndIf
 num_pages% = num_files% \ FILES_PER_PAGE + ((num_files% Mod FILES_PER_PAGE) > 0)
 cur_page% = 1
End Sub

Sub update_menu_data()
 Local i%, j%, s$
 s$ = drives$(drive_idx%) + Choice(Right$(drives$(drive_idx%), 1) = "/", "", "/")
 If Len(s$) > menu.width% - 10 Then
  s$ = Left$(s$, 3) + "..." + Right$(s$, menu.width% - 16)
 EndIf
 If num_drives% > 1 Then
  menu.items$(i%) = str.decode$(" \x95 " + s$ + " \x94 ") + "|cmd_drive"
 Else
  menu.items$(i%) = " " + s$ + " |"
 EndIf
 Inc i%
 menu.items$(i%) = "|"
 Inc i%
 If num_files% = -1 Then
  menu.items$(i%) = "Drive not found|"
  Inc i%
 Else
  Const begin% = (cur_page% - 1) * FILES_PER_PAGE
  For j% = begin% To begin% + FILES_PER_PAGE - 1
   If j% >= num_files% Then Exit For
   s$ = file_list$(j%)
   If file.is_directory%(drives$(drive_idx%) + "/" + file_list$(j%)) Then Cat s$, "/"
   If Len(s$) > menu.width% - 10 Then
    s$ = Left$(s$, 3) + "..." + Right$(s$, menu.width% - 16)
   EndIf
   menu.items$(i%) = " " + str.rpad$(s$, 26) + " |cmd_open|" + Str$(j%)
   If i% = Bound(menu.items$(), 1) - 1 Then Exit For
   Inc i%
  Next
 EndIf
 For i% = i% To Bound(menu.items$(), 1) - 1 : menu.items$(i%) = "|" : Next
 If "Game*Mite" = "Game*Mite" Then
  menu.items$(i%) = str.decode$("Use \x95 \x94 \x92 \x93 and SELECT|")
 Else
  menu.items$(i%) = str.decode$("Use \x95 \x94 \x92 \x93 and SPACE to select|")
 EndIf
 menu.item_count% = i% + 1
 If Not Len(Field$(menu.items$(menu.selection%), 2, "|")) Then
  For menu.selection% = 0 To menu.item_count% - 1
   If Len(Field$(menu.items$(menu.selection%), 2, "|")) Then Exit For
  Next
 EndIf
End Sub

Sub menu_cb(cb_data$)
 Select Case Field$(cb_data$, 1, "|")
  Case "selection_changed"
  Case "render"
   on_render(cb_data$)
  Case Else
   Error "Invalid state"
 End Select
End Sub

Sub on_render(cb_data$)
 Const s$ = "Page " + Str$(cur_page%) + "/" + Str$(num_pages%)
 twm.print_at(menu.width% - Len(s$) - 2, menu.height% - 2, s$)
End Sub

Sub cmd_drive(key%)
 Local update% = 0
 Select Case key%
  Case ctrl.B
   If Len(file.get_parent$(drives$(drive_idx%))) Then
    update% = 1
    drives$(drive_idx%) = file.get_parent$(drives$(drive_idx%))
   EndIf
  Case ctrl.LEFT, ctrl.RIGHT
   update% = 1
   Inc drive_idx%, Choice(key% = ctrl.RIGHT, 1, -1)
   Select Case drive_idx%
    Case < 0 : drive_idx% = num_drives% - 1
    Case >= num_drives% : drive_idx% = 0
   End Select
  Case ctrl.START
   on_start()
   Exit Sub
 End Select
 If update% Then
  menu.play_valid_fx(1)
  update_files()
  update_menu_data()
  menu.render()
 Else
  menu.play_invalid_fx(1)
 EndIf
End Sub

Sub cmd_open(key%)
 Select Case key%
  Case ctrl.A, ctrl.SELECT
   Local f$ = Field$(menu.items$(menu.selection%), 1, "|")
   Local file_idx% = Val(Field$(menu.items$(menu.selection%), 3, "|"))
   If Right$(f$, 1) = "/" Then
    menu.play_valid_fx(1)
    If f$ = "../" Then
     drives$(drive_idx%) = file.get_parent$(drives$(drive_idx%))
    Else
     If Right$(drives$(drive_idx%), 1) <> "/" Then Cat drives$(drive_idx%), "/"
     Cat drives$(drive_idx%), file_list$(file_idx%)
    EndIf
    update_files()
    update_menu_data()
    menu.render()
   Else
    Select Case LCase$(file.get_extension$(file_list$(file_idx%)))
     Case ".bas"
      menu.play_valid_fx(1)
      open_bas(file_idx%)
     Case ".flac", ".mod", ".wav"
      menu.play_valid_fx(1)
      play_music(file_idx%)
     Case Else
      menu.play_invalid_fx(1)
    End Select
   EndIf
  Case ctrl.B
   If Len(file.get_parent$(drives$(drive_idx%))) Then
    menu.play_valid_fx(1)
    drives$(drive_idx%) = file.get_parent$(drives$(drive_idx%))
    update_files()
    update_menu_data()
    menu.render()
   Else
    menu.play_invalid_fx(1)
   EndIf
  Case ctrl.LEFT, ctrl.RIGHT
   If num_pages% = 1 Then
    menu.play_invalid_fx(1)
   Else
    menu.play_valid_fx(1)
    Inc cur_page%, Choice(key% = ctrl.RIGHT, 1, -1)
    Select Case cur_page%
     Case < 1 : cur_page% = num_pages%
     Case > num_pages% : cur_page% = 1
    End Select
    update_menu_data()
    menu.render()
   EndIf
  Case ctrl.START
   on_start()
  Case Else
   menu.play_invalid_fx(1)
 End Select
End Sub

Sub open_bas(file_idx%)
 Local f$ = drives$(drive_idx%)
 If Right$(f$, 1) <> "/" Then Cat f$, "/"
 Cat f$, file_list$(file_idx%)
 menu.term("Loading " + file.get_name$(f$) + " ...")
 If MM.Info(Exists File f$) Then
  Run f$
 Else
  menu.term(file.get_name$(f$) + " not found")
  End
 EndIf
 Error "Invalid state"
End Sub

Sub play_music(file_idx%)
 Local f$ = drives$(drive_idx%), key%
 If Right$(f$, 1) <> "/" Then Cat f$, "/"
 Cat f$, file_list$(file_idx%)
 Local ext$ = LCase$(file.get_extension$(f$))
 sound.term()
 If ext$ = ".flac" And Instr(MM.DEVICE$, "PicoMite") Then
  Local old_cur_page% = cur_page%
  Erase file_list$()
  Erase menu.items$()
  FRAMEBUFFER Close F
 EndIf
 Select Case ext$
  Case ".flac"
   Play Flac f$, play_stop_cb
  Case ".mod"
   Play Modfile f$
  Case ".wav"
   Play Wav f$, play_stop_cb
  Case Else
   Error "Invalid state"
 End Select
 ctrl.wait_until_idle(menu.ctrl$)
 Do While key% = 0
  If sys.break_flag% Then menu.on_break()
  Call menu.ctrl$, key%
  If Not key% Then keys_cursor(key%)
 Loop
 play_stop_cb()
 If ext$ = ".flac" And Instr(MM.DEVICE$, "PicoMite") Then
  FRAMEBUFFER Create
  FRAMEBUFFER Write F
  Dim file_list$(MAX_FILES + 1) Length 64
  Dim menu.items$(15) Length 127
  update_files()
  cur_page% = old_cur_page%
  update_menu_data()
  menu.render(1)
 EndIf
 menu.process_key(key%)
End Sub

Sub play_stop_cb()
 Play Stop
 sound.init()
End Sub

Sub on_start()
 menu.play_valid_fx(1)
 Const msg$ = str.decode$("\nAre you sure you want to quit this program?")
 Select Case YES_NO_BTNS$(menu.msgbox%(msg$, YES_NO_BTNS$(), 1))
  Case "Yes"
   gamemite.end()
  Case "No"
   twm.switch(menu.win1%)
   twm.redraw()
   FRAMEBUFFER Copy F , N
  Case Else
   Error "Invalid state"
 End Select
End Sub
                           