' Transpiled on 07-01-2024 18:06:50
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Option Base 0
Option Default None
Option Explicit On
Const VERSION = 101301
' ../../mmbasic-lazer-cycle/src/splib/system.inc ++++
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' Preprocessor value GAMEMITE defined
Const sys.VERSION = 102201
Const sys.NO_DATA$ = Chr$(&h7F)
Const sys.CRLF$ = Chr$(13) + Chr$(10)
Const sys.FIRMWARE = Int(1000000 * Mm.Info(Version))
Const sys.SUCCESS = 0
Const sys.FAILURE = -1
Dim sys.break_flag%
Dim sys.err$

Function sys.format_version$(v%)
 Const v_% = Choice(v%, v%, sys.VERSION), a% = v_%\10^5, b% = (v_%-a%*10^5)\10^3
 Local c% = v_%-a%*10^5-b%*10^3, s$ = Str$(a%) + "." + Str$(b%)
 Select Case c%
  Case < 100 : Cat s$, " alpha "
  Case < 200 : Cat s$, " beta " : Inc c%, -100
  Case < 300 : Cat s$, " RC " : Inc c%, -200
  Case Else  : Cat s$, "." : Inc c%, -300
 End Select
 sys.format_version$ = s$ + Str$(c%)
End Function

Sub sys.override_break(callback$)
 sys.break_flag% = 0
 Option Break 4
 If Len(callback$) Then
  Execute "On Key 3, " + callback$ + "()"
 Else
  On Key 3, sys.break_handler()
 EndIf
End Sub

Sub sys.break_handler()
 Inc sys.break_flag%
 If sys.break_flag% > 1 Then
  sys.restore_break()
  End
 EndIf
End Sub

Sub sys.restore_break()
 sys.break_flag% = 0
 On Key 3, 0
 Option Break 3
End Sub

' ---- ../../mmbasic-lazer-cycle/src/splib/system.inc
' ../../mmbasic-lazer-cycle/src/splib/ctrl.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const ctrl.R      = &h01
Const ctrl.START  = &h02
Const ctrl.HOME   = &h04
Const ctrl.SELECT = &h08
Const ctrl.L      = &h10
Const ctrl.DOWN   = &h20
Const ctrl.RIGHT  = &h40
Const ctrl.UP     = &h80
Const ctrl.LEFT   = &h100
Const ctrl.ZR     = &h200
Const ctrl.X      = &h400
Const ctrl.A      = &h800
Const ctrl.Y      = &h1000
Const ctrl.B      = &h2000
Const ctrl.ZL     = &h4000
Const ctrl.OPEN  = -1
Const ctrl.CLOSE = -2
Const ctrl.SOFT_CLOSE = -3
Const ctrl.PULSE = 0.001
Const ctrl.UI_DELAY = 200
Dim ctrl.open_drivers$
Dim ctrl.key_type%
Dim ctrl.key_map%(31 + Mm.Info(Option Base))

Sub ctrl.init_keys(use_inkey%, period%, nbr%)
 ctrl.term_keys()
 ctrl.key_type% = 0
 On Key ctrl.on_key()
End Sub

Sub ctrl.on_key()
 Poke Var ctrl.key_map%(), Asc(LCase$(Inkey$)), 1
End Sub

Sub ctrl.term()
 ctrl.term_keys()
 On Error Ignore
 Do While Len(ctrl.open_drivers$)
  Call Field$(ctrl.open_drivers$, 1), ctrl.CLOSE
 Loop
 On Error Abort
End Sub

Sub ctrl.term_keys()
 On Key 0
 Memory Set Peek(VarAddr ctrl.key_map%()), 0, 256
 Do While Inkey$ <> "" : Loop
End Sub

Function ctrl.keydown%(i%)
 ctrl.keydown% = Peek(Var ctrl.key_map%(), i%)
 If ctrl.key_type% = 0 Then Poke Var ctrl.key_map%(), i%, 0
End Function

Function ctrl.poll_multiple$(drivers$(), mask%, duration%, key%)
 Local expires% = Choice(duration%, Timer + duration%, &h7FFFFFFFFFFFFFFF), i%
 Do
  For i% = Bound(drivers$(), 0) To Bound(drivers$(), 1)
   key% = ctrl.poll_single%(drivers$(i%), mask%)
   If key% Then ctrl.poll_multiple$ = drivers$(i%) : Exit Do
  Next
 Loop While Timer < expires%
 If duration% Then duration% = Max(0, expires% - Timer)
End Function

Function ctrl.poll_single%(driver$, mask%)
 On Error Ignore
 Call driver$, ctrl.OPEN
 If Mm.ErrNo = 0 Then
  Local key%, t% = Timer + 5
  Do
   Call driver$, key%
   ctrl.poll_single% = key% And mask%
   If ctrl.poll_single% Then
    Do While key% : Pause 5 : Call driver$, key% : Loop
    Exit Do
   EndIf
  Loop While Timer < t%
  Call driver$, ctrl.SOFT_CLOSE
 EndIf
 On Error Abort
End Function

Sub ctrl.wait_until_idle(d1$, d2$, d3$, d4$, d5$)
 Local k%
 Do
  Call d1$, k%
  If Not k% Then If Len(d2$) Then Call d2$, k%
  If Not k% Then If Len(d3$) Then Call d3$, k%
  If Not k% Then If Len(d4$) Then Call d4$, k%
  If Not k% Then If Len(d5$) Then Call d5$, k%
  If Not k% Then Exit Do
  Pause 5
 Loop
End Sub

Sub keys_cursor(x%)
 If x% < 0 Then Exit Sub
 x% =    ctrl.keydown%(32)  * ctrl.A
 Inc x%, ctrl.keydown%(128) * ctrl.UP
 Inc x%, ctrl.keydown%(129) * ctrl.DOWN
 Inc x%, ctrl.keydown%(130) * ctrl.LEFT
 Inc x%, ctrl.keydown%(131) * ctrl.RIGHT
End Sub

Sub ctrl.open_driver(d$)
 Cat ctrl.open_drivers$, d$ + ","
End Sub

Sub ctrl.close_driver(d$)
 Local idx% = InStr(ctrl.open_drivers$, d$)
 Select Case idx%
  Case 0
  Case 1
   ctrl.open_drivers$ = Mid$(ctrl.open_drivers$, Len(d$) + 2)
  Case Else
   ctrl.open_drivers$ = Mid$(ctrl.open_drivers$, 1, idx% - 1) + Mid$(ctrl.open_drivers$, idx% + Len(d$) + 1)
 End Select
End Sub

Sub ctrl.gamemite(x%)
 Select Case x%
  Case >= 0
   x% = Port(GP12,2,GP11,2,GP8,1,GP8,1,GP11,1,GP10,1,GP9,1,GP13,3,GP13,3)
   x% = (x% Xor &h7FFF) And &h29EA
   Exit Sub
  Case ctrl.OPEN
   Local i%
   For i% = 8 To 15 : SetPin Mm.Info(PinNo "GP" + Str$(i%)), Din, PullUp : Next
   ctrl.open_driver("ctrl.gamemite")
  Case ctrl.CLOSE, ctrl.SOFT_CLOSE
   Local i%
   For i% = 8 To 15 : SetPin Mm.Info(PinNo "GP" + Str$(i%)), Off : Next
   ctrl.close_driver("ctrl.gamemite")
 End Select
End Sub

' ---- ../../mmbasic-lazer-cycle/src/splib/ctrl.inc
' ../../mmbasic-lazer-cycle/src/splib/sound.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const sound.FX_FLAG% = &b01
Const sound.MUSIC_FLAG% = &b10
Dim sound.F!(127)
Dim sound.FX_BLART%(2)
Dim sound.FX_SELECT%(2)
Dim sound.FX_DIE%(4)
Dim sound.FX_WIPE%(4)
Dim sound.FX_READY_STEADY_GO%(13)
Dim sound.enabled%
Dim sound.fx_int$
Dim sound.fx_ptr%
Dim sound.fx_start_ptr%
Dim sound.music_done_cb$
Dim sound.music_int$
Dim sound.music_ptr%
Dim sound.music_start_ptr%
Dim sound.music_tick% = 200
Dim sound.music_volume% = 15

Sub sound.init(fx_int$, music_int$)
 sound.fx_int$ = Choice(Len(fx_int$), fx_int$, "sound.fx_default_int")
 sound.music_int$ = Choice(Len(music_int$), music_int$, "sound.music_default_int")
 Const offset% = Choice("Game*Mite" = "Game*Mite", 12, 0)
 Local i%
 For i% = 0 To 127
  sound.F!(i%) = Max(440 * 2^((i% - 58 + offset%) / 12.0), 16.35)
 Next
 sound.load_data("sound.blart_data", sound.FX_BLART%())
 sound.load_data("sound.select_data", sound.FX_SELECT%())
 sound.load_data("sound.die_data", sound.FX_DIE%())
 sound.load_data("sound.wipe_data", sound.FX_WIPE%())
 sound.load_data("sound.ready_steady_go_data", sound.FX_READY_STEADY_GO%())
 sound.enable(sound.FX_FLAG% Or sound.MUSIC_FLAG%)
End Sub

Sub sound.enable(flags%)
 Local raw%
 If flags% And sound.MUSIC_FLAG% Then
  If Not (sound.enabled% And sound.MUSIC_FLAG%) Then
   Execute "SetTick " + Str$(sound.music_tick%) + ", " + sound.music_int$ + ", 1"
   sound.enabled% = sound.enabled% Or sound.MUSIC_FLAG%
  EndIf
 Else
  If sound.enabled% And sound.MUSIC_FLAG% Then
   sound.enabled% = sound.enabled% Xor sound.MUSIC_FLAG%
   SetTick 0, 0, 1
   If raw% Then
    Pwm 2, sound.F!(0), 0
   Else
    Play Sound 1, B, O
    Play Sound 2, B, O
    Play Sound 3, B, O
   EndIf
   sound.music_ptr% = 0
  EndIf
 EndIf
 If flags% And sound.FX_FLAG% Then
  If Not (sound.enabled% And sound.FX_FLAG%) Then
   Execute "SetTick 40, " + sound.fx_int$ + ", 2"
   sound.enabled% = sound.enabled% Or sound.FX_FLAG%
  EndIf
 Else
  If sound.enabled% And sound.FX_FLAG% Then
   sound.enabled% = sound.enabled% Xor sound.FX_FLAG%
   SetTick 0, 0, 2
   If raw% Then
    Pwm 3, sound.F!(0), 0
   Else
    Play Sound 4, B, O
   EndIf
   sound.fx_ptr% = 0
  EndIf
 EndIf
End Sub

Function sound.is_playing%(flags%)
 Select Case flags%
  Case 0
   sound.is_playing% = (sound.music_ptr% > 0) Or (sound.fx_ptr% > 0)
  Case sound.FX_FLAG%
   sound.is_playing% = sound.fx_ptr% > 0
  Case sound.MUSIC_FLAG%
   sound.is_playing% = sound.music_ptr% > 0
 End Select
End Function

Sub sound.term()
 sound.enable(&h00)
 Play Stop
End Sub

Sub sound.load_data(data_label$, notes%())
 Local i%, num_notes%, num_channels%
 Read Save
 Restore data_label$
 Read num_notes%, num_channels%
 Const max_notes% = 8 * (Bound(notes%(), 1) - Bound(notes%(), 0))
 If num_notes% > max_notes% Then
  Local err$ = "Too much DATA; "
  Error err$ + "expected " + Str$(max_notes%) + " bytes but found " + Str$(num_notes%)
 EndIf
 notes%(0) = num_channels% + 256 * num_notes%
 For i% = 1 To num_notes% \ 8 : Read notes%(i%) : Next
 Read Restore
End Sub

Sub sound.play_music(music%(), done_cb$)
 If Not (sound.enabled% And sound.MUSIC_FLAG%) Then Exit Sub
 sound.music_start_ptr% = Peek(VarAddr music%())
 sound.music_ptr% = sound.music_start_ptr% + 8
 sound.music_done_cb$ = done_cb$
End Sub

Sub sound.music_default_int()
 If Not sound.music_ptr% Then Exit Sub
 Local n% = Peek(Byte sound.music_ptr%)
 If n% < 255 Then
  Play Sound 1, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  n% = Peek(Byte sound.music_ptr% + 1)
  Play Sound 2, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  n% = Peek(Byte sound.music_ptr% + 2)
  Play Sound 3, B, S, sound.F!(n%), (n% > 0) * sound.music_volume%
  Inc sound.music_ptr%, 3
  Exit Sub
 EndIf
 sound.music_ptr% = 0
 If Len(sound.music_done_cb$) Then Call sound.music_done_cb$
End Sub

Sub sound.play_fx(fx%(), block%)
 If Not (sound.enabled% And sound.FX_FLAG%) Then Exit Sub
 sound.fx_start_ptr% = Peek(VarAddr fx%())
 sound.fx_ptr% = sound.fx_start_ptr% + 8
 If block% Then Do While sound.fx_ptr% > 0 : Loop
End Sub

Sub sound.fx_default_int()
 If Not sound.fx_ptr% Then Exit Sub
 Local n% = Peek(Byte sound.fx_ptr%)
 If n% < 255 Then
  Play Sound 4, B, S, sound.F!(n%), (n% > 0) * 25
  Inc sound.fx_ptr%
 Else
  sound.fx_ptr% = 0
 EndIf
End Sub

sound.blart_data:
Data 16
Data 1
Data &hFFFFFF0036373C3D, &hFFFFFFFFFFFFFFFF
sound.select_data:
Data 16
Data 1
Data &hFFFFFFFF0048443C, &hFFFFFFFFFFFFFFFF
sound.die_data:
Data 32
Data 1
Data &h4748494A4B4C4D4E, &h3F40414243444546, &h0038393A3B3C3D3E, &hFFFFFFFFFFFFFFFF
sound.wipe_data:
Data 32
Data 1
Data &h3F3E3D3C3B3A3938, &h4746454443424140, &h004E4D4C4B4A4948, &hFFFFFFFFFFFFFFFF
sound.ready_steady_go_data:
Data 104
Data 1
Data &h3C3C3C3C3C3C3C3C, &h3C3C3C3C3C3C3C3C, &h0000000000000000, &h0000000000000000
Data &h3C3C3C3C3C3C3C3C, &h3C3C3C3C3C3C3C3C, &h0000000000000000, &h0000000000000000
Data &h4848484848484848, &h4848484848484848, &h4848484848484848, &h0000000048484848
Data &hFFFFFFFFFFFFFFFF
' ---- ../../mmbasic-lazer-cycle/src/splib/sound.inc
' ../../mmbasic-lazer-cycle/src/splib/string.inc ++++
' Copyright (c) 2020-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Function str.centre$(s$, x%)
 If Len(s$) < x% Then
  str.centre$ = s$ + Space$((x% - Len(s$)) \ 2)
  str.centre$ = Space$(x% - Len(str.centre$)) + str.centre$
 Else
  str.centre$ = s$
 EndIf
End Function

Function str.lpad$(s$, x%)
 str.lpad$ = s$
 If Len(s$) < x% Then str.lpad$ = Space$(x% - Len(s$)) + s$
End Function

Function str.rpad$(s$, x%)
 str.rpad$ = s$
 If Len(s$) < x% Then str.rpad$ = s$ + Space$(x% - Len(s$))
End Function

Function str.trim$(s$)
 Local st%, en%
 For st% = 1 To Len(s$)
  If Peek(Var s$, st%) <> 32 Then Exit For
 Next
 For en% = Len(s$) To 1 Step -1
  If Peek(Var s$, en%) <> 32 Then Exit For
 Next
 If en% >= st% Then str.trim$ = Mid$(s$, st%, en% - st% + 1)
End Function

Function str.decode$(s$)
 Local ad%, ch%, prev%, state%, t$
 For ad% = Peek(VarAddr s$) + 1 TO Peek(VarAddr s$) + Len(s$)
  ch% = Peek(Byte ad%)
  Select Case ch%
   Case &h5C
    state% = 1
   Case Else
    Select Case state%
     Case 0
      Cat t$, Chr$(ch%)
     Case 1
      Select Case ch%
       Case &h22 : ch% = &h22
       Case &h27 : ch% = &h27
       Case &h30 : ch% = &h00
       Case &h3F : ch% = &h3F
       Case &h5C : ch% = &h5C
       Case &h61 : ch% = &h07
       Case &h62 : ch% = &h08
       Case &h65 : ch% = &h1B
       Case &h66 : ch% = &h0C
       Case &h6E : ch% = &h0A
       Case &h71 : ch% = &h22
       Case &h72 : ch% = &h0D
       Case &h74 : ch% = &h09
       Case &h76 : ch% = &h0B
       Case &h78 : state% = 2
       Case Else : Cat t$, "\"
      End Select
      If state% = 1 Then
       Cat t$, Chr$(ch%)
       state% = 0
      EndIf
     Case 2
      Select Case ch%
       Case &h30 To &h39, &h41 To &h46, &h61 To &h66
        prev% = ch%
        state% = 3
       Case Else
        Cat t$, "\x" + Chr$(ch%)
        state% = 0
      End Select
     Case 3
      Select Case ch%
       Case &h30 To &h39, &h41 To &h46, &h61 To &h66
        Cat t$, Chr$(Val("&h" + Chr$(prev%) + Chr$(ch%)))
       Case Else
        Cat t$, "\x" + Chr$(prev%) + Chr$(ch%)
      End Select
      state% = 0
    End Select
  End Select
 Next
 Select Case state%
  Case 1 : Cat t$, "\"
  Case 2 : Cat t$, "\x"
  Case 3 : Cat t$, "\x" + Chr$(prev%)
 End Select
 str.decode$ = t$
End Function

Function str.wwrap$(s$, p%, len%)
 Const slen% = Len(s$)
 Local ch%, q%, word$
 For q% = p% To slen% + 1
  ch% = Choice(q% > slen%, 0, Peek(Var s$, q%))
  Select Case ch%
   Case 0, &h0A, &h0D, &h20
    If Len(str.wwrap$) + Len(word$) > len% Then
     If Len(word$) > len% Then
      word$ = Left$(word$, len% - Len(str.wwrap$))
      Cat str.wwrap$, word$
      Inc p%, Len(word$)
     EndIf
     Exit For
    EndIf
    Cat str.wwrap$, word$
    p% = q% + 1
    Select Case ch%
     Case &h0D
      If Choice(p% > slen%, 0, Peek(Var s$, p%)) = &h0A Then Inc p%
      Exit For
     Case &h20
      If Len(str.wwrap$) = len% Then Exit For
      Cat str.wwrap$, " "
      word$ = ""
     Case Else
      Exit For
    End Select
   Case Else
    Cat word$, Chr$(ch%)
  End Select
 Next
 p% = Min(p%, slen% + 1)
End Function

' ---- ../../mmbasic-lazer-cycle/src/splib/string.inc
' ../../mmbasic-lazer-cycle/src/splib/msgbox.inc ++++
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const msgbox.NO_PAGES = &h01

Function msgbox.show%(x%, y%, w%, h%, msg$, buttons$(), default%, ctrl$, fg%, bg%, frame%, flags%)
 Const base% = Mm.Info(Option Base), num% = Bound(buttons$(), 1) - base% + 1
 Local i%, btn_x%(num%), p% = 1
 btn_x%(base%) = x% + 2
 For i% = base% + 1 To base% + num% - 1
  btn_x%(i%) = btn_x%(i% - 1) + Len(buttons$(i% - 1)) + 5
 Next
 msgbox.box(x%, y%, w%, h%, 1, Choice(frame% = -1, fg%, frame%), bg%)
 i% = y% + 2
 Do While p% <= Len(msg$)
  msgbox.print_at(x% + 2, i%, str.wwrap$(msg$, p%, w% - 4), fg%, bg%)
  Inc i%
 Loop
 Local key%, released%, valid% = 1
 msgbox.show% = default%
 Do
  If sys.break_flag% Then msgbox.show% = default% : Exit Function
  If valid% Then
   For i% = base% To base% + num% - 1
    msgbox.button(btn_x%(i%), y% + h% - 4, buttons$(i%), i% = msgbox.show%, fg%, bg%)
   Next
   If Not flags% And msgbox.NO_PAGES Then ? ;
   valid% = 0
  EndIf
  Call ctrl$, key%
  If Not key% Then keys_cursor(key%)
  If Not key% Then released% = 1 : Continue Do
  If Not released% Then key% = 0 : Continue Do
  valid% = 0
  Select Case key%
   Case ctrl.A, ctrl.SELECT
    key% = ctrl.SELECT
    valid% = 1
   Case ctrl.LEFT
    If msgbox.show% > 0 Then Inc msgbox.show%, -1 : valid% =1
   Case ctrl.RIGHT
    If msgbox.show% < num% - 1 Then Inc msgbox.show% : valid% =1
  End Select
  msgbox.beep(valid%)
  Pause ctrl.UI_DELAY - 100
 Loop Until key% = ctrl.SELECT
 ctrl.wait_until_idle(ctrl$, "keys_cursor")
End Function

Sub msgbox.button(x%, y%, txt$, selected%, fg%, bg%)
 msgbox.box(x%, y%, Len(txt$) + 4, 3, 0, fg%, bg%)
 Const fg_% = Choice(selected%, bg%, fg%)
 Const bg_% = Choice(selected%, fg%, bg%)
 msgbox.print_at(x% + 2, y% + 1, txt$, fg_%, bg_%)
End Sub

Sub msgbox.box(x%, y%, w%, h%, dbl%, fg%, bg%)
 Const fh% = Mm.Info(FontHeight), fw% = Mm.Info(FontWidth)
 Local d% = fw% \ 2
 Box x% * fw%, y% * fh%, w% * fw%, h% * fh%, , bg%, bg%
 Box x% * fw% + d%, y% * fh% + d%, w% * fw% - 2 * d%, h% * fh% - 2 * d%, 1, fg%
 Inc d%, d%
 If dbl% Then Box x% * fw% + d%, y% * fh% + d%, w% * fw% - 2 * d%, h% * fh% - 2 * d%, 1, fg%
End Sub

Sub msgbox.print_at(x%, y%, s$, fg%, bg%)
 Text x% * Mm.Info(FontWidth), y% * Mm.Info(FontHeight), s$, , , , fg%, bg%
End Sub

Sub msgbox.beep(valid%)
 If valid% Then
  Local notes!(3) = (987.77, 1567.98, 1975.53, 30.87)
 Else
  Local notes!(4) = (1046.50, 987.77, 739.99, 698.46, 30.87)
 EndIf
 Play Stop
 Local i%
 For i% = Bound(notes!(), 0) To Bound(notes!(), 1)
  If notes!(i%) > 16.0 Then Play Sound 4, B, S, notes!(i%), 25
  Pause 40
 Next
 Play Stop
End Sub

' ---- ../../mmbasic-lazer-cycle/src/splib/msgbox.inc
' ../../mmbasic-lazer-cycle/src/highscr.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Const highscr.MAX_NAME_LEN% = 8
Dim highscr.values$(9) Length highscr.MAX_NAME_LEN% + 7
Dim highscr.ALLOWED_CHARS%(3) = (&h03FF440100000000, &h000000007FFFFFF, &h0000020080001E00, &h0)

Sub highscr.init(filename$, data_label$)
 Local i%
 Read Save
 Restore data_label$
 For i% = 0 To Bound(highscr.values$(), 1)
  Read highscr.values$(i%)
 Next
 Read Restore
 If Not highscr.has_fs%() Then Exit Sub
 Local ok% = 0
 Select Case UCase$(Left$(filename$, 1))
  Case "/", "\" : ok% = 1
  Case "A" To "Z" : If Mid$(filename$, 2, 1) = ":" Then ok% = 1
 End Select
 If Not ok% Then Error "Expected absolute file path '" + filename$ + "'"
 Local ch$, parent$
 For i% = 1 To Len(filename$)
  ch$ = Mid$(filename$, i%, 1)
  If InStr("/\", ch$) And parent$ <> "" And UCase$(parent$) <> "A:" Then
   if Mm.Info(exists dir parent$)=0 then MkDir parent$
  EndIf
  Cat parent$, ch$
 Next
 Select Case Mm.Info(exists file filename$)
  Case -1: Error "Expected file but found directory '" + parent$ + "'"
  Case 0: Exit Sub
 End Select
 Open filename$ For Input As #1
 For i% = 0 To Bound(highscr.values$())
  Line Input #1, highscr.values$(i%)
  ok% = ok% And Field$(highscr.values$(i%), 1) <> ""
  ok% = ok% And Field$(highscr.values$(i%), 2) <> ""
  ok% = ok% And Val(Field$(highscr.values$(i%), 2)) > 0
  If Not ok% Then Error "Invalid high-score file '" + filename$ + "'"
 Next
 Close #1
End Sub

Function highscr.has_fs%()
 highscr.has_fs% = 1
End Function

Sub highscr.save(filename$)
 If Not highscr.has_fs%() Then Exit Sub
 Open filename$ For Output As #1
 Local i%
 For i% = 0 To Bound(highscr.values$())
  Print #1, highscr.values$(i%)
 Next
 Close #1
End Sub

Function highscr.show_table$(ctrls$(), duration%)
 Const ch$ = Chr$(205), X_OFFSET% = MM.HRes \ 2, Y_OFFSET% = MM.VRes \ 2
 Local col_idx%, i%, name$, score$, y%
 Local expires% = Choice(duration%, Timer + duration%, &h7FFFFFFFFFFFFFFF)
 Local colours%(3) = (Rgb(Red), Rgb(Yellow), Rgb(Cyan), Rgb(Green))
 Cls
 Text X_OFFSET%, Y_OFFSET% - 95, ch$ + ch$ + " HIGH SCORES " + ch$ + ch$, "CT", 1, 1, Rgb(White)
 ctrl.init_keys()
 Do While Timer < expires% And highscr.show_table$ = ""
  For i% = 0 To Bound(highscr.values$(), 1) + 1
   If i% <= Bound(highscr.values$(), 1) Then
    name$ = str.rpad$(Field$(highscr.values$(i%), 1), highscr.MAX_NAME_LEN%)
    score$ = str.lpad$(Field$(highscr.values$(i%), 2), 5)
    y% = Y_OFFSET% - 75 + 15 * i%
    Text X_OFFSET%, y%, score$ + "  " + name$, "CT", 1, 1, colours%(col_idx%)
   EndIf
   col_idx% = (col_idx% + 1) Mod 4
  Next
  ? ;
  highscr.show_table$ = ctrl.poll_multiple$(ctrls$(), ctrl.A Or ctrl.SELECT Or ctrl.START, 200)
 Loop
End Function

Sub highscr.edit(player%, idx%, colour_%, ctrl$)
 Const HEIGHT% = 16, WIDTH% = 16
 Const X_ORIGIN% = (MM.HRes - 10 * WIDTH%) \ 2
 Const Y_ORIGIN% = Int(5.5 * HEIGHT%)
 Const Y_FOOTER% = Mm.VRes - 16
 Local bg%, ch%, count%, fg%, grid$, i%, key%, p%, t%
 Local name$, s$, space_is_fire%, state%, x%, x_new%, y%, y_new%
 Local footer$(1) Length 40
 footer$(0) = str.decode$("Use \x95 \x94 \x92 \x93 and ")
 Cat footer$(0), Choice("Game*Mite" = "Game*Mite", "SELECT", "FIRE to select")
 footer$(1) = "Or, type name and ENTER to confirm"
 Cls
 s$ = Chr$(205)
 Text Mm.HRes \ 2, 25, s$ + s$ + " " + Str$(idx% + 1) + "   PLACE HIGH SCORE " + s$ + s$, "CT", 1, 1, Rgb(White)
 Select Case idx%
  Case 0 : s$ = "ST"
  Case 1 : s$ = "ND"
  Case 2 : s$ = "RD"
  Case Else : s$ = "TH"
 End Select
 Text Choice(idx% = 9, 100, 96), 25, s$, "CT", 7, 1, Rgb(White)
 Text Mm.HRes \ 2, 3 * HEIGHT%, "PLAYER " + Str$(player%), "CT", 1, 1, colour_%
 Restore highscr.grid_data
 For y% = 0 To 4
  For x% = 0 To 9
   Read ch%
   Cat grid$, Chr$(ch%)
   If ch% = 10 Then ch% = 32
   If ch% = 8 Then ch% = 149
   Text X_ORIGIN% + x% * WIDTH% + 4, Y_ORIGIN% + y% * HEIGHT% + 3, Chr$(ch%), , 1, 1, Rgb(White)
  Next
 Next
 Text X_ORIGIN% + 9 * WIDTH% + 2, Y_ORIGIN + 4 * HEIGHT% + 5, "OK", , 7, 1, Rgb(White)
 ctrl.term_keys()
 Call ctrl$, ctrl.OPEN
 Do While highscr.get_input%(ctrl$) : Loop
 space_is_fire% = 1 : t% = Timer
 x% = -1 : x_new% = 0 : y% = -1 : y_new% = 0
 Do
  p% = Mm.HRes / 2
  Inc p%, -4 * Min(Len(name$) + 1, highscr.MAX_NAME_LEN%)
  Text p% - 8, 4 * HEIGHT%, " ", , 1, 1
  For i% = 1 To Min(Len(name$) + 1, highscr.MAX_NAME_LEN%)
   bg% = Choice((i% = Len(name$) + 1) And (count% And &b1), colour_%, Rgb(Black))
   fg% = Choice((i% = Len(name$) + 1) And (count% And &b1), Rgb(Black), colour_%)
   s$ = Choice(i% = Len(name$) + 1, " ", Mid$(name$, i%, 1))
   Text p%, 4 * HEIGHT%, s$, , 1, 1, fg%, bg%
   Inc p%, 8
  Next
  Text p%, 4 * HEIGHT%, " ", , 1, 1
  Inc p%, 8
  If x% <> x_new% Or y% <> y_new% Then
   If x% <> -1 Then
    Box X_ORIGIN% + x% * WIDTH%, Y_ORIGIN% + y% * HEIGHT% + 1, WIDTH%, HEIGHT%, 1, Rgb(Black)
   EndIf
   x% = x_new% : y% = y_new%
   Box X_ORIGIN% + x% * WIDTH%, Y_ORIGIN% + y% * HEIGHT% + 1, WIDTH%, HEIGHT%, 1, colour_%
  EndIf
  If Timer > t% + 500 Then
   count% = (count% + 1) Mod 10
   t% = Timer
  EndIf
  s$ = str.centre$(footer$(count% >= 5), 38)
  Text Mm.HRes \ 2, Y_FOOTER%, s$, "CT", 1, 1, Rgb(White), Rgb(Black)
  ? ;
  key% = highscr.get_input%(ctrl$)
  state% = key% > 0
  If key% > 0 Then
   If key% <> &h20 Then space_is_fire% = (key% > &h7F)
   If (key% = &h20) And space_is_fire% Then key% = ctrl.A << 8
   If (key% = ctrl.A << 8) Or (key% = ctrl.SELECT << 8) Then
    key% = Asc(Mid$(grid$, x% + y% * 10 + 1, 1))
   EndIf
   Select Case key%
    Case &h08, &h7F
     If Len(name$) > 0 Then name$ = Left$(name$, Len(name$) - 1) Else state% = 2
    Case &h0A, &h0D
     key% = -1
    Case ctrl.UP << 8
     If y% > 0 Then y_new% = y% - 1 Else y_new% = 4
    Case ctrl.DOWN << 8
     If y% < 4 Then y_new% = y% + 1 Else y_new% = 0
    Case ctrl.LEFT << 8
     If x% > 0 Then x_new% = x% - 1 Else x_new% = 9
    Case ctrl.RIGHT << 8
     If x% < 9 Then x_new% = x% + 1 Else x_new% = 0
    Case &h20 To &hFF
     If Len(name$) < highscr.MAX_NAME_LEN% Then Cat name$, Chr$(key%) Else state% = 2
    Case Else
     state% = 2
   End Select
  End If
  If state% Then
   If state% = 1 Then sound.play_fx(sound.FX_SELECT%()) Else sound.play_fx(sound.FX_BLART%())
   Pause ctrl.UI_DELAY
  EndIf
 Loop While key% <> -1
 Text Mm.HRes \ 2, Y_FOOTER%, Space$(40), "CT", 1, 1, Rgb(White), Rgb(Black)
 name$ = str.trim$(name$)
 If name$ = "" Then name$ = "PLAYER " + Str$(player%)
 highscr.values$(idx%) = name$ + ", " + Field$(highscr.values$(idx%), 2)
End Sub

highscr.grid_data:
Data  65,  66,  67,  68,  69,  70,  71,  72, 73,  74
Data  75,  76,  77,  78,  79,  80,  78,  82, 83,  84
Data  85,  86,  87,  88,  89,  90,  42, 46,  64, 169
Data  48,  49,  50,  51,  52,  53,  54,  55, 56,  57
Data  63,  45, 137, 138, 139, 140, 159, 32,   8,  10

Function highscr.get_input%(ctrl$)
 highscr.get_input% = Asc(UCase$(Inkey$))
 If highscr.get_input% Then
  Select Case highscr.get_input%
   Case &h80 : highscr.get_input% = ctrl.UP << 8
   Case &h81 : highscr.get_input% = ctrl.DOWN << 8
   Case &h82 : highscr.get_input% = ctrl.LEFT << 8
   Case &h83 : highscr.get_input% = ctrl.RIGHT << 8
  End Select
 Else
  Call ctrl$, highscr.get_input%
  If highscr.get_input% Then highscr.get_input% = highscr.get_input% << 8
 EndIf
End Function

Function highscr.get_directory$()
 Const drive$ = Choice(InStr(Mm.Device$, "PicoMite"), Mm.Info$(Drive), "A:")
 Const root$ = Choice(InStr(Mm.Device$, "PicoMite"), "A:/", Mm.Info$(Path))
 highscr.get_directory$ = root$ + "high-scores"
 If drive$ <> "A:" Then Drive "A:"
 On Error Skip
 MkDir highscr.get_directory$
 If drive$ <> "A:" Then Drive drive$
End Function

' ---- ../../mmbasic-lazer-cycle/src/highscr.inc
' ../../mmbasic-lazer-cycle/src/menu.inc ++++
' Copyright (c) 2022-2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Function menu.show%(ui_ctrl$, ctrls$(), colours%())
 Const x% = X_OFFSET% - 100
 Static initialised% = 0
 Local key%, i%, item% = 0, update% = 1
 Local sounds$(3) Length 10 = ("NONE      ", "FX ONLY   ", "MUSIC ONLY", "MUSIC & FX")
 Local sound_setting% = sound.enabled%
 Local music_ptr_bak%
 Local difficulties$(5) Length 6 = ("NOVICE", "1     ", "2     ", "3     ", "4     ", "5     ")
 If Not initialised% Then
  Dim menu.ctrls$(Bound(ctrls$(), 1)) Length 32
  initialised% = 1
 EndIf
 For i% = Bound(ctrls$(), 0) To Bound(ctrls$(), 1)
  menu.ctrls$(i%) = ctrls$(i%)
 Next
 Local count% = 0
 For i% = Bound(menu.ctrls$(), 0) To Bound(menu.ctrls$(), 1)
  Inc count%, Not InStr("ai_control|no_control", menu.ctrls$(i%))
 Next
 If count% = 0 Then menu.ctrls$(0) = ui_ctrl$
 Text X_OFFSET%, Y_OFFSET% + 90, VERSION_STRING$, "CM", 7, 1, Rgb(Cyan)
 ctrl.init_keys()
 Call ui_ctrl$, ctrl.OPEN
 Do
  If update% Then
   Text x%, Y_OFFSET% - 95, "START GAME", , 1, 1, Rgb(White)
   menu.txt_for_controller(x%, Y_OFFSET% - 75, 0, colours%(0))
   menu.txt_for_controller(x%, Y_OFFSET% - 55, 1, colours%(1))
   menu.txt_for_controller(x%, Y_OFFSET% - 35, 2, colours%(2))
   menu.txt_for_controller(x%, Y_OFFSET% - 15, 3, colours%(3))
   Text x%, Y_OFFSET% + 5,  "DIFFICULTY: " + difficulties$(difficulty%), , 1, 1, Rgb(White)
   Text x%, Y_OFFSET% + 25, "SOUND:      " + sounds$(sound_setting%), , 1, 1, Rgb(White)
   Text x%, Y_OFFSET% + 45, "QUIT", , 1, 1, Rgb(White)
   Text x% - 15, Y_OFFSET% - 95 + item% * 20, Chr$(137), , 1, 1, Rgb(Cyan)
   ? ;
   update% = 0
   Pause ctrl.UI_DELAY
  EndIf
  key% = 0
  Do While Not key%
   Call ui_ctrl$, key%
   If Not key% Then keys_cursor(key%)
  Loop
  If key% = ctrl.A Then key% = ctrl.SELECT
  Select Case key%
   Case ctrl.START
    menu.show% = 1
    Exit Do
   Case ctrl.UP
    Text x% - 15, Y_OFFSET% - 95 + item% * 20, Chr$(137), , 1, 1, Rgb(Black)
    Inc item%, -1
    If item% < 0 Then item% = 7
    update% = 1
   Case ctrl.DOWN
    Text x% - 15, Y_OFFSET% - 95 + item% * 20, Chr$(137), , 1, 1, Rgb(Black)
    Inc item%
    If item% > 7 Then item% = 0
    update% = 1
   Case ctrl.LEFT, ctrl.RIGHT, ctrl.SELECT
    Select Case item%
     Case 0
      If key% = ctrl.SELECT Then menu.show% = 1 : Exit Do
     Case 1,2,3,4
      menu.update_controller(item% - 1, Choice(key% = ctrl.LEFT, -1, +1), ui_ctrl$)
      update% = 1
     Case 5
      Inc difficulty%, Choice(key% = ctrl.LEFT, -1, 1)
      If difficulty% < 0 Then difficulty% = 5
      If difficulty% > 5 Then difficulty% = 0
      update% = 1
     Case 6
      If sound_setting% And sound.MUSIC_FLAG% Then music_ptr_bak% = sound.music_ptr%
      Inc sound_setting%, Choice(key% = ctrl.LEFT, -1, 1)
      If sound_setting% < 0 Then sound_setting% = 3
      If sound_setting% > 3 Then sound_setting% = 0
      sound.enable(sound_setting%)
      If sound_setting% And sound.MUSIC_FLAG% Then sound.music_ptr% = music_ptr_bak%
      update% = 1
     Case 7
      If key% = ctrl.SELECT Then Exit Do
    End Select
   Case Else
    sound.play_fx(sound.FX_BLART%())
    Pause ctrl.UI_DELAY
  End Select
  If update% Then sound.play_fx(sound.FX_SELECT%())
 Loop
 Call ui_ctrl$, ctrl.CLOSE
 For i% = Bound(ctrls$(), 0) To Bound(ctrls$(), 1)
  ctrls$(i%) = menu.ctrls$(i%)
 Next
 For i% = Bound(ctrls$(), 0) To Bound(ctrls$(), 1)
  If ctrls$(i%) <> "ai_control" And ctrls$(i%) <> "no_control" Then
   ui_ctrl$ = ctrls$(i%)
   Exit For
  EndIf
 Next
End Function

Sub menu.txt_for_controller(x%, y%, idx%, colour%)
 Local txt$ = "PLAYER " + Str$(idx% + 1) + ":   " + menu.get_controller_name$(idx%)
 Text x%, y%, txt$, , 1, 1, colour%
End Sub

Function menu.get_controller_name$(idx%)
 Local i%
 For i% = Bound(CTRL_DRIVER$(), 0) To Bound(CTRL_DRIVER$(), 1)
  If menu.ctrls$(idx%) = CTRL_DRIVER$(i%) Then
   menu.get_controller_name$ = CTRL_DESCRIPTION$(i%)
   Do While Len(menu.get_controller_name$) < 14 : Cat menu.get_controller_name$, " " : Loop
   Exit Function
  Endif
 Next
 Error "Unknown controller: " + menu.ctrls$(idx%)
End Function

Sub menu.update_controller(idx%, delta%, ui_ctrl$)
 Call ui_ctrl$, ctrl.CLOSE
 Local i%
 For i% = Bound(CTRL_DRIVER$(), 0) To Bound(CTRL_DRIVER$(), 1)
  If menu.ctrls$(idx%) = CTRL_DRIVER$(i%) Then Exit For
 Next
 If i% = Bound(CTRL_DRIVER$(), 1) + 1 Then Error "Unknown controller: " + menu.ctrls$(idx%)
 Local ok%
 Do
  Inc i%, delta%
  If i% < 0 Then i% = Bound(CTRL_DRIVER$(), 1)
  If i% > Bound(CTRL_DRIVER$(), 1) Then i% = 0
  If Not InStr("ai_control|no_control", CTRL_DRIVER$(i%)) Then
   If menu.has_controller%(idx%, CTRL_DRIVER$(i%)) Then Continue Do
  EndIf
  Select Case CTRL_DRIVER$(i%)
   Case "atari_a"   : ok% = Not menu.has_controller%(idx%, "nes_a")
   Case "atari_b"   : ok% = Not menu.has_controller%(idx%, "nes_b")
   Case "atari_dx"  : ok% = Not menu.has_controller%(idx%, "nes_dx")
   Case "nes_a"     : ok% = Not menu.has_controller%(idx%, "atari_a")
   Case "nes_b"     : ok% = Not menu.has_controller%(idx%, "atari_b")
   Case "nes_dx"    : ok% = Not menu.has_controller%(idx%, "atari_dx")
   Case "keys_cegg" : ok% = Not menu.has_controller%(idx%, "keys_azxc", "keys_punc")
   Case "keys_azxc" : ok% = Not menu.has_controller%(idx%, "keys_cegg")
   Case "keys_punc" : ok% = Not menu.has_controller%(idx%, "keys_cegg")
   Case Else        : ok% = 1
  End Select
  If ok% Then ok% = menu.can_open_controller%(CTRL_DRIVER$(i%))
 Loop Until ok%
 menu.ctrls$(idx%) = CTRL_DRIVER$(i%)
 Call ui_ctrl$, ctrl.OPEN
End Sub

Function menu.can_open_controller%(ctrl$)
 On Error Ignore
 Call ctrl$, ctrl.OPEN
 On Error Abort
 Local ok% = Mm.ErrNo = 0
 If ok% Then Call ctrl$, ctrl.CLOSE
 menu.can_open_controller% = ok%
End Function

Function menu.has_controller%(idx%, ctrl1$, ctrl2$)
 Local i%
 For i% = Bound(menu.ctrls$(), 0) To Bound(menu.ctrls$(), 1)
  If i% = idx% Then Continue For
  If menu.ctrls$(i%) = ctrl1$ Then menu.has_controller% = 1
  If menu.ctrls$(i%) = ctrl2$ Then menu.has_controller% = 1
 Next
End Function

' ---- ../../mmbasic-lazer-cycle/src/menu.inc
' ../../mmbasic-lazer-cycle/src/splib/gamemite.inc ++++
' Copyright (c) 2023 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
Function gamemite.file$(f$)
 If InStr("A:/B:/", UCase$(Left$(f$, 3))) Then
  gamemite.file$ = f$
 Else
  Local f_$ = "A:/GameMite" + Choice(f$ = "", "", "/" + f$), x%
  x% = Mm.Info(Exists File f_$)
  If Not x% Then
   f_$ = "B" + Mid$(f_$, 2)
   On Error Skip
   x% = Mm.Info(Exists File f_$)
  EndIf
  If Not x% Then f_$ = "A" + Mid$(f_$, 2)
  gamemite.file$ = f_$
 EndIf
End Function

Sub gamemite.end(break%)
 FrameBuffer Write N
 Colour Rgb(White), Rgb(Black)
 Cls
 sys.restore_break()
 On Error Skip : sound.term()
 On Error Skip : ctrl.term()
 On Error Skip
 twm.enable_cursor(1)
 If break% Then
  Const f$ = "", msg$ = "Exited due to Ctrl-C"
 Else
  Const f$ = gamemite.file$("menu.bas")
  Const x% = Mm.Info(Exists File f$)
  Const msg$ = Choice(x%, "Loading menu ...", "Menu program not found!")
 EndIf
 Text 160, 110, msg$, CM
 If Len(f$) Then Run f$ Else End
End Sub

' ---- ../../mmbasic-lazer-cycle/src/splib/gamemite.inc
If InStr(Mm.Device$, "PicoMite") Then
 If Val(Mm.Info(CpuSpeed)) < 252000000 Then
  Error "Requires OPTION CPUSPEED 252000 or 378000"
 EndIf
EndIf
Const USE_CONTROLLERS$ = "controller_data_gamemite"
Const START_TEXT$ = str.centre$("Press START to play", 40)
Const VERSION_STRING$ = "Game*Mite Version " + sys.format_version$(VERSION)
Const HIGHSCORE_FILENAME$ = highscr.get_directory$() + "/lazer-cycle.csv"
Const A_START_SELECT = ctrl.A Or ctrl.START Or ctrl.SELECT
? ;
Const WIDTH% = Mm.HRes \ 3
Const HEIGHT% = (Mm.VRes - 20) \ 3
Const X_OFFSET% = MM.HRes \ 2
Const Y_OFFSET% = MM.VRes \ 2
Const NORTH% = 0, EAST% = 1, SOUTH% = 2, WEST% = 3
Const MAX_CYCLE_IDX% = 3
Const SCORE_Y% = Mm.VRes - 16
Const STATE_OK%    = &b000
Const STATE_DYING% = &b100
Const STATE_DEAD%  = &b101
Const HORIZONTAL_MASK% = ctrl.LEFT Or ctrl.RIGHT
Const VERTICAL_MASK%   = ctrl.UP Or ctrl.DOWN
Const DIRECTION_MASK%  = HORIZONTAL_MASK% Or VERTICAL_MASK%
Dim NEXT_DIR%(7)        = (EAST%, NORTH%, WEST%, SOUTH%, EAST%, NORTH%, WEST%, SOUTH%)
Dim SCORE_X%(3)         = (35, 105, 175, 245)
Dim DIRECTIONS%(3)      = (-WIDTH%, 1, WIDTH%, -1)
Dim COMPASS_TO_CTRL%(3) = (ctrl.UP, ctrl.RIGHT, ctrl.DOWN, ctrl.LEFT)
Dim FRAME_DURATIONS%(5) = (42, 38, 34, 30, 26, 22)
Dim MUSIC_ENTERTAINER%(792 \ 8)
Dim MUSIC_BLACK_WHITE_RAG%(888 \ 8)
Dim ui_ctrl$
Dim attract_mode% = 1
Dim score%
Dim difficulty% = "Game*Mite" <> "Game*Mite"
Dim frame_duration%
Dim next_frame%
Dim music_track% = 1
Dim arena%(HEIGHT% * WIDTH% \ 8)
Dim cycle.current%
Dim cycle.score%(MAX_CYCLE_IDX%)
Dim cycle.nxt%(MAX_CYCLE_IDX%)
Dim cycle.pos%(MAX_CYCLE_IDX%)
Dim cycle.dir%(MAX_CYCLE_IDX%)
Dim cycle.colour%(MAX_CYCLE_IDX%) = (Rgb(Red), Rgb(Yellow), Rgb(Cyan), Rgb(Green))
Dim cycle.ctrl$(MAX_CYCLE_IDX%) Length 32
Dim cycle.ctrl_setting$(MAX_CYCLE_IDX%) Length 32
Dim cycle.state%(MAX_CYCLE_IDX%)
Dim cycle.last_key%(MAX_CYCLE_IDX%)
Dim num_alive%
Dim num_humans%
sys.override_break("break_cb")
init_globals()
clear_display()
sound.init()
sound.play_music(MUSIC_ENTERTAINER%(), "music_done_cb")
outer_loop()
End

Sub outer_loop()
 Local attract_mode% = 1, i%
 Do
  If attract_mode% Then wipe() : attract_mode% = Not show_title%(5000)
  If attract_mode% Then wipe() : attract_mode% = Not show_instructions%(15000)
  If attract_mode% Then wipe() : attract_mode% = Not show_highscore%(5000)
  If Not attract_mode% Then
   wipe()
   If Not menu.show%(ui_ctrl$, cycle.ctrl_setting$(), cycle.colour%()) Then end_program()
  EndIf
  wipe()
  init_game(attract_mode%)
  draw_arena()
  draw_score()
  If Not attract_mode% Then ready_steady_go()
  If game_loop%() Then
   attract_mode% = 0
  ElseIf Not attract_mode% Then
   show_game_over()
   attract_mode% = Not show_highscore%(5000)
  EndIf
  ctrl.term()
 Loop
End Sub

Sub break_cb()
 end_program(1)
End Sub

Sub end_program(break%)
 gamemite.end(break%)
End Sub

Sub music_done_cb()
 If music_track% = 1 Then
  sound.play_music(MUSIC_BLACK_WHITE_RAG%(), "music_done_cb")
  music_track% = 2
 Else
  sound.play_music(MUSIC_ENTERTAINER%(), "music_done_cb")
  music_track% = 1
 EndIf
End Sub

Sub init_globals()
 Local a%, i%, j%
 Restore USE_CONTROLLERS$
 Local num_ctrl%, num_poll%
 Read num_ctrl%, num_poll%
 Dim CTRL_DESCRIPTION$(num_ctrl% - 1)
 Dim CTRL_DRIVER$(num_ctrl% - 1)
 'Dim CTRLS_TO_POLL$(Max(1, num_poll% - 1))
 dim ctrls_to_poll$(10)
 j% = 0
 For i% = 0 To num_ctrl% - 1
  Read CTRL_DESCRIPTION$(i%), CTRL_DRIVER$(i%), a%
  If a% Then
   CTRLS_TO_POLL$(j%) = CTRL_DRIVER$(i%)
   Inc j%
  EndIf
 Next
 If num_poll% = 1 Then CTRLS_TO_POLL$(1) = CTRLS_TO_POLL$(0)
 cycle.ctrl_setting$(0) = "ai_control"
 cycle.ctrl_setting$(1) = "ai_control"
 For i% = 2 To MAX_CYCLE_IDX% : cycle.ctrl_setting$(i%) = "no_control" : Next
 highscr.init(HIGHSCORE_FILENAME$, "highscore_data")
 sound.load_data("entertainer_music_data", MUSIC_ENTERTAINER%())
 sound.load_data("black_white_rag_music_data", MUSIC_BLACK_WHITE_RAG%())
End Sub

Function show_title%(duration%)
 Text X_OFFSET%, Y_OFFSET% - 27, "LAZER CYCLE", "CM", 1, 2, Rgb(White)
 Text X_OFFSET%, Y_OFFSET% - 10, VERSION_STRING$, "CM", 7, 1, Rgb(Cyan)
 Text X_OFFSET%, Y_OFFSET% + 8, "(c) 2022-2023 Thomas Hugo Williams", "CM", 7, 1, Rgb(Cyan)
 Text X_OFFSET%, Y_OFFSET% + 20, "www.sockpuppetstudios.com", "CM", 7, 1, Rgb(Cyan)
 Text X_OFFSET%, Y_OFFSET% + 40, START_TEXT$, "CM", 1, 1, Rgb(White)
 ? ;
 show_title% = twait%(duration%)
End Function

Function show_instructions%(duration%)
 Const ch$ = Chr$(205)
 Local y% = Y_OFFSET% - 95
 Text X_OFFSET%, y%, ch$ + ch$ + " LAZER CYCLE " + ch$ + ch$, "CT", 1, 1, Rgb(White) : Inc y%, 20
 Text X_OFFSET%, y%, "An arcade game for 1-4 players.", "CT", 1, 1, Rgb(Red) : Inc y%, 20
 Text X_OFFSET%, y%, "Pilot your cycle around the", "CT", 1, 1, Rgb(Yellow) : Inc y%, 12
 Text X_OFFSET%, y%, "arena leaving a trail of light", "CT", 1, 1, Rgb(Yellow) : Inc y%, 12
 Text X_OFFSET%, y%, "behind, longest trail wins.", "CT", 1, 1, Rgb(Yellow) : Inc y%, 20
 Text X_OFFSET%, y%, "You are eliminated if you hit the", "CT", 1, 1, Rgb(Cyan) : Inc y%, 12
 Text X_OFFSET%, y%, "arena wall, or one of the trails.", "CT", 1, 1, Rgb(Cyan) : Inc y%, 20
 Text X_OFFSET%, y%, "Use keyboard, joystick or gamepad", "CT", 1, 1, Rgb(Green) : Inc y%, 12
 Text X_OFFSET%, y%, "UP, DOWN, LEFT and RIGHT to steer.", "CT", 1, 1, Rgb(Green) : Inc y%, 20
 Text X_OFFSET%, y%, "Good Luck!", "CT", 1, 1, Rgb(White)
 ? ;
 show_instructions% = twait%(duration%)
End Function

Function show_highscore%(duration%)
 Local ctrl$ = highscr.show_table$(CTRLS_TO_POLL$(), 5000)
 If ctrl$ <> "" Then
  If ui_ctrl$ = "" Then ui_ctrl$ = ctrl$
  show_highscore% = 1
 EndIf
End Function

Sub clear_display()
 Box 0, 0, Mm.HRes, Mm.VRes, 1, Rgb(Black), Rgb(Black)
 ? ;
End Sub

Function twait%(duration%)
 ctrl.init_keys()
 Local ctrl$ = ctrl.poll_multiple$(CTRLS_TO_POLL$(), A_START_SELECT, duration%)
 If Len(ctrl$) Then ui_ctrl$ = Choice(ui_ctrl$ = "", ctrl$, ui_ctrl$)
 twait% = Len(ctrl$)
End Function

Function on_quit%(ctrl$)
 msgbox.beep(1)
 Local buttons$(1) Length 3 = ("Yes", "No")
 Const msg$ = Choice(num_alive%, "Return to game menu?", "    Quit game?")
 Const x% = 9, y% = 5, fg% = Rgb(White), bg% = Rgb(Black), frame% = Rgb(Cyan)
 ? ;
 If InStr(Mm.Device$, "PicoMite") Then
  FrameBuffer Create
  If Mm.Device$ = "PicoMiteVGA" Then FrameBuffer Copy N, F, B Else FrameBuffer Copy N, F
 EndIf
 Const a% = msgbox.show%(x%, y%, 22, 9, msg$, buttons$(), 1, ctrl$, fg%, bg%, frame%, msgbox.NO_PAGES)
 If buttons$(a%) = "Yes" Then
  If num_alive% > 0 Then
   on_quit% = 1
   num_alive% = 0
  Else
   end_program()
  EndIf
 EndIf
 ? ;
 If InStr(Mm.Device$, "PicoMite") Then
  If Mm.Device$ = "PicoMiteVGA" Then FrameBuffer Copy F, N, B Else FrameBuffer Copy F, N
  FrameBuffer Close F
 EndIf
 ctrl.wait_until_idle(ctrl$)
End Function

Sub init_game(attract_mode%)
 frame_duration% = FRAME_DURATIONS%(difficulty%)
 num_alive% = 0
 num_humans% = 0
 score% = 0
 Local p_arena% = Peek(VarAddr arena%())
 Memory Set p_arena%, 0, HEIGHT% * WIDTH%
 Memory Set p_arena%, 128, WIDTH%
 Memory Set p_arena% + (HEIGHT% - 1) * WIDTH%, 128, WIDTH%
 Local y%
 For y% = 1 To HEIGHT% - 2
  Poke Byte p_arena% + y% * WIDTH%, 128
  Poke Byte p_arena% + (y% + 1) * WIDTH% - 1, 128
 Next
 Local i%
 For i% = 0 To MAX_CYCLE_IDX%
  cycle.ctrl$(i%) = Choice(attract_mode%, "ai_control", cycle.ctrl_setting$(i%))
  On Error Ignore
  Call cycle.ctrl$(i%), ctrl.OPEN
  If Mm.ErrNo <> 0 Then cycle.ctrl$(i%) = "no_control"
  On Error Abort
 Next
 ctrl.init_keys()
 cycle.dir%(0) = EAST%
 cycle.dir%(1) = WEST%
 cycle.dir%(2) = SOUTH%
 cycle.dir%(3) = NORTH%
 cycle.pos%(0) = WIDTH * (HEIGHT% \ 2) + 5
 cycle.pos%(1) = WIDTH% * (HEIGHT% \ 2) + WIDTH% - 6
 cycle.pos%(2) = 5.5 * WIDTH%
 cycle.pos%(3) = WIDTH% * (HEIGHT% - 6) + WIDTH% \ 2
 For i% = 0 To MAX_CYCLE_IDX%
  cycle.score%(i%) = 0
  cycle.last_key%(i%) = 0
  If cycle.ctrl$(i%) = "no_control" Then
   cycle.pos%(i%) = -1
   cycle.nxt%(i%) = -1
   cycle.state%(i%) = STATE_DEAD%
  Else
   Inc num_alive%
   Inc num_humans%, cycle.ctrl$(i%) <> "ai_control"
   cycle.nxt%(i%) = cycle.pos%(i%) + DIRECTIONS%(cycle.dir%(i%))
   Poke Byte p_arena% + cycle.pos%(i%), (cycle.dir%(i%) << 3) + (i% << 1) + 1
   Poke Byte p_arena% + cycle.nxt%(i%), (cycle.dir%(i%) << 3) + (i% << 1) + 1
   cycle.state%(i%) = STATE_OK%
  EndIf
 Next
End Sub

Sub draw_arena()
 Local x%
 For x% = 0 To (HEIGHT% * WIDTH%) - 1
  If Peek(Var arena%(), x%) <> 128 Then Continue For
  Box 3 * (x% Mod WIDTH%), 3 * (x% \ WIDTH%), 2, 2, , Rgb(Lilac)
 Next
End Sub

Sub ready_steady_go()
 Local music_ptr_bak% = sound.music_ptr%, enabled_bak% = sound.enabled%
 sound.enable(sound.enabled% Xor sound.MUSIC_FLAG%)
 sound.play_fx(sound.FX_READY_STEADY_GO%())
 Local i%
 For i% = 0 To MAX_CYCLE_IDX%
  If Not (cycle.state%(i%) And &b11) Then cycle.draw(i%)
 Next
 Local msg$(2) = ("READY", "STEADY", "GO!")
 For i% = 0 To 2
  Text X_OFFSET%, Y_OFFSET% - 10, msg$(i%), "CM", 1, 2, Rgb(White)
  ? ;
  Pause 1280
  If i% = 2 Then Pause 240
  Text X_OFFSET%, Y_OFFSET% - 10, msg$(i%), "CM", 1, 2, Rgb(Black)
 Next
 sound.enable(enabled_bak%)
 sound.music_ptr% = music_ptr_bak%
End Sub

Function game_loop%()
 Local d%, i%, key%, next_frame% = Timer + frame_duration%, tf%, tmp$
 If 0 Then
  num_alive% = 0
  cycle.score%(0) = 3175
  cycle.score%(1) = 2175
  cycle.score%(2) = 1175
  cycle.score%(3) = 975
  score% = 3175
 EndIf
 Do While num_alive% > 0
  tf% = Timer
  Inc score%, 1
  If score% Mod 5 = 0 Then draw_score()
  For i% = 0 To MAX_CYCLE_IDX%
   If cycle.state%(i%) = STATE_DYING% Then cycle.dying(i%)
  Next
  For i% = 0 To MAX_CYCLE_IDX%
   If Not (cycle.state%(i%) And &b11) Then cycle.draw(i%)
  Next
  ? ;
  For i% = 0 To MAX_CYCLE_IDX%
   If Not (cycle.state%(i%) And &b11) Then cycle.pos%(i%) = cycle.nxt%(i%)
  Next
  For i% = 0 To MAX_CYCLE_IDX%
   cycle.current% = i%
   Call cycle.ctrl$(i%), key%
   d% = cycle.dir%(i%)
   If key% And ctrl.START Then game_loop% = on_quit%(cycle.ctrl$(i%))
   key% = key% And DIRECTION_MASK%
   If key% <> cycle.last_key%(i%) Then
    Select Case d%
     Case NORTH%, SOUTH%
      Select Case key% And HORIZONTAL_MASK%
       Case ctrl.LEFT : d% = WEST%
       Case ctrl.RIGHT : d% = EAST%
      End Select
     Case EAST%, WEST%
      Select Case key% And VERTICAL_MASK%
       Case ctrl.UP : d% = NORTH%
       Case ctrl.DOWN : d% = SOUTH%
      End Select
    End Select
    cycle.dir%(i%) = d%
    cycle.last_key%(i%) = key%
   EndIf
   cycle.nxt%(i%) = cycle.pos%(i%) + DIRECTIONS%(d%)
   If cycle.state%(i%) <> STATE_DEAD% Then cycle.check_collision(i%)
  Next
  Do While Timer < next_frame% : Loop
  next_frame% = Timer + frame_duration%
  If num_humans% > 0 Then Continue Do
  tmp$ = CTRLS_TO_POLL$(score% Mod (Bound(CTRLS_TO_POLL$(), 1) + 1))
  If ctrl.poll_single%(tmp$, A_START_SELECT) Then
   ui_ctrl$ = Choice(ui_ctrl$ = "", tmp$, ui_ctrl$)
   num_alive% = 0
   game_loop% = 1
  EndIf
 Loop
 ? ;
 Do While sound.is_playing%(sound.FX_FLAG%) : Loop
End Function

Sub draw_score()
 If num_humans% > 0 Or ((score% \ 100) And &b1) Then
  Local i%, s$ = Str$(score%, 5, 0, "0")
  For i% = 0 To MAX_CYCLE_IDX%
   If cycle.state%(i%) < STATE_DYING% Then
    Text SCORE_X%(i%), SCORE_Y%, s$, , 1, 1, cycle.colour%(i%)
   EndIf
  Next
  Exit Sub
 EndIf
 If score% Mod 100 = 95 Then
  Local i%, sc%
  Text Mm.HRes \ 2, SCORE_Y%, Space$(40), "C", 1, 1, Rgb(White)
  For i% = 0 To MAX_CYCLE_IDX%
   sc% = Choice(cycle.state%(i%) < STATE_DYING%, score%, (cycle.score%(i%) \ 5) * 5)
   Text SCORE_X%(i%), SCORE_Y%, Str$(sc%, 5, 0, "0"), , 1, 1, cycle.colour%(i%)
  Next
 Else
  Text Mm.HRes \ 2, SCORE_Y%, START_TEXT$, "C", 1, 1, Rgb(White)
 EndIf
End Sub

Sub wipe()
 Local y%
 sound.play_fx(sound.FX_WIPE%())
 For y% = 0 To Mm.VRes \ 2 Step 5
  Box Mm.HRes \ 2 - y% * 1.2, Mm.VRes \ 2 - y%, 2.4 * y%, 2 * y%, 5, Rgb(Cyan), Rgb(Black)
  ? ;
  Pause 30
 Next
 clear_display()
End Sub

Sub cycle.draw(idx%)
 Local p% = cycle.pos%(idx%), n% = cycle.nxt%(idx%), xn% = 3*(n% Mod WIDTH%), yn% = 3*(n%\WIDTH%)
 Local colour_% = cycle.colour%(idx%) * (cycle.state%(idx%) <> STATE_DYING%)
 Line 3*(p% Mod WIDTH%), 3*(p%\WIDTH%), xn%, yn%, 2, colour_%
 Box xn%, yn%, 2, 2, , colour_%
End Sub

Sub cycle.dying(idx%)
 cycle.draw(idx%)
 cycle.pos%(idx%) = cycle.nxt%(idx%)
 cycle.current% = idx%
 Local key%
 die_control(key%)
 Select Case key%
  Case ctrl.UP:    cycle.dir%(idx%) = NORTH%
  Case ctrl.DOWN:  cycle.dir%(idx%) = SOUTH%
  Case ctrl.LEFT:  cycle.dir%(idx%) = WEST%
  Case ctrl.RIGHT: cycle.dir%(idx%) = EAST%
 End Select
 cycle.nxt%(idx%) = cycle.pos%(idx%) + DIRECTIONS%(cycle.dir%(idx%))
 cycle.check_collision(idx%)
End Sub

Sub show_game_over()
 Local dummy%, i%, idx%(MAX_CYCLE_IDX%), j%, k%, winner%
 Sort cycle.score%(), idx%(), 1
 For i% = 0 To MAX_CYCLE_IDX%
  cycle.score%(i%) = (cycle.score%(i%) \ 5) * 5
 Next
 winner% = idx%(0)
 Box X_OFFSET% - 130, Y_OFFSET% - 50, 260, 80, 0, Rgb(Black), Rgb(Black)
 If cycle.ctrl_setting$(winner%) = "ai_control" Then
  Text X_OFFSET%, Y_OFFSET% - 10, " COMPUTER WINS ", "CM", 1, 2, cycle.colour%(winner%)
  ? ;
 Else
  Local txt$ = " PLAYER " + Str$(winner% + 1) + " WINS "
  Text X_OFFSET%, Y_OFFSET% - 25, txt$, "CM", 1, 2, cycle.colour%(winner%)
  Text X_OFFSET%, Y_OFFSET% + 5, " SCORE: " + Str$(cycle.score%(0)) + " ", "CM", 1, 2, cycle.colour%(winner%)
  ? ;
  Local multiplier% = -30 + 10 * difficulty%
  For i% = 0 To MAX_CYCLE_IDX%
   If cycle.ctrl_setting$(i%) <> "no_control" Then Inc multiplier%, 10
  Next
  If multiplier% > 1 Then
   Pause 1500
   Local s$ = " BONUS +" + Str$(multiplier%) + "% "
   Text X_OFFSET%, Y_OFFSET% + 5, s$ , "CM", 1, 2, RGB(White)
   ? ;
   Pause 1500
   Local bonus% = cycle.score%(0) * (1 + multiplier% / 100)
   Do While cycle.score%(0) < bonus%
    Select Case bonus% - cycle.score%(0)
     Case > 60 : Inc cycle.score%(0), 50
     Case > 15 : Inc cycle.score%(0), 10
     Case Else : Inc cycle.score%(0), 5
    End Select
    s$ = " SCORE: " + Str$(cycle.score%(0)) + " "
    Text X_OFFSET%, Y_OFFSET% + 5, s$, "CM", 1, 2, cycle.colour%(winner%)
    ? ;
    sound.play_fx(sound.FX_SELECT%())
    Pause 100
   Loop
  EndIf
 EndIf
 dummy% = twait%(5000)
 wipe()
 Local new_highscore%, player%
 For i% = 0 To 0
  player% = idx%(i%)
  If cycle.ctrl_setting$(player%) = "ai_control" Then Continue For
  For j% = 0 To Bound(highscr.values$())
   If cycle.score%(i%) > Val(Field$(highscr.values$(j%), 2)) Then
    For k% = Bound(highscr.values$(), 1) To j% Step -1
     If k% <> 0 Then highscr.values$(k%) = highscr.values$(k% - 1)
    Next
    highscr.values$(j%) = ", " + Str$(cycle.score%(i%))
    highscr.edit(player% + 1, j%, cycle.colour%(player%), cycle.ctrl_setting$(player%))
    new_highscore% = 1
    Exit For
   EndIf
  Next
 Next
 If new_highscore% Then highscr.save(HIGHSCORE_FILENAME$)
End Sub

Sub ai_control(x%)
 If x% < 0 Then Exit Sub
 Local idx% = cycle.current%
 Local d% = cycle.dir%(idx%)
 Local i% = Int(500 * Rnd)
 If i% < 4 Then d% = NEXT_DIR%(i% + idx%)
 Local nxt%
 For i% = 0 To MAX_CYCLE_IDX%
  nxt% = cycle.pos%(idx%) + DIRECTIONS%(d%)
  If Not Peek(Var arena%(), nxt%)) Then Exit For
  d% = NEXT_DIR%(i% + idx%)
 Next
 x% = COMPASS_TO_CTRL%(d%)
End Sub

Sub die_control(x%)
 If x% < 0 Then Exit Sub
 Local bits% = Peek(Var arena%(), cycle.pos%(cycle.current%)) >> 1
 If (bits% And &b11) = cycle.current% Then
  bits% = (bits% >> 2) And &b11
  x% = COMPASS_TO_CTRL%((bits% + 2) Mod 4)
 EndIf
End Sub

Sub no_control(x%)
 x% = 0
End Sub

Sub cycle.check_collision(idx%)
 If cycle.state%(idx%) = STATE_DYING% Then
  Poke Var arena%(), cycle.pos%(idx%), 0
  Local mask% = (idx% << 1) + 1
  If (Peek(Var arena%(), cycle.nxt%(idx%)) And mask%) <> mask% Then
   cycle.ctrl$(idx%) = "no_control"
   cycle.state%(idx%) = STATE_DEAD%
   cycle.pos%(idx%) = -1
  EndIf
  Exit Sub
 EndIf
 If Not Peek(Var arena%(), cycle.nxt%(idx%)) Then
  Poke Var arena%(), cycle.nxt%(idx%), (cycle.dir%(idx%) << 3) + (idx% << 1) + 1
  cycle.state%(idx%) = STATE_OK%
  Exit Sub
 EndIf
 Inc cycle.state%(idx%)
 If cycle.state%(idx%) < STATE_DYING% Then Exit Sub
 Inc num_alive%, -1
 If cycle.ctrl$(idx%) <> "ai_control" Then Inc num_humans%, -1
 cycle.ctrl$(idx%) = "die_control"
 cycle.nxt%(idx%) = cycle.pos%(idx%)
 cycle.score%(idx%) = score%
 sound.play_fx(sound.FX_DIE%())
End Sub

controller_data_gamemite:
Data 3, 1
Data "GAMEPAD",      "ctrl.gamemite", 1
Data "AI",           "ai_control",    0
Data "NONE",         "no_control",    0
highscore_data:
Data "MICKEY, 4000"
Data "MIKE, 3000"
Data "PETER, 2000"
Data "DAVEY, 1500"
Data "JOHN, 1250"
Data "PAUL, 1000"
Data "GEORGE, 800"
Data "RINGO, 600"
Data "MIDDLE, 400"
Data "MODAL, 200"
entertainer_music_data:
Data 792
Data 3
Data &h3135000034000033, &h3500253D00313D00, &h00283D00283D0025, &h2A3D00293D002935
Data &h3D002C3D002A3D00, &h002E3D002E00002C, &h314100304000303F, &h4100293F00313D00
Data &h002A3C002A410029, &h253D002C3F002C3F, &h3D002C3D00253D00, &h00313D00313D002C
Data &h3135000034003133, &h3500253D00313D00, &h00283D00283D0025, &h2A3D00293D002935
Data &h3D002C3D002A3D00, &h002E3D002E3D002C, &h273700263800263A, &h41002B3D00273A00
Data &h002E3F002E41002B, &h2C3F00273A00273D, &h3F002A3F002C3F00, &h00293F00293F002A
Data &h2535002734002733, &h3500313D00253D00, &h00283D00283D0031, &h2A3D00293D002935
Data &h3D002C3D002A3D00, &h002E3D002E00002C, &h314100304000303F, &h4100293F00313D00
Data &h002A3C002A410029, &h313D002C3F002C3F, &h3D002C3D00313D00, &h00250000253D002C
Data &h3D4100253F00253D, &h41313D3F00003D31, &h00003D2F3B410000, &h3A4100003D2F3B3F
Data &h412E3A3F00003D2E, &h00003D2D39410000, &h384100003D2D393F, &h412C383F00003D2C
Data &h2C383C2C38410000, &h003D00003F00003F, &h3D202C3D00003D00, &h00003525313D202C
Data &h3538000037000036, &h3800363A31353831, &h2C00352C35380035, &h3538000037000036
Data &h3800363A31353831, &h2C00412C35380035, &h003A00003800003D, &h3F2A003D2A003C2A
Data &h2C003F2C00412A00, &h00382C003F2C003D, &h382C003825003825, &h3100353100382C00
Data &h3538000037000036, &h3800363A31353831, &h2C00352C35380035, &h3538000037000036
Data &h3800363A31353831, &h2C00382C35380035, &h003C32003B32003A, &h3C33000033003C33
Data &h27003A27003C3300, &h0038270033270037, &h382C00380000382C, &h2E00352E00380000
Data &h3538300037300036, &h3800363A31353831, &h2C00352C35380035, &h3538000037000036
Data &h3800363A31353831, &h2C00412C35380035, &h003A00003800003D, &h3F2A003D2A003C2A
Data &h2C003F2C00412A00, &h003D2C003F2C003D, &h3D2C003D25003D25, &h31003831003D2C00
Data &h363D000038000037, &h3D2A363A00003D2A, &h00003A2B373D0000, &h383800003A2B373D
Data &h442C384100003D2C, &h0000412935440000, &h363A00003829353D, &h3D2A363D00003A2A
Data &h00003F2C38410000, &h3D3D00003F2C383F, &h3D2C383D313D3D31, &h25313D25313D2C38
Data &hFFFF000000000000, &hFFFFFFFFFFFFFFFF, &hFFFFFFFFFFFFFFFF
black_white_rag_music_data:
Data 888
Data 3
Data &h3440003A41003A41, &h3E00354100354100, &h00333F00333F0026, &h303C00303C002F3B
Data &h35002C38002A3600, &h0027330027330029, &h333F00330000333F, &h0000273F00330000
Data &h00330000333F0027, &h273F00273F00273F, &h3F00003F00270000, &h00003E00003E0000
Data &h314216273F16273D, &h4211223F27313D27, &h27313F27313D1122, &h283F1B273D1B2742
Data &h411D29421C28421C, &h1E2A3D1E2A3F1D29, &h3041202C3C202C3B, &h4120273C27303B27
Data &h27303C27303B2027, &h2D3C202C3B202C41, &h3F222E41212D4121, &h24303824303C222E
Data &h313C222E3D222E3D, &h3C1B273D27313D27, &h27313D27313D1B27, &h3139222E3A222E3A
Data &h391B273A27313A27, &h27313A27313A1B27, &h2400002000000000, &h0000000000270000
Data &h003300003000002C, &h003C000038000000, &h4400000000003F00, &h00004B0000480000
Data &h31421B273F1B273D, &h4216223F27313D27, &h27313F27313D1622, &h283F1B273D1B2742
Data &h411D29421C28421C, &h1E2A3D1E2A3F1D29, &h3041202C3C202C3B, &h411B273C27303B27
Data &h27303C27303B1B27, &h2D3C202C3B202C41, &h3F222E41212D4121, &h24303C24303D222E
Data &h3339212D35212D41, &h3F1D294129334129, &h29333C29333D1D29, &h313A222E39222E3A
Data &h3F1D294129314129, &h29313A29313D1D29, &h30381B27331B273C, &h381B273C27303C27
Data &h1E2A3A1E2A3A1B27, &h3338303338303338, &h443C3F4430333830, &h0000000000003C3F
Data &h314216273F16273D, &h4211223F27313D27, &h27313F27313D1122, &h283F1B273D1B2742
Data &h411D29421C28421C, &h1E2A3D1E2A3F1D29, &h3041202C3C202C3B, &h4120273C27303B27
Data &h27303C27303B2027, &h2D3C202C3B202C41, &h3F222E41212D4121, &h24303824303C222E
Data &h313C222E3D222E3D, &h3C1B273D27313D27, &h27313D27313D1B27, &h3139222E3A222E3A
Data &h391B273A27313A27, &h27313A27313A1B27, &h2400002000000000, &h0000000000270000
Data &h003300003000002C, &h003C000038000000, &h4400000000003F00, &h00004B0000480000
Data &h31421B273F1B273D, &h4216223F27313D27, &h27313F27313D1622, &h283F1B273D1B2742
Data &h411D29421C28421C, &h1E2A3D1E2A3F1D29, &h3041202C3C202C3B, &h411B273C27303B27
Data &h27303C27303B1B27, &h2D3C202C3B202C41, &h3F222E41212D4121, &h24303C24303D222E
Data &h3339212D35212D41, &h3F1D294129334129, &h29333C29333D1D29, &h313A222E39222E3A
Data &h3F1D294129314129, &h29313A29313D1D29, &h30381B27331B273C, &h381B273C27303C27
Data &h1E2A3A1E2A3A1B27, &h3338303338303338, &h443C3F4430333830, &h0000000000003C3F
Data &hFFFFFFFFFFFFFFFF, &hFFFFFFFFFFFFFFFF, &hFFFFFFFFFFFFFFFF
