  ' sat_riseset.bas         March 23, 2017
  
  ' rise and set of Earth satellites
  
  ' SGP4 orbit propagation method
  
  ' Micromite eXtreme version
  
  '''''''''''''''''''''''''''
  
  option default float
  
  option base 1
  
  ' dimension global arrays and variables
  
  dim rsat(3), vsat(3)
  
  dim month$(12) as string
  
  dim cline1 as string, cline2 as string, cline3 as string
  
  dim xnut(11, 13), trr, obliq, elmin
  
  dim jdsaved, jdprint, obslat, obslong, obsalt
  
  dim jdutci, i as integer, j as integer
  
  dim cmonth, cday, cyear, ndays
  
  dim utc_hr, utc_min, utc_sec
  
  dim iflag as integer, srange, rangerate, azdot, eldot
  
  ' global constants
  
  const pi2 = 2.0 * pi, pidiv2 = 0.5 * pi, dtr = pi / 180.0, rtd = 180.0 / pi
  
  const seccon = 206264.8062470964

  ' astronomical unit (kilometers)
  
  const aunit = 149597870.691
    
  ' earth gravitational constant (kilometers^3/second^2)
  
  const mu = 398600.436233
  
  ' equatorial radius of the earth (kilometers)
  
  const reqm = 6378.1363
  
  ' earth flattening factor (nd)
  
  const flat = 1.0 / 298.257
  
  ' earth inertial rotation rate (radians/second)
  
  const omegae = 7.292115486e-5
  
  ' SGP4 utility constants
  
  const e6a = 0.000001
  const qo = 120.0
  const so = 78.0
  const tothrd = 2.0 / 3.0
  const x3pio2 = 3.0 * pi / 2.0
  const j2 = 0.0010826158
  const j3 = -0.00000253881
  const j4 = -0.00000165597
  const xke = 0.0743669161
  const xkmper = 6378.135
  const xmnpda = 1440.0
  const ae = 1.0
  const ck2 = 0.5 * j2 * ae * ae
  const ck4 = -0.375 * j4 * ae * ae * ae * ae
  const ssgp = ae * (1.0 + so / xkmper)
  
  ' sgp4 globals
  
  dim jdtle, xmo, xnodeo, omegao, eo, xincl
  
  dim xno, xndt2o, xndd6o, bstar
  
  dim qoms2t = ((qo - so) * ae / xkmper) ^ 4
  
  dim xmdot, omgdot, xnodot, xlcof, aycof, cosio, sinio
  
  dim aodp, xnodp, sinmo, delmo, eta, omgcof, xmcof, xnodcf, isimp
  
  dim c1, c4, c5, d2, d3, d4, t2cof, t3cof, t4cof, t5cof
  
  dim x1mth2, x3thm1, x7thm1
  
  ' read subset of IAU2000 nutation data
  
  for j = 1 to 13
    
    for i = 1 to 11
      
      read xnut(i, j)
      
    next i
    
  next j
  
  data  0,  0, 0,  0, 1, -172064161, -174666, 92052331,  9086,  33386, 15377
  data  0,  0, 2, -2, 2,  -13170906,   -1675,  5730336, -3015, -13696, -4587
  data  0,  0, 2,  0, 2,   -2276413,    -234,   978459,  -485,   2796,  1374
  data  0,  0, 0,  0, 2,    2074554,     207,  -897492,   470,   -698,  -291
  data  0,  1, 0,  0, 0,    1475877,   -3633,    73871,  -184,  11817, -1924
  data  0,  1, 2, -2, 2,    -516821,    1226,   224386,  -677,   -524,  -174
  data  1,  0, 0,  0, 0,     711159,      73,    -6750,     0,   -872,   358
  data  0,  0, 2,  0, 1,    -387298,    -367,   200728,    18,    380,   318
  data  1,  0, 2,  0, 2,    -301461,     -36,   129025,   -63,    816,   367
  data  0, -1, 2, -2, 2,     215829,    -494,   -95929,   299,    111,   132
  data  0,  0, 2, -2, 1,     128227,     137,   -68982,    -9,    181,    39
  data -1,  0, 2,  0, 2,     123457,      11,   -53311,    32,     19,    -4
  data -1,  0, 0,  2, 0,     156994,      10,    -1235,     0,   -168,    82
  
  ' calendar months
  
  month$(1) = "January"
  month$(2) = "February"
  month$(3) = "March"
  month$(4) = "April"
  month$(5) = "May"
  month$(6) = "June"
  month$(7) = "July"
  month$(8) = "August"
  month$(9) = "September"
  month$(10) = "October"
  month$(11) = "November"
  month$(12) = "December"
  
  ''''''''''
  ' TLE data
  ''''''''''
  
  cline1 = "ISS (ZARYA)"
  
  cline2 = "1 25544U 98067A   17077.64837083  .00001665  00000-0  32320-4 0  9993"
  
  cline3 = "2 25544  51.6420 129.6155 0006880 308.3866 194.6632 15.54222906 47676"
  
  ' read the TLE data and extract information
  
  readtle
  
  ' initialization call to SGP4
  
  iflag = 1
  
  tsince = 0.0
  
  sgp4(tsince, rsat(), vsat())
  
  ''''''''''''''''''
  ' begin simulation
  ''''''''''''''''''
  
  print " "
  print "rise and set of Earth satellites"
  print "================================"
  print " "
  
  readtle
  
  print " "
  print "TLE epoch"
  print " "
  
  jd2str(jdtle)
  
  print " "
  print cline1
  print cline2
  print cline3
  
  ' request initial calendar date (month, day, year)
  
  getdate(cmonth, cday, cyear)
  
  ' request initial UTC time (hours, minutes, seconds)
  
  getutc(utc_hr, utc_min, utc_sec)
  
  ' request observer coordinates
  
  print " "
  
  observer(obslat, obslong, obsalt)
  
  ' compute initial utc julian day
  
  julian(cmonth, cday, cyear, jdutc)
  
  jdutci = jdutc + utc_hr / 24.0 + utc_min / 1440.0 + utc_sec / 86400.0
  
  ' request search duration (days)
  
  print " "
  
  print "please input the number of days to simulate"
  
  input ndays
  
  ' request minimum elevation constraint (degrees)
  
  print " "
  
  print "please input the minimum elevation angle (degrees)"
  
  input elmin
  
  elmin = dtr * elmin
  
  print " "
  print "searching for rise and set conditions ..."
  print " "
  
  ' set orbital period (days)
  
  period = xno
  
  ' initialize search parameters
  
  ti = 0.0
  
  tf = ndays
  
  dt = period / 4.0
  
  dtsml = 10.0 / 86400.0
  
  ' find satellite rise/set conditions
  
  rs_event(ti, tf, dt, dtsml)
  
end
  
  '''''''''''''''
  '''''''''''''''
  
sub rsfunc(x, fx)
  
  ' rise/set objective function
  
  ' input
  
  '  x = current search value
  
  ' output
  
  '  fx = current objective function
  
  ''''''''''''''''''''''''''''''''''
  
  local jdutc, tsince, rsat(3), vsat(3), gast, azim, elev
  
  iflag = 0
  
  ' time since initialization (minutes)
  
  tsince = 1440.0 * ((jdutci - jdtle) + x)
  
  ' compute eci position vector of the satellite
  
  sgp4(tsince, rsat(), vsat())
  
  ' compute topocentric coordinates of the satellite
  
  jdutc = jdutci + x
  
  gast2(jdutc, gast)
  
  eci2topo(gast, rsat(), vsat(), azim, elev)
  
  ' evaluate objective function (radians)
  
  fx = -elev + elmin
  
end sub
  
  ''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''
  
sub rs_event (ti, tf, dt, dtsml)
  
  ' predict rise/set events
  
  ' input
  
  '  ti    = initial simulation time
  '  tf    = final simulation time
  '  dt    = step size used for bounding minima
  '  dtsml = small step size used to determine whether
  '          the function is increasing or decreasing
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL tolm, iend as integer
  
  local fmin1, tmin1
  
  LOCAL ftemp, df, dflft
  
  local el, er
  
  LOCAL t, ft
  
  local iter1 as integer, iter2 as integer
  
  local iter3 as integer
  
  ' initialization
  
  tolm = 1.0e-8
  
  iend = 0
  
  ' check the initial time for a minimum
  
  rsfunc(ti, fmin1)
  
  tmin1 = ti
  
  rsfunc(ti + dtsml, ftemp)
  
  df = ftemp - fmin1
  
  dflft = df
  
  el = ti
  
  er = el
  
  t = ti
  
  ' if the slope is positive and the minimum is
  ' negative, calculate event conditions at the initial time
  
  if (df > 0.0 and fmin1 < 0.0) then
    
    events1(ti, tf, tmin1)
    
  end if
  
  for iter1 = 1 to 1000
    
    ' find where function first starts decreasing
    
    for iter2 = 1 to 1000
      
      if (df <= 0.0) then
        
        exit for
        
      end if
      
      t = t + dt
      
      if (t >= tf) then
        
        ' check final time for a minimum
        
        if (iend = 1) then exit sub
        
        rsfunc(tf, fmin1)
        
        rsfunc(tf - dtsml, ftemp)
        
        df = fmin1 - ftemp
        
        ' set minimum time to final simulation time
        
        tmin1 = tf
        
        if (df < 0.0) then
          
          ' if both the slope and minimum are negative,
          ' calculate the event conditions at the final
          ' simulation time
          
          if (fmin1 < 0.0) then
            
            events1(ti, tf, tmin1)
            
          end if
          
          ' otherwise, we're finished
          
          exit sub
          
        end if
        
        if (dflft > 0.0) then exit sub
        
        er = tf
        
        iend = 1
        
      end if
      
      rsfunc(t, ft)
      
      rsfunc(t - dtsml, ftemp)
      
      df = ft - ftemp
      
    next iter2
    
    ' function decreasing - find where function
    ' first starts increasing
    
    for iter3 = 1 to 1000
      
      el = t
      
      dflft = df
      
      t = t + dt
      
      if (t >= tf) then
        
        ' check final time for a minimum
        
        if (iend = 1) then exit sub
        
        rsfunc(tf, fmin1)
        
        rsfunc(tf - dtsml, ftemp)
        
        df = fmin1 - ftemp
        
        ' set minimum time to final simulation time
        
        tmin1 = tf
        
        if (df < 0.0) then
          
          ' if both the slope and minimum are negative,
          ' calculate the event conditions at the final
          ' simulation time
          
          if (fmin1 < 0.0) then
            
            events1(ti, tf, tmin1)
            
          end if
          
          ' otherwise, we're finished
          
          exit sub
          
        end if
        
        if (dflft > 0.0) then exit sub
        
        er = tf
        
        iend = 1
        
      end if
      
      rsfunc(t, ft)
      
      rsfunc(t - dtsml, ftemp)
      
      df = ft - ftemp
      
      if (df > 0.0) then exit for
      
    next iter3
    
    er = t
    
    ' calculate minimum using Brent's method
    
    minima(el, er, tolm, tmin1, fmin1)
    
    el = er
    
    dflft = df
    
    ' if the minimum is negative,
    ' calculate event conditions for this minimum
    
    if (fmin1 < 0.0) then
      
      events1(ti, tf, tmin1)
      
      t = trr
      
    end if
    
  next iter1
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub events1 (ti, tf, topt)
  
  ' compute and display rise/set events
  
  ' input
  
  '  ti   = initial simulation time
  '  tf   = final simulation time
  '  topt = extrema time
  
  ''''''''''''''''''''''
  
  ' define root-bracketing and root-finding control parameters
  
  LOCAL factor = 0.25            ' geometric acceleration factor
  
  LOCAL dxmax = 300.0 / 86400.0  ' rectification interval
  
  LOCAL rtol = 1.0e-8            ' root-finding convergence tolerance
  
  LOCAL t1in, t2in
  
  local t1out, t2out
  
  LOCAL troot, froot, jdate
  
  local iter as integer
  
  ' compute and display rise conditions
  
  t1in = topt
  
  t2in = t1in - 10.0 / 86400.0
  
  broot(t1in, t2in, factor, dxmax, t1out, t2out)
  
  realroot(t1out, t2out, rtol, troot, froot)
  
  ' set to initial time if before ti
  
  if (troot < ti) then
    
    troot = ti
    
    rsfunc(ti, froot)
    
  end if
  
  jdate = jdutci + troot
  
  rsprint(1, jdate)
  
  ' display maximum elevation conditions
  
  jdate = jdutci + topt
  
  rsprint(2, jdate)
  
  ' compute and display set conditions
  
  t2in = t1in + 10.0 / 86400.0
  
  broot(t1in, t2in, factor, dxmax, t1out, t2out)
  
  realroot(t1out, t2out, rtol, troot, froot)
  
  ' set to final time if after tf
  
  if (troot > tf) then
    
    troot = tf
    
    rsfunc(tf, froot)
    
  end if
  
  trr = troot
  
  jdate = jdutci + troot
  
  rsprint(3, jdate)
  
END sub
  
  '''''''''''''''''''''''
  '''''''''''''''''''''''
  
sub rsprint(iflag, jdutc)
  
  ' print rise and set conditions
  
  '''''''''''''''''''''''''''''''
  
  local rsat(3), vsat(3)
  
  LOCAL dms$ as string
  
  local deltat, gast, azimuth, elevation
  
  if (iflag = 1) then
    
    print " "
    print "rise conditions"
    PRINT "---------------"
    print " "
    
    jdprint = jdutc
    
  end if
  
  if (iflag = 2) then
    
    print " "
    print "maximum elevation conditions"
    PRINT "----------------------------"
    print " "
    
  end if
  
  if (iflag = 3) then
    
    print " "
    print "set conditions"
    PRINT "--------------"
    print " "
    
  end if
  
  ' display UTC julian date
  
  jd2str(jdutc)
  
  PRINT " "
  
  print "UTC julian day     ", str$(jdutc, 0, 8)
  
  ' compute and display topocentric coordinates of the satellite
  
  gast2(jdutc, gast)
  
  tsince = 1440.0 * (jdutc - jdtle)
  
  sgp4(tsince, rsat(), vsat())
  
  eci2topo(gast, rsat(), vsat(), azimuth, elevation)
  
  PRINT " "
  
  print "topocentric coordinates"
  
  PRINT " "
  
  deg2str(rtd * azimuth, dms$)
  
  print "azimuth angle      ", dms$
  
  PRINT " "
  
  deg2str(rtd * elevation, dms$)
  
  print "elevation angle    ", dms$
  
  print " "
  
  print "slant range        ", str$(srange, 0, 4), " kilometers"
  
  print " "
  
  print "azimuth rate       ", str$(rtd * azdot, 0, 4), " degrees/second"
  
  print " "
  
  print "elevation rate     ", str$(rtd * eldot, 0, 4), " degrees/second"
  
  print " "
  
  print "range rate         ", str$(rangerate, 0, 4), " kilometers/second"
  
  shadow(jdutc, rsat())
  
  ' determine and display event duration
  
  if (iflag = 3) then
    
    deltat = 24.0 * (jdutc - jdprint)
    
    PRINT " "
    
    hrs2str(deltat)
    
    print " "
    
  end if
  
END sub
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
sub sgp4 (tsince, rs(), vs())
  
  ' SGP4 orbit propagation
  
  ' input
  
  '  tsince = time since initialization (minutes)
  
  ' output
  
  '  rs = position vector (kilometers)
  '  vs = velocity vector (kilometers/second)
  
  '''''''''''''''''''''''''''''''''''''''''''
  
  if (iflag = 1) then
    
    ' initialization
    
    a1 = (xke / xno) ^ tothrd
    
    cosio = cos(xincl)
    
    theta2 = cosio * cosio
    
    x3thm1 = 3.0 * theta2 - 1.0
    
    eosq = eo * eo
    
    betao2 = 1.0 - eosq
    
    betao = sqr(betao2)
    
    del1 = 1.5 * ck2 * x3thm1 / (a1 * a1 * betao * betao2)
    
    ao = a1 * (1.0 - del1 * (0.5 * tothrd + del1 * (1.0 + 134.0 / 81.0 * del1)))
    
    delo = 1.5 * ck2 * x3thm1 / (ao * ao * betao * betao2)
    
    xnodp = xno / (1.0 + delo)
    
    aodp = ao / (1.0 - delo)
    
    isimp = 0
    
    if ((aodp * (1.0 - eo) / ae) < (220.0 / xkmper + ae)) then
      
      isimp = 1
      
    end if
    
    s4 = ssgp
    
    qoms24 = qoms2t
    
    perige = (aodp * (1.0 - eo) - ae) * xkmper
    
    if (perige >= 156.0) then
      
      ' null
      
    else
      
      s4 = perige - 78.0
      
      if (perige > 98.0) then
        
        ' null
        
      else
        
        s4 = 20.0
        
      end if
      
      qoms24 = ((120.0 - s4) * ae / xkmper) ^ 4
      
      s4 = s4 / xkmper + ae
      
    end if
    
    pinvsq = 1.0 / (aodp * aodp * betao2 * betao2)
    
    tsi = 1.0 / (aodp - s4)
    
    eta = aodp * eo * tsi
    
    etasq = eta * eta
    
    eeta = eo * eta
    
    psisq = abs(1.0 - etasq)
    
    coef = qoms24 * tsi ^ 4
    
    coef1 = coef / psisq ^ 3.5
    
    c2 = coef1 * xnodp * (aodp * (1.0 + 1.5 * etasq + eeta * (4.0 + etasq)) + 0.75 * ck2 * tsi / psisq * x3thm1 * (8.0 + 3.0 * etasq * (8.0 + etasq)))
    
    c1 = bstar * c2
    
    sinio = sin(xincl)
    
    a3ovk2 = -j3 / ck2 * ae ^ 3
    
    c3 = coef * tsi * a3ovk2 * xnodp * ae * sinio / eo
    
    x1mth2 = 1.0 - theta2
    
    c4 = 2.0 * xnodp * coef1 * aodp * betao2
    
    tmp1 = eta * (2.0 + 0.5 * etasq) + eo * (0.5 + 2.0 * etasq)
    
    tmp2 = 1.0 - 2.0 * eeta + etasq * (1.5 - 0.5 * eeta)
    
    tmp3 = 0.75 * x1mth2 * (2.0 * etasq - eeta * (1.0 + etasq))
    
    c4 = c4 * (tmp1 - 2.0 * ck2 * tsi / (aodp * psisq) * (-3.0 * x3thm1 * tmp2 + tmp3 * cos(2.0 * omegao)))
    
    c5 = 2.0 * coef1 * aodp * betao2 * (1.0 + 2.75 * (etasq + eeta) + eeta * etasq)
    
    theta4 = theta2 * theta2
    
    temp1 = 3.0 * ck2 * pinvsq * xnodp
    temp2 = temp1 * ck2 * pinvsq
    temp3 = 1.25 * ck4 * pinvsq * pinvsq * xnodp
    
    xmdot = xnodp + 0.5 * temp1 * betao * x3thm1 + 0.0625 * temp2 * betao * (13.0 - 78.0 * theta2 + 137.0 * theta4)
    
    x1m5th = 1.0 - 5.0 * theta2
    
    omgdot = -0.5 * temp1 * x1m5th + 0.0625 * temp2 * (7.0 - 114.0 * theta2 + 395.0 * theta4)
    
    omgdot = omgdot + temp3 * (3.0 - 36.0 * theta2 + 49.0 * theta4)
    
    xhdot1 = -temp1 * cosio
    
    xnodot = xhdot1 + (0.5 * temp2 * (4.0 - 19.0 * theta2) + 2.0 * temp3 * (3.0 - 7.0 * theta2)) * cosio
    
    omgcof = bstar * c3 * cos(omegao)
    
    xmcof = -tothrd * coef * bstar * ae / eeta
    
    xnodcf = 3.5 * betao2 * xhdot1 * c1
    
    t2cof = 1.5 * c1
    
    xlcof = 0.125 * a3ovk2 * sinio * (3.0 + 5.0 * cosio) / (1.0 + cosio)
    
    aycof = 0.25 * a3ovk2 * sinio
    
    delmo = (1.0 + eta * cos(xmo)) ^ 3
    
    sinmo = sin(xmo)
    
    x7thm1 = 7.0 * theta2 - 1.0
    
    if (isimp = 1) then
      
      ' null
      
    else
      
      c1sq = c1 * c1
      
      d2 = 4.0 * aodp * tsi * c1sq
      
      temp = d2 * tsi * c1 / 3.0
      
      d3 = (17.0 * aodp + s4) * temp
      
      d4 = 0.5 * temp * aodp * tsi * (221.0 * aodp + 31.0 * s4) * c1
      
      t3cof = d2 + 2.0 * c1sq
      
      t4cof = 0.25 * (3.0 * d3 + c1 * (12.0 * d2 + 10.0 * c1sq))
      
      t5cof = 0.2 * (3.0 * d4 + 12.0 * c1 * d3 + 6.0 * d2 * d2 + 15.0 * c1sq * (2.0 * d2 + c1sq))
      
    end if
    
  end if
  
  xmdf = xmo + xmdot * tsince
  
  omgadf = omegao + omgdot * tsince
  
  xnoddf = xnodeo + xnodot * tsince
  
  omega = omgadf
  
  xmp = xmdf
  
  tsq = tsince * tsince
  
  xnode = xnoddf + xnodcf * tsq
  
  tempa = 1.0 - c1 * tsince
  
  tempe = bstar * c4 * tsince
  
  templ = t2cof * tsq
  
  if (isimp = 1) then
    
    ' null
    
  else
    
    delomg = omgcof * tsince
    
    delm = xmcof * ((1 + eta * cos(xmdf)) ^ 3 - delmo)
    
    temp = delomg + delm
    
    xmp = xmdf + temp
    
    omega = omgadf - temp
    
    tcube = tsq * tsince
    
    tfour = tsince * tcube
    
    tempa = tempa - d2 * tsq - d3 * tcube - d4 * tfour
    
    tempe = tempe + bstar * c5 * (sin(xmp) - sinmo)
    
    templ = templ + t3cof * tcube + tfour * (t4cof + tsince * t5cof)
    
  end if
  
  a = aodp * tempa ^ 2
  
  e = eo - tempe
  
  xl = xmp + omega + xnode + xnodp * templ
  
  beta = sqr(1.0 - e * e)
  
  xn = xke / a ^ 1.5
  
  axn = e * cos(omega)
  
  temp = 1.0 / (a * beta * beta)
  
  xll = temp * xlcof * axn
  
  aynl = temp * aycof
  
  xlt = xl + xll
  
  ayn = e * sin(omega) + aynl
  
  capu = modulo(xlt - xnode)
  
  temp2 = capu
  
  ' solve Kepler's equation
  
  for i% = 1 to 10
    
    sinepw = sin(temp2)
    cosepw = cos(temp2)
    
    temp3 = axn * sinepw
    temp4 = ayn * cosepw
    temp5 = axn * cosepw
    temp6 = ayn * sinepw
    
    epw = (capu - temp4 + temp3 - temp2) / (1.0 - temp5 - temp6) + temp2
    
    ' check for convergence
    
    if (abs(epw - temp2) <= e6a) then
      
      exit for
      
    end if
    
    temp2 = epw
    
  next i%
  
  ecose = temp5 + temp6
  
  esine = temp3 - temp4
  
  elsq = axn * axn + ayn * ayn
  
  temp = 1.0 - elsq
  
  pl = a * temp
  
  r = a * (1.0 - ecose)
  
  temp1 = 1.0 / r
  
  rdot = xke * sqr(a) * esine * temp1
  
  rfdot = xke * sqr(pl) * temp1
  
  temp2 = a * temp1
  
  betal = sqr(temp)
  
  temp3 = 1.0 / (1.0 + betal)
  
  cosu = temp2 * (cosepw - axn + ayn * esine * temp3)
  
  sinu = temp2 * (sinepw - ayn - axn * esine * temp3)
  
  u = atan3(sinu, cosu)
  
  sin2u = 2.0 * sinu * cosu
  cos2u = 2.0 * cosu * cosu - 1.0
  
  temp = 1.0 / pl
  
  temp1 = ck2 * temp
  
  temp2 = temp1 * temp
  
  rk = r * (1.0 - 1.5 * temp2 * betal * x3thm1) + 0.5 * temp1 * x1mth2 * cos2u
  
  uk = u - 0.25 * temp2 * x7thm1 * sin2u
  
  xnodek = xnode + 1.5 * temp2 * cosio * sin2u
  
  xinck = xincl + 1.5 * temp2 * cosio * sinio * cos2u
  
  rdotk = rdot - xn * temp1 * x1mth2 * sin2u
  
  rfdotk = rfdot + xn * temp1 * (x1mth2 * cos2u + 1.5 * x3thm1)
  
  sinuk = sin(uk)
  cosuk = cos(uk)
  
  sinik = sin(xinck)
  cosik = cos(xinck)
  
  sinnok = sin(xnodek)
  cosnok = cos(xnodek)
  
  xmx = -sinnok * cosik
  xmy = cosnok * cosik
  
  ux = xmx * sinuk + cosnok * cosuk
  uy = xmy * sinuk + sinnok * cosuk
  uz = sinik * sinuk
  
  vx = xmx * cosuk - cosnok * sinuk
  vy = xmy * cosuk - sinnok * sinuk
  vz = sinik * cosuk
  
  ' position vector
  
  rs(1) = rk * ux * xkmper
  rs(2) = rk * uy * xkmper
  rs(3) = rk * uz * xkmper
  
  ' velocity vector
  
  vconst = xkmper / ae * xmnpda / 86400.0
  
  vs(1) = vconst * (rdotk * ux + rfdotk * vx)
  vs(2) = vconst * (rdotk * uy + rfdotk * vy)
  vs(3) = vconst * (rdotk * uz + rfdotk * vz)
  
end sub
  
  '''''''''''''''''''''''
  '''''''''''''''''''''''
  
sub shadow(jdate, rsat())

  ' determine shadow conditions subroutine
  
  ' input
  
  '  jdate  = julian day
  '  rsat() = eci position vector of satellite
  
  ''''''''''''''''''''''''''''''''''''''''''''
  
  local a, b, c, d, e, u, v, w, xx, yy
  
  local rsun(3), usat(3), usun(3)
  
  local rmsun, rmsat, rmatm

  ' compute ECI position vector of the sun
    
  sun(jdate, rsun())
  
  rmsun = vecmag(rsun())
  
  ' geocentric radius of the satellite
  
  rmsat = vecmag(rsat())
  
  ' compute unit vectors
  
  uvector(rsun(), usun())
  
  uvector(rsat(), usat())
  
  ' determine shadow conditions
  
  a = usat(2) * usun(3) - usat(3) * usun(2)
  
  b = usat(3) * usun(1) - usat(1) * usun(3)
  
  c = usat(1) * usun(2) - usat(2) * usun(1)
  
  d = sqr(a * a + b * b + c * c)
  
  e = vdot(usat(), usun())

  ' umbra shadow angle
    
  u = asin(0.00460983743 / rmsun)
  
  ' penumbra shadow angle
  
  p = asin(0.0046951089 / rmsun)
  
  if (e > 0.0) then
    
    q = -d
    
  else
    
    q = d
    
  end if
  
  rmatm = reqm + 90.0
  
  b = asin(rmatm / rmsat)
  
  v = b - u
  w = b + p
  
  xx = sin(v)
  yy = sin(w)
  
  ' determine shadow conditions
  
  if (q <= yy and q > xx) then
    
    ' penumbra
    
    print " "
    print "penumbra shadow"
    
  else if (q <= xx and q >= 0.0) then
    
    ' umbra
    
    print " "
    print "umbra shadow"
    
  else
    
    ' sunlight
    
  end if
  
end sub

  ''''''''''''''''''''
  ''''''''''''''''''''
  
sub sun(jdate, rsun())
  
  ' solar ephemeris
  
  ' input
  
  '  jdate = julian day
  
  ' output
  
  '  rsun() = eci position vector of the sun (kilometers)
  
  ' note
  
  '  coordinates are inertial, geocentric,
  '  equatorial and true-of-date
  
  '''''''''''''''''''''''''''''''
  
  local djd, t, plon, obliq
  
  local gs, lm, ls, g2, g4, g5, rm
  
  ' time arguments
  
  djd = jdate - 2451545.0
  
  t = (djd / 36525.0) + 1.0
  
  ' fundamental arguments (radians)
  
  gs = r2r(0.993126 + 0.0027377785 * djd)
  
  lm = r2r(0.606434 + 0.03660110129 * djd)
  
  ls = r2r(0.779072 + 0.00273790931 * djd)
  
  g2 = r2r(0.140023 + 0.00445036173 * djd)
  
  g4 = r2r(0.053856 + 0.00145561327 * djd)
  
  g5 = r2r(0.056531 + 0.00023080893 * djd)
  
  rm = r2r(0.347343 - 0.00014709391 * djd)
  
  ' geocentric, ecliptic longitude of the sun (radians)
  
  plon = 6910.0 * sin(gs) + 72.0 * sin(2.0 * gs) - 17.0 * t * sin(gs)
  
  plon = plon - 7.0 * cos(gs - g5) + 6.0 * sin(lm - ls) + 5.0 * sin(4.0 * gs - 8.0 * g4 + 3.0 * g5)
  
  plon = plon - 5.0 * cos(2.0 * (gs - g2)) - 4.0 * (sin(gs - g2) - cos(4.0 * gs - 8.0 * g4 + 3.0 * g5))
  
  plon = plon + 3.0 * (sin(2.0 * (gs - g2)) - sin(g5) - sin(2 * (gs - g5)))
  
  plon = ls + atr * (plon - 17.0 * sin(rm))
  
  ' obliquity of the ecliptic (radians)
  
  obliq = atr * (84428.0 - 47.0 * t + 9.0 * cos(rm))
  
  ' geocentric, equatorial right ascension and declination (radians)
  
  a = sin(plon) * cos(obliq)
  
  b = cos(plon)
  
  rasc = atan3(a, b)
  
  decl = asin(sin(obliq) * sin(plon))
  
  rmsun = aunit * (1.00014 - 0.01675 * cos(gs) - 0.00014 * cos(2.0 * gs))
  
  rsun(1) = rmsun * cos(rasc) * cos(decl)
  
  rsun(2) = rmsun * sin(rasc) * cos(decl)
  
  rsun(3) = rmsun * sin(decl)
  
end sub
  
  ''''''''''''''''''''''
  ''''''''''''''''''''''
  
function r2r(x) as float
  
  ' revolutions to radians function
  
  r2r = pi2 * (x - fix(x))
  
end function
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub minima(a, b, tolm, xmin, fmin)
  
  ' one-dimensional minimization
  
  ' Brent's method
  
  ' input
  
  '  a    = initial x search value
  '  b    = final x search value
  '  tolm = convergence criterion
  
  ' output
  
  '  xmin = minimum x value
  
  ' note
  
  '  user-defined objective subroutine
  '  coded as usr_func(x, fx)
  
  ' remember: a maximum is simply a minimum
  '           with a negative attitude!
  
  '''''''''''''''''''''''''''''''''''''
  
  ' machine epsilon
  
  LOCAL epsm = 2.23e-16
  
  ' golden number
  
  LOCAL c = 0.38196601125
  
  LOCAL iter as integer, d, e
  
  LOCAL t2, p, q
  
  local r, u, fu
  
  LOCAL x, xm, w
  
  local v, fx, fw
  
  LOCAL tol1, fv
  
  x = a + c * (b - a)
  
  w = x
  
  v = w
  
  e = 0.0
  p = 0.0
  q = 0.0
  r = 0.0
  
  rsfunc(x, fx)
  
  fw = fx
  
  fv = fw
  
  for iter = 1 to 100
    
    if (iter > 50) then
      
      print ("error in function minima!")
      print ("(more than 50 iterations)")
      
    end if
    
    xm = 0.5 * (a + b)
    
    tol1 = tolm * abs(x) + epsm
    
    t2 = 2.0 * tol1
    
    if (abs(x - xm) <= (t2 - 0.5 * (b - a))) then
      
      xmin = x
      
      fmin = fx
      
      exit sub
      
    end if
    
    if (abs(e) > tol1) then
      
      r = (x - w) * (fx - fv)
      
      q = (x - v) * (fx - fw)
      
      p = (x - v) * q - (x - w) * r
      
      q = 2.0 * (q - r)
      
      if (q > 0.0) then
        
        p = -p
        
      end if
      
      q = abs(q)
      
      r = e
      
      e = d
      
    end if
    
    if ((abs(p) >= abs(0.5 * q * r)) or (p <= q * (a - x)) or (p >= q * (b - x))) then
      
      if (x >= xm) then
        
        e = a - x
        
      else
        
        e = b - x
        
      end if
      
      d = c * e
      
    else
      
      d = p / q
      
      u = x + d
      
      if ((u - a) < t2) or ((b - u) < t2) then
        
        d = sgn(xm - x) * tol1
        
      end if
      
    end if
    
    if (abs(d) >= tol1) then
      
      u = x + d
      
    else
      
      u = x + sgn(d) * tol1
      
    end if
    
    rsfunc(u, fu)
    
    if (fu <= fx) then
      
      if (u >= x) then
        
        a = x
        
      else
        
        b = x
        
      end if
      
      v = w
      
      fv = fw
      
      w = x
      
      fw = fx
      
      x = u
      
      fx = fu
      
    else
      
      if (u < x) then
        
        a = u
        
      else
        
        b = u
        
      end if
      
      if ((fu <= fw) Or (w = x)) then
        
        v = w
        
        fv = fw
        
        w = u
        
        fw = fu
        
      elseif ((fu <= fv) Or (v = x) Or (v = w)) then
        
        v = u
        
        fv = fu
        
      end if
      
    end if
    
  next iter
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub gsite (angle, rsite())
  
  ' ground site position vector
  
  ' input
  
  '  angle  = local sidereal time or east longitude
  '           (radians; 0 <= angle <= 2*pi)
  
  ' input via global
  
  '  obslat = geodetic latitude (radians)
  '           (+north, -south; -pi/2 <= lat <= -pi/2)
  '  obsalt = geodetic altitude (meters)
  '           (+ above sea level, - below sea level)
  
  ' output
  
  '  rsite = ground site position vector (kilometers)
  
  ' special notes
  
  '  (1) eci coordinates if angle = local sidereal time
  
  '  (2) ecf coordinates if angle = east longitude
  
  '  (3) geocentric, equatorial coordinates
  
  '''''''''''''''''''''''''''''''''''''''''
  
  LOCAL slat, clat
  
  local sangle, cangle
  
  LOCAL b, c, d
  
  slat = sin(obslat)
  
  clat = cos(obslat)
  
  sangle = sin(angle)
  
  cangle = cos(angle)
  
  ' compute geodetic constants
  
  b = sqr(1.0 - (2.0 * flat - flat * flat) * slat * slat)
  
  c = reqm / b + 0.001 * obsalt
  
  d = reqm * (1.0 - flat) * (1.0 - flat) / b + 0.001 * obsalt
  
  ' compute ECI x, y and z components of position vector (kilometers)
  
  rsite(1) = c * clat * cangle
  
  rsite(2) = c * clat * sangle
  
  rsite(3) = d * slat
  
END sub
  
  ''''''''''''''''''''
  ''''''''''''''''''''
  
sub gast2(jdate, gast)
  
  ' greenwich apparent sidereal time
  
  ' input
  
  '  jdate = julian date
  
  ' output
  
  '  gast = greenwich apparent sidereal time (radians)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL t, t2, t3, eqeq, dpsi, deps
  
  LOCAL th, tl, obm, obt, st, x
  
  local tjdh as integer, tjdl
  
  tjdh = int(jdate)
  
  tjdl = jdate - tjdh
  
  th = (tjdh - 2451545.0) / 36525
  
  tl = tjdl / 36525.0
  
  t = th + tl
  
  t2 = t * t
  
  t3 = t2 * t
  
  ' obtain equation of the equinoxes
  
  eqeq = 0.0
  
  ' obtain nutation parameters in seconds of arc
  
  nut2000_lp(jdate, dpsi, deps)
  
  ' compute mean obliquity of the ecliptic in seconds of arc
  
  obm = 84381.4480 - 46.8150 * t - 0.00059 * t2 + 0.001813 * t3
  
  ' compute true obliquity of the ecliptic in seconds of arc
  
  obliq = obm + deps
  
  ' compute equation of the equinoxes in seconds of time
  
  eqeq = (dpsi / 15.0) * cos(obliq / seccon)
  
  st = eqeq - 6.2e-6 * t3 + 0.093104 * t2 + 67310.54841 + 8640184.812866 * tl + 3155760000.0 * tl + 8640184.812866 * th + 3155760000.0 * th
  
  ' modulo 24 hours
  
  x = st / 3600.0
  
  gast = x - 24.0 * fix(x / 24.0)
  
  if (gast < 0.0) then
    
    gast = gast + 24.0
    
  end if
  
  ' convert to radians
  
  gast = pi2 * (gast / 24.0)
  
END sub
  
  ''''''''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''''''''
  
sub eci2topo(gast, robj(), vobj(), azim, elev)
  
  ' convert eci position vector to topocentric coordinates
  
  ' input
  
  '  gast = Greenwich apparent sidereal time (radians)
  '  robj = eci position vector of satellite (kilometers)
  '  vobj = eci velocity vector of satellite (km/sec)
  
  ' output
  
  '  azim = topocentric azimuth (radians)
  '  elev = topocentric elevation (radians)
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local rsite(3), rhoijk(3), rhohatijk(3), rhohatsez(3)
  
  local wxr(3), rhodotijk(3), rhosez(3), rhodotsez(3)
  
  local i as integer, tmatrix(3, 3)
  
  LOCAL obslst, sobslat
  
  local cobslat, sobslst, cobslst
  
  ' observer local sidereal time (radians)
  
  obslst = modulo(gast + obslong)
  
  gsite(obslst, rsite())
  
  ' eci vector from observer to satellite (kilometers)
  
  for i = 1 to 3
    
    rhoijk(i) = robj(i) - rsite(i)
    
  next i
  
  ' omega cross r_sat
  
  wxr(1) = -omegae * robj(2)
  wxr(2) = omegae * robj(1)
  wxr(3) = 0.0
  
  ' eci range-rate
  
  for i = 1 to 3
    
    rhodotijk(i) = vobj(i) - wxr(i)
    
  next i
  
  ' observer-to-satellite slant range (kilometers)
  
  srange = vecmag(rhoijk())
  
  ' compute topocentric unit pointing vector
  
  uvector(rhoijk(), rhohatijk())
  
  ' compute eci-to-sez transformation matrix
  
  sobslat = sin(obslat)
  cobslat = cos(obslat)
  
  sobslst = sin(obslst)
  cobslst = cos(obslst)
  
  tmatrix(1, 1) = sobslat * cobslst
  tmatrix(1, 2) = sobslat * sobslst
  tmatrix(1, 3) = -cobslat
  
  tmatrix(2, 1) = -sobslst
  tmatrix(2, 2) = cobslst
  tmatrix(2, 3) = 0.0
  
  tmatrix(3, 1) = cobslat * cobslst
  tmatrix(3, 2) = cobslat * sobslst
  tmatrix(3, 3) = sobslat
  
  ' compute sez coordinates
  
  matxvec(tmatrix(), rhohatijk(), rhohatsez())
  
  matxvec(tmatrix(), rhoijk(), rhosez())
  
  matxvec(tmatrix(), rhodotijk(), rhodotsez())
  
  ' topocentric elevation (radians)
  
  elev = asin(rhohatsez(3))
  
  ' topocentric azimuth (radians)
  
  azim = atan3(rhohatsez(2), -rhohatsez(1))
  
  ' topocentric range-rate
  
  tmp = vdot(rhosez(), rhodotsez())
  
  rangerate = tmp / srange
  
  ' compute topocentric azimuth and elevation rates
  ' with check for satellite at zenith
  
  tmp1 = rhosez(1) * rhosez(1) + rhosez(2) * rhosez(2)
  
  if (abs(tmp1) > 0.000001) then
    
    azdot = (rhodotsez(1) * rhosez(2) - rhodotsez(2) * rhosez(1)) / tmp1
    
    eldot = (rhodotsez(3) - rangerate * sin(elevation)) / sqr(tmp1)
    
  else
    
    azdot = 0.0
    
    eldot = 0.0
    
  end if
  
END sub
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
sub broot(x1in, x2in, factor, dxmax, x1out, x2out)
  
  ' bracket a single root of a nonlinear equation
  
  ' input
  
  '  x1in   = initial guess for first bracketing x value
  '  x2in   = initial guess for second bracketing x value
  '  factor = acceleration factor (non-dimensional)
  '  dxmax  = rectification interval
  
  ' output
  
  '  xout1 = final value for first bracketing x value
  '  xout2 = final value for second bracketing x value
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL f1, f2
  
  local x3, dx
  
  ' evaluate objective function at initial value
  
  rsfunc(x1in, f1)
  
  ' save initial value
  
  x3 = x1in
  
  ' save initial delta-x
  
  dx = x2in - x1in
  
  ' perform bracketing until the product of the
  ' two function values is negative
  
  do
    
    ' geometrically accelerate the second point
    
    x2in = x2in + factor * (x2in - x3)
    
    ' evaluate objective function at x2
    
    rsfunc(x2in, f2)
    
    ' check to see if rectification is required
    
    if (abs(x2in - x3) > dxmax) then
      
      x3 = x2in - dx
      
    end if
    
    ' is the root bracketed?
    
    if ((f1 * f2) < 0.0) then exit do
    
  loop
  
  x1out = x1in
  
  x2out = x2in
  
END sub
  
  '''''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''''
  
sub realroot(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub rsfunc
  
  '''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  rsfunc(a, fa)
  
  rsfunc(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0.0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0.0) then q = -q
      
      p = abs(p)
      
      min = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (min < tmp) then min = tmp
      
      if (2.0 * p < min) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    rsfunc(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  '''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''
  
sub nut2000_lp(jdate, dpsi, deps)
  
  ' low precison nutation based on iau 2000a
  
  ' this function evaluates a short nutation series and returns approximate
  ' values for nutation in longitude and nutation in obliquity for a given
  ' tdb julian date. in this mode, only the largest 13 terms of the iau 2000a
  ' nutation series are evaluated.
  
  ' input
  
  '  jdate = tdb julian date
  
  ' output
  
  '  dpsi = nutation in longitude in arcseconds
  
  '  deps = nutation in obliquity in arcseconds
  
  '''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL rev = 1296000.0
  
  LOCAL el, elp, f, d, omega
  
  LOCAL i as integer, arg
  
  LOCAL t = (jdate - 2451545.0) / 36525.0
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' computation of fundamental (delaunay) arguments from simon et al. (1994)
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' mean anomaly of the moon
  
  el = (485868.249036 + t * (1717915923.2178 + t * (31.8792 + t * (0.051635 + t * (-0.00024470)))) mod rev) / seccon
  
  ' mean anomaly of the sun
  
  elp = (1287104.79305 + t * (129596581.0481 + t * (-0.5532 + t * (0.000136 + t * (- 0.00001149)))) mod rev) / seccon
  
  ' mean argument of the latitude of the moon
  
  f = (335779.526232 + t * (1739527262.8478 + t * (-12.7512 + t * (-0.001037 + t * (0.00000417)))) mod rev) / seccon
  
  ' mean elongation of the moon from the sun
  
  d = (1072260.70369 + t * (1602961601.2090 + t * (- 6.3706 + t * (0.006593 + t * (- 0.00003169)))) mod rev) / seccon
  
  ' mean longitude of the ascending node of the moon (from simon section 3.4(b.3), precession = 5028.8200 arcsec/cy)
  
  omega = (450160.398036 + t * (- 6962890.5431 + t * (7.4722 + t * (0.007702 + t * (- 0.00005939)))) mod rev) / seccon
  
  dpsi = 0.0
  
  deps = 0.0
  
  ' sum nutation series terms
  
  for i = 13 to 1 step -1
    
    arg = xnut(1, i) * el + xnut(2, i) * elp + xnut(3, i) * f + xnut(4, i) * d + xnut(5, i) * omega
    
    dpsi = (xnut(6, i) + xnut(7, i) * t) * sin(arg) + xnut(10, i) * cos(arg) + dpsi
    
    deps = (xnut(8, i) + xnut(9, i) * t) * cos(arg) + xnut(11, i) * sin(arg) + deps
    
  next i
  
  dpsi = 1.0e-7 * dpsi
  
  deps = 1.0e-7 * deps
  
  ' add in out-of-phase component of principal (18.6-year) term
  ' (to avoid small but long-term bias in results)
  
  dpsi = dpsi + 0.0033 * cos(omega)
  
  deps = deps + 0.0015 * sin(omega)
  
END sub
  
  ''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''
  
sub getdate (month, day, year)
  
  ' request calendar date subroutine
  
  do
    print " "
    print "please input the calendar date"
    print " "
    print "(month [1 - 12], day [1 - 31], year [yyyy])"
    print "< for example, october 21, 1986 is input as 10,21,1986 >"
    print "< b.c. dates are negative, a.d. dates are positive >"
    print "< the day of the month may also include a decimal part >"
    print " "
    input month, day, year
    
  loop until (month >= 1 and month <= 12) and (day >= 1 and day <= 31)
  
end sub
  
  ''''''''''''''''''''''''''
  ''''''''''''''''''''''''''
  
sub getutc (thr, tmin, tsec)
  
  ' request time subroutine
  
  do
    print " "
    print "please input the UTC time"
    print " "
    print "(hours [0 - 24], minutes [0 - 60], seconds [0 - 60])"
    print " "
    input thr, tmin, tsec
    
  loop until ((thr >= 0.0 and thr <= 24.0) and (tmin >= 0.0 and tmin <= 60.0) and (tsec >= 0.0 and tsec <= 60.0))
  
end sub
  
  '''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''
  
sub observer(obslat, obslong, obsalt)
  
  ' interactive request of latitude, longitude and altitude subroutine
  
  ' output
  
  '  obslat  = latitude (radians)
  '  obslong = longitude (radians)
  '  obsalt  = altitude (meters)
  
  ''''''''''''''''''''''''''''''
  
  do
    
    print "please input the geographic latitude of the observer"
    print "(degrees [-90 to +90], minutes [0 - 60], seconds [0 - 60])"
    print "(north latitudes are positive, south latitudes are negative)"
    
    input obslat.deg$, obslat.min, obslat.sec
    
  loop until (abs(val(obslat.deg$)) <= 90.0 and (obslat.min >= 0.0 and obslat.min <= 60.0) and (obslat.sec >= 0.0 and obslat.sec <= 60.0))
  
  if (left$(obslat.deg$, 2) = "-0") then
    
    obslat = -dtr * (obslat.min / 60.0 + obslat.sec / 3600.0)
    
  elseif (val(obslat.deg$) = 0.0) then
    
    obslat = dtr * (obslat.min / 60.0 + obslat.sec / 3600.0)
    
  else
    
    obslat = dtr * (sgn(val(obslat.deg$)) * (abs(val(obslat.deg$)) + obslat.min / 60.0 + obslat.sec / 3600.0))
    
  endif
  
  do
    
    print
    print "please input the geographic longitude of the observer"
    print "(degrees [0 - 360], minutes [0 - 60], seconds [0 - 60])"
    print "(east longitude is positive, west longitude is negative)"
    
    input obslong.deg$, obslong.min, obslong.sec
    
  loop until (abs(val(obslong.deg$)) >= 0.0 and abs(val(obslong.deg$)) <= 360.0) and (obslong.min >= 0.0 and obslong.min <= 60.0) and (obslong.sec >= 0.0 and obslong.sec <= 60.0)
  
  if (left$(obslong.deg$, 2) = "-0") then
    
    obslong = -dtr * (obslong.min / 60 + obslong.sec / 3600)
    
  elseif (val(obslong.deg$) = 0.0) then
    
    obslong = dtr * (obslong.min / 60.0 + obslong.sec / 3600.0)
    
  else
    
    obslong = dtr * (sgn(val(obslong.deg$)) * (abs(val(obslong.deg$)) + obslong.min / 60.0 + obslong.sec / 3600.0))
    
  endif
  
  print " "
  
  print "please input the altitude of the observer (meters)"
  print "(positive above sea level, negative below sea level)"
  
  input obsalt
  
end sub
  
  '''''''''
  '''''''''
  
sub readtle
  
  ' read tle data subroutine
  
  ' input (via global)
  
  '  cline2, cline3 = TLE data
  
  ''''''''''''''''''''''''''''
  
  local iyr, iyear, dayofyear, xjdtmp
  
  iyr = val(mid$(cline2, 19, 2))
  
  dayofyear = VAL(MID$(cline2, 21, 12))
  
  xndt2o = VAL(MID$(cline2, 34, 10))
  
  ixndd6o = VAL(MID$(cline2, 45, 6))
  
  iexp = VAL(MID$(cline2, 51, 2))
  
  ibstar = VAL(MID$(cline2, 54, 6))
  
  ibexp = VAL(MID$(cline2, 60, 2))
  
  xincl = VAL(MID$(cline3, 9, 8))
  
  xnodeo = VAL(MID$(cline3, 18, 8))
  
  iecc = VAL(MID$(cline3, 27, 7))
  
  omegao = VAL(MID$(cline3, 35, 8))
  
  xmo = VAL(MID$(cline3, 44, 8))
  
  xno = VAL(MID$(cline3, 53, 11))
  
  if (iyr < 50.0) then
    
    iyear = 2000.0 + iyr
    
  else
    
    iyear = 1900.0 + iyr
    
  end if
  
  julian(1, 0, iyear, xjdtmp)
  
  jdtle = xjdtmp + dayofyear
  
  xndd6o = 0.00001 * ixndd6o
  
  xndd6o = xndd6o * (10^iexp)
  
  xnodeo = xnodeo * dtr
  
  omegao = omegao * dtr
  
  xmo = xmo * dtr
  
  xincl = xincl * dtr
  
  temp = pi2 / xmnpda / xmnpda
  
  xno = xno * temp * xmnpda
  
  xndt2o = xndt2o * temp
  
  xndd6o = xndd6o * temp / xmnpda
  
  eo = 1.0e-7 * iecc
  
  bstar = 1.0e-5 * ibstar
  
  bstar = bstar * (10^ibexp) / ae
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub julian(month, day, year, jday)
  
  ' Gregorian date to julian day subroutine
  
  ' input
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year (all four digits)
  
  ' output
  
  '  jday = julian day
  
  ' special notes
  
  '  (1) calendar year must include all digits
  
  '  (2) will report October 5, 1582 to October 14, 1582
  '      as invalid calendar dates and exit
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local a, b, c, m, y
  
  y = year
  
  m = month
  
  b = 0.0
  
  c = 0.0
  
  if (m <= 2.0) then
    
    y = y - 1.0
    
    m = m + 12.0
    
  end if
  
  if (y < 0.0) then c = -0.75
  
  if (year < 1582.0) then
    
    ' null
    
  elseif (year > 1582.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (month < 10.0) then
    
    ' null
    
  elseif (month > 10.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (day <= 4.0) then
    
    ' null
    
  elseif (day > 14.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  else
    
    print "this date does not exist!!"
    
    exit
    
  end if
  
  jday = fix(365.25 * y + c) + fix(30.6001 * (m + 1.0)) + day + b + 1720994.5
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub gdate (jday, month, day, year)
  
  ' Julian day to Gregorian date subroutine
  
  ' input
  
  '  jday = julian day
  
  ' output
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year
  
  ''''''''''''''''''''''''
  
  local a, b, c, d, e, f, z, alpha
  
  z = fix(jday + 0.5)
  
  f = jday + 0.5 - z
  
  if (z < 2299161.0) then
    
    a = z
    
  else
    
    alpha = fix((z - 1867216.25) / 36524.25)
    
    a = z + 1.0 + alpha - fix(alpha / 4.0)
    
  end if
  
  b = a + 1524.0
  
  c = fix((b - 122.1) / 365.25)
  
  d = fix(365.25 * c)
  
  e = fix((b - d) / 30.6001)
  
  day = b - d - fix(30.6001 * e) + f
  
  if (e < 13.5) then
    
    month = e - 1.0
    
  else
    
    month = e - 13.0
    
  end if
  
  if (month > 2.5) then
    
    year = c - 4716.0
    
  else
    
    year = c - 4715.0
    
  end if
  
end sub
  
  '''''''''''''''''''''''''
  '''''''''''''''''''''''''
  
function modulo(x) as float
  
  ' modulo 2 pi function
  
  ''''''''''''''''''''''
  
  local a
  
  a = x - pi2 * fix(x / pi2)
  
  if (a < 0.0) then
    
    a = a + pi2
    
  end if
  
  modulo = a
  
end function
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
function atan3(a, b) as float
  
  ' four quadrant inverse tangent function
  
  ' input
  
  '  a = sine of angle
  '  b = cosine of angle
  
  ' output
  
  '  atan3 = angle (0 =< atan3 <= 2 * pi; radians)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
  local c
  
  if (abs(a) < 1.0e-10) then
    
    atan3 = (1.0 - sgn(b)) * pidiv2
    
    exit function
    
  else
    
    c = (2.0 - sgn(a)) * pidiv2
    
  endif
  
  if (abs(b) < 1.0e-10) then
    
    atan3 = c
    
    exit function
    
  else
    
    atan3 = c + sgn(a) * sgn(b) * (abs(atn(a / b)) - pidiv2)
    
  endif
  
end function
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
function vecmag(a()) as float
  
  ' vector magnitude function
  
  ' input
  
  '  { a } = column vector ( 3 rows by 1 column )
  
  ' output
  
  '  vecmag = scalar magnitude of vector { a }
  
  vecmag = sqr(a(1) * a(1) + a(2) * a(2) + a(3) * a(3))
  
end function
  
  ''''''''''''''''''''
  ''''''''''''''''''''
  
sub uvector (a(), b())
  
  ' unit vector subroutine
  
  ' input
  
  '  a = column vector (3 rows by 1 column)
  
  ' output
  
  '  b = unit vector (3 rows by 1 column)
  
  '''''''''''''''''''''''''''''''''''''''
  
  local i as integer, amag
  
  amag = vecmag(a())
  
  for i = 1 to 3
    
    if (amag <> 0.0) then
      
      b(i) = a(i) / amag
      
    else
      
      b(i) = 0.0
      
    end if
    
  next i
  
end sub
  
  ''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''
  
function vdot(a(), b()) as float
  
  ' vector dot product function
  
  ' c = { a } dot { b }
  
  ' input
  
  '  n%    = number of rows
  '  { a } = column vector with n rows
  '  { b } = column vector with n rows
  
  ' output
  
  '  vdot = dot product of { a } and { b }
  
  ''''''''''''''''''''''''''''''''''''''''
  
  local c = 0.0
  
  for i% = 1 to 3
    
    c = c + a(i%) * b(i%)
    
  next i%
  
  vdot = c
  
end function
  
  '''''''''''''''''''''''
  '''''''''''''''''''''''
  
sub vcross(a(), b(), c())
  
  ' vector cross product subroutine
  
  ' { c } = { a } x { b }
  
  ' input
  
  '  { a } = vector a ( 3 rows by 1 column )
  '  { b } = vector b ( 3 rows by 1 column )
  
  ' output
  
  '  { c } = { a } x { b } ( 3 rows by 1 column )
  
  c(1) = a(2) * b(3) - a(3) * b(2)
  c(2) = a(3) * b(1) - a(1) * b(3)
  c(3) = a(1) * b(2) - a(2) * b(2)
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub matxvec(a(), b(), c())
  
  ' matrix/vector multiplication subroutine
  
  ' { c } = [ a ] * { b }
  
  ' input
  
  '  a  = matrix a ( 3 rows by 3 columns )
  '  b  = vector b ( 3 rows )
  
  ' output
  
  '  c = vector c ( 3 rows )
  
  ''''''''''''''''''''''''''
  
  local s, i%, j%
  
  for i% = 1 to 3
    
    s = 0.0
    
    for j% = 1 to 3
      
      s = s + a(i%, j%) * b(j%)
      
    next j%
    
    c(i%) = s
    
  next i%
  
end sub
  
  ''''''''''''''''''''''
  ''''''''''''''''''''''
  
sub transpose (a(), b())
  
  ' matrix traspose subroutine
  
  ' input
  
  '  m = number of rows in matrix [ a ]
  '  n = number of columns in matrix [ a ]
  '  a = matrix a ( 3 rows by 3 columns )
  
  ' output
  
  '  b = matrix transpose (3 rows by 3 columns)
  
  '''''''''''''''''''''''''''''''''''''''''''''
  
  local i as integer, j as integer
  
  for i = 1 to 3
    
    for j = 1 to 3
      
      b(i, j) = a(j, i)
      
    next j
    
  next i
  
end sub
  
  '''''''''''''''
  '''''''''''''''
  
sub jd2str(jdutc)
  
  ' convert julian day to calendar date and UTC time
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''
  
  gdate (jdutc, cmonth, day, year)
  
  print "calendar date      ", month$(cmonth), " ", STR$(int(day)), " ", str$(year)
  
  print " "
  
  thr0 = 24.0 * (day - int(day))
  
  thr = int(thr0)
  
  tmin0 = 60.0 * (thr0 - thr)
  
  tmin = int(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 59.9) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1.0
    
  end if
  
  print "UTC                ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub deg2str(dd, dms$)
  
  ' convert decimal degrees to degrees,
  ' minutes, and seconds string
  
  ' input
  
  '  dd = angle in decimal degrees
  
  ' output
  
  '  dms$ = string equivalent
  
  '''''''''''''''''''''''''''
  
  local dwrk, tmin0, tmin, tsec
  
  dwrk = fix(dd)
  
  tmin0 = 60.0 * (dd - dwrk)
  
  tmin = fix(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 59.9) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    dwrk = dwrk + 1.0
    
  end if
  
  dms$ = str$(dwrk) + " deg " + str$(tmin) + " min " + str$(tsec, 0, 2) + " sec"
  
end sub
  
  ''''''''''''''''
  ''''''''''''''''
  
sub hrs2str(hours)
  
  ' convert hours to equivalent string
  
  ''''''''''''''''''''''''''''''''''''
  
  local thr, tmin0, tmin, tsec
  
  thr = fix(hours)
  
  tmin0 = 60.0 * (hours - thr)
  
  tmin = fix(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 59.9) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1
    
  end if
  
  print "event duration     ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub

