  ' transit.bas      June 28, 2017
  
  ' transits of Mercury and Venus
  
  ' Micromite eXtreme version
  
  '''''''''''''''''''''''''''
  
  option default float
  
  option base 1
  
  ' dimension global arrays and variables
  
  dim xsl(50), xsr(50), xsa(50), xsb(50)
  
  dim cl(88), al(88), bl(88)
  
  dim jdleap(28), leapsec(28), sdia(2)
  
  dim month$(12) as string, rm_sun, rm_planet
  
  dim xnut(11, 13), trr, sd_sun, sd_planet
  
  dim jdtdbi, jdprint, jdsaved, obslat, obslong, obsalt
  
  dim cmonth, cday, cyear, ndays
  
  dim iplanet%
  
  ' global constants
  
  const pi2 = 2.0 * pi, pidiv2 = 0.5 * pi, dtr = pi / 180.0, rtd = 180.0 / pi
  
  const atr = pi / 648000.0, seccon = 206264.8062470964
  
  ' astronomical unit (kilometers)
  
  const aunit = 149597870.691
  
  ' equatorial radius of the earth (kilometers)
  
  const reqm = 6378.14
  
  ' earth flattening factor (nd)
  
  const flat = 1.0 / 298.257

  ' radius of the sun (au)

  const dsun = 696000.0 / aunit

  ' semidiameter of the planets (radians)

  sdia(1) = atr * 3.36

  sdia(2) = atr * 8.41

  ' read ephemeris and leap second data
    
  read_data
      
  ''''''''''''''''''
  ' begin simulation
  ''''''''''''''''''
  
  print " "
  print "transits of Mercury and Venus"
  print "============================="
  print " "

  ' request planet of interest

  do
    
    print " <1> Mercury"
    print " "
    print " <2> Venus"
    print " "
    print "please input the planet of interest"
    
    input iplanet%
    
    if (iplanet% = 1 or iplanet% = 2) then
      
      exit do
      
    end if
    
  loop
   
  ' request initial calendar date (month, day, year)
  
  getdate(cmonth, cday, cyear)
  
  ' request observer coordinates
  
  print " "
  
  observer(obslat, obslong, obsalt)
    
  ' initial tdb julian day
  
  julian(cmonth, cday, cyear, jdtdbi)
  
  ' search duration (days)
  
  print " "
  
  print "please input the search duration in days"
  
  input ndays
  
  print " "
  print "searching for transit conditions ..."
  print " "
  
  ' define search parameters
  
  ti = 0.0
  
  tf = ndays
  
  dt = 0.25
  
  dtsml = 0.1
  
  ' find and display transit conditions
  
  t_event(ti, tf, dt, dtsml)
  
end
  
  ''''''''''''''
  ''''''''''''''
  
sub tfunc(x, fx)
  
  ' transit objective function
  
  ''''''''''''''''''''''''''''
  
  local rpg_sun(3), rpg_planet(3)
  
  local jdtdb, jdutc, gast, cpsi, sd_sun, sd_planet
  
  local dl, dr, rasc, decl, usun(3), uplanet(3)
  
  ' current TDB julian day
  
  jdtdb = jdtdbi + x
  
  ' compute topocentric coordinates of the sun

  sun(jdtdb, dl, dr, rasc, decl, rpg_sun())
      
  ' iterative solution for UTC julian day
  
  tdb2utc(jdtdb, jdutc)
  
  ' greenwich apparent sidereal time
  
  gast2(jdutc, gast)
  
  eci2topo(gast, rpg_sun(), azim_sun, elev_sun)
 
  ' geocentric distance of the sun (au)
   
  rm_sun = vecmag(rpg_sun())
  
  ' compute topocentric coordinates of the planet

  ephem(iplanet%, jdtdb, rpg_planet(), rasc, decl)
    
  eci2topo(gast, rpg_planet(), azim_planet, elev_planet)  
  
  ' geocentric distance of the planet (au)
  
  rm_planet = vecmag(rpg_planet())
  
  ' compute cosine of sun/planet separation angle

  uvector(rpg_sun(), usun())
  
  uvector(rpg_planet(), uplanet())
    
  cpsi = vdot(usun(), uplanet())
    
  ' semi-diameter of the sun and planet
  
  sd_planet = sdia(iplanet%) / rm_planet
  
  sd_sun = asin(dsun / rm_sun)

  ' evaluate objective function
    
  fx = acos(cpsi) - (sd_sun + sd_planet)
  
end sub
  
  '''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''
  
sub t_event (ti, tf, dt, dtsml)
  
  ' predict transit events
  
  ' input
  
  '  ti    = initial simulation time
  '  tf    = final simulation time
  '  dt    = step size used for bounding minima
  '  dtsml = small step size used to determine whether
  '          the function is increasing or decreasing
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL tolm, iend as integer
  
  local fmin1, tmin1
  
  LOCAL ftemp, df, dflft
  
  local el, er
  
  LOCAL t, ft
  
  local iter1 as integer, iter2 as integer
  
  local iter3 as integer
  
  ' initialization
  
  tolm = 1.0e-8
  
  iend = 0
  
  ' check the initial time for a minimum
  
  tfunc(ti, fmin1)
  
  tmin1 = ti
  
  tfunc(ti + dtsml, ftemp)
  
  df = ftemp - fmin1
  
  dflft = df
  
  el = ti
  
  er = el
  
  t = ti
  
  ' if the slope is positive and the minimum is
  ' negative, calculate event conditions at the initial time
  
  if (df > 0.0 and fmin1 < 0.0) then
    
    events1(ti, tf, tmin1)
    
  end if
  
  for iter1 = 1 to 1000
    
    ' find where function first starts decreasing
    
    for iter2 = 1 to 1000
      
      if (df <= 0.0) then
        
        exit for
        
      end if
      
      t = t + dt
      
      if (t >= tf) then
        
        ' check final time for a minimum
        
        if (iend = 1) then exit sub
        
        tfunc(tf, fmin1)
        
        tfunc(tf - dtsml, ftemp)
        
        df = fmin1 - ftemp
        
        ' set minimum time to final simulation time
        
        tmin1 = tf
        
        if (df < 0.0) then
          
          ' if both the slope and minimum are negative,
          ' calculate the event conditions at the final
          ' simulation time
          
          if (fmin1 < 0.0) then
            
            events1(ti, tf, tmin1)
            
          end if
          
          ' otherwise, we're finished
          
          exit sub
          
        end if
        
        if (dflft > 0.0) then exit sub
        
        er = tf
        
        iend = 1
        
      end if
      
      tfunc(t, ft)
      
      tfunc(t - dtsml, ftemp)
      
      df = ft - ftemp
      
    next iter2
    
    ' function decreasing - find where function
    ' first starts increasing
    
    for iter3 = 1 to 1000
      
      el = t
      
      dflft = df
      
      t = t + dt
      
      if (t >= tf) then
        
        ' check final time for a minimum
        
        if (iend = 1) then exit sub
        
        tfunc(tf, fmin1)
        
        tfunc(tf - dtsml, ftemp)
        
        df = fmin1 - ftemp
        
        ' set minimum time to final simulation time
        
        tmin1 = tf
        
        if (df < 0.0) then
          
          ' if both the slope and minimum are negative,
          ' calculate the event conditions at the final
          ' simulation time
          
          if (fmin1 < 0.0) then
            
            events1(ti, tf, tmin1)
            
          end if
          
          ' otherwise, we're finished
          
          exit sub
          
        end if
        
        if (dflft > 0.0) then exit sub
        
        er = tf
        
        iend = 1
        
      end if
      
      tfunc(t, ft)
      
      tfunc(t - dtsml, ftemp)
      
      df = ft - ftemp
      
      if (df > 0.0) then exit for
      
    next iter3
    
    er = t
    
    ' calculate minimum using Brent's method
    
    minima(el, er, tolm, tmin1, fmin1)
    
    el = er
    
    dflft = df
    
    ' if the minimum is negative,
    ' calculate event conditions for this minimum
    
    if (fmin1 < 0.0) then
      
      events1(ti, tf, tmin1)
      
      t = trr
      
    end if
    
  next iter1
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub events1 (ti, tf, topt)
  
  ' compute and display transit events
  
  ' input
  
  '  ti   = initial simulation time
  '  tf   = final simulation time
  '  topt = extrema time
  
  ''''''''''''''''''''''
  
  ' define root-bracketing and root-finding control parameters
  
  LOCAL factor = 0.25            ' geometric acceleration factor
  
  LOCAL dxmax = 600.0 / 86400.0  ' rectification interval (days)
  
  LOCAL rtol = 1.0e-8            ' root-finding convergence tolerance
  
  LOCAL t1in, t2in
  
  local t1out, t2out
  
  LOCAL troot, froot, jdtdb

  if (rm_planet > rm_sun) then
  
     ' planet is not between sun and observer
  
     trr = topt
     
     return
     
  end if
    
  ' compute and display ingress conditions
  
  t1in = topt
  
  t2in = t1in - 60.0 / 86400.0
  
  broot(t1in, t2in, factor, dxmax, t1out, t2out)
  
  realroot1(t1out, t2out, rtol, troot, froot)
  
  ' set to initial time if before ti
  
  if (troot < ti) then
    
    troot = ti
    
    tfunc(ti, froot)
    
  end if
  
  jdtdb = jdtdbi + troot
  
  tprint(1, jdtdb)
  
  ' compute and display least distance conditions
  
  jdtdb = jdtdbi + topt
  
  tprint(2, jdtdb)
  
  ' compute and display egress conditions
  
  t2in = t1in + 60.0 / 86400.0
  
  broot(t1in, t2in, factor, dxmax, t1out, t2out)
  
  realroot1(t1out, t2out, rtol, troot, froot)
  
  ' set to final time if after tf
  
  if (troot > tf) then
    
    troot = tf
    
    tfunc(tf, froot)
    
  end if
  
  jdtdb = jdtdbi + troot
  
  tprint(3, jdtdb)
  
  trr = troot
  
END sub
  
  '''''''''''''''''''''''
  '''''''''''''''''''''''
  
sub tprint(iflag%, jdtdb)
  
  ' print transit conditions
  
  '''''''''''''''''''''''''''''''
  
  local rpg_sun(3)
  
  LOCAL jdutc, dms$ as string
  
  local deltat, gast, azim_sun, elev_sun
  
  if (iflag% = 1) then
    
     if (iplanet% = 1) then
     
        print " "
        print "ingress - exterior contact - Mercury"
        print "------------------------------------"
        print " "
     
     else
     
        print " "
        print "ingress - exterior contact - Venus"
        print "----------------------------------"
        print " "
                
     end if
         
     jdprint = jdtdb
    
  end if
  
  if (iflag% = 2) then
    
     if (iplanet% = 1) then
     
        print " "
        print " "
        print "least distance of the Sun and Mercury"
        print "-------------------------------------"
        print " "
     
     else
     
        print " "
        print " "
        print "least distance of the Sun and Venus"
        print "-----------------------------------"
        print " "
                
     end if
    
  end if
  
  if (iflag% = 3) then
    
     if (iplanet% = 1) then
     
        print " "
        print " "
        print "engress - exterior contact - Mercury"
        print "------------------------------------"
        print " "
     
     else
     
        print " "
        print " "
        print "engress - exterior contact - Venus"
        print "----------------------------------"
        print " "
                
     end if
    
  end if
  
  ' compute and display UTC julian date
  
  tdb2utc(jdtdb, jdutc)
  
  jd2str(jdutc)
  
  PRINT " "
  
  print "UTC julian day     ", str$(jdutc, 0, 8)
  
  ' compute and display topocentric coordinates of the sun

  sun(jdtdb, dl, dr, rasc, decl, rpg_sun())
  
  gast2(jdutc, gast)
  
  eci2topo(gast, rpg_sun(), azim_sun, elev_sun)
  
  PRINT " "
  
  print "topocentric coordinates of the Sun"
  
  PRINT " "
  
  deg2str(rtd * azim_sun, dms$)
  
  print "azimuth angle    ", dms$
  
  PRINT " "
  
  deg2str(rtd * elev_sun, dms$)
  
  print "elevation angle  ", dms$
  
  ' determine and display event duration
  
  if (iflag% = 3) then
    
    deltat = 24.0 * (jdtdb - jdprint)
    
    PRINT " "
    
    hrs2str(deltat)
    
    print " "
    
  end if
  
END sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub minima(a, b, tolm, xmin, fmin)
  
  ' one-dimensional minimization
  
  ' Brent's method
  
  ' input
  
  '  a    = initial x search value
  '  b    = final x search value
  '  tolm = convergence criterion
  
  ' output
  
  '  xmin = minimum x value
  
  ' note
  
  '  user-defined objective subroutine
  '  coded as usr_func(x, fx)
  
  ' remember: a maximum is simply a minimum
  '           with a negative attitude!
  
  '''''''''''''''''''''''''''''''''''''
  
  ' machine epsilon
  
  LOCAL epsm = 2.23e-16
  
  ' golden number
  
  LOCAL c = 0.38196601125
  
  LOCAL d, e
  
  LOCAL t2, p, q
  
  local r, u, fu
  
  LOCAL x, xm, w
  
  local v, fx, fw
  
  LOCAL tol1, fv
  
  x = a + c * (b - a)
  
  w = x
  
  v = w
  
  e = 0.0
  p = 0.0
  q = 0.0
  r = 0.0
  
  tfunc(x, fx)
  
  fw = fx
  
  fv = fw
  
  for iter% = 1 to 100
    
    if (iter% > 50) then
      
      print ("error in function minima!")
      print ("(more than 50 iterations)")
      
    end if
    
    xm = 0.5 * (a + b)
    
    tol1 = tolm * abs(x) + epsm
    
    t2 = 2.0 * tol1
    
    if (abs(x - xm) <= (t2 - 0.5 * (b - a))) then
      
      xmin = x
      
      fmin = fx
      
      exit sub
      
    end if
    
    if (abs(e) > tol1) then
      
      r = (x - w) * (fx - fv)
      
      q = (x - v) * (fx - fw)
      
      p = (x - v) * q - (x - w) * r
      
      q = 2.0 * (q - r)
      
      if (q > 0.0) then
        
        p = -p
        
      end if
      
      q = abs(q)
      
      r = e
      
      e = d
      
    end if
    
    if ((abs(p) >= abs(0.5 * q * r)) or (p <= q * (a - x)) or (p >= q * (b - x))) then
      
      if (x >= xm) then
        
        e = a - x
        
      else
        
        e = b - x
        
      end if
      
      d = c * e
      
    else
      
      d = p / q
      
      u = x + d
      
      if ((u - a) < t2) or ((b - u) < t2) then
        
        d = sgn(xm - x) * tol1
        
      end if
      
    end if
    
    if (abs(d) >= tol1) then
      
      u = x + d
      
    else
      
      u = x + sgn(d) * tol1
      
    end if
    
    tfunc(u, fu)
    
    if (fu <= fx) then
      
      if (u >= x) then
        
        a = x
        
      else
        
        b = x
        
      end if
      
      v = w
      
      fv = fw
      
      w = x
      
      fw = fx
      
      x = u
      
      fx = fu
      
    else
      
      if (u < x) then
        
        a = u
        
      else
        
        b = u
        
      end if
      
      if ((fu <= fw) Or (w = x)) then
        
        v = w
        
        fv = fw
        
        w = u
        
        fw = fu
        
      elseif ((fu <= fv) Or (v = x) Or (v = w)) then
        
        v = u
        
        fv = fu
        
      end if
      
    end if
    
  next iter%
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub gsite (angle, rsite())
  
  ' ground site position vector
  
  ' input
  
  '  angle  = local sidereal time or east longitude
  '           (radians; 0 <= angle <= 2*pi)
  
  ' input via global
  
  '  obslat = geodetic latitude (radians)
  '           (+north, -south; -pi/2 <= lat <= -pi/2)
  '  obsalt = geodetic altitude (meters)
  '           (+ above sea level, - below sea level)
  
  ' output
  
  '  rsite = ground site position vector (kilometers)
  
  ' special notes
  
  '  (1) eci coordinates if angle = local sidereal time
  
  '  (2) ecf coordinates if angle = east longitude
  
  '  (3) geocentric, equatorial coordinates
  
  '''''''''''''''''''''''''''''''''''''''''
  
  LOCAL slat, clat, sangle, cangle, b, c, d
  
  slat = sin(obslat)
  
  clat = cos(obslat)
  
  sangle = sin(angle)
  
  cangle = cos(angle)
  
  ' compute geodetic constants
  
  b = sqr(1.0 - (2.0 * flat - flat * flat) * slat * slat)
  
  c = reqm / b + 0.001 * obsalt
  
  d = reqm * (1.0 - flat) * (1.0 - flat) / b + 0.001 * obsalt
  
  ' compute x, y and z components of position vector (kilometers)
  
  rsite(1) = c * clat * cangle
  
  rsite(2) = c * clat * sangle
  
  rsite(3) = d * slat
    
END sub
  
  ''''''''''''''''''''
  ''''''''''''''''''''
  
sub gast2(jdate, gast)
  
  ' greenwich apparent sidereal time
  
  ' input
  
  '  jdate = julian date
  
  ' output
  
  '  gast = greenwich apparent sidereal time (radians)
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL t, t2, t3, eqeq, dpsi, deps
  
  LOCAL th, tl, obm, obt, st, x
  
  local tjdh as integer, tjdl
  
  tjdh = int(jdate)
  
  tjdl = jdate - tjdh
  
  th = (tjdh - 2451545.0) / 36525
  
  tl = tjdl / 36525.0
  
  t = th + tl
  
  t2 = t * t
  
  t3 = t2 * t
  
  ' obtain equation of the equinoxes
  
  eqeq = 0.0
  
  ' obtain nutation parameters in seconds of arc
  
  nut2000_lp(jdate, dpsi, deps)
  
  ' compute mean obliquity of the ecliptic in seconds of arc
  
  obm = 84381.4480 - 46.8150 * t - 0.00059 * t2 + 0.001813 * t3
  
  ' compute true obliquity of the ecliptic in seconds of arc
  
  obliq = obm + deps
  
  ' compute equation of the equinoxes in seconds of time
  
  eqeq = (dpsi / 15.0) * cos(obliq / seccon)
  
  st = eqeq - 6.2e-6 * t3 + 0.093104 * t2 + 67310.54841 + 8640184.812866 * tl + 3155760000.0 * tl + 8640184.812866 * th + 3155760000.0 * th
  
  ' modulo 24 hours
  
  x = st / 3600.0
  
  gast = x - 24.0 * fix(x / 24.0)
  
  if (gast < 0.0) then
    
    gast = gast + 24.0
    
  end if
  
  ' convert to radians
  
  gast = pi2 * (gast / 24.0)
  
END sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub tdb2utc (jdtdb, jdutc)
  
  ' convert TDB julian day to UTC julian day subroutine
  
  ' input
  
  '  jdtdb = TDB julian day
  
  ' output
  
  '  jdutc = UTC julian day
  
  '''''''''''''''''''''''''
  
  local x1, x2, xroot, froot
  
  jdsaved = jdtdb
  
  ' set lower and upper bounds
  
  x1 = jdsaved - 0.1
  
  x2 = jdsaved + 0.1
  
  ' solve for UTC julian day using Brent's method
  
  realroot2(x1, x2, 1.0e-8, xroot, froot)
  
  jdutc = xroot
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub jdfunc (jdin, fx)
  
  ' objective function for tdb2utc
  
  ' input
  
  '  jdin = current value for UTC julian day
  
  ' output
  
  '  fx = delta julian day
  
  ''''''''''''''''''''''''
  
  local jdwrk
  
  utc2tdb(jdin, jdwrk)
  
  fx = jdwrk - jdsaved
  
end sub

  ''''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''''
  
sub sun(jdtdb, dl, dr, rasc, decl, rsun())
  
  ' precision ephemeris of the Sun
  
  ' input
  
  '  jdtdb = julian ephemeris day
  
  ' output
  
  '  dl   = ecliptic longitude of the sun (radians)
  '         (0 <= dl <= 2 pi)
  '  dr   = geocentric distance of the sun (AU)
  '  rasc = right ascension of the Sun (radians)
  '         (0 <= rasc <= 2 pi)
  '  decl = declination of the Sun (radians)
  '         (-pi/2 <= decl <= pi/2)
  '  rsun = geocentric equatorial position vector
  
  '''''''''''''''''''''''''''''''''''''''''''''''
  
  local u, a1, a2, psi, deps, meps, eps, seps, ceps
  
  local w, srl, crl, srb, crb, sra, cra, i%
  
  u = (jdtdb - 2451545.0) / 3652500.0
  
  ' compute nutation in longitude
  
  a1 = 2.18 + u * (-3375.7 + u * 0.36)
  
  a2 = 3.51 + u * (125666.39 + u * 0.1)
  
  psi = 0.0000001 * (-834.0 * sin(a1) - 64.0 * sin(a2))
  
  ' compute nutation in obliquity
  
  deps = 0.0000001 * u * (-226938 + u * (-75 + u * (96926 + u * (-2491 - u * 12104))))
  
  meps = 0.0000001 * (4090928.0 + 446.0 * cos(a1) + 28.0 * cos(a2))
  
  eps = meps + deps
  
  obliq = eps
  
  seps = sin(eps)
  
  ceps = cos(eps)
  
  dl = 0.0
  
  dr = 0.0
  
  for i% = 1 to 50
    
    w = xsa(i%) + xsb(i%) * u
    
    dl = dl + xsl(i%) * sin(w)
      
    dr = dr + xsr(i%) * cos(w)
    
  next i%
  
  dl = modulo(dl * 0.0000001 + 4.9353929 + 62833.196168 * u)
  
  dr = dr * 0.0000001 + 1.0001026
  
  rlsun = modulo(dl + 0.0000001 * (-993.0 + 17.0 * cos(3.1 + 62830.14 * u)) + psi)
  
  rb = 0.0
  
  ' compute geocentric declination and right ascension
  
  crl = cos(rlsun)
  srl = sin(rlsun)
  crb = cos(rb)
  srb = sin(rb)
  
  decl = asin(ceps * srb + seps * crb * srl)
  
  sra = -seps * srb + ceps * crb * srl
  
  cra = crb * crl
  
  rasc = atan3(sra, cra)
  
  ' geocentric equatorial position vector of the Sun (au)
  
  rsun(1) = dr * cos(rasc) * cos(decl)
  
  rsun(2) = dr * sin(rasc) * cos(decl)
  
  rsun(3) = dr * sin(decl)
  
end sub
    
  ''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''
  
sub eci2topo(gast, robj(), azim, elev)
  
  ' convert eci position vector to topocentric coordinates
  
  ' input
  
  '  gast = Greenwich apparent sidereal time (radians)
  '  robj = eci position vector of object (au)
  
  ' output
  
  '  azim = topocentric azimuth (radians)
  '  elev = topocentric elevation (radians)
  
  ''''''''''''''''''''''''''''''''''''''''''''''
  
  local rsite(3), rhoijk(3), rhohatijk(3), rhohatsez(3)
  
  local tmatrix(3, 3), i%
  
  LOCAL obslst, srange, sobslat
  
  local cobslat, sobslst, cobslst
  
  ' observer local sidereal time
  
  obslst = modulo(gast + obslong)
  
  gsite(obslst, rsite())
  
  ' eci position vector from observer to object
  
  for i% = 1 to 3
    
    rhoijk(i%) = aunit * robj(i%) - rsite(i%)
    
  next i%
  
  ' observer-to-object slant range (kilometers)
  
  srange = vecmag(rhoijk())
  
  ' compute topocentric unit pointing vector
  
  uvector(rhoijk(), rhohatijk())
  
  ' compute eci-to-sez transformation matrix
  
  sobslat = sin(obslat)
  cobslat = cos(obslat)
  
  sobslst = sin(obslst)
  cobslst = cos(obslst)
  
  tmatrix(1, 1) = sobslat * cobslst
  tmatrix(1, 2) = sobslat * sobslst
  tmatrix(1, 3) = -cobslat
  
  tmatrix(2, 1) = -sobslst
  tmatrix(2, 2) = cobslst
  tmatrix(2, 3) = 0.0
  
  tmatrix(3, 1) = cobslat * cobslst
  tmatrix(3, 2) = cobslat * sobslst
  tmatrix(3, 3) = sobslat
  
  ' compute sez coordinates
  
  matxvec(tmatrix(), rhohatijk(), rhohatsez())
  
  ' topocentric elevation (radians)
  
  elev = asin(rhohatsez(3))
  
  ' topocentric azimuth (radians)
  
  azim = atan3(rhohatsez(2), -rhohatsez(1))
  
END sub
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
sub broot(x1in, x2in, factor, dxmax, x1out, x2out)
  
  ' bracket a single root of a nonlinear equation
  
  ' input
  
  '  x1in   = initial guess for first bracketing x value
  '  x2in   = initial guess for second bracketing x value
  '  factor = acceleration factor (non-dimensional)
  '  dxmax  = rectification interval
  
  ' output
  
  '  xout1 = final value for first bracketing x value
  '  xout2 = final value for second bracketing x value
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL f1, f2
  
  local x3, dx
  
  ' evaluate objective function at initial value
  
  tfunc(x1in, f1)
  
  ' save initial value
  
  x3 = x1in
  
  ' save initial delta-x
  
  dx = x2in - x1in
  
  ' perform bracketing until the product of the
  ' two function values is negative
  
  do
    
    ' geometrically accelerate the second point
    
    x2in = x2in + factor * (x2in - x3)
    
    ' evaluate objective function at x2
    
    tfunc(x2in, f2)
    
    ' check to see if rectification is required
    
    if (abs(x2in - x3) > dxmax) then
      
      x3 = x2in - dx
      
    end if
    
    ' is the root bracketed?
    
    if ((f1 * f2) < 0.0) then exit do
    
  loop
  
  x1out = x1in
  
  x2out = x2in
  
END sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot1(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub tfunc
  
  ''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  tfunc(a, fa)
  
  tfunc(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0.0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0.0) then q = -q
      
      p = abs(p)
      
      min = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (min < tmp) then min = tmp
      
      if (2.0 * p < min) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    tfunc(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot2(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub jdfunc
  
  '''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  jdfunc(a, fa)
  
  jdfunc(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0) then q = -q
      
      p = abs(p)
      
      xmin = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (xmin < tmp) then xmin = tmp
      
      if (2.0 * p < xmin) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    jdfunc(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub

  '''''''''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''''''''
  
sub ephem(ibody%, jdtdb, rpg(), rasc, decl)
  
  ' sun and inner planet ephemeris subroutine
  
  ' input
  
  '  ibody% = celestial body index
  '  jdtdb  = tdb julian day
  
  ' output
  
  '  rpg() = geocentric position vector (au)
  '  rasc  = geocentric right ascension (radians)
  '  decl  = geocentric declination (radians)
  
  '''''''''''''''''''''''''''''''''''''''''''
  
  local dls, drs, gl, gb, pl, pb, pr, alon, alat
  
  local rpgm, rsun(3)
  
  ' compute coordinates of the sun
  
  sun(jdtdb, dls, drs, rasun, decsun, rsun())
  
  gl = dls
  
  ' compute coordinates of the planet
  
  select case ibody%
      
    case 1
      
      mercury(jdtdb, pl, pb, pr)
      
    case 2
      
      venus(jdtdb, pl, pb, pr)
            
  end select
  
  ' compute geocentric mean coordinates
  
  latlong(dls, drs, pl, pb, pr, gl, gb, rpgm)
  
  ' apparent geocentric equatorial right ascension and declination
  
  abernu(jdtdb, ibody%, gl, gb, alon, alat, rasc, decl)
  
  ' compute geocentric equatorial position vector of planet (au)
  
  rpg(1) = rpgm * cos(rasc) * cos(decl)
  
  rpg(2) = rpgm * sin(rasc) * cos(decl)
  
  rpg(3) = rpgm * sin(decl)
  
end sub

  ''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''
  
sub mercury(jdtdb, pl, pb, pr)
  
  '  computation of the heliocentric coordinates of mercury
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  local u, h1, w1
   
  u = (jdtdb - 2451545.0) / 3652500.0
  
  pl = 0.0
  
  for i% = 1 to 25
    
    pl = pl + cl(i%) * sin(al(i%) + bl(i%) * u)
    
  next i%
  
  pl = pl * 0.0000001 + 4.4429839 + u * 260881.4701279
  
  h1 = 0.000001 * (409894.2 + u * (2435.0 + u * (-1408.0 + u * (114.0 + u * (233.0 - u * 88.0)))))
  
  w1 = 3.053817 + u * (260878.756773 + u * (-0.001093 + u * (-9.3e-04 + u * (0.00043 + u * 0.00014))))
  
  pl = modulo(pl + h1 * sin(w1))
  
  pb = 0.0
  
  for i% = 26 to 43
    
    pb = pb + cl(i%) * sin(al(i%) + bl(i%) * u)
    
  next i%
  
  pb = pb * 0.0000001
  
  pr = 0.0
  
  for i% = 44 to 57
    
    pr = pr + cl(i%) * cos(al(i%) + bl(i%) * u)
    
  next i%
  
  pr = pr * 0.0000001 + 0.395202
  
end sub

  ''''''''''''''''''''''''''
  ''''''''''''''''''''''''''
  
sub venus(jdtdb, pl, pb, pr)
  
  '  computation of the heliocentric coordinates of venus
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  local u, t1, h1, t2, w1, t3, h2, w2, t4
  
  u = (jdtdb - 2451545.0) / 3652500.0
  
  pl = 0.0
  
  for i% = 58 to 77
    
    pl = pl + cl(i%) * sin(al(i%) + bl(i%) * u)
    
  next i%
  
  pl = pl * 0.0000001 + 3.2184413 + u * 102135.2937764
  
  t1 = -51.0 + u * 10.0
  
  h1 = 0.000001 * (13539.7 + u * (-9570.0 + u * (1987.0 + u * (927.0 + u * (230.0 + u * t1)))))
  
  t2 = 0.0017 + u * 0.0151
  
  w1 = 0.88074 + u * (102132.84648 + u * (0.24082 + u * (0.1004 + u * (0.0355 - u * t2))))
  
  t3 = -79.0 + u * 56.0
  
  h2 = 0.000001 * (898.9 + u * (112.4 + u * (-170.0 + u * (113.0 + u * (34.0 + u * t3)))))
  
  w2 = 0.5941 + u * (204267.313 + u * (0.014 + u * (0.123 + u * (-0.146 + u * 0.052))))
  
  pl = modulo(pl + h1 * sin(w1) + h2 * sin(w2))
  
  pb = 0.0
  
  for i% = 78 to 83
    
    pb = pb + cl(i%) * sin(al(i%) + bl(i%) * u)
    
  next i%
  
  h1 = 0.0000001 * (4011.0 + u * (-2713.0 + u * (490.0 + u * (290.0 + u * 90.0))))
  
  w1 = 2.7182 + u * (204266.568 + u * (0.225 + u * (0.102 + u * 0.035)))
  
  h2 = 0.0000001 * (101.0 + u * (26.0 - u * 64.0))
  
  w2 = 2.66 + u * (306400.49 + u * 0.45)
  
  pb = pb * 0.0000001 + h1 * sin(w1) + h2 * sin(w2)
  
  pr = 0.0
  
  for i% = 84 to 88
    
    pr = pr + cl(i%) * cos(al(i%) + bl(i%) * u)
    
  next i%
  
  pr = pr * 0.0000001 + 0.7235481
  
  h1 = 0.0000001 * (48982.0 + u * (-34549.0 + u * (7096.0 + u * (3360.0 + u * (890.0 - u * 210.0)))))
  
  t4 = -0.0013 - u * 0.015
  
  w1 = 4.02152 + u * (102132.84695 + u * (0.242 + u * (0.0994 + u * (0.0351 + u * t4))))
  
  h2 = 0.0000001 * (166.0 + u * (-234.0 + u * 131.0))
  
  w2 = 4.9 + u * (204265.69 + u * (0.48 + u * 0.2))
  
  pr = pr + h1 * cos(w1) + h2 * cos(w2)
  
end sub
   
  '''''''''''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''''''''''
  
sub latlong(sl, sr, pl, pb, pr, gl, gb, rpgm)
  
  ' geocentric mean coordinates subroutine
  
  ''''''''''''''''''''''''''''''''''''''''
  
  local xs, ys, xp, yp, zp, x, y, z
  
  ' heliocentric ecliptic position of the sun (au)
  
  xs = sr * cos(sl)
  
  ys = sr * sin(sl)
  
  ' heliocentric ecliptic position of planet (au)
  
  xp = pr * cos(pb) * cos(pl)
  
  yp = pr * cos(pb) * sin(pl)
  
  zp = pr * sin(pb)
  
  ' geocentric ecliptic position of planet (au)
  
  x = xp + xs
  
  y = yp + ys
  
  z = zp
  
  ' mean geocentric longitude of planet (radians)
  
  gl = atan3(y, x)
  
  ' mean geocentric latitude of planet (radians)
  
  gb = atan2(z, sqr(x^2 + y^2))
  
  ' geocentric distance of planet (au)
  
  rpgm = sqr(x * x + y * y + z * z)
  
end sub
  
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
sub abernu(jdtdb, ibody%, gl, gb, alon, alat, rasc, decl)
  
  ' aberration and nutation corrections subroutine
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
  local u, a1, a2, dpsi, deps, epsi
  
  local xce, xse, xcl, xsl, xcb, xsb
  
  ' fundamental time argument
  
  u = (jdtdb - 2451545.0) / 3652500.0
  
  select case ibody%
      
    case 1
      
      ' mercury
      
      alon = gl + 0.0000001 * (-1261.0 + 1485.0 * cos(2.649 + 198048.273 * u))
      
      alon = alon + 0.0000001 * (305.0 * cos(5.71 + 458927.03 * u) + 230.0 * cos(5.3 + 396096.55 * u))
      
      alat = gb + 0.000019 * cos(0.42 + 260879.41 * u)
      
    case 2
      
      ' venus
      
      alon = gl + 0.0000001 * (-1304.0 + 1016.0 * cos(1.423 + 39302.097 * u))
      
      alon = alon + 0.0000001 * (224.0 * cos(2.85 + 78604.19 * u) + 98.0 * cos(4.27 + 117906.29 * u))
      
      alat = gb
                  
  end select
  
  ' nutation corrections
  
  a1 = 2.18 + u * (-3375.7 + u * 0.36)
  
  a2 = 3.51 + u * (125666.39 + u * 0.1)
  
  dpsi = 0.0000001 * (-834.0 * sin(a1) - 64.0 * sin(a2))
  
  deps = 0.0000001 * u * (-226938.0 + u * (-75.0 + u * (96926.0 + u * (-2491.0 - u * 12104.0))))
  
  epsi = 0.0000001 * (4090928.0 + 446.0 * cos(a1) + 28.0 * cos(a2)) + deps
  
  alon = modulo(alon + dpsi)
  
  ' compute right ascension and declination (radians)
  
  xce = cos(epsi)
  
  xse = sin(epsi)
  
  xcl = cos(alon)
  
  xsl = sin(alon)
  
  xcb = cos(alat)
  
  xsb = sin(alat)
  
  decl = asin(xce * xsb + xse * xcb * xsl)
  
  rasc = atan3(-xse * xsb + xce * xcb * xsl, xcb * xcl)
  
end sub
  
  '''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''
  
sub nut2000_lp(jdate, dpsi, deps)
  
  ' low precison nutation based on iau 2000a
  
  ' this function evaluates a short nutation series and returns approximate
  ' values for nutation in longitude and nutation in obliquity for a given
  ' tdb julian date. in this mode, only the largest 13 terms of the iau 2000a
  ' nutation series are evaluated.
  
  ' input
  
  '  jdate = tdb julian date
  
  ' output
  
  '  dpsi = nutation in longitude in arcseconds
  
  '  deps = nutation in obliquity in arcseconds
  
  '''''''''''''''''''''''''''''''''''''''''''''
  
  LOCAL rev = 360.0 * 3600.0
  
  LOCAL el, elp, f, d, omega
  
  LOCAL i%, arg
  
  LOCAL t = (jdate - 2451545.0) / 36525.0
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  ' computation of fundamental (delaunay) arguments from simon et al. (1994)
  ''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
  
  ' mean anomaly of the moon
  
  el = (485868.249036 + t * (1717915923.2178 + t * (31.8792 + t * (0.051635 + t * (-0.00024470)))) mod rev) / seccon
  
  ' mean anomaly of the sun
  
  elp = (1287104.79305 + t * (129596581.0481 + t * (-0.5532 + t * (0.000136 + t * (- 0.00001149)))) mod rev) / seccon
  
  ' mean argument of the latitude of the moon
  
  f = (335779.526232 + t * (1739527262.8478 + t * (-12.7512 + t * (-0.001037 + t * (0.00000417)))) mod rev) / seccon
  
  ' mean elongation of the moon from the sun
  
  d = (1072260.70369 + t * (1602961601.2090 + t * (- 6.3706 + t * (0.006593 + t * (- 0.00003169)))) mod rev) / seccon
  
  ' mean longitude of the ascending node of the moon (from simon section 3.4(b.3), precession = 5028.8200 arcsec/cy)
  
  omega = (450160.398036 + t * (- 6962890.5431 + t * (7.4722 + t * (0.007702 + t * (- 0.00005939)))) mod rev) / seccon
  
  dpsi = 0.0
  
  deps = 0.0
  
  ' sum nutation series terms
  
  for i% = 13 to 1 step -1
    
    arg = xnut(1, i%) * el + xnut(2, i%) * elp + xnut(3, i%) * f + xnut(4, i%) * d + xnut(5, i%) * omega
    
    dpsi = (xnut(6, i%) + xnut(7, i%) * t) * sin(arg) + xnut(10, i%) * cos(arg) + dpsi
    
    deps = (xnut(8, i%) + xnut(9, i%) * t) * cos(arg) + xnut(11, i%) * sin(arg) + deps
    
  next i%
  
  dpsi = 1.0e-7 * dpsi
  
  deps = 1.0e-7 * deps
  
  ' add in out-of-phase component of principal (18.6-year) term
  ' (to avoid small but long-term bias in results)
  
  dpsi = dpsi + 0.0033 * cos(omega)
  
  deps = deps + 0.0015 * sin(omega)
  
END sub
  
  ''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''
  
sub getdate (month, day, year)
  
  ' request calendar date subroutine
  
  do
    print " "
    print "please input the calendar date"
    print " "
    print "(month [1 - 12], day [1 - 31], year [yyyy])"
    print "< for example, october 21, 1986 is input as 10,21,1986 >"
    print "< b.c. dates are negative, a.d. dates are positive >"
    print "< the day of the month may also include a decimal part >"
    print " "
    input month, day, year
    
  loop until (month >= 1 and month <= 12) and (day >= 1 and day <= 31)
  
end sub
  
  '''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''
  
sub observer(obslat, obslong, obsalt)
  
  ' interactive request of latitude, longitude and altitude subroutine
  
  ' output
  
  '  obslat  = latitude (radians)
  '  obslong = longitude (radians)
  '  obsalt  = altitude (meters)
  
  ''''''''''''''''''''''''''''''
  
  do
    
    print "please input the geographic latitude of the observer"
    print "(degrees [-90 to +90], minutes [0 - 60], seconds [0 - 60])"
    print "(north latitudes are positive, south latitudes are negative)"
    
    input obslat.deg$, obslat.min, obslat.sec
    
  loop until (abs(val(obslat.deg$)) <= 90.0 and (obslat.min >= 0.0 and obslat.min <= 60.0) and (obslat.sec >= 0.0 and obslat.sec <= 60.0))
  
  if (left$(obslat.deg$, 2) = "-0") then
    
    obslat = -dtr * (obslat.min / 60.0 + obslat.sec / 3600.0)
    
  elseif (val(obslat.deg$) = 0.0) then
    
    obslat = dtr * (obslat.min / 60.0 + obslat.sec / 3600.0)
    
  else
    
    obslat = dtr * (sgn(val(obslat.deg$)) * (abs(val(obslat.deg$)) + obslat.min / 60.0 + obslat.sec / 3600.0))
    
  endif
  
  do
    
    print
    print "please input the geographic longitude of the observer"
    print "(degrees [0 - 360], minutes [0 - 60], seconds [0 - 60])"
    print "(east longitude is positive, west longitude is negative)"
    
    input obslong.deg$, obslong.min, obslong.sec
    
  loop until (abs(val(obslong.deg$)) >= 0.0 and abs(val(obslong.deg$)) <= 360.0) and (obslong.min >= 0.0 and obslong.min <= 60.0) and (obslong.sec >= 0.0 and obslong.sec <= 60.0)
  
  if (left$(obslong.deg$, 2) = "-0") then
    
    obslong = -dtr * (obslong.min / 60 + obslong.sec / 3600)
    
  elseif (val(obslong.deg$) = 0.0) then
    
    obslong = dtr * (obslong.min / 60.0 + obslong.sec / 3600.0)
    
  else
    
    obslong = dtr * (sgn(val(obslong.deg$)) * (abs(val(obslong.deg$)) + obslong.min / 60.0 + obslong.sec / 3600.0))
    
  endif
  
  print " "
  
  print "please input the altitude of the observer (meters)"
  print "(positive above sea level, negative below sea level)"
  
  input obsalt
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub julian(month, day, year, jday)
  
  ' Gregorian date to julian day subroutine
  
  ' input
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year (all four digits)
  
  ' output
  
  '  jday = julian day
  
  ' special notes
  
  '  (1) calendar year must include all digits
  
  '  (2) will report October 5, 1582 to October 14, 1582
  '      as invalid calendar dates and exit
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local a, b, c, m, y
  
  y = year
  
  m = month
  
  b = 0.0
  
  c = 0.0
  
  if (m <= 2.0) then
    
    y = y - 1.0
    
    m = m + 12.0
    
  end if
  
  if (y < 0.0) then c = -0.75
  
  if (year < 1582.0) then
    
    ' null
    
  elseif (year > 1582.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (month < 10.0) then
    
    ' null
    
  elseif (month > 10.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (day <= 4.0) then
    
    ' null
    
  elseif (day > 14.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  else
    
    print "this date does not exist!!"
    
    exit
    
  end if
  
  jday = fix(365.25 * y + c) + fix(30.6001 * (m + 1.0)) + day + b + 1720994.5
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub gdate (jday, month, day, year)
  
  ' Julian day to Gregorian date subroutine
  
  ' input
  
  '  jday = julian day
  
  ' output
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year
  
  ''''''''''''''''''''''''
  
  local a, b, c, d, e, f, z, alpha
  
  z = fix(jday + 0.5)
  
  f = jday + 0.5 - z
  
  if (z < 2299161) then
    
    a = z
    
  else
    
    alpha = fix((z - 1867216.25) / 36524.25)
    
    a = z + 1.0 + alpha - fix(alpha / 4.0)
    
  end if
  
  b = a + 1524.0
  
  c = fix((b - 122.1) / 365.25)
  
  d = fix(365.25 * c)
  
  e = fix((b - d) / 30.6001)
  
  day = b - d - fix(30.6001 * e) + f
  
  if (e < 13.5) then
    
    month = e - 1.0
    
  else
    
    month = e - 13.0
    
  end if
  
  if (month > 2.5) then
    
    year = c - 4716.0
    
  else
    
    year = c - 4715.0
    
  end if
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub utc2tdb (jdutc, jdtdb)
  
  ' convert UTC julian date to TDB julian date
  
  ' input
  
  '  jdutc   = UTC julian day
  
  ' output
  
  '  jdtdb = TDB julian day
  
  ' Reference Frames in Astronomy and Geophysics
  ' J. Kovalevsky et al., 1989, pp. 439-442
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local corr, jdtt, t, leapsecond
  
  ' find current number of leap seconds
  
  findleap(jdutc, leapsecond)
  
  ' compute TDT julian date
  
  corr = (leapsecond + 32.184) / 86400.0
  
  jdtt = jdutc + corr
  
  ' time argument for correction
  
  t = (jdtt - 2451545.0) / 36525.0
  
  ' compute correction in microseconds
  
  corr = 1656.675 * sin(dtr * (35999.3729 * t + 357.5287))
  corr = corr + 22.418     * sin(dtr * (32964.467  * t + 246.199))
  corr = corr + 13.84      * sin(dtr * (71998.746  * t + 355.057))
  corr = corr +  4.77      * sin(dtr * ( 3034.906  * t +  25.463))
  corr = corr +  4.677     * sin(dtr * (34777.259  * t + 230.394))
  corr = corr + 10.216 * t * sin(dtr * (35999.373  * t + 243.451))
  corr = corr +  0.171 * t * sin(dtr * (71998.746  * t + 240.98 ))
  corr = corr +  0.027 * t * sin(dtr * ( 1222.114  * t + 194.661))
  corr = corr +  0.027 * t * sin(dtr * ( 3034.906  * t + 336.061))
  corr = corr +  0.026 * t * sin(dtr * (  -20.186  * t +   9.382))
  corr = corr +  0.007 * t * sin(dtr * (29929.562  * t + 264.911))
  corr = corr +  0.006 * t * sin(dtr * (  150.678  * t +  59.775))
  corr = corr +  0.005 * t * sin(dtr * ( 9037.513  * t + 256.025))
  corr = corr +  0.043 * t * sin(dtr * (35999.373  * t + 151.121))
  
  ' convert corrections to days
  
  corr = 0.000001 * corr / 86400.0
  
  ' TDB julian date
  
  jdtdb = jdtt + corr
  
end sub
  
  ''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''
  
sub findleap(jday, leapsecond)
  
  ' find number of leap seconds for utc julian day
  
  ' input
  
  '  jday = utc julian day
  
  ' input via global
  
  '  jdleap  = array of utc julian dates
  '  leapsec = array of leap seconds
  
  ' output
  
  '  leapsecond = number of leap seconds
  
  ''''''''''''''''''''''''''''''''''''''
  
  if (jday <= jdleap(1)) then
    
    ' date is <= 1972; set to first data element
    
    leapsecond = leapsec(1)
    
    exit sub
    
  end if
  
  if (jday >= jdleap(28)) then
    
    ' date is >= end of current data
    ' set to last data element
    
    leapsecond = leapsec(28)
    
    exit sub
    
  end if
  
  ' find data within table
  
  for i% = 1 to 27
    
    if (jday >= jdleap(i%) and jday < jdleap(i% + 1)) then
      
      leapsecond = leapsec(i%)
      
      exit sub
      
    end if
    
  next i%
  
end sub
  
  '''''''''''''''''''''''''
  '''''''''''''''''''''''''
  
function modulo(x) as float
  
  ' modulo 2 pi function
  
  ''''''''''''''''''''''
  
  local a
  
  a = x - pi2 * fix(x / pi2)
  
  if (a < 0.0) then
    
    a = a + pi2
    
  end if
  
  modulo = a
  
end function
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
function atan3(a, b) as float
  
  ' four quadrant inverse tangent function
  
  ' input
  
  '  a = sine of angle
  '  b = cosine of angle
  
  ' output
  
  '  atan3 = angle (0 =< atan3 <= 2 * pi; radians)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
  local c
  
  if (abs(a) < 1.0e-10) then
    
    atan3 = (1.0 - sgn(b)) * pidiv2
    
    exit function
    
  else
    
    c = (2.0 - sgn(a)) * pidiv2
    
  endif
  
  if (abs(b) < 1.0e-10) then
    
    atan3 = c
    
    exit function
    
  else
    
    atan3 = c + sgn(a) * sgn(b) * (abs(atn(a / b)) - pidiv2)
    
  endif
  
end function
  
  ''''''''''''''''''
  ''''''''''''''''''
  
function vecmag(a())
  
  ' vector magnitude function
  
  ' input
  
  '  { a } = column vector ( 3 rows by 1 column )
  
  ' output
  
  '  vecmag = scalar magnitude of vector { a }
  
  vecmag = sqr(a(1) * a(1) + a(2) * a(2) + a(3) * a(3))
  
end function
  
  ''''''''''''''''''''
  ''''''''''''''''''''
  
sub uvector (a(), b())
  
  ' unit vector subroutine
  
  ' input
  
  '  a = column vector (3 rows by 1 column)
  
  ' output
  
  '  b = unit vector (3 rows by 1 column)
  
  '''''''''''''''''''''''''''''''''''''''
  
  local amag = vecmag(a())
  
  for i% = 1 to 3
    
    if (amag <> 0.0) then
      
      b(i%) = a(i%) / amag
      
    else
      
      b(i%) = 0.0
      
    end if
    
  next i%
  
end sub
  
  '''''''''''''''''''''
  '''''''''''''''''''''
  
function vdot(a(), b())
  
  ' vector dot product function
  
  ' c = { a } dot { b }
  
  ' input
  
  '  n%    = number of rows
  '  { a } = column vector with n rows
  '  { b } = column vector with n rows
  
  ' output
  
  '  vdot = dot product of { a } and { b }
  
  ''''''''''''''''''''''''''''''''''''''''
  
  local c = 0.0
  
  for i% = 1 to 3
    
    c = c + a(i%) * b(i%)
    
  next i%
  
  vdot = c
  
end function
  
  '''''''''''''''''''''''
  '''''''''''''''''''''''
  
sub vcross(a(), b(), c())
  
  ' vector cross product subroutine
  
  ' { c } = { a } x { b }
  
  ' input
  
  '  { a } = vector a ( 3 rows by 1 column )
  '  { b } = vector b ( 3 rows by 1 column )
  
  ' output
  
  '  { c } = { a } x { b } ( 3 rows by 1 column )
  
  c(1) = a(2) * b(3) - a(3) * b(2)
  c(2) = a(3) * b(1) - a(1) * b(3)
  c(3) = a(1) * b(2) - a(2) * b(2)
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub matxvec(a(), b(), c())
  
  ' matrix/vector multiplication subroutine
  
  ' { c } = [ a ] * { b }
  
  ' input
  
  '  a  = matrix a ( 3 rows by 3 columns )
  '  b  = vector b ( 3 rows )
  
  ' output
  
  '  c = vector c ( 3 rows )
  
  ''''''''''''''''''''''''''
  
  local s, i%, j%
  
  for i% = 1 to 3
    
    s = 0.0
    
    for j% = 1 to 3
      
      s = s + a(i%, j%) * b(j%)
      
    next j%
    
    c(i%) = s
    
  next i%
  
end sub
  
  ''''''''''''''''''''''
  ''''''''''''''''''''''
  
sub transpose (a(), b())
  
  ' matrix traspose subroutine
  
  ' input
  
  '  m = number of rows in matrix [ a ]
  '  n = number of columns in matrix [ a ]
  '  a = matrix a ( 3 rows by 3 columns )
  
  ' output
  
  '  b = matrix transpose ( 3 rows by 3 columns )
  
  '''''''''''''''''''''''''''''''''''''''''''''''
  
  local i%, j%
  
  for i% = 1 to 3
    
    for j% = 1 to 3
      
      b(i%, j%) = a(j%, i%)
      
    next j%
    
  next i%
  
end sub
  
  '''''''''''''''
  '''''''''''''''
  
sub jd2str(jdutc)
  
  ' convert julian day to calendar date and UTC time
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''
  
  gdate (jdutc, cmonth, day, year)
  
  print "calendar date  ", month$(cmonth) + " " + STR$(int(day)) + " " + str$(year)
  
  print " "
  
  thr0 = 24.0 * (day - int(day))
  
  thr = int(thr0)
  
  tmin0 = 60.0 * (thr0 - thr)
  
  tmin = int(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 60.0) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1.0
    
  end if
  
  print "UTC time       ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub deg2str(dd, dms$)
  
  ' convert decimal degrees to degrees,
  ' minutes, seconds string
  
  ' input
  
  '  dd = angle in decimal degrees
  
  ' output
  
  '  dms$ = string equivalent
  
  '''''''''''''''''''''''''''
  
  local d1, d, m, s
  
  d1 = abs(dd)
  
  d = fix(d1)
  
  d1 = (d1 - d) * 60.0
  
  m = fix(d1)
  
  s = (d1 - m) * 60.0
  
  if (dd < 0.0) then
    
    if (d <> 0.0) then
      
      d = -d
      
    elseif (m <> 0.0) then
      
      m = -m
      
    else
      
      s = -s
      
    end if
    
  end if
  
  dms$ = str$(d) + " deg " + str$(m) + " min " + str$(s, 0, 2) + " sec"
  
end sub
  
  ''''''''''''''''
  ''''''''''''''''
  
sub hrs2str(hours)
  
  ' convert hours to equivalent string
  
  ''''''''''''''''''''''''''''''''''''
  
  local thr, tmin0, tmin, tsec
  
  thr = fix(hours)
  
  tmin0 = 60.0 * (hours - thr)
  
  tmin = fix(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 60.0) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1
    
  end if
  
  print " "
  
  print "event duration  ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub

  ''''''''''
  ''''''''''
  
sub keycheck

    ' check user response subroutine
    
    ''''''''''''''''''''''''''''''''
    
    local a$

    print ""
    
    print "< press Enter key to continue >"
    
    a$ = ""
    
    do while a$ = ""
    
       a$ = inkey$
       
    loop

end sub

  '''''''''''
  '''''''''''
  
sub read_data
  
  ' read data subroutine
  
  ''''''''''''''''''''''
  
  local i%, j%
  
  ' read data for the sun
  
  for i% = 1 to 50
    
    read xsl(i%), xsr(i%), xsa(i%), xsb(i%)
    
  next i%

  ' read data for Mercury and Venus
    
  for i% = 1 to 88
    
    read cl(i%), al(i%), bl(i%)
      
  next i%
  
  ' data for the sun - longitude and radius vector
  
  data 403406,      0, 4.721964,      1.621043
  data 195207, -97597, 5.937458,  62830.348067
  data 119433, -59715, 1.115589,  62830.821524
  data 112392, -56188, 5.781616,  62829.634302
  data   3891,  -1556, 5.5474  , 125660.5691
  data   2819,  -1126, 1.5120  , 125660.9845
  data   1721,   -861, 4.1897  ,  62832.4766
  data      0,    941, 1.163   ,      0.813
  data    660,   -264, 5.415   , 125659.310
  data    350,   -163, 4.315   ,  57533.850
  data    334,      0, 4.553   ,    -33.931
  data    314,    309, 5.198   , 777137.715
  data    268,   -158, 5.989   ,  78604.191
  data    242,      0, 2.911   ,      5.412
  data    234,    -54, 1.423   ,  39302.098
  data    158,      0, 0.061   ,    -34.861
  data    132,    -93, 2.317   , 115067.698
  data    129,    -20, 3.193   ,  15774.337
  data    114,      0, 2.828   ,   5296.670
  data     99,    -47, 0.52    ,  58849.27
  data     93,      0, 4.65    ,   5296.11
  data     86,      0, 4.35    ,  -3980.70
  data     78,    -33, 2.75    ,  52237.69
  data     72,    -32, 4.50    ,  55076.47
  data     68,      0, 3.23    ,    261.08
  data     64,    -10, 1.22    ,  15773.85
  data     46,    -16, 0.14    , 188491.03
  data     38,      0, 3.44    ,  -7756.55
  data     37,      0, 4.37    ,    264.89
  data     32,    -24, 1.14    , 117906.27
  data     29,    -13, 2.84    ,  55075.75
  data     28,      0, 5.96    ,  -7961.39
  data     27,     -9, 5.09    , 188489.81
  data     27,      0, 1.72    ,   2132.19
  data     25,    -17, 2.56    , 109771.03
  data     24,    -11, 1.92    ,  54868.56
  data     21,      0, 0.09    ,  25443.93
  data     21,     31, 5.98    , -55731.43
  data     20,    -10, 4.03    ,  60697.74
  data     18,      0, 4.27    ,   2132.79
  data     17,    -12, 0.79    , 109771.63
  data     14,      0, 4.24    ,  -7752.82
  data     13,     -5, 2.01    , 188491.91
  data     13,      0, 2.65    ,    207.81
  data     13,      0, 4.98    ,  29424.63
  data     12,      0, 0.93    ,     -7.99
  data     10,      0, 2.21    ,  46941.14
  data     10,      0, 3.59    ,    -68.29
  data     10,      0, 1.50    ,  21463.25
  data     10,     -9, 2.55    , 157208.40
  
  ' data for mercury - heliocentric longitude, latitude and radius vector
  
  data 510728, 6.09670 ,  521757.52364 , 404847, 4.72189,       1.62027
  data  91048, 2.8946  ,  782636.2744  ,  30594, 4.1535 ,  521758.6270
  data  15769, 5.8003  , 1043515.073   ,  13726, 0.4656 ,  521756.9570
  data  11582, 1.0266  ,  782637.2016  ,   7633, 3.517  ,  521759.335
  data   5247, 0.418   ,  782638.007   ,   4001, 3.993  , 1043516.352
  data   3299, 2.791   , 1304393.680   ,   3212, 0.209  , 1043514.724
  data   1690, 2.067   , 1304394.627   ,   1482, 6.174  , 1304395.168
  data   1233, 3.606   ,  782635.409   ,   1152, 5.856  , 1565272.646
  data    845, 2.63    , 1043516.88    ,    654, 3.40   , 1565273.50
  data    359, 2.66    , 1826151.56    ,    356, 3.08   ,   11094.77
  data    257, 6.27    , 1826152.20    ,    246, 2.89   ,       5.41
  data    180, 5.67    ,   56613.61    ,    159, 4.57   ,  250285.49
  data    137, 6.17    ,  271973.50
  
  data 680303, 3.82625 ,  260879.17693 , 538354, 3.30009,  260879.66625
  data 176935, 3.74070 ,       0.40005 , 143323, 0.58073,  521757.92658
  data 105214, 0.0545  ,  521758.44880 ,  91011, 3.3915 ,       0.9954
  data  47427, 1.9266  ,  260878.2610  ,  41669, 3.5084 ,  782636.7624
  data  19826, 3.1539  ,  782637.4813  ,  12963, 0.2455 , 1043515.6610
  data   8233, 4.886   ,  521756.972   ,   6399, 0.358  ,  782637.769
  data   3196, 3.253   , 1304394.380   ,   1536, 4.824  , 1043516.451
  data    824, 0.04    , 1565273.15    ,    819, 1.84   ,  782635.45
  data    324, 1.60    , 1304395.53    ,    201, 2.92   , 1826151.86
  
  data 780141, 6.202782,  260878.753962,  78942, 2.98062,  521757.50830
  data  12000, 6.0391  ,  782636.2640  ,   9839, 4.8422 ,  260879.3808
  data   2355, 5.062   ,       0.734   ,   2019, 2.898  , 1043514.987
  data   1974, 1.588   ,  521758.140   ,   1859, 0.805  ,  260877.716
  data    426, 4.601   ,  782636.915   ,    397, 5.976  , 1304393.735
  data    382, 3.86    ,  521756.47    ,    306, 1.87   , 1043515.34
  data    102, 0.62    ,  782635.28    ,     92, 2.60   , 1565272.52
  
  ' data for venus - heliocentric longitude, latitude and radius vector
  
  data 423015, 4.722173,      1.600752, 548  , 5.987  ,  78604.195
  data    346, 4.27    , 117906.29    , 253  , 2.95   ,      5.37
  data    237, 4.56    ,  39302.10    , 181  , 0.05   ,  15774.33
  data    153, 2.14    , 306400.25    , 144  , 5.73   ,  96835.94
  data     99, 0.09    ,    261.08    ,  98  , 6.18   , 306399.50
  data     89, 4.34    ,  15773.85    ,  85  , 2.86   ,  94378.51
  data     69, 2.85    ,   5296.67    ,  56  , 5.71   ,   1915.95
  data     55, 1.23    ,    264.89    ,  55  , 2.85   ,   7756.55
  data     50, 5.69    , 157208.38    ,  48  , 4.62   ,   5296.12
  data     43, 5.16    , 193671.89    ,  39  , 0.85   ,  94377.98
  
  data 590350, 1.759897, 102133.735253, 34737, 3.17478, 102133.01934
  data  13104, 0.2705  ,      2.0678  , 12910, 3.7446 , 102134.2721
  data   8591, 3.7878  ,      1.5631  ,  7015, 3.3730 ,      2.2248
  
  data   2101, 2.828   ,      0.361   ,   163, 2.85   ,  78604.20
  data    138, 1.13    , 117906.29    ,    50, 2.59   ,  96835.94
  data     37, 1.42    ,  39302.10
  
  ' read subset of IAU2000 nutation data
  
  for j% = 1 to 13
    
    for i% = 1 to 11
      
      read xnut(i%, j%)
      
    next i%
    
  next j%
  
  data  0,  0, 0,  0, 1, -172064161, -174666, 92052331,  9086,  33386, 15377
  data  0,  0, 2, -2, 2,  -13170906,   -1675,  5730336, -3015, -13696, -4587
  data  0,  0, 2,  0, 2,   -2276413,    -234,   978459,  -485,   2796,  1374
  data  0,  0, 0,  0, 2,    2074554,     207,  -897492,   470,   -698,  -291
  data  0,  1, 0,  0, 0,    1475877,   -3633,    73871,  -184,  11817, -1924
  data  0,  1, 2, -2, 2,    -516821,    1226,   224386,  -677,   -524,  -174
  data  1,  0, 0,  0, 0,     711159,      73,    -6750,     0,   -872,   358
  data  0,  0, 2,  0, 1,    -387298,    -367,   200728,    18,    380,   318
  data  1,  0, 2,  0, 2,    -301461,     -36,   129025,   -63,    816,   367
  data  0, -1, 2, -2, 2,     215829,    -494,   -95929,   299,    111,   132
  data  0,  0, 2, -2, 1,     128227,     137,   -68982,    -9,    181,    39
  data -1,  0, 2,  0, 2,     123457,      11,   -53311,    32,     19,    -4
  data -1,  0, 0,  2, 0,     156994,      10,    -1235,     0,   -168,    82

  ' calendar months
  
  month$(1) = "January"
  month$(2) = "February"
  month$(3) = "March"
  month$(4) = "April"
  month$(5) = "May"
  month$(6) = "June"
  month$(7) = "July"
  month$(8) = "August"
  month$(9) = "September"
  month$(10) = "October"
  month$(11) = "November"
  month$(12) = "December"
  
  ' read leap second data
  
  for i% = 1 to 28
    
    read jdleap(i%), leapsec(i%)
    
  next i%
  
  data 2441317.5,  10.0
  data 2441499.5,  11.0
  data 2441683.5,  12.0
  data 2442048.5,  13.0
  data 2442413.5,  14.0
  data 2442778.5,  15.0
  data 2443144.5,  16.0
  data 2443509.5,  17.0
  data 2443874.5,  18.0
  data 2444239.5,  19.0
  data 2444786.5,  20.0
  data 2445151.5,  21.0
  data 2445516.5,  22.0
  data 2446247.5,  23.0
  data 2447161.5,  24.0
  data 2447892.5,  25.0
  data 2448257.5,  26.0
  data 2448804.5,  27.0
  data 2449169.5,  28.0
  data 2449534.5,  29.0
  data 2450083.5,  30.0
  data 2450630.5,  31.0
  data 2451179.5,  32.0
  data 2453736.5,  33.0
  data 2454832.5,  34.0
  data 2456109.5,  35.0
  data 2457204.5,  36.0
  data 2457754.5,  37.0
     
end sub
