  ' moon_phases.bas         July 17, 2017
  
  ' this MMBASIC program calculates the calendar date
  ' and UTC time of the phases of the Moon
  
  ' MMBASIC eXtreme, RasPi and DOS version
  
  ''''''''''''''''''''''''''''''''''''''''
  
  option default float
  
  option base 1
  
  ' dimension global arrays
  
  dim jdleap(28), leapsec(28), month$(12)
  
  dim nmdata(25), fmdata(25), qdata(25)
  
  dim iphase%, jdayi
  
  ' global constants
  
  const pi2 = 2.0 * pi, pidiv2 = 0.5 * pi, dtr = pi / 180.0
  
  CONST rtd = 180.0 / pi, atr = dtr / 3600.0
  
  ' read initial guess data
  
  for i% = 1 to 25
  
    ' new moon, full moon, first and last quarter
    
    read nmdata(i%), fmdata(i%), qdata(i%)
    
  next i%
  
  DATA -0.40720, -0.40614, -0.62801
  DATA  0.17241,  0.17302,  0.17172
  DATA  0.01608,  0.01614, -0.01183
  DATA  0.01039,  0.01043,  0.00862
  DATA  0.00739,  0.00734,  0.00804
  DATA -0.00514, -0.00515, -0.00454
  DATA  0.00208,  0.00209,  0.00204
  DATA -0.00111, -0.00111, -0.00180
  DATA -0.00057, -0.00057, -0.00070
  DATA  0.00056,  0.00056, -0.00040
  DATA -0.00042, -0.00042, -0.00034
  DATA  0.00042,  0.00042,  0.00032
  DATA  0.00038,  0.00038,  0.00032
  DATA -0.00024, -0.00024, -0.00028
  DATA -0.00017, -0.00017,  0.00027
  DATA -0.00007, -0.00007, -0.00017
  DATA  0.00004,  0.00004, -0.00005
  DATA  0.00004,  0.00004,  0.00004
  DATA  0.00003,  0.00003, -0.00004
  DATA  0.00003,  0.00003,  0.00004
  DATA -0.00003, -0.00003,  0.00003
  DATA  0.00003,  0.00003,  0.00003
  DATA -0.00002, -0.00002, -0.00002
  DATA -0.00002, -0.00002,  0.00002
  DATA  0.00002,  0.00002, -0.00002
  
  ' read leap second data
  
  for i% = 1 to 28
    
    read jdleap(i%), leapsec(i%)
    
  next i%
  
  data 2441317.5,  10.0
  data 2441499.5,  11.0
  data 2441683.5,  12.0
  data 2442048.5,  13.0
  data 2442413.5,  14.0
  data 2442778.5,  15.0
  data 2443144.5,  16.0
  data 2443509.5,  17.0
  data 2443874.5,  18.0
  data 2444239.5,  19.0
  data 2444786.5,  20.0
  data 2445151.5,  21.0
  data 2445516.5,  22.0
  data 2446247.5,  23.0
  data 2447161.5,  24.0
  data 2447892.5,  25.0
  data 2448257.5,  26.0
  data 2448804.5,  27.0
  data 2449169.5,  28.0
  data 2449534.5,  29.0
  data 2450083.5,  30.0
  data 2450630.5,  31.0
  data 2451179.5,  32.0
  data 2453736.5,  33.0
  data 2454832.5,  34.0
  DATA 2456109.5,  35.0
  data 2457204.5,  36.0
  data 2457754.5,  37.0
  
  ' calendar months
  
  month$(1) = "January"
  month$(2) = "February"
  month$(3) = "March"
  month$(4) = "April"
  month$(5) = "May"
  month$(6) = "June"
  month$(7) = "July"
  month$(8) = "August"
  month$(9) = "September"
  month$(10) = "October"
  month$(11) = "November"
  month$(12) = "December"
  
  ''''''''''''''''''
  ' begin simulation
  ''''''''''''''''''
  
  print " "
  print "phases of the Moon"
  print "=================="
  print " "
  
  do
    
    ' request calendar month
    
    print "please input the calendar month (1 = January, 2 = February, etc.)"
    
    input cmonthi
    
  loop until(cmonthi >= 1 and cmonthi <= 12)
  
  ' request calendar year
  
  print " "
  
  print "please input the calendar year (include all digits)"
  
  input cyeari
  
  ' process each lunar phase
  
  for iphase% = 1 to 4
    
    print " "
    
    select case iphase%
        
      case (1)
        
        print "new Moon"
        print "--------"
        
        phase_jday(cmonthi, cyeari, 0.0, jdtdb)
        
      case (2)
        
        print "first quarter"
        print "-------------"
        
        phase_jday(cmonthi, cyeari, 0.25, jdtdb)
        
      case (3)
        
        print "full Moon"
        print "---------"
        
        phase_jday(cmonthi, cyeari, 0.5, jdtdb)
        
      case (4)
        
        print "last quarter"
        print "------------"
        
        phase_jday(cmonthi, cyeari, 0.75, jdtdb)
        
    end select
    
    jdayi = jdtdb
    
    ' convergence criterion
    
    tol = 1.0e-8
    
    ' search interval in days
    
    x1 = -0.25
    
    x2 = 0.25
    
    realroot1(x1, x2, tol, xroot, froot)
    
    ' TDB julian day of event
    
    jdtdb = jdayi + xroot
    
    ' compute UTC julian day
    
    tdb2utc(jdtdb, jdutc)
    
    ' print results for each phase
    
    print " "
    
    jd2str(jdutc)
    
  next iphase%
  
  print " "
  
end
  
  '''''''''''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''''''''''
  
sub phase_jday(cmonth, cyear, deltak, jday)
  
  ' initial guess for phase epoch subroutine
  
  ' "Astronomical Algorithms" bj J. Meeus
  
  '''''''''''''''''''''''''''''''''''''''
  
  local ddays = 0, k = 0
  
  local tc = 0, tc2 = 0, tc3 = 0, tc4 = 0
  
  local ecc, m, mp, f, raan
  
  local a1, a2, a3, a4, a5, a6, a7
  
  local a8, a9, a10, a11, a12, a13, a14
  
  k = fix(12.3685 * (cyear + cmonth / 12.0 - 2000)) + deltak
  
  ' julian centuries
  
  tc = k / 1236.85
  
  tc2 = tc * tc
  
  tc3 = tc * tc * tc
  
  tc4 = tc * tc * tc * tc
  
  ' initial guess for TDB julian day
  
  jday = 2451550.09765 + 29.530588853 * k
  
  jday = jday + 0.0001337 * tc2
  
  jday = jday - 0.000000150 * tc3
  
  jday = jday + 0.00000000073 * tc4
  
  ' fundamental arguments (radians)
  
  ecc = 1.0 - 0.002516 * tc - 0.0000074 * tc2
  
  m =    dtr * (2.5534   + 29.10535669 * k  - 0.0000218 * tc2 - 0.00000011 * tc3)
  
  mp =   dtr * (201.5643 + 385.81693528 * k + 0.0107438 * tc2 + 0.00001239 * tc3 - 0.000000058 * tc4)
  
  f =    dtr * (160.7108 + 390.67050274     - 0.0016341 * tc2 - 0.00000227 * tc3 + 0.000000011 * tc4)
  
  raan = dtr * (124.7746 - 1.56375580 * k   + 0.0020691 * tc2 + 0.00000215 * tc3)
  
  a1 =  dtr * (299.77 + 0.107408  * k - 0.009173 * tc2)
  a2 =  dtr * (251.88 + 0.016321  * k)
  a3 =  dtr * (251.83 + 26.651886 * k)
  a4 =  dtr * (349.42 + 36.412478 * k)
  a5 =  dtr * (84.66  + 18.206239 * k)
  a6 =  dtr * (141.74 + 53.303771 * k)
  a7 =  dtr * (207.14 + 2.453732  * k)
  a8 =  dtr * (154.84 + 7.306860  * k)
  a9 =  dtr * (34.52  + 27.261239 * k)
  a10 = dtr * (207.19 + 0.121824  * k)
  a11 = dtr * (291.34 + 1.844379  * k)
  a12 = dtr * (161.72 + 24.198154 * k)
  a13 = dtr * (239.56 + 25.513099 * k)
  a14 = dtr * (331.55 + 3.592518  * k)
  
  ' correction for all phases in days
  
  ddays =         0.000325 * sin(a1) + 0.000056 * sin(a8)
  
  ddays = ddays + 0.000165 * sin(a2) + 0.000047 * sin(a9)
  
  ddays = ddays + 0.000164 * sin(a3) + 0.000042 * sin(a10)
  
  ddays = ddays + 0.000126 * sin(a4) + 0.000040 * sin(a11)
  
  ddays = ddays + 0.000110 * sin(a5) + 0.000037 * sin(a12)
  
  ddays = ddays + 0.000062 * sin(a6) + 0.000035 * sin(a13)
  
  ddays = ddays + 0.000060 * sin(a7) + 0.000023 * sin(a14)
  
  jday = jday + ddays
  
  if (iphase% = 2 or iphase% = 4) then
    
    ' w-correction for quarter phases only
    
    w = 0.00306 - 0.00038 * ecc * cos(m) + 0.00026 * cos(mp)
    
    w = w - 0.00002 * cos(mp - m) + 0.00002 * cos(mp + m)
    
    w = w + 0.00002 * cos(2 * f)
    
  end if
  
  if (iphase% = 2) then
    
    ' w-correction for first quarter
    
    jday = jday + w
    
  end if
  
  if (iphase% = 4) then
    
    ' w-correction for last quarter
    
    jday = jday - w
    
  end if
  
  if (iphase% = 1) then
    
    ' new moon correction
    
    ddays =         nmdata(1) * sin(mp)                + nmdata(2) * ecc * sin(m)           + nmdata(3) * sin(2 * mp)
    
    ddays = ddays + nmdata(4) * sin(2 * f)             + nmdata(5) * ecc * sin(mp - m)      + nmdata(6) * ecc * sin(mp + m)
    
    ddays = ddays + nmdata(7) * ecc * ecc * sin(2 * m) + nmdata(8) * sin(mp - 2 * f)        + nmdata(9) * sin(mp + 2 * f)
    
    ddays = ddays + nmdata(10) * ecc * sin(2 * mp + m) + nmdata(11) * sin(3 * mp)           + nmdata(12) * ecc * sin(m + 2 * f)
    
    ddays = ddays + nmdata(13) * ecc * sin(m - 2 * f)  + nmdata(14) * ecc * sin(2 * mp - m) + nmdata(15) * sin(raan)
    
    ddays = ddays + nmdata(16) * sin(mp + 2 * m)       + nmdata(17) * sin(2 * mp - 2 * f)   + nmdata(18) * sin(3 * m)
    
    ddays = ddays + nmdata(19) * sin(mp + m - 2 * f)   + nmdata(20) * sin(2 * mp + 2 * f)   + nmdata(21) * sin(mp + m + 2 * f)
    
    ddays = ddays + nmdata(22) * sin(mp + m + 2 * f)   + nmdata(23) * sin(mp - m - 2 * f)   + nmdata(24) * sin(3 * mp + m)
    
    ddays = ddays + nmdata(25) * sin(4 * mp)
    
    jday = jday + ddays
    
  end if
  
  if (iphase% = 3) then
    
    ' full moon correction in days
    
    ddays =         fmdata(1) * sin(mp)                + fmdata(2) * ecc * sin(m)           + fmdata(3) * sin(2 * mp)
    
    ddays = ddays + fmdata(4) * sin(2 * f)             + fmdata(5) * ecc * sin(mp - m)      + fmdata(6) * ecc * sin(mp + m)
    
    ddays = ddays + fmdata(7) * ecc * ecc * sin(2 * m) + fmdata(8) * sin(mp - 2 * f)        + fmdata(9) * sin(mp + 2 * f)
    
    ddays = ddays + fmdata(10) * ecc * sin(2 * mp + m) + fmdata(11) * sin(3 * mp)           + fmdata(12) * ecc * sin(m + 2 * f)
    
    ddays = ddays + fmdata(13) * ecc * sin(m - 2 * f)  + fmdata(14) * ecc * sin(2 * mp - m) + fmdata(15) * sin(raan)
    
    ddays = ddays + fmdata(16) * sin(mp + 2 * m)       + fmdata(17) * sin(2 * mp - 2 * f)   + fmdata(18) * sin(3 * m)
    
    ddays = ddays + fmdata(19) * sin(mp + m - 2 * f)   + fmdata(20) * sin(2 * mp + 2 * f)   + fmdata(21) * sin(mp + m + 2 * f)
    
    ddays = ddays + fmdata(22) * sin(mp + - m + 2 * f) + fmdata(23) * sin(mp - m - 2 * f)   + fmdata(24) * sin(3 * mp + m)
    
    ddays = ddays + fmdata(25) * sin(4 * mp)
    
    jday = jday + ddays
    
  end if
  
  if (iphase% = 2 or iphase% = 4) then
    
    ' first and last quarter correction
    
    ddays =         qdata(1) * sin(mp)                + qdata(2) * ecc * sin(m)           + qdata(3) * ecc * sin(mp + m)
    
    ddays = ddays + qdata(4) * sin(2 * mp)            + qdata(5) * sin(2 * f)             + qdata(6) * ecc * sin(mp - m)
    
    ddays = ddays + qdata(7) * ecc * ecc * sin(2 * m) + qdata(8) * sin(mp - 2 * f)        + qdata(9) * sin(mp + 2 * f)
    
    ddays = ddays + qdata(10) * sin(3 * mp)           + qdata(11) * ecc * sin(2 * mp - m) + qdata(12) * ecc * sin(m + 2 * f)
    
    ddays = ddays + qdata(13) * ecc * sin(m - 2 * f)  + qdata(14) * ecc * ecc * sin(mp + 2 * m) + qdata(15) * ecc * sin(2 * mp + m)
    
    ddays = ddays + qdata(16) * sin(raan)             + qdata(17) * sin(mp - m - 2 * f)   + qdata(18) * sin(2 * mp + 2 * f)
    
    ddays = ddays + qdata(19) * sin(mp + m + 2 * f)   + qdata(20) * sin(mp - 2 * m)       + qdata(21) * sin(mp + m - 2 * f)
    
    ddays = ddays + qdata(22) * sin(3 * mp)           + qdata(23) * sin(2 * mp - 2 * f)   + qdata(24) * sin(mp - m + 2 * f)
    
    ddays = ddays + qdata(25) * sin(3 * mp + m)
    
    jday = jday + ddays
    
  end if
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub phase_func(x, fx)
  
  ' lunar phase objective function
  
  ''''''''''''''''''''''''''''''''
  
  local jday, plon_sun, plon_moon
  
  ' current tdb julian day
  
  jday = jdayi + x
  
  ' compute solar and lunar geocentric ecliptic longitudes
  
  sun_moon(jday, plon_sun, plon_moon)
  
  ' compute value of current objective function
  
  select case iphase%
      
    case (1)
      
      ' new moon
      
      fx = plon_sun - plon_moon
      
    case (2)
      
      ' first quarter
      
      fx = (plon_moon - plon_sun) - pidiv2
      
    case (3)
      
      ' full moon
      
      fx = abs(plon_moon - plon_sun) - pi
      
    case (4)
      
      ' last quarter
      
      fx = (plon_sun - plon_moon) - pidiv2
      
  end select
  
end sub
  
  '''''''''''''''''''''''''''''''''
  '''''''''''''''''''''''''''''''''
  
sub utc2tdb (jdutc, tai_utc, jdtdb)
  
  ' convert UTC julian date to TDB julian date
  
  ' input
  
  '  jdutc   = UTC julian day
  '  tai_utc = TAI-UTC (seconds)
  
  ' output
  
  '  jdtdb = TDB julian day
  
  ' Reference Frames in Astronomy and Geophysics
  ' J. Kovalevsky et al., 1989, pp. 439-442
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local corr, jdtdt, t
  
  ' TDT julian date
  
  corr = (tai_utc + 32.184) / 86400.0
  
  jdtdt = jdutc + corr
  
  ' time argument for correction
  
  t = (jdtdt - 2451545.0) / 36525.0
  
  ' compute correction in microseconds
  
  corr = 1656.675 * sin(dtr * (35999.3729 * t + 357.5287))
  corr = corr + 22.418     * sin(dtr * (32964.467  * t + 246.199))
  corr = corr + 13.84      * sin(dtr * (71998.746  * t + 355.057))
  corr = corr +  4.77      * sin(dtr * ( 3034.906  * t +  25.463))
  corr = corr +  4.677     * sin(dtr * (34777.259  * t + 230.394))
  corr = corr + 10.216 * t * sin(dtr * (35999.373  * t + 243.451))
  corr = corr +  0.171 * t * sin(dtr * (71998.746  * t + 240.98 ))
  corr = corr +  0.027 * t * sin(dtr * ( 1222.114  * t + 194.661))
  corr = corr +  0.027 * t * sin(dtr * ( 3034.906  * t + 336.061))
  corr = corr +  0.026 * t * sin(dtr * (  -20.186  * t +   9.382))
  corr = corr +  0.007 * t * sin(dtr * (29929.562  * t + 264.911))
  corr = corr +  0.006 * t * sin(dtr * (  150.678  * t +  59.775))
  corr = corr +  0.005 * t * sin(dtr * ( 9037.513  * t + 256.025))
  corr = corr +  0.043 * t * sin(dtr * (35999.373  * t + 151.121))
  
  ' convert corrections to days
  
  corr = 0.000001 * corr / 86400.0
  
  ' TDB julian date
  
  jdtdb = jdtdt + corr
  
end sub
  
  ''''''''''''''''''''''''
  ''''''''''''''''''''''''
  
sub tdb2utc (jdtdb, jdutc)
  
  ' convert TDB julian day to UTC julian day subroutine
  
  ' input
  
  '  jdtdb = TDB julian day
  
  ' output
  
  '  jdutc = UTC julian day
  
  '''''''''''''''''''''''''
  
  local x1, x2, xroot, froot
  
  jdsaved = jdtdb
  
  ' set lower and upper bounds
  
  x1 = jdsaved - 0.1
  
  x2 = jdsaved + 0.1
  
  ' solve for UTC julian day using Brent's method
  
  realroot2(x1, x2, 1.0e-8, xroot, froot)
  
  jdutc = xroot
  
end sub
  
  '''''''''''''''''''
  '''''''''''''''''''
  
sub jdfunc (jdin, fx)
  
  ' objective function for tdb2utc
  
  ' input
  
  '  jdin = current value for UTC julian day
  
  ' output
  
  '  fx = delta julian day
  
  ''''''''''''''''''''''''
  
  local jdwrk, tai_utc
  
  findleap(jdin, tai_utc)
  
  utc2tdb(jdin, tai_utc, jdwrk)
  
  fx = jdwrk - jdsaved
  
end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub sun_moon (jday, plon_sun, plon_moon)
  
  ' lunar and solar longitudes subroutine
  
  '''''''''''''''''''''''''''''''''''''''
  
  local djd, l, t
  
  djd = jday - 2451545.0
  
  t = (djd / 36525.0) + 1.0
  
  gm = r2r(0.374897 + 0.03629164709 * djd)
  gm2 = modulo(2 * gm)
  gm3 = modulo(3 * gm)
  fm = r2r(0.259091 + 0.0367481952 * djd)
  fm2 = modulo(2 * fm)
  em = r2r(0.827362 + 0.03386319198 * djd)
  em2 = modulo(2 * em)
  em4 = modulo(4 * em)
  gs = r2r(0.993126 + 0.0027377785 * djd)
  lv = r2r(0.505498 + 0.00445046867 * djd)
  lm = r2r(0.606434 + 0.03660110129 * djd)
  ls = r2r(0.779072 + 0.00273790931 * djd)
  g2 = r2r(0.140023 + 0.00445036173 * djd)
  g4 = r2r(0.053856 + 0.00145561327 * djd)
  g5 = r2r(0.056531 + 0.00023080893 * djd)
  rm = r2r(0.347343 - 0.00014709391 * djd)
  
  l = 22640 * sin(gm) - 4586 * sin(gm - em2) + 2370 * sin(em2)
  l = l + 769 * sin(gm2) - 668 * sin(gs) - 412 * sin(fm2)
  l = l - 212 * sin(gm2 - em2) - 206 * sin(gm - em2 + gs)
  l = l + 192 * sin(gm + em2) + 165 * sin(em2 - gs)
  l = l + 148 * sin(gm - gs) - 125 * sin(em) - 110 * sin(gm + gs)
  l = l - 55 * sin(fm2 - em2) - 45 * sin(gm + fm2) + 40 * sin(gm - fm2)
  l = l - 38 * sin(gm - em4) + 36 * sin(gm3) - 31 * sin(gm2 - em4)
  l = l + 28 * sin(gm - em2 - gs) - 24 * sin(em2 + gs) + 19 * sin(gm - em)
  l = l + 18 * sin(em + gs) + 15 * sin(gm + em2 - gs) + 14 * sin(gm2 + em2)
  l = l + 14 * sin(em4) - 13 * sin(gm3 - em2) - 17 * sin(rm)
  l = l - 11 * sin(gm + 16 * ls - 18 * lv) + 10 * sin(gm2 - gs) + 9 * sin(gm - fm2 - em2)
  l = l + 9 * (cos(gm + 16 * ls - 18 * lv) - sin(gm2 - em2 + gs)) - 8 * sin(gm + em)
  l = l + 8 * (sin(2 * (em - gs)) - sin(gm2 + gs)) - 7 * (sin(2 * gs) + sin(gm - 2 * (em - gs)) - sin(rm))
  l = l - 6 * (sin(gm - fm2 + em2) + sin(fm2 + em2)) - 4 * (sin(gm - em4 + gs) - t * cos(gm + 16 * ls - 18 * lv))
  l = l - 4 * (sin(gm2 + fm2) - t * sin(gm + 16 * ls - 18 * lv))
  l = l + 3 * (sin(gm - 3 * em) - sin(gm + em2 + gs) - sin(gm2 - em4 + gs) + sin(gm - 2 * gs) + sin(gm - em2 - 2 * gs))
  l = l - 2 * (sin(gm2 - em2 - gs) + sin(fm2 - em2 + gs) - sin(gm + em4))
  l = l + 2 * (sin(4 * gm) + sin(em4 - gs) + sin(gm2 - em))
  
  plon_moon = lm + atr * l
  
  l = 6910 * sin(gs) + 72 * sin(2 * gs) - 17 * t * sin(gs)
  l = l - 7 * cos(gs - g5) + 6 * sin(lm - ls) + 5 * sin(4 * gs - 8 * g4 + 3 * g5)
  l = l - 5 * cos(2 * (gs - g2)) - 4 * (sin(gs - g2) - cos(4 * gs - 8 * g4 + 3 * g5))
  l = l + 3 * (sin(2 * (gs - g2)) - sin(g5) - sin(2 * (gs - g5)))
  
  plon_sun = ls + atr * (l - 17.0 * sin(rm))
  
end sub
  
  ''''''''''''''''''''''
  ''''''''''''''''''''''
  
function r2r(x) as float
  
  ' revolutions to radians function
  
  ' input
  
  '  x = argument (revolutions; 0 <= x <= 1)
  
  ' output
  
  '  r2r = equivalent x (radians; 0 <= y <= 2 pi)
  
  '''''''''''''''''''''''''''''''
  
  r2r = pi2 * (x - fix(x))
  
end function
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot1(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub phase_func
  
  '''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  phase_func(a, fa)
  
  phase_func(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0.0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0.0) then q = -q
      
      p = abs(p)
      
      min = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (min < tmp) then min = tmp
      
      if (2.0 * p < min) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    phase_func(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  ''''''''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''''''''
  
sub realroot2(x1, x2, tol, xroot, froot)
  
  ' real root of a single non-linear function subroutine
  
  ' input
  
  '  x1  = lower bound of search interval
  '  x2  = upper bound of search interval
  '  tol = convergence criter%ia
  
  ' output
  
  '  xroot = real root of f(x) = 0
  '  froot = function value
  
  ' note: requires sub jdfunc
  
  '''''''''''''''''''''''''''
  
  local eps, a, b, c, d, e, fa, fb, fcc, tol1
  
  local xm, p, q, r, s, xmin, tmp
  
  eps = 2.23e-16
  
  e = 0.0
  
  a = x1
  
  b = x2
  
  jdfunc(a, fa)
  
  jdfunc(b, fb)
  
  fcc = fb
  
  for iter% = 1 to 50
    
    if (fb * fcc > 0.0) then
      
      c = a
      
      fcc = fa
      
      d = b - a
      
      e = d
      
    end if
    
    if (abs(fcc) < abs(fb)) then
      
      a = b
      
      b = c
      
      c = a
      
      fa = fb
      
      fb = fcc
      
      fcc = fa
      
    end if
    
    tol1 = 2.0 * eps * abs(b) + 0.5 * tol
    
    xm = 0.5 * (c - b)
    
    if (abs(xm) <= tol1 or fb = 0) then exit for
    
    if (abs(e) >= tol1 and abs(fa) > abs(fb)) then
      
      s = fb / fa
      
      if (a = c) then
        
        p = 2.0 * xm * s
        
        q = 1.0 - s
        
      else
        
        q = fa / fcc
        
        r = fb / fcc
        
        p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))
        
        q = (q - 1.0) * (r - 1.0) * (s - 1.0)
        
      end if
      
      if (p > 0) then q = -q
      
      p = abs(p)
      
      xmin = abs(e * q)
      
      tmp = 3.0 * xm * q - abs(tol1 * q)
      
      if (xmin < tmp) then xmin = tmp
      
      if (2.0 * p < xmin) then
        
        e = d
        
        d = p / q
        
      else
        
        d = xm
        
        e = d
        
      end if
      
    else
      
      d = xm
      
      e = d
      
    end if
    
    a = b
    
    fa = fb
    
    if (abs(d) > tol1) then
      
      b = b + d
      
    else
      
      b = b + sgn(xm) * tol1
      
    end if
    
    jdfunc(b, fb)
    
  next iter%
  
  froot = fb
  
  xroot = b
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub julian(month, day, year, jday)
  
  ' Gregorian date to julian day subroutine
  
  ' input
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year (all four digits)
  
  ' output
  
  '  jday = julian day
  
  ' special notes
  
  '  (1) calendar year must include all digits
  
  '  (2) will report October 5, 1582 to October 14, 1582
  '      as invalid calendar dates and exit
  
  '''''''''''''''''''''''''''''''''''''''''
  
  local a, b, c, m, y
  
  y = year
  
  m = month
  
  b = 0.0
  
  c = 0.0
  
  if (m <= 2.0) then
    
    y = y - 1.0
    
    m = m + 12.0
    
  end if
  
  if (y < 0.0) then c = -0.75
  
  if (year < 1582.0) then
    
    ' null
    
  elseif (year > 1582.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (month < 10.0) then
    
    ' null
    
  elseif (month > 10.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  elseif (day <= 4.0) then
    
    ' null
    
  elseif (day > 14.0) then
    
    a = fix(y / 100.0)
    
    b = 2.0 - a + fix(a / 4.0)
    
  else
    
    print "this date does not exist!!"
    
    exit
    
  end if
  
  jday = fix(365.25 * y + c) + fix(30.6001 * (m + 1.0)) + day + b + 1720994.5
  
end sub
  
  ''''''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''''''
  
sub gdate (jday, month, day, year)
  
  ' Julian day to Gregorian date subroutine
  
  ' input
  
  '  jday = julian day
  
  ' output
  
  '  month = calendar month
  '  day   = calendar day
  '  year  = calendar year
  
  ''''''''''''''''''''''''
  
  local a, b, c, d, e, f, z, alpha
  
  z = fix(jday + 0.5)
  
  f = jday + 0.5 - z
  
  if (z < 2299161) then
    
    a = z
    
  else
    
    alpha = fix((z - 1867216.25) / 36524.25)
    
    a = z + 1.0 + alpha - fix(alpha / 4.0)
    
  end if
  
  b = a + 1524.0
  
  c = fix((b - 122.1) / 365.25)
  
  d = fix(365.25 * c)
  
  e = fix((b - d) / 30.6001)
  
  day = b - d - fix(30.6001 * e) + f
  
  if (e < 13.5) then
    
    month = e - 1.0
    
  else
    
    month = e - 13.0
    
  end if
  
  if (month > 2.5) then
    
    year = c - 4716.0
    
  else
    
    year = c - 4715.0
    
  end if
  
end sub
  
  '''''''''''''''
  '''''''''''''''
  
sub jd2str(jdutc)
  
  ' convert julian day to calendar date and UTC time
  
  ''''''''''''''''''''''''''''''''''''''''''''''''''
  
  local cmonth, day, year
  
  gdate(jdutc, cmonth, day, year)
  
  print "calendar date  ", month$(cmonth) + " " + STR$(int(day)) + " " + str$(year)
  
  print " "
  
  thr0 = 24.0 * (day - int(day))
  
  thr = int(thr0)
  
  tmin0 = 60.0 * (thr0 - thr)
  
  tmin = int(tmin0)
  
  tsec = 60.0 * (tmin0 - tmin)
  
  ' fix seconds and minutes for rollover
  
  if (tsec >= 60.0) then
    
    tsec = 0.0
    
    tmin = tmin + 1.0
    
  end if
  
  ' fix minutes for rollover
  
  if (tmin >= 60.0) then
    
    tmin = 0.0
    
    thr = thr + 1.0
    
  end if
  
  print "UTC time       ", str$(thr) + " hours " + str$(tmin) + " minutes " + str$(tsec, 0, 2) + " seconds"
  
end sub
  
  ''''''''''''''''''''''''''''
  ''''''''''''''''''''''''''''
  
sub findleap(jday, leapsecond)
  
  ' find number of leap seconds for utc julian day
  
  ' input
  
  '  jday = utc julian day
  
  ' input via global
  
  '  jdleap  = array of utc julian dates
  '  leapsec = array of leap seconds
  
  ' output
  
  '  leapsecond = number of leap seconds
  
  ''''''''''''''''''''''''''''''''''''''
  
  if (jday <= jdleap(1)) then
    
    ' date is <= 1972; set to first data element
    
    leapsecond = leapsec(1)
    
    exit sub
    
  end if
  
  if (jday >= jdleap(28)) then
    
    ' date is >= end of current data
    ' set to last data element
    
    leapsecond = leapsec(28)
    
    exit sub
    
  end if
  
  ' find data within table
  
  for i% = 1 to 27
    
    if (jday >= jdleap(i%) and jday < jdleap(i% + 1)) then
      
      leapsecond = leapsec(i%)
      
      exit sub
      
    end if
    
  next i%
  
end sub
  
  '''''''''''''''''''''''''
  '''''''''''''''''''''''''
  
function modulo(x) as float
  
  ' modulo 2 pi function
  
  ''''''''''''''''''''''
  
  local a
  
  a = x - pi2 * fix(x / pi2)
  
  if (a < 0.0) then
    
    a = a + pi2
    
  end if
  
  modulo = a
  
end function
  
  '''''''''''''''''''''''''''
  '''''''''''''''''''''''''''
  
function atan3(a, b) as float
  
  ' four quadrant inverse tangent function
  
  ' input
  
  '  a = sine of angle
  '  b = cosine of angle
  
  ' output
  
  '  atan3 = angle (0 =< atan3 <= 2 * pi; radians)
  
  ''''''''''''''''''''''''''''''''''''''''''''''''
  
  local c
  
  if (abs(a) < 1.0e-10) then
    
    atan3 = (1.0 - sgn(b)) * pidiv2
    
    exit function
    
  else
    
    c = (2.0 - sgn(a)) * pidiv2
    
  endif
  
  if (abs(b) < 1.0e-10) then
    
    atan3 = c
    
    exit function
    
  else
    
    atan3 = c + sgn(a) * sgn(b) * (abs(atn(a / b)) - pidiv2)
    
  endif
  
end function
  
  ''''''''''
  ''''''''''
  
sub keycheck
  
  ' check user response subroutine
  
  ''''''''''''''''''''''''''''''''
  
  local a$
  
  print " "
  
  print "< press Enter key to continue >"
  
  a$ = ""
  
  do while a$ = ""
    
    a$ = inkey$
    
  loop
  
end sub
  

