' seasons.bas            November 3, 2021

' calendar date and UTC time of the seasons

' MMBASIC double precision

''''''''''''''''''''''''''

' dimension global arrays

Dim sl(50) As float, sr(50) As float, sa(50) As float, sb(50) As float

Dim jdleap(28) As float, leapsec(28) As float, month$(12) As string

' global constants

Const pi2 = 2.0 * Pi, pidiv2 = 0.5 * Pi, dtr = Pi / 180.0

' read solar ephemeris data

For i% = 1 To 50

Read sl(i%), sr(i%), sa(i%), sb(i%)

Next i%

Data 403406,      0, 4.721964,      1.621043
Data 195207, -97597, 5.937458,  62830.348067
Data 119433, -59715, 1.115589,  62830.821524
Data 112392, -56188, 5.781616,  62829.634302
Data   3891,  -1556, 5.5474  , 125660.5691
Data   2819,  -1126, 1.5120  , 125660.9845
Data   1721,   -861, 4.1897  ,  62832.4766
Data      0,    941, 1.163   ,      0.813
Data    660,   -264, 5.415   , 125659.310
Data    350,   -163, 4.315   ,  57533.850
Data    334,      0, 4.553   ,    -33.931
Data    314,    309, 5.198   , 777137.715
Data    268,   -158, 5.989   ,  78604.191
Data    242,      0, 2.911   ,      5.412
Data    234,    -54, 1.423   ,  39302.098
Data    158,      0, 0.061   ,    -34.861
Data    132,    -93, 2.317   , 115067.698
Data    129,    -20, 3.193   ,  15774.337
Data    114,      0, 2.828   ,   5296.670
Data     99,    -47, 0.52    ,  58849.27
Data     93,      0, 4.65    ,   5296.11
Data     86,      0, 4.35    ,  -3980.70
Data     78,    -33, 2.75    ,  52237.69
Data     72,    -32, 4.50    ,  55076.47
Data     68,      0, 3.23    ,    261.08
Data     64,    -10, 1.22    ,  15773.85
Data     46,    -16, 0.14    , 188491.03
Data     38,      0, 3.44    ,  -7756.55
Data     37,      0, 4.37    ,    264.89
Data     32,    -24, 1.14    , 117906.27
Data     29,    -13, 2.84    ,  55075.75
Data     28,      0, 5.96    ,  -7961.39
Data     27,     -9, 5.09    , 188489.81
Data     27,      0, 1.72    ,   2132.19
Data     25,    -17, 2.56    , 109771.03
Data     24,    -11, 1.92    ,  54868.56
Data     21,      0, 0.09    ,  25443.93
Data     21,     31, 5.98    , -55731.43
Data     20,    -10, 4.03    ,  60697.74
Data     18,      0, 4.27    ,   2132.79
Data     17,    -12, 0.79    , 109771.63
Data     14,      0, 4.24    ,  -7752.82
Data     13,     -5, 2.01    , 188491.91
Data     13,      0, 2.65    ,    207.81
Data     13,      0, 4.98    ,  29424.63
Data     12,      0, 0.93    ,     -7.99
Data     10,      0, 2.21    ,  46941.14
Data     10,      0, 3.59    ,    -68.29
Data     10,      0, 1.50    ,  21463.25
Data     10,     -9, 2.55    , 157208.40

' read leap second data

For i% = 1 To 28

Read jdleap(i%), leapsec(i%)

Next i%

Data 2441317.5,  10.0
Data 2441499.5,  11.0
Data 2441683.5,  12.0
Data 2442048.5,  13.0
Data 2442413.5,  14.0
Data 2442778.5,  15.0
Data 2443144.5,  16.0
Data 2443509.5,  17.0
Data 2443874.5,  18.0
Data 2444239.5,  19.0
Data 2444786.5,  20.0
Data 2445151.5,  21.0
Data 2445516.5,  22.0
Data 2446247.5,  23.0
Data 2447161.5,  24.0
Data 2447892.5,  25.0
Data 2448257.5,  26.0
Data 2448804.5,  27.0
Data 2449169.5,  28.0
Data 2449534.5,  29.0
Data 2450083.5,  30.0
Data 2450630.5,  31.0
Data 2451179.5,  32.0
Data 2453736.5,  33.0
Data 2454832.5,  34.0
Data 2456109.5,  35.0
Data 2457204.5,  36.0
Data 2457754.5,  37.0

' calendar months

month$(1) = "January"
month$(2) = "Febuary"
month$(3) = "March"
month$(4) = "April"
month$(5) = "May"
month$(6) = "June"
month$(7) = "July"
month$(8) = "August"
month$(9) = "September"
month$(10) = "October"
month$(11) = "November"
month$(12) = "December"

''''''''''''''''''
' begin simulation
''''''''''''''''''

CLS

Font 4

Colour RGB(yellow)

Text 0, 0, "----- TIME OF THE SEASONS -----"

' request calendar year

GUI numberbox #1, 160, 35, 70, 40

CtrlVal(#1) = 0

Text 0, 45, "calendar year?"

Text 0, 100, "(include all four digits)"

Do

If (CtrlVal(#1) > 0) Then

  Exit

EndIf

Loop

year = CtrlVal(#1)

' process each season

CLS

For iequsol% = 1 To 4

Select Case iequsol%

  Case (1)

    cmonth = 3

    day = 15

    julian(cmonth, day, year, jdayi)

  Case (2)

    cmonth = 6

    day = 15

    julian(cmonth, day, year, jdayi)

    along2 = 0.5 * Pi

  Case (3)

    cmonth = 9

    day = 15

    julian(cmonth, day, year, jdayi)

  Case (4)

    cmonth = 12

    day = 15

    julian(cmonth, day, year, jdayi)

    along2 = 1.5 * Pi

End Select

' find event

x1 = 0.0

x2 = 10.0

realroot1(x1, x2, 1.0e-8, xroot, froot)

' TDB julian day of event

jdtdb = jdayi + xroot

' compute UTC julian day

tdb2utc(jdtdb, jdutc)

' print results for this event

Select Case iequsol%

  Case (1)

    Text 0, 0, "SPRING EQUINOX"

    jd2str(jdutc, cdate$, utc$)

    Text 0, 18, "date       " + cdate$

    Text 0, 33, "UTC time   " + utc$

  Case (2)

    Text 0, 57, "SUMMER SOLSTICE"

    jd2str(jdutc, cdate$, utc$)

    Text 0, 75, "date       " + cdate$

    Text 0, 90, "UTC time   " + utc$

  Case (3)

    Text 0, 114, "FALL EQUINOX"

    jd2str(jdutc, cdate$, utc$)

    Text 0, 132, "date       " + cdate$

    Text 0, 147, "UTC time   " + utc$

  Case (4)

    Text 0, 171, "WINTER SOLSTICE"

    jd2str(jdutc, cdate$, utc$)

    Text 0, 189, "date       " + cdate$

    Text 0, 204, "UTC time   " + utc$

    check4touch

End Select

Next iequsol%

' return to main menu

' Flash run 1

End

'''''''''''''''
'''''''''''''''

Sub esfunc(x, fx)

' equinox/solstice objective function

'''''''''''''''''''''''''''''''''''''

Local jday, rlsun, rasc, decl

jday = jdayi + x

solar(jday, rlsun, rasc, decl)

If (iequsol% = 1 Or iequsol% = 3) Then

fx = decl

Else

fx = along2 - rlsun

EndIf

End Sub

''''''''''''''''''''''''
''''''''''''''''''''''''

Sub tdb2utc(jdtdb, jdutc)

' convert TDB julian day to UTC julian day subroutine

' input

'  jdtdb = TDB julian day

' output

'  jdutc = UTC julian day

'''''''''''''''''''''''''

Local x1, x2, xroot, froot

jdsaved = jdtdb

' set lower and upper bounds

x1 = jdsaved - 0.1

x2 = jdsaved + 0.1

' solve for UTC julian day using Brent's method

realroot2(x1, x2, 1.0e-8, xroot, froot)

jdutc = xroot

End Sub

'''''''''''''''''''
'''''''''''''''''''

Sub jdfunc(jdin, fx)

' objective function for tdb2utc

' input

'  jdin = current value for UTC julian day

' output

'  fx = delta julian day

''''''''''''''''''''''''

Local jdwrk, tai_utc

findleap(jdin, tai_utc)

utc2tdb(jdin, tai_utc, jdwrk)

fx = jdwrk - jdsaved

End Sub

'''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''

Sub solar(jd, rlsun, rasc, decl)

' precision ephemeris of the Sun

' input

'  jd = julian ephemeris day

' output

'  rlsun = ecliptic longitude of the sun
'          (0 <= rlsun <= 2 pi)
'  rasc  = right ascension of the Sun (radians)
'          (0 <= rasc <= 2 pi)
'  decl  = declination of the Sun (radians)
'          (-pi/2 <= decl <= pi/2)

''''''''''''''''''''''''''''''''''

Local u, a1, a2, psi, deps, meps, eps, seps, ceps

Local dl, dr, w, srl, crl, srb, crb, sra, cra

u = (jd - 2451545.0) / 3652500.0

' compute nutation in longitude

a1 = 2.18 + u * (-3375.7 + u * 0.36)

a2 = 3.51 + u * (125666.39 + u * 0.1)

psi = 0.0000001 * (-834.0 * Sin(a1) - 64.0 * Sin(a2))

' compute nutation in obliquity

deps = 0.0000001 * u * (-226938 + u * (-75 + u * (96926 + u * (-2491 - u * 12104))))

meps = 0.0000001 * (4090928.0 + 446.0 * Cos(a1) + 28.0 * Cos(a2))

eps = meps + deps

seps = Sin(eps)

ceps = Cos(eps)

dl = 0.0

dr = 0.0

For i% = 1 To 50

w = sa(i%) + sb(i%) * u

dl = dl + sl(i%) * Sin(w)

If (sr(i%) <> 0.0) Then

  dr = dr + sr(i%) * Cos(w)

EndIf

Next i%

dl = modulo(dl * 0.0000001 + 4.9353929 + 62833.196168 * u)

dr = 149597870.691 * (dr * 0.0000001 + 1.0001026)

rlsun = modulo(dl + 0.0000001 * (-993.0 + 17.0 * Cos(3.1 + 62830.14 * u)) + psi)

rb = 0.0

' compute geocentric declination and right ascension

crl = Cos(rlsun)
srl = Sin(rlsun)
crb = Cos(rb)
srb = Sin(rb)

decl = ASin(ceps * srb + seps * crb * srl)

sra = -seps * srb + ceps * crb * srl

cra = crb * crl

rasc = atan3(sra, cra)

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot1(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub esfunc

'''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

esfunc(a, fa)

esfunc(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0.0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0.0) Then q = -q

  p = Abs(p)

  min = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (min < tmp) Then min = tmp

  If (2.0 * p < min) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

esfunc(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot2(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub jdfunc

'''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

jdfunc(a, fa)

jdfunc(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0) Then q = -q

  p = Abs(p)

  xmin = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (xmin < tmp) Then xmin = tmp

  If (2.0 * p < xmin) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

jdfunc(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub julian(month, day, year, jday)

' Gregorian date to julian day subroutine

' input

'  month = calendar month
'  day   = calendar day
'  year  = calendar year (all four digits)

' output

'  jday = julian day

' special notes

'  (1) calendar year must include all digits

'  (2) will report October 5, 1582 to October 14, 1582
'      as invalid calendar dates and exit

'''''''''''''''''''''''''''''''''''''''''

Local a, b, c, m, y

y = year

m = month

b = 0.0

c = 0.0

If (m <= 2.0) Then

y = y - 1.0

m = m + 12.0

EndIf

If (y < 0.0) Then c = -0.75

If (year < 1582.0) Then

' null

ElseIf (year > 1582.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (month < 10.0) Then

' null

ElseIf (month > 10.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (day <= 4.0) Then

' null

ElseIf (day > 14.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

Else

Print "this date does not exist!!"

Exit

EndIf

jday = Fix(365.25 * y + c) + Fix(30.6001 * (m + 1.0)) + day + b + 1720994.5

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub gdate(jday, month, day, year)

' Julian day to Gregorian date subroutine

' input

'  jday = julian day

' output

'  month = calendar month
'  day   = calendar day
'  year  = calendar year

''''''''''''''''''''''''

Local a, b, c, d, e, f, z, alpha

z = Fix(jday + 0.5)

f = jday + 0.5 - z

If (z < 2299161) Then

a = z

Else

alpha = Fix((z - 1867216.25) / 36524.25)

a = z + 1.0 + alpha - Fix(alpha / 4.0)

EndIf

b = a + 1524.0

c = Fix((b - 122.1) / 365.25)

d = Fix(365.25 * c)

e = Fix((b - d) / 30.6001)

day = b - d - Fix(30.6001 * e) + f

If (e < 13.5) Then

month = e - 1.0

Else

month = e - 13.0

EndIf

If (month > 2.5) Then

year = c - 4716.0

Else

year = c - 4715.0

EndIf

End Sub

'''''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''''

Sub utc2tdb(jdutc, tai_utc, jdtdb)

' convert UTC julian date to TDB julian date

' input

'  jdutc   = UTC julian day
'  tai_utc = TAI-UTC (seconds)

' output

'  jdtdb = TDB julian day

' Reference Frames in Astronomy and Geophysics
' J. Kovalevsky et al., 1989, pp. 439-442

'''''''''''''''''''''''''''''''''''''''''

Local corr, jdtdt, t

' TDT julian date

corr = (tai_utc + 32.184) / 86400.0

jdtdt = jdutc + corr

' time argument for correction

t = (jdtdt - 2451545.0) / 36525.0

' compute correction in microseconds

corr = 1656.675 * Sin(dtr * (35999.3729 * t + 357.5287))
corr = corr + 22.418     * Sin(dtr * (32964.467  * t + 246.199))
corr = corr + 13.84      * Sin(dtr * (71998.746  * t + 355.057))
corr = corr +  4.77      * Sin(dtr * ( 3034.906  * t +  25.463))
corr = corr +  4.677     * Sin(dtr * (34777.259  * t + 230.394))
corr = corr + 10.216 * t * Sin(dtr * (35999.373  * t + 243.451))
corr = corr +  0.171 * t * Sin(dtr * (71998.746  * t + 240.98 ))
corr = corr +  0.027 * t * Sin(dtr * ( 1222.114  * t + 194.661))
corr = corr +  0.027 * t * Sin(dtr * ( 3034.906  * t + 336.061))
corr = corr +  0.026 * t * Sin(dtr * (  -20.186  * t +   9.382))
corr = corr +  0.007 * t * Sin(dtr * (29929.562  * t + 264.911))
corr = corr +  0.006 * t * Sin(dtr * (  150.678  * t +  59.775))
corr = corr +  0.005 * t * Sin(dtr * ( 9037.513  * t + 256.025))
corr = corr +  0.043 * t * Sin(dtr * (35999.373  * t + 151.121))

' convert corrections to days

corr = 0.000001 * corr / 86400.0

' TDB julian date

jdtdb = jdtdt + corr

End Sub

''''''''''''''''''''''''''''
''''''''''''''''''''''''''''

Sub findleap(jday, leapsecond)

' find number of leap seconds for utc julian day

' input

'  jday = utc julian day

' input via global

'  jdleap  = array of utc julian dates
'  leapsec = array of leap seconds

' output

'  leapsecond = number of leap seconds

''''''''''''''''''''''''''''''''''''''

If (jday <= jdleap(1)) Then

' date is <= 1972; set to first data element

leapsecond = leapsec(1)

Exit Sub

EndIf

If (jday >= jdleap(28)) Then

' date is >= end of current data
' set to last data element

leapsecond = leapsec(28)

Exit Sub

EndIf

' find data within table

For i% = 1 To 27

If (jday >= jdleap(i%) And jday < jdleap(i% + 1)) Then

  leapsecond = leapsec(i%)

  Exit Sub

EndIf

Next i%

End Sub

'''''''''''''''''''''''''
'''''''''''''''''''''''''

Function modulo(x) As float

' modulo 2 pi function

''''''''''''''''''''''

Local a

a = x - pi2 * Fix(x / pi2)

If (a < 0.0) Then

a = a + pi2

EndIf

modulo = a

End Function

'''''''''''''''''''''''''''
'''''''''''''''''''''''''''

Function atan3(a, b) As float

' four quadrant inverse tangent function

' input

'  a = sine of angle
'  b = cosine of angle

' output

'  atan3 = angle (0 <= atan3 <= 2 * pi; radians)

''''''''''''''''''''''''''''''''''''''''''''''''

Local c

If (Abs(a) < 1.0e-10) Then

atan3 = (1.0 - Sgn(b)) * pidiv2

Exit Function

Else

c = (2.0 - Sgn(a)) * pidiv2

EndIf

If (Abs(b) < 1.0e-10) Then

atan3 = c

Exit Function

Else

atan3 = c + Sgn(a) * Sgn(b) * (Abs(Atn(a / b)) - pidiv2)

EndIf

End Function

'''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''

Sub jd2str(jdutc, cdate$, utc$)

' convert julian date to date and time strings

Local thr, tmin0, tmin, tsec

'''''''''''''''''''''''''''''

gdate(jdutc, cmonth, day, year)

lmonth = Len(month$(cmonth))

cdate$ = Left$(month$(cmonth), lmonth) + " "

cdate$ = cdate$ + Str$(Int(day)) + ", " + Str$(year)

thours = 24.0 * (day - Int(day))

thr = Fix(thours)

tmin0 = 60.0 * (thours - thr)

tmin = Fix(tmin0)

tsec = 60.0 * (tmin0 - tmin)

' fix seconds and minutes for rollover

If (tsec >= 60.0) Then

  tsec = 0.0

  tmin = tmin + 1.0

EndIf

' fix minutes for rollover

If (tmin >= 60.0) Then

 tmin = 0.0

 thr = thr + 1

EndIf

thour$ = Str$(thr)

If (thr < 10) Then

  thour$ = "0" + thour$

EndIf

tminute$ = Str$(tmin)

If (tmin < 10) Then

  tminute$ = "0" + tminute$

EndIf

tsecond$ = Str$(Int(tsec))

If (tsec < 10) Then

  tsecond$ = "0" + tsecond$

EndIf

utc$ = thour$ + ":" + tminute$ + ":" + tsecond$

End Sub

'''''''''''''''
'''''''''''''''

Sub check4touch

' check for user touching screen

Font 1

Text 40, 225, "touch screen to continue"

Font 4

Do

   If (Touch(down) <> 0) Then

     Exit

   EndIf

Loop

End Sub
