' sun_riseset.bas        November 3, 2021

' rise and set of the sun

' MMBASIC double precision

''''''''''''''''''''''''''

Option default float

Option base 1

' dimension global arrays and variables

Dim xsl(50), xsr(50), xsa(50), xsb(50)

Dim jdleap(28), leapsec(28)

Dim month$(12) As string

Dim xnut(11, 13), trr, elev_minima

Dim jdsaved, jdprint, obslat, obslong, obsalt

Dim jdtdbi, i As integer, j As integer

Dim cmonth, cday, cyear, ndays

Dim rootflag% As integer

' global constants

Const pi2 = 2.0 * Pi, pidiv2 = 0.5 * Pi, dtr = Pi / 180.0, rtd = 180.0 / Pi

Const atr = Pi / 648000.0, seccon = 206264.8062470964

' astronomical unit (kilometers)

Const aunit = 149597870.691

' equatorial radius of the earth (kilometers)

Const reqm = 6378.14

' earth flattening factor (nd)

Const flat = 1.0 / 298.257

' read solar ephemeris data

For i = 1 To 50

Read xsl(i), xsr(i), xsa(i), xsb(i)

Next i

Data 403406,      0, 4.721964,      1.621043
Data 195207, -97597, 5.937458,  62830.348067
Data 119433, -59715, 1.115589,  62830.821524
Data 112392, -56188, 5.781616,  62829.634302
Data   3891,  -1556, 5.5474  , 125660.5691
Data   2819,  -1126, 1.5120  , 125660.9845
Data   1721,   -861, 4.1897  ,  62832.4766
Data      0,    941, 1.163   ,      0.813
Data    660,   -264, 5.415   , 125659.310
Data    350,   -163, 4.315   ,  57533.850
Data    334,      0, 4.553   ,    -33.931
Data    314,    309, 5.198   , 777137.715
Data    268,   -158, 5.989   ,  78604.191
Data    242,      0, 2.911   ,      5.412
Data    234,    -54, 1.423   ,  39302.098
Data    158,      0, 0.061   ,    -34.861
Data    132,    -93, 2.317   , 115067.698
Data    129,    -20, 3.193   ,  15774.337
Data    114,      0, 2.828   ,   5296.670
Data     99,    -47, 0.52    ,  58849.27
Data     93,      0, 4.65    ,   5296.11
Data     86,      0, 4.35    ,  -3980.70
Data     78,    -33, 2.75    ,  52237.69
Data     72,    -32, 4.50    ,  55076.47
Data     68,      0, 3.23    ,    261.08
Data     64,    -10, 1.22    ,  15773.85
Data     46,    -16, 0.14    , 188491.03
Data     38,      0, 3.44    ,  -7756.55
Data     37,      0, 4.37    ,    264.89
Data     32,    -24, 1.14    , 117906.27
Data     29,    -13, 2.84    ,  55075.75
Data     28,      0, 5.96    ,  -7961.39
Data     27,     -9, 5.09    , 188489.81
Data     27,      0, 1.72    ,   2132.19
Data     25,    -17, 2.56    , 109771.03
Data     24,    -11, 1.92    ,  54868.56
Data     21,      0, 0.09    ,  25443.93
Data     21,     31, 5.98    , -55731.43
Data     20,    -10, 4.03    ,  60697.74
Data     18,      0, 4.27    ,   2132.79
Data     17,    -12, 0.79    , 109771.63
Data     14,      0, 4.24    ,  -7752.82
Data     13,     -5, 2.01    , 188491.91
Data     13,      0, 2.65    ,    207.81
Data     13,      0, 4.98    ,  29424.63
Data     12,      0, 0.93    ,     -7.99
Data     10,      0, 2.21    ,  46941.14
Data     10,      0, 3.59    ,    -68.29
Data     10,      0, 1.50    ,  21463.25
Data     10,     -9, 2.55    , 157208.40

' read subset of IAU2000 nutation data

For j = 1 To 13

For i = 1 To 11

  Read xnut(i, j)

Next i

Next j

Data  0,  0, 0,  0, 1, -172064161, -174666, 92052331,  9086,  33386, 15377
Data  0,  0, 2, -2, 2,  -13170906,   -1675,  5730336, -3015, -13696, -4587
Data  0,  0, 2,  0, 2,   -2276413,    -234,   978459,  -485,   2796,  1374
Data  0,  0, 0,  0, 2,    2074554,     207,  -897492,   470,   -698,  -291
Data  0,  1, 0,  0, 0,    1475877,   -3633,    73871,  -184,  11817, -1924
Data  0,  1, 2, -2, 2,    -516821,    1226,   224386,  -677,   -524,  -174
Data  1,  0, 0,  0, 0,     711159,      73,    -6750,     0,   -872,   358
Data  0,  0, 2,  0, 1,    -387298,    -367,   200728,    18,    380,   318
Data  1,  0, 2,  0, 2,    -301461,     -36,   129025,   -63,    816,   367
Data  0, -1, 2, -2, 2,     215829,    -494,   -95929,   299,    111,   132
Data  0,  0, 2, -2, 1,     128227,     137,   -68982,    -9,    181,    39
Data -1,  0, 2,  0, 2,     123457,      11,   -53311,    32,     19,    -4
Data -1,  0, 0,  2, 0,     156994,      10,    -1235,     0,   -168,    82

' read leap second data

For i = 1 To 28

Read jdleap(i), leapsec(i)

Next i

Data 2441317.5,  10.0
Data 2441499.5,  11.0
Data 2441683.5,  12.0
Data 2442048.5,  13.0
Data 2442413.5,  14.0
Data 2442778.5,  15.0
Data 2443144.5,  16.0
Data 2443509.5,  17.0
Data 2443874.5,  18.0
Data 2444239.5,  19.0
Data 2444786.5,  20.0
Data 2445151.5,  21.0
Data 2445516.5,  22.0
Data 2446247.5,  23.0
Data 2447161.5,  24.0
Data 2447892.5,  25.0
Data 2448257.5,  26.0
Data 2448804.5,  27.0
Data 2449169.5,  28.0
Data 2449534.5,  29.0
Data 2450083.5,  30.0
Data 2450630.5,  31.0
Data 2451179.5,  32.0
Data 2453736.5,  33.0
Data 2454832.5,  34.0
Data 2456109.5,  35.0
Data 2457204.5,  36.0
Data 2457754.5,  37.0

' calendar months

month$(1) = "January"
month$(2) = "February"
month$(3) = "March"
month$(4) = "April"
month$(5) = "May"
month$(6) = "June"
month$(7) = "July"
month$(8) = "August"
month$(9) = "September"
month$(10) = "October"
month$(11) = "November"
month$(12) = "December"

''''''''''''''''''
' begin simulation
''''''''''''''''''

CLS

Font 4

Colour RGB(yellow)

Text 0, 0, "------ SUN RISE AND SET ------"

' request calendar month

GUI numberbox #1, 160, 35, 70, 40

CtrlVal(#1) = 0

Text 0, 47, "calendar month?"

Text 0, 100, "(1 = January, 2 = February, etc)"

Do

If (CtrlVal(#1) >= 1 And CtrlVal(#1) <= 12) Then

  Exit

EndIf

Loop

cmonth = CtrlVal(#1)

CLS

Text 0, 0, "------ SUN RISE AND SET ------"

' request calendar day

GUI numberbox #2, 155, 35, 70, 40

CtrlVal(#2) = 0

Text 0, 47, "calendar day?"

Text 0, 100, "(1 <= calendar day <= 31)"

Do

If (CtrlVal(#2) >= 1 And CtrlVal(#2) <= 31) Then

  Exit

EndIf

Loop

cday = CtrlVal(#2)

CLS

Text 0, 0, "------ SUN RISE AND SET ------"

' request calendar year

GUI numberbox #3, 155, 35, 70, 40

CtrlVal(#3) = 0

Text 0, 47, "calendar year?"

Text 0, 100, "(include all four digits)"

Do

If (CtrlVal(#3) <> 0) Then

  Exit

EndIf

Loop

cyear = CtrlVal(#3)

CLS

' read observer coordinates

observer(obslat, obslong, obsalt)

' initial utc julian day

julian(cmonth, cday, cyear, jdutc)

' compute initial tdb julian date

utc2tdb(jdutc, jdtdb)

jdtdbi = jdtdb

ndays = 1

' define search parameters

ti = 0.0

tf = ndays

dt = 0.1

dtsml = 0.01

rootflag% = 0

' find sun rise/set conditions

rs_event(ti, tf, dt, dtsml)

' return to main menu

' Flash run 1

End

'''''''''''''''''''''''
'''''''''''''''''''''''

Sub rsprint(iflag, jdtdb)

' print rise and set conditions

'''''''''''''''''''''''''''''''

Local rsun(3)

Local jdutc, dms$ As string

Local deltat, gast, azimuth, elevation

' compute UTC julian date

tdb2utc(jdtdb, jdutc)

jd2str(jdutc, cdate$, utc$)

' compute topocentric coordinates of the sun

gast2(jdutc, gast)

sun(jdtdb, rsun())

eci2topo(gast, rsun(), azimuth, elevation)

deg2str(rtd * azimuth, az_dms$)

deg2str(rtd * elevation, el_dms$)

If (iflag = 1) Then

 Text 0, 0, "SUN RISE CONDITIONS"

 Text 0, 20, "date      " + cdate$

 Text 0, 40, "UTC time  " + utc$

 Text 0, 60, "azimuth   " + az_dms$

 Text 0, 80, "elevation " + el_dms$

EndIf

If (iflag = 2) Then

 Text 0, 107, "SUN MAX ELEVATION CONDITIONS"

 Text 0, 127, "date      " + cdate$

 Text 0, 147, "UTC time  " + utc$

 Text 0, 167, "azimuth   " + az_dms$

 Text 0, 187, "elevation " + el_dms$

 check4touch

EndIf

If (iflag = 3) Then

 CLS

 Text 0, 0, "SUN SET CONDITIONS"

 Text 0, 20, "date      " + cdate$

 Text 0, 40, "UTC time  " + utc$

 Text 0, 60, "azimuth   " + az_dms$

 Text 0, 80, "elevation " + el_dms$

 check4touch

EndIf

' return to main menu

' flash run 1

End Sub

'''''''''''''''
'''''''''''''''

Sub rsfunc(x, fx)

' rise/set objective function

'''''''''''''''''''''''''''''

Local jdtdb, rsun(3), gast, azim, elev

Local dis, dref, sdia

' current TDB julian day

jdtdb = jdtdbi + x

' compute eci position vector and distance of the sun

sun(jdtdb, rsun())

dis = Math(magnitude rsun()) / aunit

' compute topocentric coordinates of the sun

tdb2utc(jdtdb, jdutc)

gast2(jdutc, gast)

eci2topo(gast, rsun(), azim, elev)

If (rootflag% = 1) Then

' correct for horizontal refraction

dref = (34.0 / 60.0)

' correct for semidiameter of the sun

tsdia = rtd * ASin((0.5 * 696000.0 / aunit) / dis)

Else

dref = 0.0

tsdia = 0.0

EndIf

' evaluate objective function

fx = -(rtd * elev + tsdia + dref)

End Sub

''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''

Sub rs_event(ti, tf, dt, dtsml)

' predict rise/set events

' input

'  ti    = initial simulation time
'  tf    = final simulation time
'  dt    = step size used for bounding minima
'  dtsml = small step size used to determine whether
'          the function is increasing or decreasing

'''''''''''''''''''''''''''''''''''''''''''''''''''

Local tolm, iend As integer

Local fmin1, tmin1

Local ftemp, df, dflft

Local el, er

Local t, ft

Local iter1 As integer, iter2 As integer

Local iter3 As integer

' initialization

tolm = 1.0e-8

iend = 0

' check the initial time for a minimum

rsfunc(ti, fmin1)

tmin1 = ti

rsfunc(ti + dtsml, ftemp)

df = ftemp - fmin1

dflft = df

el = ti

er = el

t = ti

' if the slope is positive and the minimum is
' negative, calculate event conditions at the initial time

If (df > 0.0 And fmin1 < 0.0) Then

events1(ti, tf, tmin1)

EndIf

For iter1 = 1 To 1000

' find where function first starts decreasing

For iter2 = 1 To 1000

  If (df <= 0.0) Then

 Exit For

  EndIf

  t = t + dt

  If (t >= tf) Then

 ' check final time for a minimum

 If (iend = 1) Then Exit Sub

 rsfunc(tf, fmin1)

 rsfunc(tf - dtsml, ftemp)

 df = fmin1 - ftemp

 ' set minimum time to final simulation time

 tmin1 = tf

 If (df < 0.0) Then

   ' if both the slope and minimum are negative,
   ' calculate the event conditions at the final
   ' simulation time

   If (fmin1 < 0.0) Then

  events1(ti, tf, tmin1)

   EndIf

   ' otherwise, we're finished

   Exit Sub

 EndIf

 If (dflft > 0.0) Then Exit Sub

 er = tf

 iend = 1

  EndIf

  rsfunc(t, ft)

  rsfunc(t - dtsml, ftemp)

  df = ft - ftemp

Next iter2

' function decreasing - find where function
' first starts increasing

For iter3 = 1 To 1000

  el = t

  dflft = df

  t = t + dt

  If (t >= tf) Then

 ' check final time for a minimum

 If (iend = 1) Then Exit Sub

 rsfunc(tf, fmin1)

 rsfunc(tf - dtsml, ftemp)

 df = fmin1 - ftemp

 ' set minimum time to final simulation time

 tmin1 = tf

 If (df < 0.0) Then

   ' if both the slope and minimum are negative,
   ' calculate the event conditions at the final
   ' simulation time

   If (fmin1 < 0.0) Then

  events1(ti, tf, tmin1)

   EndIf

   ' otherwise, we're finished

   Exit Sub

 EndIf

 If (dflft > 0.0) Then Exit Sub

 er = tf

 iend = 1

  EndIf

  rsfunc(t, ft)

  rsfunc(t - dtsml, ftemp)

  df = ft - ftemp

  If (df > 0.0) Then Exit For

Next iter3

er = t

' calculate minimum using Brent's method

minima(el, er, tolm, tmin1, fmin1)

el = er

dflft = df

' if the minimum is negative,
' calculate event conditions for this minimum

If (fmin1 < 0.0) Then

  events1(ti, tf, tmin1)

  t = trr

EndIf

Next iter1

End Sub

''''''''''''''''''''''''
''''''''''''''''''''''''

Sub events1(ti, tf, topt)

' compute and display rise/set events

' input

'  ti   = initial simulation time
'  tf   = final simulation time
'  topt = extrema time

''''''''''''''''''''''

' define root-bracketing and root-finding control parameters

Local factor = 0.25            ' geometric acceleration factor

Local dxmax = 0.1              ' rectification interval

Local rtol = 1.0e-8            ' root-finding convergence tolerance

Local t1in, t2in

Local t1out, t2out

Local troot, froot, jdate

Local iter As integer

' compute and display rise conditions

rootflag% = 1

t1in = topt

t2in = t1in - 0.05

broot(t1in, t2in, factor, dxmax, t1out, t2out)

realroot1(t1out, t2out, rtol, troot, froot)

jdate = jdtdbi + troot

rsprint(1, jdate)

' display maximum elevation conditions

rootflag% = 0

jdate = jdtdbi + topt

rsprint(2, jdate)

' compute and display set conditions

rootflag% = 1

t2in = t1in + 0.05

broot(t1in, t2in, factor, dxmax, t1out, t2out)

realroot1(t1out, t2out, rtol, troot, froot)

trr = troot

jdate = jdtdbi + troot

rsprint(3, jdate)

rootflag% = 0

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub minima(a, b, tolm, xmin, fmin)

' one-dimensional minimization

' Brent's method

' input

'  a    = initial x search value
'  b    = final x search value
'  tolm = convergence criterion

' output

'  xmin = minimum x value

' note

'  user-defined objective subroutine
'  coded as usr_func(x, fx)

' remember: a maximum is simply a minimum
'           with a negative attitude!

'''''''''''''''''''''''''''''''''''''

' machine epsilon

Local epsm = 2.23e-16

' golden number

Local c = 0.38196601125

Local iter As integer, d, e

Local t2, p, q

Local r, u, fu

Local x, xm, w

Local v, fx, fw

Local tol1, fv

x = a + c * (b - a)

w = x

v = w

e = 0.0
p = 0.0
q = 0.0
r = 0.0

rsfunc(x, fx)

fw = fx

fv = fw

For iter = 1 To 100

If (iter > 50) Then

  Print ("error in function minima!")
  Print ("(more than 50 iterations)")

EndIf

xm = 0.5 * (a + b)

tol1 = tolm * Abs(x) + epsm

t2 = 2.0 * tol1

If (Abs(x - xm) <= (t2 - 0.5 * (b - a))) Then

  xmin = x

  fmin = fx

  Exit Sub

EndIf

If (Abs(e) > tol1) Then

  r = (x - w) * (fx - fv)

  q = (x - v) * (fx - fw)

  p = (x - v) * q - (x - w) * r

  q = 2.0 * (q - r)

  If (q > 0.0) Then

 p = -p

  EndIf

  q = Abs(q)

  r = e

  e = d

EndIf

If ((Abs(p) >= Abs(0.5 * q * r)) Or (p <= q * (a - x)) Or (p >= q * (b - x))) Then

  If (x >= xm) Then

 e = a - x

  Else

 e = b - x

  EndIf

  d = c * e

Else

  d = p / q

  u = x + d

  If ((u - a) < t2) Or ((b - u) < t2) Then

 d = Sgn(xm - x) * tol1

  EndIf

EndIf

If (Abs(d) >= tol1) Then

  u = x + d

Else

  u = x + Sgn(d) * tol1

EndIf

rsfunc(u, fu)

If (fu <= fx) Then

  If (u >= x) Then

 a = x

  Else

 b = x

  EndIf

  v = w

  fv = fw

  w = x

  fw = fx

  x = u

  fx = fu

Else

  If (u < x) Then

 a = u

  Else

 b = u

  EndIf

  If ((fu <= fw) Or (w = x)) Then

 v = w

 fv = fw

 w = u

 fw = fu

  ElseIf ((fu <= fv) Or (v = x) Or (v = w)) Then

 v = u

 fv = fu

  EndIf

EndIf

Next iter

End Sub

'''''''''''''''''''''''
'''''''''''''''''''''''

Sub gsite(angle, rsite())

' ground site position vector

' input

'  angle  = local sidereal time or east longitude
'           (radians; 0 <= angle <= 2*pi)

' input via global

'  obslat = geodetic latitude (radians)
'           (+north, -south; -pi/2 <= lat <= -pi/2)
'  obsalt = geodetic altitude (meters)
'           (+ above sea level, - below sea level)

' output

'  rsite = ground site position vector (kilometers)

' special notes

'  (1) eci coordinates if angle = local sidereal time

'  (2) ecf coordinates if angle = east longitude

'  (3) geocentric, equatorial coordinates

'''''''''''''''''''''''''''''''''''''''''

Local slat, clat

Local sangle, cangle

Local b, c, d

slat = Sin(obslat)

clat = Cos(obslat)

sangle = Sin(angle)

cangle = Cos(angle)

' compute geodetic constants

b = Sqr(1.0 - (2.0 * flat - flat * flat) * slat * slat)

c = reqm / b + 0.001 * obsalt

d = reqm * (1.0 - flat) * (1.0 - flat) / b + 0.001 * obsalt

' compute x, y and z components of position vector (kilometers)

rsite(1) = c * clat * cangle

rsite(2) = c * clat * sangle

rsite(3) = d * slat

End Sub

''''''''''''''''''''
''''''''''''''''''''

Sub gast2(jdate, gast)

' greenwich apparent sidereal time

' input

'  jdate = julian date

' output

'  gast = greenwich apparent sidereal time (radians)

'''''''''''''''''''''''''''''''''''''''''''''''''''

Local t, t2, t3, eqeq, dpsi, deps

Local th, tl, obm, obt, st, x

Local tjdh As integer, tjdl

tjdh = Int(jdate)

tjdl = jdate - tjdh

th = (tjdh - 2451545.0) / 36525

tl = tjdl / 36525.0

t = th + tl

t2 = t * t

t3 = t2 * t

' obtain equation of the equinoxes

eqeq = 0.0

' obtain nutation parameters in seconds of arc

nut2000_lp(jdate, dpsi, deps)

' compute mean obliquity of the ecliptic in seconds of arc

obm = 84381.4480 - 46.8150 * t - 0.00059 * t2 + 0.001813 * t3

' compute true obliquity of the ecliptic in seconds of arc

obliq = obm + deps

' compute equation of the equinoxes in seconds of time

eqeq = (dpsi / 15.0) * Cos(obliq / seccon)

st = eqeq - 6.2e-6 * t3 + 0.093104 * t2 + 67310.54841 + 8640184.812866 * tl + 3155760000.0 * tl + 8640184.812866 * th + 3155760000.0 * th

' modulo 24 hours

x = st / 3600.0

gast = x - 24.0 * Fix(x / 24.0)

If (gast < 0.0) Then

gast = gast + 24.0

EndIf

' convert to radians

gast = pi2 * (gast / 24.0)

End Sub

''''''''''''''''''''''''
''''''''''''''''''''''''

Sub tdb2utc(jdtdb, jdutc)

' convert TDB julian day to UTC julian day subroutine

' input

'  jdtdb = TDB julian day

' output

'  jdutc = UTC julian day

'''''''''''''''''''''''''

Local x1, x2, xroot, froot

jdsaved = jdtdb

' set lower and upper bounds

x1 = jdsaved - 0.1

x2 = jdsaved + 0.1

' solve for UTC julian day using Brent's method

realroot2(x1, x2, 1.0e-8, xroot, froot)

jdutc = xroot

End Sub

'''''''''''''''''''
'''''''''''''''''''

Sub jdfunc(jdin, fx)

' objective function for tdb2utc

' input

'  jdin = current value for UTC julian day

' output

'  fx = delta julian day

''''''''''''''''''''''''

Local jdwrk

utc2tdb(jdin, jdwrk)

fx = jdwrk - jdsaved

End Sub

'''''''''''''''''
'''''''''''''''''

Sub sun(jd, rsun())

' precision ephemeris of the Sun

' input

'  jd = julian ephemeris day

' output

'  rsun = eci position of the sun

'''''''''''''''''''''''''''''''''

Local u, a1, a2, psi, deps, meps, eps, seps, ceps

Local dl, dr, w, srl, crl, srb, crb, sra, cra

u = (jd - 2451545.0) / 3652500.0

' compute nutation in longitude

a1 = 2.18 + u * (-3375.7 + u * 0.36)

a2 = 3.51 + u * (125666.39 + u * 0.1)

psi = 0.0000001 * (-834.0 * Sin(a1) - 64.0 * Sin(a2))

' compute nutation in obliquity

deps = 0.0000001 * u * (-226938 + u * (-75 + u * (96926 + u * (-2491 - u * 12104))))

meps = 0.0000001 * (4090928.0 + 446.0 * Cos(a1) + 28.0 * Cos(a2))

eps = meps + deps

obliq = eps

seps = Sin(eps)

ceps = Cos(eps)

dl = 0.0

dr = 0.0

For i% = 1 To 50

w = xsa(i%) + xsb(i%) * u

dl = dl + xsl(i%) * Sin(w)

If (xsr(i%) <> 0.0) Then

  dr = dr + xsr(i%) * Cos(w)

EndIf

Next i%

dl = modulo(dl * 0.0000001 + 4.9353929 + 62833.196168 * u)

dr = aunit * (dr * 0.0000001 + 1.0001026)

rlsun = modulo(dl + 0.0000001 * (-993.0 + 17.0 * Cos(3.1 + 62830.14 * u)) + psi)

rb = 0.0

' compute geocentric declination and right ascension

crl = Cos(rlsun)
srl = Sin(rlsun)
crb = Cos(rb)
srb = Sin(rb)

decl = ASin(ceps * srb + seps * crb * srl)

sra = -seps * srb + ceps * crb * srl

cra = crb * crl

rasc = Math(atan3 sra, cra)

' geocentric equatorial position vector of the Sun (au)

rsun(1) = dr * Cos(rasc) * Cos(decl)

rsun(2) = dr * Sin(rasc) * Cos(decl)

rsun(3) = dr * Sin(decl)

End Sub

''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''

Sub eci2topo(gast, robj(), azim, elev)

' convert eci position vector to topocentric coordinates

' input

'  gast = Greenwich apparent sidereal time (radians)
'  robj = eci position vector of object (kilometers)

' output

'  azim = topocentric azimuth (radians)
'  elev = topocentric elevation (radians)

''''''''''''''''''''''''''''''''''''''''''''''

Local rsite(3), rhoijk(3), rhohatijk(3), rhohatsez(3)

Local i As integer, tmatrix(3, 3), tmatrixt(3, 3)

Local obslst, srange, sobslat

Local cobslat, sobslst, cobslst

' observer local sidereal time

obslst = modulo(gast + obslong)

gsite(obslst, rsite())

' eci vector from observer to moon

For i = 1 To 3

rhoijk(i) = aunit * robj(i) - rsite(i)

Next i

' observer-to-object slant range (kilometers)

srange = Math(magnitude rhoijk())

' compute topocentric unit pointing vector

Math v_normalise rhoijk(), rhohatijk()

' compute eci-to-sez transformation matrix

sobslat = Sin(obslat)
cobslat = Cos(obslat)

sobslst = Sin(obslst)
cobslst = Cos(obslst)

tmatrix(1, 1) = sobslat * cobslst
tmatrix(1, 2) = sobslat * sobslst
tmatrix(1, 3) = -cobslat

tmatrix(2, 1) = -sobslst
tmatrix(2, 2) = cobslst
tmatrix(2, 3) = 0.0

tmatrix(3, 1) = cobslat * cobslst
tmatrix(3, 2) = cobslat * sobslst
tmatrix(3, 3) = sobslat

' transpose tmatrix

Math m_transpose tmatrix(), tmatrixt()

' compute sez coordinates

Math v_mult tmatrixt(), rhohatijk(), rhohatsez()

' topocentric elevation (radians)

elev = ASin(rhohatsez(3))

' topocentric azimuth (radians)

azim = Math(atan3 rhohatsez(2), -rhohatsez(1))

End Sub

''''''''''''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''''''''''''

Sub broot(x1in, x2in, factor, dxmax, x1out, x2out)

' bracket a single root of a nonlinear equation

' input

'  x1in   = initial guess for first bracketing x value
'  x2in   = initial guess for second bracketing x value
'  factor = acceleration factor (non-dimensional)
'  dxmax  = rectification interval

' output

'  xout1 = final value for first bracketing x value
'  xout2 = final value for second bracketing x value

''''''''''''''''''''''''''''''''''''''''''''''''''''

Local f1, f2

Local x3, dx

' evaluate objective function at initial value

rsfunc(x1in, f1)

' save initial value

x3 = x1in

' save initial delta-x

dx = x2in - x1in

' perform bracketing until the product of the
' two function values is negative

Do

' geometrically accelerate the second point

x2in = x2in + factor * (x2in - x3)

' evaluate objective function at x2

rsfunc(x2in, f2)

' check to see if rectification is required

If (Abs(x2in - x3) > dxmax) Then

  x3 = x2in - dx

EndIf

' is the root bracketed?

If ((f1 * f2) < 0.0) Then Exit

Loop

x1out = x1in

x2out = x2in

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot1(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub rsfunc

'''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

rsfunc(a, fa)

rsfunc(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0.0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0.0) Then q = -q

  p = Abs(p)

  min = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (min < tmp) Then min = tmp

  If (2.0 * p < min) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

rsfunc(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

''''''''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''''''''

Sub realroot2(x1, x2, tol, xroot, froot)

' real root of a single non-linear function subroutine

' input

'  x1  = lower bound of search interval
'  x2  = upper bound of search interval
'  tol = convergence criter%ia

' output

'  xroot = real root of f(x) = 0
'  froot = function value

' note: requires sub jdfunc

'''''''''''''''''''''''''''

Local eps, a, b, c, d, e, fa, fb, fcc, tol1

Local xm, p, q, r, s, xmin, tmp

eps = 2.23e-16

e = 0.0

a = x1

b = x2

jdfunc(a, fa)

jdfunc(b, fb)

fcc = fb

For iter% = 1 To 50

If (fb * fcc > 0.0) Then

  c = a

  fcc = fa

  d = b - a

  e = d

EndIf

If (Abs(fcc) < Abs(fb)) Then

  a = b

  b = c

  c = a

  fa = fb

  fb = fcc

  fcc = fa

EndIf

tol1 = 2.0 * eps * Abs(b) + 0.5 * tol

xm = 0.5 * (c - b)

If (Abs(xm) <= tol1 Or fb = 0) Then Exit For

If (Abs(e) >= tol1 And Abs(fa) > Abs(fb)) Then

  s = fb / fa

  If (a = c) Then

 p = 2.0 * xm * s

 q = 1.0 - s

  Else

 q = fa / fcc

 r = fb / fcc

 p = s * (2.0 * xm * q * (q - r) - (b - a) * (r - 1.0))

 q = (q - 1.0) * (r - 1.0) * (s - 1.0)

  EndIf

  If (p > 0) Then q = -q

  p = Abs(p)

  xmin = Abs(e * q)

  tmp = 3.0 * xm * q - Abs(tol1 * q)

  If (xmin < tmp) Then xmin = tmp

  If (2.0 * p < xmin) Then

 e = d

 d = p / q

  Else

 d = xm

 e = d

  EndIf

Else

  d = xm

  e = d

EndIf

a = b

fa = fb

If (Abs(d) > tol1) Then

  b = b + d

Else

  b = b + Sgn(xm) * tol1

EndIf

jdfunc(b, fb)

Next iter%

froot = fb

xroot = b

End Sub

'''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''

Sub nut2000_lp(jdate, dpsi, deps)

' low precison nutation based on iau 2000a

' this function evaluates a short nutation series and returns approximate
' values for nutation in longitude and nutation in obliquity for a given
' tdb julian date. in this mode, only the largest 13 terms of the iau 2000a
' nutation series are evaluated.

' input

'  jdate = tdb julian date

' output

'  dpsi = nutation in longitude in arcseconds

'  deps = nutation in obliquity in arcseconds

'''''''''''''''''''''''''''''''''''''''''''''

Local rev = 360.0 * 3600.0

Local el, elp, f, d, omega

Local i%, arg

Local t = (jdate - 2451545.0) / 36525.0

''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' computation of fundamental (delaunay) arguments from simon et al. (1994)
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' mean anomaly of the moon

el = (485868.249036 + t * (1717915923.2178 + t * (31.8792 + t * (0.051635 + t * (-0.00024470)))) Mod rev) / seccon

' mean anomaly of the sun

elp = (1287104.79305 + t * (129596581.0481 + t * (-0.5532 + t * (0.000136 + t * (- 0.00001149)))) Mod rev) / seccon

' mean argument of the latitude of the moon

f = (335779.526232 + t * (1739527262.8478 + t * (-12.7512 + t * (-0.001037 + t * (0.00000417)))) Mod rev) / seccon

' mean elongation of the moon from the sun

d = (1072260.70369 + t * (1602961601.2090 + t * (- 6.3706 + t * (0.006593 + t * (- 0.00003169)))) Mod rev) / seccon

' mean longitude of the ascending node of the moon (from simon section 3.4(b.3), precession = 5028.8200 arcsec/cy)

omega = (450160.398036 + t * (- 6962890.5431 + t * (7.4722 + t * (0.007702 + t * (- 0.00005939)))) Mod rev) / seccon

dpsi = 0.0

deps = 0.0

' sum nutation series terms

For i% = 13 To 1 Step -1

arg = xnut(1, i%) * el + xnut(2, i%) * elp + xnut(3, i%) * f + xnut(4, i%) * d + xnut(5, i%) * omega

dpsi = (xnut(6, i%) + xnut(7, i%) * t) * Sin(arg) + xnut(10, i%) * Cos(arg) + dpsi

deps = (xnut(8, i%) + xnut(9, i%) * t) * Cos(arg) + xnut(11, i%) * Sin(arg) + deps

Next i%

dpsi = 1.0e-7 * dpsi

deps = 1.0e-7 * deps

' add in out-of-phase component of principal (18.6-year) term
' (to avoid small but long-term bias in results)

dpsi = dpsi + 0.0033 * Cos(omega)

deps = deps + 0.0015 * Sin(omega)

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub julian(month, day, year, jday)

' Gregorian date to julian day subroutine

' input

'  month = calendar month
'  day   = calendar day
'  year  = calendar year (all four digits)

' output

'  jday = julian day

' special notes

'  (1) calendar year must include all digits

'  (2) will report October 5, 1582 to October 14, 1582
'      as invalid calendar dates and exit

'''''''''''''''''''''''''''''''''''''''''

Local a, b, c, m, y

y = year

m = month

b = 0.0

c = 0.0

If (m <= 2.0) Then

y = y - 1.0

m = m + 12.0

EndIf

If (y < 0.0) Then c = -0.75

If (year < 1582.0) Then

' null

ElseIf (year > 1582.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (month < 10.0) Then

' null

ElseIf (month > 10.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

ElseIf (day <= 4.0) Then

' null

ElseIf (day > 14.0) Then

a = Fix(y / 100.0)

b = 2.0 - a + Fix(a / 4.0)

Else

Print "this date does not exist!!"

Exit

EndIf

jday = Fix(365.25 * y + c) + Fix(30.6001 * (m + 1.0)) + day + b + 1720994.5

End Sub

''''''''''''''''''''''''''''''''
''''''''''''''''''''''''''''''''

Sub gdate(jday, month, day, year)

' Julian day to Gregorian date subroutine

' input

'  jday = julian day

' output

'  month = calendar month
'  day   = calendar day
'  year  = calendar year

''''''''''''''''''''''''

Local a, b, c, d, e, f, z, alpha

z = Fix(jday + 0.5)

f = jday + 0.5 - z

If (z < 2299161) Then

a = z

Else

alpha = Fix((z - 1867216.25) / 36524.25)

a = z + 1.0 + alpha - Fix(alpha / 4.0)

EndIf

b = a + 1524.0

c = Fix((b - 122.1) / 365.25)

d = Fix(365.25 * c)

e = Fix((b - d) / 30.6001)

day = b - d - Fix(30.6001 * e) + f

If (e < 13.5) Then

month = e - 1.0

Else

month = e - 13.0

EndIf

If (month > 2.5) Then

year = c - 4716.0

Else

year = c - 4715.0

EndIf

End Sub

''''''''''''''''''''''''
''''''''''''''''''''''''

Sub utc2tdb(jdutc, jdtdb)

' convert UTC julian date to TDB julian date

' input

'  jdutc   = UTC julian day

' output

'  jdtdb = TDB julian day

' Reference Frames in Astronomy and Geophysics
' J. Kovalevsky et al., 1989, pp. 439-442

'''''''''''''''''''''''''''''''''''''''''

Local corr, jdtt, t, leapsecond

' find current number of leap seconds

findleap(jdutc, leapsecond)

' compute TDT julian date

corr = (leapsecond + 32.184) / 86400.0

jdtt = jdutc + corr

' time argument for correction

t = (jdtt - 2451545.0) / 36525.0

' compute correction in microseconds

corr = 1656.675 * Sin(dtr * (35999.3729 * t + 357.5287))
corr = corr + 22.418     * Sin(dtr * (32964.467  * t + 246.199))
corr = corr + 13.84      * Sin(dtr * (71998.746  * t + 355.057))
corr = corr +  4.77      * Sin(dtr * ( 3034.906  * t +  25.463))
corr = corr +  4.677     * Sin(dtr * (34777.259  * t + 230.394))
corr = corr + 10.216 * t * Sin(dtr * (35999.373  * t + 243.451))
corr = corr +  0.171 * t * Sin(dtr * (71998.746  * t + 240.98 ))
corr = corr +  0.027 * t * Sin(dtr * ( 1222.114  * t + 194.661))
corr = corr +  0.027 * t * Sin(dtr * ( 3034.906  * t + 336.061))
corr = corr +  0.026 * t * Sin(dtr * (  -20.186  * t +   9.382))
corr = corr +  0.007 * t * Sin(dtr * (29929.562  * t + 264.911))
corr = corr +  0.006 * t * Sin(dtr * (  150.678  * t +  59.775))
corr = corr +  0.005 * t * Sin(dtr * ( 9037.513  * t + 256.025))
corr = corr +  0.043 * t * Sin(dtr * (35999.373  * t + 151.121))

' convert corrections to days

corr = 0.000001 * corr / 86400.0

' TDB julian date

jdtdb = jdtt + corr

End Sub

''''''''''''''''''''''''''''
''''''''''''''''''''''''''''

Sub findleap(jday, leapsecond)

' find number of leap seconds for utc julian day

' input

'  jday = utc julian day

' input via global

'  jdleap  = array of utc julian dates
'  leapsec = array of leap seconds

' output

'  leapsecond = number of leap seconds

''''''''''''''''''''''''''''''''''''''

If (jday <= jdleap(1)) Then

' date is <= 1972; set to first data element

leapsecond = leapsec(1)

Exit Sub

EndIf

If (jday >= jdleap(28)) Then

' date is >= end of current data
' set to last data element

leapsecond = leapsec(28)

Exit Sub

EndIf

' find data within table

For i% = 1 To 27

If (jday >= jdleap(i%) And jday < jdleap(i% + 1)) Then

  leapsecond = leapsec(i%)

  Exit Sub

EndIf

Next i%

End Sub

'''''''''''''''''''''''''
'''''''''''''''''''''''''

Function modulo(x) As float

' modulo 2 pi function

''''''''''''''''''''''

Local a

a = x - pi2 * Fix(x / pi2)

If (a < 0.0) Then

a = a + pi2

EndIf

modulo = a

End Function

'''''''''''''''''''''''''''''

Sub jd2str(jdutc, cdate$, utc$)

' convert julian date to date and time strings

Local thr, tmin0, tmin, tsec

'''''''''''''''''''''''''''''

gdate(jdutc, cmonth, day, year)

lmonth = Len(month$(cmonth))

cdate$ = Left$(month$(cmonth), lmonth) + " "

cdate$ = cdate$ + Str$(Int(day)) + ", " + Str$(year)

thours = 24.0 * (day - Int(day))

thr = Fix(thours)

tmin0 = 60.0 * (thours - thr)

tmin = Fix(tmin0)

tsec = 60.0 * (tmin0 - tmin)

' fix seconds and minutes for rollover

If (tsec >= 60.0) Then

  tsec = 0.0

  tmin = tmin + 1.0

EndIf

' fix minutes for rollover

If (tmin >= 60.0) Then

 tmin = 0.0

 thr = thr + 1

EndIf

thour$ = Str$(thr)

If (thr < 10) Then

  thour$ = "0" + thour$

EndIf

tminute$ = Str$(tmin)

If (tmin < 10) Then

  tminute$ = "0" + tminute$

EndIf

tsecond$ = Str$(Int(tsec))

If (tsec < 10) Then

  tsecond$ = "0" + tsecond$

EndIf

utc$ = thour$ + ":" + tminute$ + ":" + tsecond$

End Sub

'''''''''''''''''''
'''''''''''''''''''

Sub deg2str(dd, dms$)

' convert decimal degrees to degrees,
' minutes, seconds string

' input

'  dd = angle in decimal degrees

' output

'  dms$ = string equivalent

'''''''''''''''''''''''''''

Local d1, d, m, s

d1 = Abs(dd)

d = Fix(d1)

d1 = (d1 - d) * 60.0

m = Fix(d1)

s = (d1 - m) * 60.0

If (dd < 0.0) Then

If (d <> 0.0) Then

  d = -d

ElseIf (m <> 0.0) Then

  m = -m

Else

  s = -s

EndIf

EndIf

dms$ = Str$(d) + " deg " + Str$(m) + " min " + Str$(Int(s)) + " sec"

End Sub

'''''''''''''''''''''''''''''''''''''
'''''''''''''''''''''''''''''''''''''

Sub observer(obslat, obslong, obsalt)

' read observer latitude, longitude and altitude subroutine

' output

'  obslat  = latitude (radians)
'  obslong = longitude (radians)
'  obsalt  = altitude (meters)

''''''''''''''''''''''''''''''

Open "observer.dat" For input As #1

' read 15 lines of information

For i% = 1 To 7

  Line Input #1, rem_string$

Next i%

' latitude

Input #1, obslat_deg, obslat_min, obslat_sec

For i% = 1 To 3

  Line Input #1, rem_string$

Next i%

' longitude

Input #1, obslong_deg, obslong_min, obslong_sec

For i% = 1 To 2

  Line Input #1, rem_string$

Next i%

' altitude

Input #1, obsalt

Close #1

obslat.deg$ = Str$(obslat_deg)

If (Left$(obslat.deg$, 2) = "-0") Then

obslat = -dtr * (obslat_min / 60.0 + obslat_sec / 3600.0)

ElseIf (Val(obslat.deg$) = 0.0) Then

obslat = dtr * (obslat_min / 60.0 + obslat_sec / 3600.0)

Else

term1 = Sgn(Val(obslat.deg$))

term2 = Abs(Val(obslat.deg$))

obslat = dtr * term1 * (term2 + obslat_min / 60.0 + obslat_sec / 3600.0)

EndIf

obslong.deg$ = Str$(obslong_deg)

If (Left$(obslong.deg$, 2) = "-0") Then

obslong = -dtr * (obslong_min / 60 + obslong_sec / 3600)

ElseIf (Val(obslong.deg$) = 0.0) Then

obslong = dtr * (obslong_min / 60.0 + obslong_sec / 3600.0)

Else

term1 = Sgn(Val(obslong.deg$))

term2 = Abs(Val(obslong.deg$))

obslong = dtr * term1 * (term2 + obslong_min / 60.0 + obslong_sec / 3600.0)

EndIf

End Sub

'''''''''''''''
'''''''''''''''

Sub check4touch

' check for user touching screen

''''''''''''''''''''''''''''''''

Text 40, 215, "touch screen to continue"

Do

  If (Touch(down) <> 0) Then

    Exit

  EndIf

Loop

End Sub
