  '------------------------------------------------------------------------------'
  '       Micromite RGB Color Selector / Comparator Utility Program -            '
  '       requires a minimum 320 x 240 screen res (in Landscape mode)            '
  '       Larger screens can use either Portrait or Landscape mode               '
  '       (as long as horizontal res >= 320 and vertical res >= 240).            '
  '                                                                              '
  '       Author:   Philip Erb                                                   '
  '                                                                              '
  dim ProgTitle$ = "RGB Color Selector/Comparator"                               '
  '                                                                              '
  Dim ProgVer$   = "v1.0.5"                                                      '
  '                                                                              '
  Dim ProgDate$  = "30-Nov-2017"                                                 '
  '------------------------------------------------------------------------------'
  
  Option Explicit
  Option Default Integer
  
  Const FC% = RGB(Green)                            'Fore Color
  CONST BC% = RGB(Black)                            'Back Color
  
  Const Max5Bit% = 248                              'max value for 5 bit code (Red & Blue)
  Const Inc5Bit% = 8                                'increment between 5 bit values
  Const Max6Bit% = 252                              'max value for 6 bit code (Green)
  Const Inc6Bit% = 4                                'increment between 6 bit values
  Const Max8Bit% = 255                              'max value for 8 bit code (Red, Green & Blue in RGB888)
  Const Inc8Bit% = 1                                'increment between 8 bit values
  
  Const MaxButtons% = 24                            'Max buttons displayed at the same time
  
  Const Debug = 0                                   'Debug mode flag
  
  '================================================================================================
  ' Data used to define Button Coordinates and associated attributes and
  ' the details for the Named Color entries that will be displayed via the Named Color List screen.
  '================================================================================================
  'The following is the button coordinate and flag data for the buttons on the various screens.
  'Data for each button is a group of 5 integers representing: Left, Top, Right, Bottom and Flags.
  '
  'Flag Meaning: 0 = Disabled Button (will not react to touch - text normally Grayed out but depends on button type)
  '              1 = Enabled Button - Animated (will react to touch and provide visual indication of being pressed)
  '              2 = Enabled Button - Not Animated (will react to touch, no visual indication given)
  MainScreen:
  Data 272,   2, 311,  22, 1, 108,  18, 142,  38, 1, 178,  18, 211,  38, 1, 272,  42, 311,  62, 1
  Data 108,  58, 142,  78, 1, 178,  58, 211,  78, 1, 272,  82, 311, 102, 1, 108,  98, 142, 118, 1
  Data 178,  98, 211, 118, 1,   4, 125,  76, 145, 1,  87, 125, 199, 145, 1, 210, 125, 314, 145, 1
  Kybd:
  Data   3,   0,  76,  53, 1,  83,   0, 156,  53, 1, 163,   0, 236,  53, 1,   3,  60,  76, 113, 1
  Data  83,  60, 156, 113, 1, 163,  60, 236, 113, 1,   3, 120,  76, 173, 1,  83, 120, 156, 173, 1
  Data 163, 120, 236, 173, 1,   3, 180,  76, 233, 1,  83, 180, 156, 233, 1, 163, 180, 236, 233, 1
  Data 243, 180, 316, 233, 0
  NCButtons:
  Data   2,   4,  72,  35, 1,  80,   4, 150,  35, 1, 172,   4, 240,  35, 1, 247,   4, 315,  35, 0
  
  Data 298,  41, 315,  58, 2, 298,  61, 315,  78, 2, 298,  81, 315,  98, 2, 298, 101, 315, 118, 2
  data 298, 121, 315, 138, 2, 298, 141, 315, 158, 2, 298, 161, 315, 178, 2, 298, 181, 315, 198, 2
  Data 298, 201, 315, 218, 2, 298, 221, 315, 238, 2
  
  Data   2,  40, 275,  59, 2,   2,  60, 275,  79, 2,   2,  80, 275,  99, 2,   2, 100, 275, 119, 2
  Data   2, 120, 275, 139, 2,   2, 140, 275, 159, 2,   2, 160, 275, 179, 2,   2, 180, 275, 199, 2
  Data   2, 200, 275, 219, 2,   2, 220, 275, 239, 2
  '================================================================================================
  '
  NamedColors:
  ' Named Colors List data
  '   Flag Meaning: 0=Color Group Header (non-selectable)
  '                 1=Named Color (Selectable - White Text)
  '                 2=Named Color (Selectable - Black Text)
  '     Color Name,               R,   G,   B, Flag Index
  'RED Color Group
  Data  "RED Color Group",      248,   0,   0, 0    '0
  Data  "IndianRed",            205,  92,  92, 1    '1
  Data  "LightCoral",           240, 128, 128, 1    '2
  Data  "Salmon",               250, 128, 114, 1    '3
  Data  "DarkSalmon",           233, 150, 122, 1    '4
  Data  "LightSalmon",          255, 160, 122, 1    '5
  Data  "Crimson",              220,  20,  60, 1    '6
  Data  "Red",                  255,   0,   0, 1    '7
  Data  "FireBrick",            178,  34,  34, 1    '8
  Data  "DarkRed",              139,   0,   0, 1    '9
  'PINK Color Group
  Data  "PINK Color Group",     248,  20, 144, 0    '10
  data  "Pink",                 255, 192, 203, 1    '11
  data  "LightPink",            255, 182, 193, 1    '12
  Data  "HotPink",              255, 105, 180, 1    '13
  data  "DeepPink",             255,  20, 147, 1    '14
  data  "MediumVioletRed",      199,  21, 133, 1    '15
  Data  "PaleVioletRed",        219, 112, 147, 1    '16
  'ORANGE Color Group
  data  "ORANGE Color Group",   248, 164,   0, 0    '17
  Data  "Coral",                255, 127,  80, 1    '18
  data  "Tomato",               255,  99,  71, 1    '19
  data  "OrangeRed",            255,  69,   0, 1    '20
  data  "DarkOrange",           255, 140,   0, 1    '21
  data  "Orange",               255, 165,   0, 1    '22
  'YELLOW Color Group
  data  "YELLOW Color Group",   248, 252,   0, 0    '23
  data  "Gold",                 255, 215,   0, 2    '24
  data  "Yellow",               255, 255,   0, 2    '25
  data  "LightYellow",          255, 255, 224, 2    '26
  data  "LemonChiffon",         255, 250, 205, 2    '27
  data  "LightGoldenRodYellow", 250, 250, 210, 2    '28
  Data  "PapayaWhip",           255, 239, 213, 2    '29
  data  "Moccasin",             255, 228, 181, 2    '30
  data  "PeachPuff",            255, 218, 185, 2    '31
  data  "PaleGoldenRod",        238, 232, 170, 2    '32
  data  "Khaki",                240, 230, 140, 2    '33
  Data  "DarkKhaki",            189, 183, 107, 1    '34
  'PURPLE Color Group
  Data  "PURPLE Color Group",   248,   0, 248, 0    '35
  data  "Lavender",             230, 230, 250, 2    '36
  data  "Thistle",              216, 191, 216, 1    '37
  data  "Plum",                 221, 160, 221, 1    '38
  data  "Violet",               238, 130, 238, 1    '39
  data  "Orchid",               218, 112, 214, 1    '40
  data  "Fuchsia",              255,   0, 255, 1    '41
  data  "Magenta",              255,   0, 255, 1    '42
  data  "MediumOrchid",         186,  85, 211, 1    '43
  data  "MediumPurple",         147, 112, 219, 1    '44
  data  "BlueViolet",           138,  43, 226, 1    '45
  data  "DarkViolet",           148,   0, 211, 1    '46
  data  "DarkOrchid",           153,  50, 204, 1    '47
  data  "DarkMagenta",          139,   0, 139, 1    '48
  data  "Purple",               128,   0, 128, 1    '49
  data  "RebeccaPurple",        102,  51, 153, 1    '50
  data  "Indigo",                75,   0, 130, 1    '51
  'GREEN Color Group
  data  "GREEN Color Group",      0, 248,   0, 0    '52
  data  "GreenYellow",          173, 255,  47, 2    '53
  data  "Chartreuse",           127, 255,   0, 2    '54
  data  "LawnGreen",            124, 252,   0, 2    '55
  data  "Lime",                   0, 255,   0, 2    '56
  data  "LimeGreen",             50, 205,  50, 1    '57
  data  "PaleGreen",            152, 251, 152, 2    '58
  data  "LightGreen",           144, 238, 144, 2    '59
  data  "MediumSpringGreen",      0, 250, 154, 2    '60
  data  "SpringGreen",            0, 255, 127, 2    '61
  data  "MediumSeaGreen",        60, 179, 113, 1    '62
  data  "SeaGreen",              46, 139,  87, 1    '63
  data  "ForestGreen",           34, 139,  34, 1    '64
  data  "Green",                  0, 128,   0, 1    '65
  data  "DarkGreen",              0, 100,   0, 1    '66
  data  "YellowGreen",          154, 205,  50, 1    '67
  data  "OliveDrab",            107, 142,  35, 1    '68
  data  "Olive",                128, 128,   0, 1    '69
  data  "DarkOliveGreen",        85, 107,  47, 1    '70
  data  "MediumAquamarine",     102, 205, 170, 1    '71
  data  "DarkSeaGreen",         143, 188, 143, 1    '72
  data  "LightSeaGreen",         32, 178, 170, 1    '73
  data  "DarkCyan",               0, 139, 139, 1    '74
  data  "Teal",                   0, 128, 128, 1    '75
  'BLUE Color Group
  data  "BLUE Color Group",       0,   0, 248, 0    '76
  data  "Aqua",                   0, 255, 255, 2    '77
  data  "Cyan",                   0, 255, 255, 2    '78
  data  "LightCyan",            224, 255, 255, 2    '79
  data  "PaleTurquoise",        175, 238, 238, 2    '80
  data  "Aquamarine",           127, 255, 212, 2    '81
  data  "Turquoise",             64, 224, 208, 1    '82
  data  "MediumTurquoise",       72, 209, 204, 1    '83
  data  "DarkTurquoise",          0, 206, 209, 1    '84
  data  "CadetBlue",             95, 158, 160, 1    '85
  data  "SteelBlue",             70, 130, 180, 1    '86
  data  "LightSteelBlue",       176, 196, 222, 1    '87
  data  "PowderBlue",           176, 224, 230, 2    '88
  Data  "LightBlue",            173, 216, 230, 2    '89
  data  "SkyBlue",              135, 206, 235, 1    '90
  data  "LightSkyBlue",         135, 206, 250, 1    '91
  data  "DeepSkyBlue",            0, 191, 255, 1    '92
  data  "DodgerBlue",            30, 144, 255, 1    '93
  data  "CornflowerBlue",       100, 149, 237, 1    '94
  data  "RoyalBlue",             65, 105, 225, 1    '95
  data  "Blue",                   0,   0, 255, 1    '96
  data  "MediumBlue",             0,   0, 205, 1    '97
  data  "DarkBlue",               0,   0, 139, 1    '98
  data  "Navy",                   0,   0, 128, 1    '99
  data  "MidnightBlue",          25,  25, 112, 1    '100
  data  "SlateBlue",            106,  90, 205, 1    '101
  data  "MediumSlateBlue",      123, 104, 238, 1    '102
  data  "DarkSlateBlue",         72,  61, 139, 1    '103
  'BROWN Color Group
  data  "BROWN Color Group",    232, 128,  80, 0    '104
  data  "Cornsilk",             255, 248, 220, 2    '105
  data  "BlanchedAlmond",       255, 235, 205, 2    '106
  data  "Bisque",               255, 228, 196, 2    '107
  data  "NavajoWhite",          255, 222, 173, 2    '108
  data  "Wheat",                245, 222, 179, 2    '109
  data  "BurlyWood",            222, 184, 135, 1    '110
  data  "Tan",                  210, 180, 140, 1    '111
  data  "RosyBrown",            188, 143, 143, 1    '112
  data  "SandyBrown",           244, 164,  96, 1    '113
  data  "GoldenRod",            218, 165,  32, 1    '114
  data  "DarkGoldenRod",        184, 134,  11, 1    '115
  data  "Peru",                 205, 133,  63, 1    '116
  data  "Chocolate",            210, 105,  30, 1    '117
  data  "SaddleBrown",          139,  69,  19, 1    '118
  data  "Sienna",               160,  82,  45, 1    '119
  data  "Brown",                165,  42,  42, 1    '120
  data  "Maroon",               128,   0,   0, 1    '121
  'GRAY Color Group
  data  "GRAY Color Group",     176, 180, 176, 0    '122
  data  "Gainsboro",            220, 220, 220, 1    '123
  data  "LightGray",            211, 211, 211, 1    '124
  Data  "Silver",               192, 192, 192, 1    '125
  data  "DarkGray",             169, 169, 169, 1    '126
  data  "Gray",                 128, 128, 128, 1    '127
  data  "DimGray",              105, 105, 105, 1    '128
  data  "LightSlateGray",       119, 136, 153, 1    '129
  data  "SlateGray",            112, 128, 144, 1    '130
  data  "DarkSlateGray",         47,  79,  79, 1    '131
  data  "Black",                  0,   0,   0, 1    '132
  'WHITE Color Group
  data  "WHITE Color Group",    248, 252, 248, 0    '133
  data  "White",                255, 255, 255, 2    '134
  data  "Snow",                 255, 250, 250, 2    '135
  data  "Honeydew",             240, 255, 240, 2    '136
  data  "MintCream",            245, 255, 250, 2    '137
  data  "Azure",                240, 255, 255, 2    '138
  data  "AliceBlue",            240, 248, 255, 2    '139
  data  "GhostWhite",           248, 248, 255, 2    '140
  data  "WhiteSmoke",           245, 245, 245, 2    '141
  data  "Seashell",             255, 245, 238, 2    '142
  data  "Beige",                245, 245, 220, 2    '143
  data  "OldLace",              253, 245, 230, 2    '144
  data  "FloralWhite",          255, 250, 240, 2    '145
  data  "Ivory",                255, 255, 240, 2    '146
  data  "AntiqueWhite",         250, 235, 215, 2    '147
  data  "Linen",                250, 240, 230, 2    '148
  data  "LavenderBlush",        255, 240, 245, 2    '149
  data  "MistyRose",            255, 228, 225, 2    '150
  
  Const MaxNCIdx% = 150
  
  '================================================================================================
  ' Data Variables
  '================================================================================================
  
  Dim Integer MaxButtonIdx                          'Max button index in use on current screen
  Dim Integer B_Coords((MaxButtons - 1), 4)         'Button Co-ordinates (Top, Left, Right, Bottom, Enabled/Flags)
  
  Dim Float NC_Data(MaxNCIdx%, 3)                   'Array to store Named Color numeric data - R, G, B & Flag (using Float to conserve space)
  Dim String NC_Names(MaxNCIdx%) Length 20          'Array to store Named Color Names (string data, max 20 chars per element)
  Dim Integer NC_Groups(9, 1)                       'Array to store pointer to start of each Color Group and the display color value
  Dim Integer NC_StartIndex                         'Named Color index to start displaying in the list
  Dim Integer NC_SelectedIndex                      'Index of the named color currently selected (-1 = none selected)
  
  Dim Integer Val_Red(3), Val_Green(3), Val_Blue(3) 'Arrays to store current, previous, original and secondary color component values
  
  Dim integer ColorPanelHeight
  Dim Integer Compare, OverlayText                  'Option Flags
  
  Dim Integer DisplayType                           'MM Display Type
  Dim Integer ColorDepth                            'ColorDepth - 0 = 16-bit (RGB565), 1 = 24-bit (RGB888)
  Dim Integer CV_Max(2)                             'Active ColorValue Maximums   - Array for Red (0), Green (1) and Blue (2)
  Dim Integer CV_Inc(2)                             'Active ColorValue Increments - Array for Red (0), Green (1) and Blue (2)
  
  Dim Integer Mode
  Dim Integer Indx
  dim Integer NewValue, MaxValue, WhichColor        'for use with the Kybd for entering a new color value
  
  '================================================================================================
  ' Program Initialization
  '================================================================================================
  
  Print "Starting " + ProgTitle$ + ": " + ProgVer$ + " - " + ProgDate$
  
  DisplayType = peek(byte peek(word &H9D000090) + 20) 'Get Display Type
  
  If DisplayType > 0 And DisplayType < 7 Then       'Display is using SSD1963 controller - supports RGB888
    Print "Targetting 24-bit ColorDepth (RGB888 Encoding)"
    ColorDepth = 1                                  'RGB888 Encoding
    CV_Max(0) = Max8Bit%                            'Red ColorValue Max
    CV_Max(1) = Max8Bit%                            'Green ColorValue Max
    CV_Max(2) = Max8bit%                            'Blue ColorValue Max
    CV_Inc(0) = Inc8Bit%                            'Red ColorValue Increment
    CV_Inc(1) = Inc8Bit%                            'Green ColorValue Increment
    CV_Inc(2) = Inc8Bit%                            'Blue ColorValue Increment
  Else
    Print "Targetting 16-bit ColorDepth (RGB565 Encoding)"
    ColorDepth = 0                                  'RGB565 Encoding
    CV_Max(0) = Max5Bit%                            'Red ColorValue Max
    CV_Max(1) = Max6Bit%                            'Green ColorValue Max
    CV_Max(2) = Max5bit%                            'Blue ColorValue Max
    CV_Inc(0) = Inc5Bit%                            'Red ColorValue Increment
    CV_Inc(1) = Inc6Bit%                            'Green ColorValue Increment
    CV_Inc(2) = Inc5Bit%                            'Blue ColorValue Increment
  EndIf
  
  if mm.Vres < 240 Or MM.HRes < 320 Then
    (Text MM.HRes\2), (MM.Vres\2), "Requires 320 x 240 screen", "CM", 1, 1, RGB(Red), RGB(White)
    Print "Requires 320 x 240 screen - Terminating."
    End                                             'Terminate program
  EndIf
  
  ColorPanelHeight = 65                             'default for VRes = 240
  
  If mm.Vres > 240 Then
    If mm.Vres < 335 Then
      ColorPanelHeight = mm.Vres - 175              'expand height of color display panel (limited by VRes)
    Else
      ColorPanelHeight = 160                        'limit color display panel height to 160
    Endif
  EndIf
  
  LoadNamedColorArrays()
  NC_SelectedIndex = -1
  
  Color FC%, BC%
  'set initial displayed color to black
  Val_Red(0) = 0
  Val_Red(1) = Val_Red(0)                           'save as Previous Value also
  Val_Green(0) = 0
  Val_Green(1) = Val_Green(0)                       'save as Previous Value also
  Val_Blue(0) = 0
  Val_Blue(1) = Val_Blue(0)                         'save as Previous Value also
  
  'set initial Secondary color to black
  Val_Red(3) = 0
  Val_Green(3) = 0
  Val_Blue(3) = 0
  
  Mode = 0                                          'Mode = show Main Screen
  Compare = 0                                       'Compare Mode = Off
  OverlayText = 0                                   'OverlayText = Off
  MainScreen()                                      'Display Main Screen

  '================================================================================================
  ' Main program loop
  '================================================================================================

  do
    Watchdog 40000                                  'reset after 40 secs on hang
    'check if display is touched
    If Touch(x) <> -1 Then
      'display is touched - get input and process
      HandleTouch()
    EndIf
  Loop

  '================================================================================================
  ' Button handling procedures
  '================================================================================================

  Sub SetupMainButtons()
    'setup the Button Coordinates array for buttons on the Main screen
    Local I
    
    MaxButtonIdx = 11                               'zero relative (i.e. 11 = 12 buttons)
    Restore MainScreen
    
    For I = 0 to MaxButtonIdx
      'Read Button Coords data - Left, Top, Right, Bottom, Enabled
      Read B_Coords(I,0), B_Coords(I,1), B_Coords(I,2), B_Coords(I,3), B_Coords(I,4)
    Next I
    
    B_Coords(10,4) = Compare                        'Enabled - 'Text Overlay' is enabled when Compare Mode is On
  end Sub

  Sub SetupKybdButtons()
    'setup the Button Coordinates array for buttons on the Kybd (Color Value entry) screen
    Local I
    
    MaxButtonIdx = 12                               'zero relative (12 = 13 buttons)
    Restore Kybd
    
    For I = 0 to MaxButtonIdx
      'Read Button Coords data - Left, Top, Right, Bottom, Enabled
      Read B_Coords(I,0), B_Coords(I,1), B_Coords(I,2), B_Coords(I,3), B_Coords(I,4)
    Next I
  End Sub

  Sub SetupNCButtons()
    'setup the Button Coordinates array for buttons on the Named Color List screen
    Local I
    
    MaxButtonIdx = 23                               'zero relative (23 = 24 buttons)
    Restore NCButtons
    
    For I = 0 to MaxButtonIdx
      'Read Button Coords data - Left, Top, Right, Bottom, Enabled
      Read B_Coords(I,0), B_Coords(I,1), B_Coords(I,2), B_Coords(I,3), B_Coords(I,4)
    Next I
  End Sub

  Sub DrawButton(which as Integer, Pressed as Integer)
    'controls drawing/re-drawing any button on any screen
    Local Integer X, Y, W, H, S
    Local String Caption
    
    X = B_Coords(which,0)
    Y = B_Coords(which,1)
    W = ((B_Coords(which,2) - B_Coords(which,0)) + 1)
    H = ((B_Coords(which,3) - B_Coords(which,1)) + 1)
    S = 1                                           'default Font Scale = 1
    
    If Mode = 0 Then                                'Main screen
      Select Case which
        CAse 0, 3, 6, 9, 10, 11                     'Text/Number Button
          If which = 0 then
            Caption = Str$(Val_Red(0))
          ElseIf which = 3 then
            Caption = Str$(Val_Green(0))
          ElseIf which = 6 Then
            Caption = Str$(Val_Blue(0))
          ElseIf which = 9 Then
            Caption = "Compare"
          ElseIf which = 10 Then
            Caption = "Text Overlay"
          ElseIf which = 11 Then
            Caption = "Named Color"
          Endif
          DrawTextButton(X, Y, W, H, 0, S, Pressed, Caption, B_Coords(which,4))
          
        Case 1, 4, 7                                'Decrement Button
          DrawDecrIncrButton(0, X, Y, W, H, Pressed)
          
        Case 2, 5, 8                                'Increment Button
          DrawDecrIncrButton(1, X, Y, W, H, Pressed)
      End Select
      
    Elseif Mode = 1 Then                            'Kybd Screen
      Select Case which
        Case 0, 1, 2, 3, 4, 5, 6, 7, 8
          Caption = Str$(which +1)
          S = 2                                     'font scale = 2 for number buttons
          
        Case 9
          Caption = "Clear"
          
        Case 10
          Caption = "0"
          S = 2                                     'font scale = 2 for number buttons
          
        Case 11
          Caption = "Cancel"
          
        Case 12
          Caption = "Done"
      End Select
      
      DrawTextButton(X, Y, W, H, 1, S, Pressed, Caption, B_Coords(which,4))
      
    ElseIf Mode = 2 Then                            'Named Colors Screen
      Select Case which
        Case 0, 1                                   'PgUp / PgDn
          DrawPgUpDnButton(which, X, Y, W, H, Pressed, B_Coords(which, 4))
          
        CAse 2, 3                                   'Cancel / Done
          If which = 2 then
            Caption = "Cancel"
          Else
            Caption = "Done"
          EndIf
          DrawTextButton(X, Y, W, H, 1, S, Pressed, Caption, B_Coords(which,4))
          
        Case 4 to 13                                'Color Group Select buttons
          DrawColorGroupSelect((which - 4), X, Y, W, H)
          
        Case 14 to 23                               'Named Color buttons
          DisplayNCEntry(which, X, Y, W, H)
      End Select
    Endif
  End Sub

  Sub DrawTextButton(X as Integer, Y as Integer, W as Integer, H as Integer, Just as Integer, Scale as Integer, Pressed as Integer, Caption as String, Enabled As Integer)
    'draw a button containing text. Button and Text attributes are partially controlled via arguments
    'passed to this procedure, with some being statically defined depending on which screen the button
    'is being displayed on
    Local Integer BFC, BBC, TX, TY
    Local String J
    
    if Enabled = 0 then                             'button is disabled
      BFC = RGB(144,144,144)                        'text is gray
      BBC = BC%
    Else
      If Pressed = 0 then                           'not pressed - use 'Normal' video
        BFC = FC%
        BBC = BC%
      Else                                          'pressed - use 'Reverse' video
        BFC = BC%
        BBC = FC%
      EndIf
    EndIf
    
    If Just = 0 Then                                'Center, Top
      TX = X+(W\2)
      TY = Y+4
      J = "CT"
    Else                                            'Center, Middle
      TX = X+(W\2)
      TY = Y+(H\2)
      J = "CM"
    EndIf
    
    'button border style is determined by the screen the button is being displayed on (i.e. Mode value)
    if Mode = 0 then                                'on Main Screen
      Box X, Y, W, H, 1, FC%, BBC                   'draw filled button outline
    ElseIf Mode = 1 Then                            'on Kybd Screen - use rounded corners on button outlines
      RBox X, Y, W, H, 6, FC%, BBC                  'draw filled button outline
    elseIf Mode = 2 Then                            'on Named Color Screen - use smaller rounded corners
      RBox X, Y, W, H, 4, FC%, BBC                  'draw filled button outline
    Endif
    
    Text TX, TY, Caption, J, 1, Scale, BFC, BBC     'overlay caption text (or number)
  end sub

  Sub DrawDecrIncrButton(Type as Integer, X as Integer, Y as Integer, W As Integer, H as Integer, Pressed as Integer)
    'draw the decrement (left facing arrow) or increment (right facing arrow) below a Color Value line on the main screen
    'these arrows are used to decrement/increment the current value of the associated RGB Color component.
    Local Integer BFC, BBC, AC, AX, I, XStep
    
    If Pressed = 0 then                             'not pressed - use 'Normal' video
      BFC = FC%
      BBC = BC%
    Else                                            'pressed - use 'Reverse' video
      BFC = BC%
      BBC = FC%
    EndIf
    
    AC = Y+10
    If Type = 0 Then                                'Decrement button (Left pointing arrow)
      AX = X+4
      XStep = 3
    Else                                            'Increment button (Right pointing arrow)
      AX = X+28
      XStep = -3
    EndIf
    
    Box X, Y, W, H, 1, FC%, BBC                     'draw filled button outline
    
    For I = 0 to 8                                  'draw arrow
      If I = 0 Then
        Line AX, AC, (AX+2), AC, 1, BFC
      Else
        Line AX, (AC-I), AX, (AC+I), 3, BFC
      EndIf
      AX = AX+XStep
    Next I
  End Sub

  Sub DrawColorValuePtr(which as Integer)
    'draw the pointer showing the current value relative position between Min and Max values 
    'on a Color Value line on the main screen
    Local X, XP, Y1, Y2, C, I, CurrVal, PrevVal
    
    C = RGB(Yellow)
    
    If which = 0 Then                               'Red value
      Y1 = 4
      Y2 = 11
      CurrVal = Val_Red(0)
      PrevVal = Val_Red(1)
      If CurrVal => Max5Bit% then C = RGB(RED)      'at Max Limit
      
    Else if which = 1 Then                          'Green value
      Y1 = 44
      Y2 = 51
      CurrVal = Val_Green(0)
      PrevVal = Val_Green(1)
      If CurrVal => Max6Bit% then C = RGB(RED)      'at Max Limit
      
    Else if which = 2 Then                          'Blue value
      Y1 = 84
      Y2 = 91
      CurrVal = Val_Blue(0)
      PrevVal = Val_Blue(1)
      If CurrVal => Max5Bit% then C = RGB(RED)      'at Max Limit
    Endif
    
    If CurrVal = 0 then C = RGB(RED)                'at Min Limit
    
    XP = Fix((PrevVal/255) * 200) + 60              'calc Previous Value pointer X axis position (absolute)
    box (XP-3), Y1, 7, ((Y2-Y1)+1), 0, 0, BC%       'clear previous pointer
    
    X = Fix((CurrVal/255) * 200) + 60               'calc Current Value pointer X axis position (absolute)
    
    For I = 0 to 3                                  'draw pointer
      If I = 0 Then
        Line X, Y1, X, Y2, 1, C
      Else
        Line (X-I), Y1, (X-I), (Y2-(I*2)), 1, C
        Line (X+I), Y1, (X+I), (Y2-(I*2)), 1, C
      EndIf
    Next I
  End Sub

  Sub DrawPgUpDnButton(which as Integer, X as Integer, Y as Integer, W As Integer, H as Integer, Pressed as Integer, Enabled As Integer)
    'draw the PgUp / PgDn buttons used to navigate up and down through the Named Color List
    Local Integer BFC, BBC, AC, AX, AY, I, YStep
    Local String Caption
    if Enabled = 0 then                             'button is disabled
      BFC = RGB(144,144,144)                        'text is gray
      BBC = BC%
    Else
      If Pressed = 0 then                           'not pressed - use 'Normal' video
        BFC = FC%
        BBC = BC%
      Else                                          'pressed - use 'Reverse' video
        BFC = BC%
        BBC = FC%
      EndIf
    EndIf
    
    AX = X+15
    If which = 0 Then                               'PgUp button
      AY = Y+4
      YStep = 3
      Caption = "PgUp"
    Else                                            'PgDn button
      AY = Y+25
      YStep = -3
      Caption = "PgDn"
    EndIf
    
    RBox X, Y, W, H, 4, FC%, BBC                    'draw filled button outline
    
    For I = 0 to 7                                  'draw arrow
      If I = 0 Then
        Line AX, AY, AX, (AY+2), 1, BFC
      Else
        Line (AX-I), AY, (AX+I), AY, 3, BFC
      EndIf
      AY = AY+YStep
    Next I
    
    'Add caption text to the right of the arrow
    Text (X+48), (Y+(H\2)), Caption, "CM", 1, 1, BFC, BBC
  End Sub

  Sub DrawColorGroupSelect(Group as Integer, X As Integer, Y As Integer, W As Integer, H As Integer)
    'draw a color patch on right side of screen to be used as a shortcut to the corresponding 
    'color group in the Named Color List
    
    Box X, Y, W, H, 0, 0, NC_Groups(Group,1)
  End Sub

  sub DrawNCSelectedIndicator(which as Integer, Selected as Integer)
    'draw the Named Color Selected indicator adjacent to a Named Color List entry 
    'when the "button" that contains that entry is pressed,
    'or clears a previously displayed Indicator when the selection is changed
    Local Integer X, X1, Y, Y1, W, H, C, I
    
    X = B_Coords(which,0)                           'retreive/calculate coordinates
    Y = B_Coords(which,1)
    W = (B_Coords(which,2) - B_Coords(which,0)) + 1
    H = (B_Coords(which,3) - B_Coords(which,1)) + 1
    X1 = (X+W)
    Y1 = (Y+(H\2))
    
    If Selected = 0 then                            'clear previous selected indicator at this position
      Box X1, Y, 18, H, 0, 0, BC%
      
    Else                                            'show selected indicator at this position
      C = RGB(Yellow)
      
      For I = 0 to 8                                'draw the left facing arrow pointer
        If I = 0 Then
          Line X1, Y1, (X1+1), Y1, 1, C
        Else
          Line X1, (Y1-I), X1, (Y1+I), 2, C
        Endif
        X1 = X1 + 2
      Next I
    Endif
  end sub

  Sub DisplayNCEntry(which as Integer, X As Integer, Y As Integer, W as Integer, H as integer)
    'display a Named Color entry - this is actually a button so that it can be selected
    'but does not display a border, or show any animation when touched / released.
    Local Integer NCFC, NCBC, NC_Index, LineIdx, L
    Local String Caption
    
    LineIdx = which - 14                            'calc line index
    NC_Index = NC_StartIndex + LineIdx
    
    If NC_Data(NC_Index,3) = 0 then                 'Color Group header
      NCBC = RGB(104,104,104)                       'background = DimGray
      NCFC = RGB(Yellow)                            'text color = Yellow
      B_Coords(which,4) = 0                         'mark this button as disabled (does not react to touch)
      Caption = NC_Names(NC_Index) + " "            'setup Color Group Header caption text
      L = 33 - Len(Caption)
      Caption = Caption + String$(L, "-") 
    Else
      B_Coords(which,4) = 2                         'mark this button as enabled, no animation (does react to touch)
      'Determine background color = named color
      NCBC = CnvtToRGB(NC_Data(NC_Index,0),NC_Data(NC_Index,1),NC_Data(NC_Index,2))
      'Determine text (font) color - specified via NC_Data(n,3) flag value (1= White, 2=Black)
      If NC_Data(NC_Index,3) = 1 Then
        NCFC = RGB(White)                           'white text (for dark colors)
      Else
        NCFC = 0                                    'black text (for light colors)
      ENdif
      Caption = NC_Names(NC_Index)                  'setup Named Color entry caption
    EndIf
    
    Box X, Y, W, H, 0, 0, NCBC                      'no border on button box
    Text (X+4), (Y+3), Caption, "LT", 1, 1, NCFC, NCBC
    
    If NC_Data(NC_Index,3) <> 0 then
      'not a Color Group Header entry - also display the R,G,B values
      Text (X+168), (Y+3), "(" + Str$(NC_Data(NC_Index,0),3) + "," + STR$(NC_Data(NC_Index,1),3) + "," + Str$(NC_Data(NC_Index,2),3) + ")", "LT", 1, 1, NCFC, NCBC
    ENdif
  End Sub

  '================================================================================================
  ' Touch handling procedures
  '================================================================================================

  Sub HandleTouch
    'see which button has been pressed and process accordingly
    Local Integer which, Work
    
    which = CheckButtonPress(0,MaxButtonIdx)
    
    If Debug then
      Print "CheckButtonPress return value:" + Str$(which)
    endif
    
    if which > -1 then
      'a defined button was pressed - action it
      If Mode = 0 Then                              'on Main Screen
        Select Case which
          Case 0, 3, 6                              'one of the Color value boxes
            GetNewColorValue(which)                 'display kybd to allow a new value to be entered
            
          Case 1, 2, 4, 5, 7, 8                     'a decrement or increment button
            DecrIncrColorValue(which)               'handle decrementing or incrementing the appropriate color value
            
          Case 9                                    'Compare On/Off button
            ToggleCompareMode()
            
          Case 10                                   'Overlay Text On/Off button
            ToggleOverlayText()
            
          Case 11                                   'Named Color button
            GetNamedColorValue()
        End Select
        
      ElseIf Mode = 1 Then                          'kybd displayed
        ProcessKybdKey(which)
        
      ElseIf Mode = 2 Then                          'Named Color Select
        Select Case which
          Case 0                                    'PgUp button - calc new start index
            If NC_StartIndex < 9 then
              Work = 0
            Else
              Work = NC_StartIndex - 10
            Endif
            
            ShowNamedColorList(Work)
            
          Case 1                                    'Pgdn button - calc new start index
            If NC_StartIndex + 10 > MaxNCIdx% then
              Work = NC_StartIndex                  'cannot PgDn - already at end of list
            Else
              Work = NC_StartIndex + 10
            Endif
            
            ShowNamedColorList(Work)
            
          Case 2                                    'Cancel button
            Mode = 0                                'just exit back to the main screen
            MainScreen()                            'discarding any color that has been selected.
            
          Case 3                                    'Done button
            'save previous color values
            Val_Red(1) = Val_Red(0)
            Val_Green(1) = Val_Green(0)
            Val_Blue(1) = Val_Blue(0)
            
            'save the selected color value
            If ColorDepth = 0 then                  'use RGB565 Encoding
              Val_Red(0) = CnvtTo5Bit(NC_Data(NC_SelectedIndex, 0))
              Val_Green(0) = CnvtTo6Bit(NC_Data(NC_SelectedIndex, 1))
              Val_Blue(0) = CnvtTo5Bit(NC_Data(NC_SelectedIndex, 2))
              
            Else                                    'use RGB888 Encoding
              Val_Red(0) = NC_Data(NC_SelectedIndex, 0)
              Val_Green(0) = NC_Data(NC_SelectedIndex, 1)
              Val_Blue(0) = NC_Data(NC_SelectedIndex, 2)
            Endif
            
            'exit and return to the main screen
            Mode = 0
            MainScreen()
            
          Case 4 to 13
            ShowNamedColorList(NC_Groups((which - 4),0))
            
          Case 14 to 23
            SelectNamedColor(which)                 'select one of the Named Color entries displayed
        End Select
      EndIf
    Else
      If Debug Then
        Print "Invalid Button pressed - Mode = " + Str$(Mode) + ", Button Index = " + Str$(which)
      EndIf
    EndIf
  End Sub

  Function CheckButtonPress(startn As Integer, endn As Integer) As Integer
    Local Integer keydwn, xt, yt, n, saveTimer
    keydwn = 0
    saveTimer = Timer                               'save timer value on entry
    'check which button has been pressed, then loop until it is released again
    Do
      If Touch(x) <> -1 And Not keydwn Then
        'we have a touch
        xt = Touch(x)
        yt = Touch(y)
        'set the button coordinates to see if the touch was within the boundaries of a button
        For n = startn To endn
          if B_Coords(n,4) > 0 Then                 'only consider 'Enabled' buttons
            If xt > B_Coords(n, 0) And xt < B_Coords(n, 2) And yt > B_Coords(n, 1) And yt < B_Coords(n, 3) Then
              'we have a defined button press
              'draw the button as pressed
              If B_Coords(n,4) <> 2 then            '2 = a button that doesn't show any animation
                DrawButton n, 1
              Endif
              If Mode = 2 then                      'on Named Color selection screen
                If n <> 2 And n <> 3 then           'not 'Cancel' or 'Done' buttons
                  if NC_SelectedIndex <> -1 then    'if there is a currently selected Named Color
                    ResetNCSelectedIndicator()      'reset its selection
                  Endif
                Endif
              Endif
              keydwn = 1
              Exit For
            Endif
          EndIf
        Next n
        
        If Debug then
          Print "Touch at: " + Str$(xt) + "," + Str$(yt) + " n = " + Str$(n) + " keydwn = " + Str$(keydwn)
        endif
      EndIf
      
      'check when touch has been released or if button has been down > 30 secs
      If (Touch(x) = -1 Or (Timer - saveTimer) > 30000) Then
        If keydwn Then
          'button was down - has now been released
          If B_Coords(n,4) <> 2 then
            '2 = a button that doesn't show any animation
            'draw the button as normal (ie, not pressed)
            DrawButton n, 0
          EndIf
          'return value is the button's index
          CheckButtonPress = n
        else
          'touch did not correspond to any defined button
          'return -1 (invalid button)
          CheckButtonPress = -1
        EndIf
        Exit Function
      EndIf
    Loop
  End Function

  '================================================================================================
  ' Procedures providing program functionality components
  '================================================================================================

  Function CnvtTo5Bit(Value as Integer) As Integer
    'convert 8 bit value to 5 bit value for RGB565 code
    CnvtTo5Bit = Value AND &b11111000               'mask out the 3 LSBs
  End Function

  Function CnvtTo6Bit(Value as Integer) As Integer
    'convert 8 bit value to 6 bit value for RGB565 code
    CnvtTo6Bit = Value AND &b11111100               'mask out the 2 LSBs
  End Function
  
  Function CnvtToRGB(C_R as Integer, C_G as Integer, C_B as Integer) as Integer
    'converts the 3 component color values (Red, Green and Blue) to the 
    'equivalent RGB code using RGB888 encoding if ColorDepth indicates 24 bit (1)
    'or if ColorDepth indicates 16 bit (0) using RGB565 encoding.
    
    If ColorDepth = 1 then                          'use RGB888 encoding (24-bit)
      CnvtToRgb = RGB(C_R, C_G, C_B)                
    Else                                            'use RGB565 encoding (16-bit)
      CnvtToRGB = RGB(CnvtTo5Bit(C_R), CnvtTo6Bit(C_G), CnvtTo5Bit(C_B))
    Endif
  End Function
  
  Sub DecrIncrColorValue(which as integer)
    Local Integer Dirn, NewVal, ValStep, Max
    
    Dirn = 0                                        'default - decrement
    
    Select Case which
      Case 1                                        'decrement Red value
        Val_Red(1) = Val_Red(0)                     'save current value as previous value
        NewVal = Val_Red(0)                         'start with current value (will be RGB565 equivalent Red value)
        ValStep = -CV_Inc(0)                        'step value
        Max = CV_Max(0)                             'max value
        
      Case 2                                        'increment Red value
        Dirn = 1
        Val_Red(1) = Val_Red(0)                     'save current value as previous value
        NewVal = Val_Red(0)                         'start with current value (will be RGB565 equivalent Red value)
        ValStep = CV_Inc(0)                         'step value
        Max = CV_Max(0)                             'max value
        
      Case 4                                        'decrement Green value
        Val_Green(1) = Val_Green(0)                 'save current value as previous value
        NewVal = Val_Green(0)                       'start with current value (will be RGB565 equivalent Green value)
        ValStep = -CV_Inc(1)                        'step value
        Max = CV_Max(1)                             'max value
        
      Case 5                                        'increment Green value
        Dirn = 1
        Val_Green(1) = Val_Green(0)                 'save current value as previous value
        NewVal = Val_Green(0)                       'start with current value (will be RGB565 equivalent Green value)
        ValStep = CV_Inc(1)                         'step value
        Max = CV_Max(1)                             'max value
        
      Case 7                                        'decrement Blue value
        Val_Blue(1) = Val_Blue(0)                   'save current value as previous value
        NewVal = Val_Blue(0)                        'start with current value (will be RGB565 equivalent Blue value)
        ValStep = -CV_Inc(2)                        'step value
        Max = CV_Max(2)                             'max value
        
      Case 8                                        'increment Blue value
        Dirn = 1
        Val_Blue(1) = Val_Blue(0)                   'save current value as previous value
        NewVal = Val_Blue(0)                        'start with current value (will be RGB565 equivalent Blue value)
        ValStep = CV_Inc(2)                         'step value
        Max = CV_Max(2)                             'max value
        
      Case Else
        Exit Sub                                    'shouldn't be here
    End Select
    
    If ((Dirn = 0 and NewVal > 0) Or (Dirn = 1 And NewVal < Max)) then
      NewVal = NewVal + ValStep                     'increment or decrement by one step value (4 or 8)
    Endif
    
    If NewVal < 0 then                              'don't allow value to go negative!
      NewVal = 0
    EndIf
    
    If NewVal > Max Then                            'don't allow values above max
      NewVal = Max
    Endif
    
    'update the appropriate color value, redisplay the value in the value box,
    'move the value pointer and adjust the color displayed
    Select Case which
      Case 1, 2                                     'Red
        Val_Red(0) = NewVal
        DrawButton(0,0)
        DrawColorValuePtr(0)
        ShowColor()
        
      Case 4, 5                                     'Green
        Val_Green(0) = NewVal
        DrawButton(3,0)
        DrawColorValuePtr(1)
        ShowColor()
        
      Case 7, 8                                     'Blue
        Val_Blue(0) = NewVal
        DrawButton(6,0)
        DrawColorValuePtr(2)
        ShowColor()
    End Select
  End Sub

  Sub GetNamedColorValue()
    Mode = 2
    ShowNamedColors()
  End Sub

  Sub GetNewColorValue(which as integer)
    Local Integer CurrValue
    
    If which = 0 Then                               'Red
      WhichColor = 0
      CurrValue = Val_Red(0)
      
    ElseIf which = 3 Then                           'Green
      WhichColor = 1
      CurrValue = Val_Green(0)
      
    ElseIf which = 6 Then                           'Blue
      WhichColor = 2
      CurrValue = Val_Blue(0)
      
    Else                                            'shouldn't be here - get out
      exit sub
    endif
    
    Mode = 1
    NewValue = 0
    
    ShowKybd(WhichColor, CurrValue)                 'display the keyboard
  end sub

  Sub ProcessKybdKey(which)
    Local Integer VFC, VBC, EnableDone = 1, SaveDoneEnabled = B_Coords(12,4)
    
    Select Case which
      Case 0                                        '1 key
        NewValue = ((NewValue * 10) + 1)
      Case 1                                        '2 key
        NewValue = ((NewValue * 10) + 2)
      Case 2                                        '3 key
        NewValue = ((NewValue * 10) + 3)
      Case 3                                        '4 key
        NewValue = ((NewValue * 10) + 4)
      Case 4                                        '5 key
        NewValue = ((NewValue * 10) + 5)
      Case 5                                        '6 key
        NewValue = ((NewValue * 10) + 6)
      Case 6                                        '7 key
        NewValue = ((NewValue * 10) + 7)
      Case 7                                        '8 key
        NewValue = ((NewValue * 10) + 8)
      Case 8                                        '9 key
        NewValue = ((NewValue * 10) + 9)
      Case 9                                        'Clear key
        NewValue = 0                                'clear NewValue
        Box 241, 120, 78, 59, 0, 0, BC%             'clear Invalid notification area
        EnableDone = 0                              'this will Disable 'Done' button
      Case 10                                       '0 key
        NewValue = (NewValue * 10)
      Case 11                                       'Cancel key - exit kybd mode
        'cancel - ignore any changes made to NewValue
        Mode = 0
        MainScreen()
        exit sub
      Case 12                                       'Done key - only enabled when NewValue is valid
        'save the new color value
        Select Case WhichColor
          Case 0                                    'Red
            Val_Red(1) = Val_Red(0)                 'save previous value
            If ColorDepth = 0 Then                  'use RGB565 Encoding
              Val_Red(0) = CnvtTo5Bit(NewValue)     'change to new value with conversion to nearest 5 bit code
            Else                                    'use RGB888 Encoding
              Val_Red(0) = NewValue
            Endif
            
          Case 1                                    'Green
            Val_Green(1) = Val_Green(0)             'save previous value
            If ColorDepth = 0 then                  'Use RGB565 Encoding
              Val_Green(0) = CnvtTo6Bit(NewValue)   'change to new value with conversion to nearest 6 bit code
            Else                                    'use RGB888 Encoding
              Val_Green(0) = NewValue
            Endif
            
          Case 2                                    'Blue
            Val_Blue(1) = Val_Blue(0)               'save previous value
            If ColorDepth = 0 Then                  'use RGB565 Encoding
              Val_Blue(0) = CnvtTo5Bit(NewValue)    'change to new value with conversion to nearest 5 bit code
            Else                                    'use RGB888 Encoding
              Val_Blue(0) = NewValue
            Endif
        end select
        
        Mode = 0
        MainScreen()
        Exit Sub
    end Select
    
    'validate the NewValue and update the New Value display
    If NewValue < 0 or NewValue > 255 Then
      'NewValue is outside valid range - report invalid condition
      Box 246, 120, 66, 19, 0, 0, RGB(Red)
      Text 279, 123, "INVALID", "CT", 1, 1, RGB(White), RGB(Red)
      Text 279, 144, "Range is", "CT", 1, 1, FC%, BC%
      Text 279, 161, "0 - 255", "CT", 1, 1, FC%, BC%
      VFC = RGB(Red)
      VBC = RGB(White)
      EnableDone = 0                                'this will Disable 'Done' button
    Else                                            'NewValue is valid
      VFC = RGB(Yellow)
      VBC = BC%
    EndIf
    
    B_Coords(12,4) = EnableDone                     'Enable/Disable 'Done' button
    
    'Display the New Value
    Box 246, 88, 66, 32, 1, VFC, VBC
    Text 279, 91, Str$(NewValue), "CT", 1, 2, VFC, VBC
    
    If B_Coords(12,4) <> SaveDoneEnabled Then
      'repaint the 'Done' button (if it's enabled/disabled state changed)
      DrawButton(12,0)
    Endif
  end sub

  sub ResetNCSelectedIndicator()
    'de-select the currently selected button on the Named Colors screen
    Local Integer B_Idx
    
    B_Idx = ((NC_SelectedIndex - NC_StartIndex) + 14) 'calc Button Index for currently selected NC button
    DrawNCSelectedIndicator(B_Idx, 0)               'clear the Selected Indicator
    NC_SelectedIndex = -1                           'reset Selected Index
    'disable and redraw the 'Done' button - not available now as there is nothing selected
    B_Coords(3,4) = 0                               'button disabled
    DrawButton(3,0)
  end sub

  Sub SelectNamedColor(which)
  
    DrawNCSelectedIndicator(which, 1)               'set the Selected Indicator for this button
    NC_SelectedIndex = NC_StartIndex + (which - 14) 'save this NC item index as the selected one
    'enable and redraw the 'Done' button - now available
    B_Coords(3,4) = 1                               'button enabled
    DrawButton(3,0)
  end sub

  Sub ShowColor()
    Local Integer RGBTextCenter, RGBTextTop, ColorPanelWidth, L_R, L_G, L_B
    
    RGBTextTop = (ColorPanelHeight + 160)           '160 = Top of Color Panel + 5 (space below panel to top of text)
    
    'Note: Color values stored in the Val_Red, Val_Green and Val_Blue arrays 
    'have already been adjusted for RGB565 encoding (if necessary)
    
    If Compare = 0 Then                             'Compare Mode is Off
      'display active color right across the screen (320 pixels anyway)
      RGBTextCenter = 160
      ColorPanelWidth = 320
      L_R = Val_Red(0)
      L_G = Val_Green(0)
      L_B = Val_Blue(0)
    Else                                            'Compare Mode is On
      'Display original (Locked) color on left side (160 pixels wide)
      RGBTextCenter = 80
      ColorPanelWidth = 160
      L_R = Val_Red(2)
      L_G = Val_Green(2)
      L_B = Val_Blue(2)
    Endif
    
    'display the selected color in the color panel
    Box 0, 155, ColorPanelWidth, ColorPanelHeight, 0, 0, RGB(L_R, L_G, L_B)
    Box 0, RGBTextTop, 320, 13, 0, 0, BC%   'clear any previous RGB value display
    Text RGBTextCenter, RGBTextTop, "RGB(" + Str$(L_R) + "," + Str$(L_G) + "," + Str$(L_B) + ")", "CT", 1, 1, FC%
    
    If Compare = 1 Then                             'Compare Mode is On
      'display secondary color on the right side
      Box 160, 155, 160, ColorPanelHeight, 0, 0, RGB(Val_Red(0), Val_Green(0), Val_Blue(0))
      Text 240, RGBTextTop, "RGB(" + Str$(Val_Red(0)) + "," + Str$(Val_Green(0)) + "," + Str$(Val_Blue(0)) + ")", "CT", 1, 1, FC%
      
      If OverlayText = 1 then
        'display string "Sample Text" on the left side Color Panel, using the right side color as the font color 
        'and display the same string on the right side Color Panel, using the left side color as the font color
        Text 80, (155 + (ColorPanelHeight\2)), "Sample Text", "CM", 1, 1, RGB(Val_Red(0), Val_Green(0), Val_Blue(0)), RGB(Val_Red(2), Val_Green(2), Val_Blue(2))
        Text 240, (155 + (ColorPanelHeight\2)), "Sample Text", "CM", 1, 1, RGB(Val_Red(2), Val_Green(2), Val_Blue(2)), RGB(Val_Red(0), Val_Green(0), Val_Blue(0))
      endif
    EndIf
  End Sub

  Sub ShowNamedColorList(StartIdx as Integer)
    Local Integer I, J, MaxIndx, ListEnd = 0
    
    'clear the area used by the list display
    Box B_Coords(14,0), B_Coords(14,1), ((B_Coords(14,2)-B_Coords(14,0))+1), ((B_Coords(23,3)-B_Coords(14,1))+1), 0, 0, BC%
    
    NC_StartIndex = Startidx                        'named color index to start display from
    
    If StartIdx + 10 > MaxNCIdx% Then               'less than 10 entries to be displayed
      MaxIndx = MaxNCIdx% - StartIdx%               'calc number of lines to display
      ListEnd = 1
    Else
      MaxIndx = 9                                   'display 10 lines from specified start
    EndIf
    
    For I = 0 to MaxIndx
      DrawButton((I+14),0)                          'display the Named Color entries for this page
    Next I
    
    If ListEnd = 1 Then                             'end of list - not a full page of 10 entries displayed 
      For J = I to 9                                'cycle thru the remaining button (entry) positions
        B_Coords((J+14),4) = 0                      'setting the button flag for each to disabled
      Next J                                        'to stop them being selected
    EndIf
    
    If ListEnd = 0 Then                             'not at end of list - PgDn should be enabled
      if B_Coords(1,4) = 0 then                     'PgDn currently disabled
        B_Coords(1,4) = 1                           'Re-enable PgDn
        DrawButton(1,0)                             'and redraw button
      Endif
    Else                                            'at end of list - PgDn should be disabled
      If B_Coords(1,4) = 1 Then                     'PgDn currently enabled
        B_Coords(1,4) = 0                           'disable PgDn
        DrawButton(1,0)                             'and redraw button
      endif
    Endif
    
    If StartIdx = 0 then                            'list starts from first entry - PgUp should be disabled
      If B_Coords(0,4) = 1 then                     'PgUp is currently enabled
        B_Coords(0,4) = 0                           'disable PgUp
        DrawButton(0,0)                             'and redraw button
      endif
    else                                            'list doesn't start at first entry - PgUp should be enabled
      If B_Coords(0,4) = 0 Then                     'PgUp is currently disabled
        B_Coords(0,4) = 1                           're-enable PgUp
        DrawButton(0,0)                             'and redraw button
      EndIf
    endif
  end sub
    
  Sub ToggleCompareMode()
    If Compare = 0 then                             'Compare Mode is currently Off
      'Turn Compare Mode On
      Compare = 1
      'Save the current RGB color codes to compare against and as the previous values
      Val_Red(2) = Val_Red(0)
      Val_Green(2) = Val_Green(0)
      Val_Blue(2) = Val_Blue(0)
      Val_Red(1) = Val_Red(0)
      Val_Green(1) = Val_Green(0)
      Val_Blue(1) = Val_Blue(0)
      'set current RGB color codes to the saved secondary color
      Val_Red(0) = Val_Red(3)
      Val_Green(0) = Val_Green(3)
      Val_Blue(0) = Val_Blue(3)
      'update the Color component controls (Red, Green and Blue)
      DrawButton(0,0)
      DrawColorValuePtr(0)
      DrawButton(3,0)
      DrawColorValuePtr(1)
      DrawButton(6,0)
      DrawColorValuePtr(2)
      're-display the color panel
      ShowColor()
      'enable Text Overlay button and re-draw
      B_Coords(10,4) = 1
      DrawButton(10,0)
    Else                                            'Compare Mode is currently On
      'Turn Compare Mode Off
      Compare = 0
      'ensure OverlayText option is also turned off
      OverlayText = 0
      'save current secondary color
      Val_Red(3) = Val_Red(0)
      Val_Green(3) = Val_Green(0)
      Val_Blue(3) = Val_Blue(0)
      'restore original color as active color and set secondary as previous color
      Val_Red(1) = Val_Red(0)
      Val_Green(1) = Val_Green(0)
      Val_Blue(1) = Val_Blue(0)
      Val_Red(0) = Val_Red(2)
      Val_Green(0) = Val_Green(2)
      Val_Blue(0) = Val_Blue(2)
      'update the Color component controls (Red, Green and Blue)
      DrawButton(0,0)
      DrawColorValuePtr(0)
      DrawButton(3,0)
      DrawColorValuePtr(1)
      DrawButton(6,0)
      DrawColorValuePtr(2)
      're-display the color panel
      ShowColor()
      'disable Text Overlay button and re-draw
      B_Coords(10,4) = 0
      DrawButton(10,0)
    ENdIf
  End Sub
    
  Sub ToggleOverlayText()
    'enable/disable displaing sample text overlaid on the two Color Panel 
    'using the color of the other panel as for the font color
    If OverlayText = 0 then                         'OverlayText is currently Off
      'Turn OverlayText On
      OverlayText = 1
    Else                                            'OverlayText is currently On
      'Turn OverlayText Off
      OverlayText = 0
    Endif
    ShowColor()
  End sub
    
  '================================================================================================
  ' Procedures defining screen formats
  '================================================================================================
    
  Sub MainScreen()
    
    CLS
    SetupMainButtons()
    For Indx = 0 to MaxButtonIdx
      DrawButton(Indx, 0)
    Next Indx
    
    Text 20, 6, "Red", "LT", 1, 1, FC%, BC%
    Text 4, 46, "Green", "LT", 1, 1, FC%, BC%
    Text 12, 86, "Blue", "LT", 1, 1, FC%, BC%
    Line 60, 12, 260, 12, 1, FC%
    Line 60, 52, 260, 52, 1, FC%
    Line 60, 92, 260, 92, 1, FC%
    Text 60, 15, "0", "LT", 1, 1, FC%, BC%
    Text 260, 15, "255", "RT", 1, 1, FC%, BC%
    Text 60, 55, "0", "LT", 1, 1, FC%, BC%
    Text 260, 55, "255", "RT", 1, 1, FC%, BC%
    Text 60, 95, "0", "LT", 1, 1, FC%, BC%
    Text 260, 95, "255", "RT", 1, 1, FC%, BC%
    DrawColorValuePtr(0)
    DrawColorValuePtr(1)
    DrawColorValuePtr(2)
    ShowColor()
  End Sub
    
  Sub ShowKybd(WhichColor as Integer, CurrValue as Integer)
    Local String ColorName
    Local Integer CFC, CBC
    
    CLS
    SetupKybdButtons()
    
    For Indx = 0 To MaxButtonIdx
      DrawButton(Indx, 0)
    Next Indx
    
    Select Case WhichColor
      Case 0
        ColorName = "Red"
        CFC = RGB(White)
        CBC = RGB(Red)
      Case 1
        ColorName = "Green"
        CFC = RGB(Black)
        CBC = RGB(Green)
      Case 2
        ColorName = "Blue"
        CFC = RGB(White)
        CBC = RGB(Blue)
    End Select
    
    box 251, 0, 56, 21, 0, 0, CBC
    Text 279, 10, ColorName, "CM", 1, 1, CFC, CBC
    Text 279, 26, "Current", "CT", 1, 1, FC%
    Text 279, 41, Str$(CurrValue), "CT", 1, 1, FC%
    Text 279, 56, "New", "CT", 1, 2, RGB(Yellow)
    Box 246, 88, 66, 32, 1, RGB(Yellow), BC%
    '    Text 279, 91, "999", "CT", 1, 2, RGB(Yellow)
  End Sub
    
  Sub ShowNamedColors()
    CLS
    SetupNCButtons()
    
    For Indx = 0 to (MaxButtonIdx - 10)
      DrawButton(Indx, 0)
    Next Indx
    
    NC_SelectedIndex = -1                           'nothing selected yet
    NC_StartIndex = 0                               'display from the start of the list
    ShowNamedColorList(NC_StartIndex)
  End Sub
    
  '================================================================================================
  ' Procedures for loading the Named Color List Arrays
  '================================================================================================
    
  Sub LoadNamedColorArrays()
    Local Integer I, J
    
    Restore NamedColors
    
    J = 0
    For I = 0 to MaxNCIdx%
      Read NC_Names(I), NC_Data(I,0), NC_Data(I,1), NC_Data(I,2), NC_Data(I,3)
      
      If NC_Data(I,3) = 0 Then                      'Color Group Header
        'store pointer to the start index for this Color Group 
        NC_Groups(J,0) = I
        'also store the RGB color code (in either RGB888 or RGB565 encoding) 
        'to display the shortcut to this group.
        NC_Groups(J,1) = CnvtToRGB(NC_Data(I,0), NC_Data(I,1), NC_Data(I,2))
        J = J + 1
      EndIf
    Next I
  End Sub
