' =============================================================
' USER MAIN PROGRAM
' =============================================================
Option EXPLICIT

' 1. Define the Database to load
Dim STRING dbname$ = "us500"  ' Set this to the definition file name (no extension)
Dim STRING cmd$
Dim STRING defLine$, token$, workLine$, tmp_f$
Dim INTEGER fNum, commaPos, tokIdx, startTime, tStart
Dim STRING temp_alias$, temp_file$, temp_owner$, temp_name$
Dim INTEGER temp_len, temp_type, temp_start, temp_width, spc

' --- QUERY CONTEXT GLOBALS ---
Dim STRING Q_Filter$, Q_OrderFld$, Q_ShowFlds$, Q_OutFile$
Dim INTEGER Q_Limit, Q_OrderDesc, Q_OutType ' 0=Screen, 1=TXT, 2=CSV
Dim INTEGER Result_Count

' --- AGGREGATE GLOBALS ---
Dim INTEGER Aggr_Mode   ' 0=None, 1=Count, 2=Sum, 3=Avg, 4=Min, 5=Max
Dim INTEGER Aggr_Fld_ID ' Which field to calculate?
Dim FLOAT   Aggr_Val    ' The accumulator
Dim INTEGER Aggr_Count  ' How many items added
Dim INTEGER Action_Mode ' 0=None, 1=Delete, 2=Undelete

Dim INTEGER SQL_Mode ' 0=Native, 1=SQL
Dim INTEGER CACHE_LOADED ' 0=no, 1=yes

' 2. Initialize the Engine (Load Schema)
If dbname$ <> "" Then GoSub Init_Database

' 3. Main Command Loop
Do
    Print "DB:" + dbname$ + "> ";
    Line Input cmd$
    tStart = Timer
    Process_Command(cmd$)
    Print "Runtime: " + Str$(Timer - tStart) + " ms"
Loop

End

' =============================================================
' ENGINE STUB (Simulating the Library)
' =============================================================
Init_Database:

' --- DATABASE CONFIGURATION ---
Const MAX_TBL = 10
Const MAX_FLD = 100
Const TYPE_STR = 1
Const TYPE_INT = 2
Const TYPE_FLOAT = 3

' --- TIER 1: TABLE REGISTRY ---

Dim STRING DB_Tbl_Name$(MAX_TBL) LENGTH 32  ' Was 16
Dim STRING DB_Tbl_File$(MAX_TBL) LENGTH 32  ' Was 12
Dim INTEGER DB_Tbl_Len(MAX_TBL)
Dim INTEGER Table_Count

' --- TIER 2: FIELD REGISTRY ---
Dim STRING DB_Fld_Owner$(MAX_FLD) LENGTH 32 ' Was 16
Dim STRING DB_Fld_Name$(MAX_FLD) LENGTH 32  ' Was 32
Dim INTEGER DB_Fld_Type(MAX_FLD)   ' 1=Str, 2=Int, 3=Float
Dim INTEGER DB_Fld_Start(MAX_FLD)
Dim INTEGER DB_Fld_Width(MAX_FLD)
Dim INTEGER Field_Count, db_Record_Count

' --- INDEXING CONFIGURATION ---
Dim INTEGER MAX_RECS = 600   ' Tune this based on your RAM (US500 is 500)

Dim STRING DB_RowBuf$(MAX_TBL)   ' Holds the raw record currently being processed

' Define the Index Structure (Key + Pointer)
Type IndexItem
    Key As STRING LENGTH 31  ' with length byte, aligned to 8 bytes
    RecNo As INTEGER    ' The pointer to the .dat file record
End Type

' Create the Index Array in Memory
Dim IndexItem DB_Index(MAX_RECS)
Dim IndexItem DB_Lookup(MAX_RECS)
' Re-use the Index structure for sorting results in memory
Dim IndexItem Result_List(MAX_RECS)

' =============================================================
' DATABASE INITIALIZATION ENGINE
' =============================================================
Print "Initializing Database Engine... "+MM.Info(CURRENT)
If dbname$ <> "" Then Load_Database dbname$
Return

' =============================================================
' LIBRARY CODE
' =============================================================
Sub Process_Command(cmdIn$)
    Local STRING c$, verb$, rest$, wherePart$
    Local INTEGER i, spc, wherePos, delPos, kwPos
    Local STRING tmp_f$, tName$, fName$

    c$ = Trim$(cmdIn$,,B)

    ' 1. Pre-Parse: Comments & Mode Switching
    i = Instr(c$, "#")
    If i > 0 Then c$ = Trim$(Left$(c$, i - 1))
    If c$ = "" Then Exit Sub
    If UCase$(c$) = "EXIT" Then End
    
    ' --- MODE SWITCHING ---
    If UCase$(c$) = "MODE SQL" Then 
        SQL_Mode = 1 
        Print "Mode switched to SQL (SQLite V2.8 subset)"
        Exit Sub
    EndIf
    If UCase$(c$) = "MODE NATIVE" Then 
        SQL_Mode = 0 
        Print "Mode switched to Native"
        Exit Sub
    EndIf

    ' --- HELP ---
    If UCase$(Left$(c$, 4)) = "HELP" Or c$ = "?" Then
        If Instr(UCase$(c$), "SQL") Then: Show_Help_SQL: Else: Show_Help: endif
        Exit Sub
    EndIf

' --- DATABASE SWITCHER ---
    If Left$(c$, 7) = "use-db " Then
        Cmd_UseDB(Trim$(Mid$(c$, 8)))
        Exit Sub
    EndIf
    
' --- DATABASE BUILDER ---
    If  UCase$(Left$(c$, 8)) = "MAKE-DB " Then 
        Cmd_MakeDB(Trim$(Mid$(c$, 9))) 
        Exit Sub 
    EndIf
    
    ' --- SQL PARSER REDIRECTION ---
    If SQL_Mode = 1 Then
        ' Pass commands starting with recognized SQL verbs to the parser
        ' Pass system commands (index, batch, etc.) through to native handler
        If UCase$(Left$(c$, 7)) = "SELECT " Or UCase$(Left$(c$, 7)) = "DELETE " Or UCase$(Left$(c$, 9)) = "UNDELETE " Then
            c$ = Parse_SQL$(c$)
            ' Fall through to standard processing with the translated string
        EndIf
    EndIf
    
    ' 2. INDEX COMMAND (Smart Syntax)
    If Left$(c$, 6) = "index " Then
        tmp_f$ = Trim$(Mid$(c$, 7))

        ' Style A: "index us500 ON city" (SQL Standard)
        kwPos = Instr(UCase$(tmp_f$), " ON ")
        If kwPos > 0 Then
            tName$ = Trim$(Left$(tmp_f$, kwPos - 1))
            fName$ = Trim$(Mid$(tmp_f$, kwPos + 4))
            Build_Index(tName$, fName$)
            Exit Sub
        EndIf

        ' Style B: "index city IN us500" (Natural English)
        kwPos = Instr(UCase$(tmp_f$), " IN ")
        If kwPos > 0 Then
            fName$ = Trim$(Left$(tmp_f$, kwPos - 1))
            tName$ = Trim$(Mid$(tmp_f$, kwPos + 4))
            Build_Index(tName$, fName$)
            Exit Sub
        EndIf

        ' Style C: "index us500 city" (Legacy/Lazy)
        spc = Instr(tmp_f$, " ")
        If spc > 0 Then
            Build_Index(Left$(tmp_f$, spc-1), Mid$(tmp_f$, spc+1))
        Else
            ' Fallback: Assume table is current dbname (us500)
            Build_Index(dbname$, tmp_f$)
        EndIf
        Exit Sub
    EndIf

    ' 3. BATCH / TOOLS
    If Left$(c$, 6) = "batch " Then
        Run_Batch(Trim$(Mid$(c$, 7)))
        Exit Sub
    EndIf

    If Left$(c$, 6) = "ruler " Then Cmd_Ruler(Trim$(Mid$(c$, 7))) : Exit Sub
    If Left$(c$, 7) = "format " Then Save_Format(Mid$(c$, 8)) : Exit Sub

    ' 4. HANDLE DELETE / UNDELETE (Syntax: filter DELETE FROM table)
    Action_Mode = 0
    delPos = Instr(UCase$(c$), " DELETE FROM ")
    If delPos > 0 Then
        Action_Mode = 1
        c$ = Trim$(Left$(c$, delPos - 1)) ' Strip the action, keep filter
    EndIf
    delPos = Instr(UCase$(c$), " UNDELETE FROM ")
    If delPos > 0 Then
        Action_Mode = 2
        c$ = Trim$(Left$(c$, delPos - 1))
    EndIf

    ' 5. HANDLE AGGREGATES (Syntax: VERB field [WHERE filter])
    Aggr_Mode = 0
    spc = Instr(c$, " ")
    If spc > 0 Then
        verb$ = UCase$(Left$(c$, spc - 1))
        rest$ = Trim$(Mid$(c$, spc + 1))

        If verb$="COUNT" Then Aggr_Mode = 1
        If verb$="SUM"   Then Aggr_Mode = 2
        If verb$="AVG"   Then Aggr_Mode = 3
        If verb$="MIN"   Then Aggr_Mode = 4
        If verb$="MAX"   Then Aggr_Mode = 5

        If Aggr_Mode > 0 Then
            ' Split Field and Where
            wherePos = Instr(UCase$(rest$), " WHERE ")
            If wherePos > 0 Then
                Aggr_Fld_ID = Get_Field_ID(Trim$(Left$(rest$, wherePos - 1)))
                c$ = Trim$(Mid$(rest$, wherePos + 7)) ' The filter
            Else
                Aggr_Fld_ID = Get_Field_ID(rest$)
                c$ = "" ' No filter = Show all
            EndIf

            If Aggr_Mode > 1 And Aggr_Fld_ID = 0 Then
                Print "Error: Unknown field for aggregate."
                Exit Sub
            EndIf

            ' For COUNT, we allow field 0 (Count *)
            If Aggr_Mode = 1 And Aggr_Fld_ID = 0 Then Aggr_Fld_ID = -1
        EndIf
    EndIf

    ' 6. EXECUTE
    If Aggr_Mode > 0 Or Action_Mode > 0 Then
        Parse_And_Execute(c$)
    ElseIf Instr(c$, "=") Or Instr(c$, ">") Or Instr(c$, "<") Or Instr(c$, "~") Or Instr(c$, " LIKE ") Then
        Parse_And_Execute(c$)
    Else
        Print "Unknown command: " + c$
    EndIf
End Sub

' =============================================================
' DATABASE MANAGEMENT (USE-DB)
' =============================================================

Sub Cmd_UseDB(arg$)
    Local STRING newDB$
    newDB$ = Trim$(arg$)
    If newDB$ = "" Then Print "Usage: use-db <dbname>" : Exit Sub
    
    ' Check if .def exists before nuking the current session
    If Not MM.Info(EXISTS FILE newDB$ + ".def") Then
        Print "Error: Schema '" + newDB$ + ".def' not found."
        Exit Sub
    EndIf
    
    Print "Switching to database: " + newDB$
    Load_Database newDB$
    
    ' Update the global tracker so the prompt shows the correct name
    dbname$ = newDB$
End Sub

Sub Load_Database(dbName$)
    ' 1. CLEANUP: Close everything to prevent conflicts
    Close_All_Files
    
    ' 2. RESET GLOBALS
    Table_Count = 0
    Field_Count = 0
    db_Record_Count = 0
    Cache_Loaded = 0  ' Important if you used the RAM cache!
    
    ' 3. PARSE SCHEMA (Uses File #8 - Utility Handle)
    Local STRING defLine$, workLine$, token$
    Local INTEGER commaPos
    
    Open dbName$ + ".def" For Input As #8
    
    Do While Not Eof(#8)
        Line Input #8, defLine$
        defLine$ = Trim$(defLine$)
        
        ' Skip comments/empty
        If Left$(defLine$, 1) = "#" Or defLine$ = "" Then Continue Do
        
        workLine$ = defLine$ + ","
        
        ' Extract Token
        commaPos = Instr(workLine$, ",")
        If commaPos > 0 Then
            token$ = UCase$(Trim$(Left$(workLine$, commaPos - 1)))
            workLine$ = Mid$(workLine$, commaPos + 1)
        EndIf
        
        ' --- PARSE TABLE ---
        If token$ = "TABLE" Then
            Table_Count = Table_Count + 1
            
            ' Alias
            commaPos = Instr(workLine$, ",")
            DB_Tbl_Name$(Table_Count) = Trim$(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Filename
            commaPos = Instr(workLine$, ",")
            DB_Tbl_File$(Table_Count) = Trim$(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Length
            commaPos = Instr(workLine$, ",")
            DB_Tbl_Len(Table_Count) = Val(Left$(workLine$, commaPos - 1))
            
        ' --- PARSE FIELD ---
        ElseIf token$ = "FIELD" Then
            Field_Count = Field_Count + 1
            
            ' Owner
            commaPos = Instr(workLine$, ",")
            DB_Fld_Owner$(Field_Count) = Trim$(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Name
            commaPos = Instr(workLine$, ",")
            DB_Fld_Name$(Field_Count) = Trim$(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Type
            commaPos = Instr(workLine$, ",")
            DB_Fld_Type(Field_Count) = Val(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Start
            commaPos = Instr(workLine$, ",")
            DB_Fld_Start(Field_Count) = Val(Left$(workLine$, commaPos - 1))
            workLine$ = Mid$(workLine$, commaPos + 1)
            
            ' Width
            commaPos = Instr(workLine$, ",")
            DB_Fld_Width(Field_Count) = Val(Left$(workLine$, commaPos - 1))
        EndIf
    Loop
    Close #8
    
    ' 4. INIT RECORD COUNT
    ' Now that Table 1 is loaded, we can count the records
    db_Record_Count = Init_Record_Count()
    
    Print "Database '" + dbName$ + "' Loaded."
    Print Str$(Table_Count) + " Tables, " + Str$(Field_Count) + " Fields."
End Sub

Sub Close_All_Files
    ' Aggressively close all possible file handles
    ' "On Error Skip" ignores errors if a file is already closed
    Local INTEGER i
    For i = 1 To 10
        On Error Skip
        Close #i
    Next i
End Sub

Sub Run_Batch(fName$)
    Local STRING rawLine$
    Local INTEGER startTime
    Local FLOAT batchStartTime

    batchStartTime = Timer
    Print "Running Batch: " + fName$
    If Not MM.Info(EXISTS FILE fName$) Then Print "Error: File not found." : Exit Sub

    Open fName$ For Input As #10
    Do While Not Eof(#10)
        Line Input #10, rawLine$
        Print "BATCH> " + rawLine$

        ' Skip comments/empty lines for timing purposes
        If Left$(Trim$(rawLine$), 1) = "#" Or Trim$(rawLine$) = "" Then
             Process_Command(rawLine$)
        Else
            startTime = Timer
            Process_Command(rawLine$)
            ' FIXED PRINT SYNTAX
            Print "Runtime: " + Str$(Timer - startTime) + " ms"
        EndIf
    Loop
    Close #10
    Print "Batch Complete; Total Time: " + Str$(Timer - batchStartTime) + " ms"
End Sub

Sub Parse_And_Execute(rawQ$)
    Local STRING workQ$, token$, fmtName$
    Local INTEGER ptr

    ' --- RESET QUERY CONTEXT ---
    Q_Filter$ = "" : Q_OrderFld$ = "" : Q_ShowFlds$ = "" : Q_OutFile$ = ""
    Q_Limit = 0 : Q_OrderDesc = 0 : Q_OutType = 0

    workQ$ = rawQ$

    ' --- PARSE CLAUSES (Right to Left extraction) ---

    ' 1. CHECK FOR OUTPUT (TXT / CSV)
    ' Syntax: ... txt filename.txt
    ptr = Instr(UCase$(workQ$), " TXT ")
    If ptr > 0 Then
        Q_OutFile$ = Trim$(Mid$(workQ$, ptr + 5))
        Q_OutType = 1 ' 1 = TXT
        workQ$ = Trim$(Left$(workQ$, ptr - 1))
    EndIf

    ' Syntax: ... csv filename.csv
    ptr = Instr(UCase$(workQ$), " CSV ")
    If ptr > 0 Then
        Q_OutFile$ = Trim$(Mid$(workQ$, ptr + 5))
        Q_OutType = 2 ' 2 = CSV
        workQ$ = Trim$(Left$(workQ$, ptr - 1))
    EndIf

    ' 2. CHECK FOR FORMAT MODIFIER
    ' Syntax: ... format occupation
    ' Action: Load the format file immediately into Q_ShowFlds$
    ptr = Instr(UCase$(workQ$), " FORMAT ")
    If ptr > 0 Then
        fmtName$ = Trim$(Mid$(workQ$, ptr + 8))
        workQ$ = Trim$(Left$(workQ$, ptr - 1))

        If MM.Info(EXISTS FILE fmtName$ + ".fmt") Then
            Open fmtName$ + ".fmt" For Input As #5
            Line Input #5, Q_ShowFlds$
            Close #5
        Else
            Print "Warning: Format '" + fmtName$ + "' not found."
        EndIf
    EndIf

    ' 3. CHECK FOR LIMIT
    ptr = Instr(UCase$(workQ$), " LIMIT ")
    If ptr > 0 Then
        Q_Limit = Val(Mid$(workQ$, ptr + 7))
        workQ$ = Trim$(Left$(workQ$, ptr - 1))
    EndIf

    ' 4. CHECK FOR SHOW (SELECT)
    ' Note: If FORMAT was used above, Q_ShowFlds$ is already set.
    ' This allows manual override: "format jobs show age"
    ptr = Instr(UCase$(workQ$), " SHOW ")
    If ptr > 0 Then
        Q_ShowFlds$ = Trim$(Mid$(workQ$, ptr + 6))
        workQ$ = Trim$(Left$(workQ$, ptr - 1))
    EndIf

    ' 5. CHECK FOR ORDER BY
    ptr = Instr(UCase$(workQ$), " ORDER BY ")
    If ptr > 0 Then
        token$ = Trim$(Mid$(workQ$, ptr + 10))
        If UCase$(Right$(token$, 5)) = " DESC" Then
            Q_OrderDesc = 1
            token$ = Trim$(Left$(token$, Len(token$) - 5))
        EndIf
        Q_OrderFld$ = token$
        workQ$ = Trim$(Left$(workQ$, ptr - 1))
    EndIf

    ' 6. NORMALIZE FILTER (The remaining string is the WHERE clause)
    Q_Filter$ = Normalize_Filter$(workQ$)
' DEBUG: Print the preprocessed command
    If Q_Filter$ <> "" Then Print "DEBUG: Filter=[" + Q_Filter$ + "]"

    ' --- EXECUTE ---
    Execute_Advanced_Query
End Sub

' =============================================================
' ENGINE HELPER FUNCTIONS
' =============================================================

' Find a Field Index by its Name (Case Insensitive)
' Returns 0 if not found
Function Get_Field_ID(fName$)
    Local INTEGER i
    Local STRING target$
    target$ = UCase$(Trim$(fName$))

    For i = 1 To Field_Count
        If UCase$(DB_Fld_Name$(i)) = target$ Then
            Get_Field_ID = i
            Exit Function
        EndIf
    Next i

    Get_Field_ID = 0
End Function

' Extract value from the global DB_RowBuf$ using the Schema
Function Get_Value$(fIdx)
    Local INTEGER s, w, tIdx
    Local STRING owner$

    s = DB_Fld_Start(fIdx)
    w = DB_Fld_Width(fIdx)
    owner$ = DB_Fld_Owner$(fIdx)

    ' Find which Table ID corresponds to the owner alias
    ' (Optimization: We could store TblID in the Field Registry to avoid this loop)
    For tIdx = 1 To Table_Count
        If UCase$(DB_Tbl_Name$(tIdx)) = UCase$(owner$) Then Exit For
    Next tIdx

    ' Extract from the specific buffer for that table
    Get_Value$ = Trim$(Mid$(DB_RowBuf$(tIdx), s, w),,B)
End Function

Sub Build_Index(tblAlias$, fldName$)
    Local INTEGER tIdx, fIdx, count
    Local STRING idxFile$, canonicalTbl$, canonicalFld$

    ' 1. Resolve Field & Table to get Official Names
    fIdx = Get_Field_ID(fldName$)
    If fIdx = 0 Then Print "Error: Field not found." : Exit Sub

    ' Get Canonical Field Name from Schema
    canonicalFld$ = DB_Fld_Name$(fIdx)

    ' Resolve Table
    For tIdx = 1 To Table_Count
        If UCase$(DB_Tbl_Name$(tIdx)) = UCase$(tblAlias$) Then
            canonicalTbl$ = DB_Tbl_Name$(tIdx)
            Exit For
        EndIf
    Next tIdx
    If tIdx > Table_Count Then Print "Error: Table " + tblAlias$ + " not found." : Exit Sub

    ' 2. Build Filename using Canonical Names
    idxFile$ = canonicalTbl$ + "_" + canonicalFld$ + ".ndx"
    Print "Indexing " + canonicalTbl$ + "." + canonicalFld$ + "..."

    ' 3. Reset Array
    ReDim Preserve DB_Index(MAX_RECS)

    ' 4. Read Loop
    Open DB_Tbl_File$(tIdx) For Input As #6
    count = 0
    Do While Not Eof(#6)
        Line Input #6, DB_RowBuf$(tIdx)
        If DB_RowBuf$(tIdx) = "" Then Continue Do

        If count >= MAX_RECS Then Print "Error: MAX_RECS exceeded." : Exit Do

        ' Store Key and RecNo
        DB_Index(count).Key = UCase$(Get_Value$(fIdx))
        DB_Index(count).RecNo = count + 1

        count = count + 1
        If count Mod 100 = 0 Then Print ".";
    Loop

    Close #6
    Print " Read " + Str$(count) + " records."

    ' 5. Sort and Save
    ReDim Preserve DB_Index(count - 1)
    Struct Sort DB_Index().Key

    Open idxFile$ For Output As #7
    Struct Save #7, DB_Index()
    Close #7
    Print "Saved " + idxFile$
End Sub

Sub Load_Row(recNum)
    Local INTEGER i, recLen, jobRecNum, tIdx, id

    ' 1. Load Primary Table (US500)
    tIdx = 1
    recLen = DB_Tbl_Len(tIdx)
    Open DB_Tbl_File$(tIdx) For Random As #9
    Seek #9, (recNum - 1) * recLen + 1
    DB_RowBuf$(tIdx) = Input$(recLen, #9)
    Close #9

    ' 2. Load Secondary Table (JobSort) via Join
    id = recNum
    tIdx = 2
    recLen = DB_Tbl_Len(tIdx)

    If MM.Info(EXISTS FILE "jobsort_id_ref.ndx") Then
        Open "jobsort_id_ref.ndx" For Input As #8
        ' FIX: Load into the LOOKUP array, not the main DB_Index
        Struct Load #8, DB_Lookup()
        Close #8

        jobRecNum = -1
        ' FIX: Search the LOOKUP array
        i = Struct(Find DB_Lookup().Key, Str$(id))

        If i >= 0 Then jobRecNum = DB_Lookup(i).RecNo

        If jobRecNum > 0 Then
            Open DB_Tbl_File$(tIdx) For Random As #9
            Seek #9, (jobRecNum - 1) * recLen + 1
            DB_RowBuf$(tIdx) = Input$(recLen, #9)
            Close #9
        Else
            DB_RowBuf$(tIdx) = ""
        EndIf
    Else
        DB_RowBuf$(tIdx) = ""
    EndIf
End Sub

ow_Fast(recNum)
    Local INTEGER k, seekPos
    Static INTEGER nextRec(10)

    If nextRec(1) = 0 Then
        For k = 1 To 10 : nextRec(k) = -1 : Next k
    EndIf

    For k = 1 To Table_Count
        seekPos = (recNum - 1) * DB_Tbl_Len(k) + 1
        
        If k = 1 Then
            If recNum <> nextRec(1) Then Seek #1, seekPos
            If Not Eof(#1) Then Line Input #1, DB_RowBuf$(1) : nextRec(1) = recNum + 1 Else DB_RowBuf$(1) = ""
            
        ElseIf k = 2 Then
            If recNum <> nextRec(2) Then Seek #2, seekPos
            If Not Eof(#2) Then Line Input #2, DB_RowBuf$(2) : nextRec(2) = recNum + 1 Else DB_RowBuf$(2) = ""
            
        ElseIf k = 3 Then
            If recNum <> nextRec(3) Then Seek #3, seekPos
            If Not Eof(#3) Then Line Input #3, DB_RowBuf$(3) : nextRec(3) = recNum + 1 Else DB_RowBuf$(3) = ""
            
        ElseIf k = 4 Then
            If recNum <> nextRec(4) Then Seek #4, seekPos
            If Not Eof(#4) Then Line Input #4, DB_RowBuf$(4) : nextRec(4) = recNum + 1 Else DB_RowBuf$(4) = ""
            
        ElseIf k = 5 Then
            If recNum <> nextRec(5) Then Seek #5, seekPos
            If Not Eof(#5) Then Line Input #5, DB_RowBuf$(5) : nextRec(5) = recNum + 1 Else DB_RowBuf$(5) = ""
        EndIf
    Next k
End Sub

Sub Load_Row_Fast(recNum)
    Local INTEGER k, seekPos
    Static INTEGER nextRec(10)

    If nextRec(1) = 0 Then
        For k = 1 To 10 : nextRec(k) = -1 : Next k
    EndIf

    For k = 1 To Table_Count
        ' Only Seek if we are not at the expected sequential position
        If recNum <> nextRec(k) Then 
            seekPos = (recNum - 1) * DB_Tbl_Len(k) + 1
            Seek #k, seekPos
        EndIf
        
        ' --- READ OR GHOST ---
        If Not Eof(#k) Then
            Line Input #k, DB_RowBuf$(k)
            nextRec(k) = recNum + 1
        Else
            ' GHOST RECORD: File ended, but query loop continues (Implicit Join)
            DB_RowBuf$(k) = ""
            ' CRITICAL FIX: Increment nextRec anyway! 
            ' This tells the engine "We are virtually at the next record"
            ' preventing expensive Seeks on every subsequent ghost row.
            nextRec(k) = recNum + 1
        EndIf
    Next k
End Sub

Sub Cmd_Ruler(fName$)
    Local STRING raw$, numLine$, tickLine$, digit$
    Local INTEGER i, L

    If fName$ = "" Then Print "Usage: ruler <filename>" : Exit Sub
    If Not MM.Info(EXISTS FILE fName$) Then Print "Error: File not found." : Exit Sub

    ' Open as Input to grab the first line (record) automatically
    Open fName$ For Input As #5
    Line Input #5, raw$
    Close #5

    L = Len(raw$)
    Print "File: [" + fName$ + "]  Length: " + Str$(L)
    Print "Data: [" + raw$ + "]"

    ' --- Build the Ruler ---
    ' 1. The Tick Marks (| at 5, 10, 15...)
    tickLine$ = ""
    For i = 1 To L
        If i Mod 5 = 0 Then tickLine$ = tickLine$ + "|" Else tickLine$ = tickLine$ + " "
    Next i

    ' 2. The Numbers (0, 5, 10...) aligned under ticks
    numLine$ = ""
    For i = 1 To L
        If i Mod 5 = 0 Then
            ' Print the number centered/aligned on the tick
            digit$ = Str$(i)
            ' Backtrack in the string to overwrite spaces with the number
            ' (Basic Logic: chop off trailing spaces equal to len of number, add number)
            If Len(numLine$) >= Len(digit$) Then
                 numLine$ = Left$(numLine$, Len(numLine$) - Len(digit$) + 1) + digit$
            Else
                 numLine$ = numLine$ + digit$
            EndIf
        Else
            numLine$ = numLine$ + " "
        EndIf
    Next i

    ' Print Rulers (indented to match "Data: [" prefix which is 7 chars)
    Print "       " + tickLine$
    Print "       " + numLine$
End Sub

' --- FORMATTING: SAVE VIEW ---
Sub Save_Format(args$)
    Local STRING fmtName$, fieldList$, fName$
    Local INTEGER spc

    ' Parse "occupation field1, field2, ..."
    args$ = Trim$(args$)
    spc = Instr(args$, " ")

    If spc = 0 Then Print "Usage: format <name> <field_list>" : Exit Sub

    fmtName$ = Left$(args$, spc - 1)
    fieldList$ = Trim$(Mid$(args$, spc + 1))
    fName$ = fmtName$ + ".fmt"

    Open fName$ For Output As #5
    Print #5, fieldList$
    Close #5

    Print "Format '" + fmtName$ + "' saved to " + fName$
End Sub

' --- UTILITY: REMOVE SPACES (Respecting Quotes) & NORMALIZE OPERATORS ---
Function Normalize_Filter$(s$)
    Local INTEGER i, inQuote, likePos
    Local STRING c$, outx$, temp$

    ' 1. Pre-process: Replace " LIKE " with "~" (Case Insensitive)
    temp$ = s$
    Do
        likePos = Instr(UCase$(temp$), " LIKE ")
        If likePos = 0 Then Exit Do
        ' Replace " LIKE " with "~"
        temp$ = Left$(temp$, likePos - 1) + "~" + Mid$(temp$, likePos + 6)
    Loop

    ' 2. Standard Normalization (Strip spaces, respect quotes)
    outx$ = ""
    inQuote = 0

    For i = 1 To Len(temp$)
        c$ = Mid$(temp$, i, 1)

        ' Toggle Quote Flag
        If c$ = Chr$(34) Then inQuote = Not inQuote
        If c$ = "'" Then inQuote = Not inQuote

        ' Copy character IF it's not a space, OR if we are inside quotes
        If c$ <> " " Or inQuote <> 0 Then outx$ = outx$ + c$
    Next i

    Normalize_Filter$ = outx$
End Function

Sub Execute_Query(cmdIn$)
    Local STRING q$, fmtName$, displayFolds$, strat$, idxFile$, rawFld$
    Local STRING valStr$, ownerAlias$, canonicalTbl$
    Local FLOAT valx, rowVal
    Local INTEGER i, totalRecs, matchCount, fmtPos, sortFldIdx, k
    Local INTEGER useComplex, isMatch, useIndex, fIdx, idxPtr, opIdx
    Local STRING op$

    q$ = Trim$(cmdIn$)
    displayFolds$ = ""

    ' --- DIAGNOSTIC PRINT ---
    Print "DIAG: Analyzing Query [" + q$ + "]"

' 1. FORMAT HANDLING (Use #8)
    fmtPos = Instr(UCase$(q$), " FORMAT ")
    If fmtPos > 0 Then
        fmtName$ = Trim$(Mid$(q$, fmtPos + 8))
        q$ = Trim$(Left$(q$, fmtPos - 1))
        If MM.Info(EXISTS FILE fmtName$ + ".fmt") Then
            Open fmtName$ + ".fmt" For Input As #8
            Line Input #8, displayFolds$
            Close #8
        EndIf
    EndIf
    
    ' FIX: Ensure we use the Global Q_ShowFlds$ if no specific format was loaded
    If displayFolds$ = "" Then displayFolds$ = Q_ShowFlds$
    
    ' Fallback to default format
    If displayFolds$ = "" And MM.Info(EXISTS FILE dbname$ + "-default.fmt") Then
        Open dbname$ + "-default.fmt" For Input As #8
        Line Input #8, displayFolds$
        Close #8
    EndIf
    ' 2. ANALYZE COMPLEXITY
    useComplex = 0 : useIndex = 0 : op$ = ""
    If Instr(q$, "&") Or Instr(q$, "|") Or Instr(q$, "(") Or Instr(q$, "~") Then useComplex = 1
    If q$ = "" Then useComplex = 0

    If useComplex Then Print "DIAG: Query marked as Complex (Linear Scan forced)"

    ' 3. ATTEMPT TO FIND INDEX
    If useComplex = 0 And Action_Mode = 0 Then
        opIdx = 0
        If Instr(q$, ">=") Then: opIdx = Instr(q$, ">=") : op$ = ">=": EndIf
        If opIdx=0 Then
          If Instr(q$, "<=") Then: opIdx = Instr(q$, "<=") : op$ = "<=": EndIf
          If Instr(q$, "<>") Then: opIdx = Instr(q$, "<>") : op$ = "<>": EndIf
          If Instr(q$, "=") Then: opIdx = Instr(q$, "=") : op$ = "=": EndIf
          If Instr(q$, ">") Then: opIdx = Instr(q$, ">") : op$ = ">": EndIf
          If Instr(q$, "<") Then: opIdx = Instr(q$, "<") : op$ = "<": EndIf
        EndIf

        Print "DIAG: Operator detected: [" + op$ + "] at Index: " + Str$(opIdx)

        If opIdx > 0 And op$ <> "<>" Then
            rawFld$ = Trim$(Left$(q$, opIdx - 1))
            valStr$ = Trim$(Mid$(q$, opIdx + Len(op$)))
            If Left$(valStr$, 1) = Chr$(34) Then valStr$ = Mid$(valStr$, 2, Len(valStr$)-2)
            valStr$ = UCase$(valStr$)

            fIdx = Get_Field_ID(rawFld$)
            Print "DIAG: Parsed Field [" + rawFld$ + "] -> ID: " + Str$(fIdx)

            If fIdx > 0 Then
                ownerAlias$ = DB_Fld_Owner$(fIdx)
                canonicalTbl$ = ownerAlias$
                For k = 1 To Table_Count
                    If UCase$(DB_Tbl_Name$(k)) = UCase$(ownerAlias$) Then
                        canonicalTbl$ = DB_Tbl_Name$(k)
                        Exit For
                    EndIf
                Next k

                idxFile$ = canonicalTbl$ + "_" + DB_Fld_Name$(fIdx) + ".ndx"
                Print "DIAG: Looking for Index File: [" + idxFile$ + "]"

                If MM.Info(EXISTS FILE idxFile$) Then
                    Print "DIAG: Index Found! Switching to Indexed Strategy."
                    useIndex = 1
                Else
                    Print "DIAG: Index File NOT found on disk."
                EndIf
            EndIf
        EndIf
    EndIf

    ' OPEN TABLES #1-#5
    totalRecs = 0
    If Table_Count >= 1 Then Open DB_Tbl_File$(1) For Input As #1 : i=Lof(#1)/DB_Tbl_Len(1) : If i>totalRecs Then totalRecs=i
    If Table_Count >= 2 Then Open DB_Tbl_File$(2) For Input As #2 : i=Lof(#2)/DB_Tbl_Len(2) : If i>totalRecs Then totalRecs=i
    If Table_Count >= 3 Then Open DB_Tbl_File$(3) For Input As #3 : i=Lof(#3)/DB_Tbl_Len(3) : If i>totalRecs Then totalRecs=i
    If Table_Count >= 4 Then Open DB_Tbl_File$(4) For Input As #4 : i=Lof(#4)/DB_Tbl_Len(4) : If i>totalRecs Then totalRecs=i
    If Table_Count >= 5 Then Open DB_Tbl_File$(5) For Input As #5 : i=Lof(#5)/DB_Tbl_Len(5) : If i>totalRecs Then totalRecs=i

    ' Select Strategy
    If useComplex Then
        strat$ = "Complex Scan"
    ElseIf useIndex Then
        strat$ = "Indexed Range [" + idxFile$ + "]"
    Else
        strat$ = "Linear Scan"
    EndIf

    Aggr_Val = 0 : Aggr_Count = 0
    If Aggr_Mode = 4 Then Aggr_Val = 99999999
    If Aggr_Mode = 5 Then Aggr_Val = -99999999

    If Aggr_Mode = 0 And Action_Mode = 0 Then
        Print "Searching " + Str$(totalRecs) + " records..."
        Print "Strategy: " + strat$
        Print "------------------------------------------------"
    EndIf

    matchCount = 0 : Result_Count = 0
    If Q_OrderFld$ <> "" Then sortFldIdx = Get_Field_ID(Q_OrderFld$)

' =========================================================
    ' STRATEGY: INDEXED SEARCH
    ' =========================================================
    If useIndex = 1 Then
        ' 1. LOAD THE INDEX FILE (Critical Step!)
        ' We found the file exists, now we must load it into RAM to search it.
        If useIndex = 1 Then
            Open idxFile$ For Input As #6
            Struct Load #6, DB_Index()
            Close #6
        EndIf
        ' 2. Calculate the starting point
        idxPtr = Find_First_GE(valStr$, db_Record_Count)

        ' --- CASE A: Greater Than (>) or Greater Equal (>=) ---
        If op$ = ">" Or op$ = ">=" Then
            ' If we requested strict ">", and the first match is equal, skip it.
            If op$ = ">" Then
                Do While idxPtr <= db_Record_Count-1
                    If DB_Index(idxPtr).Key > valStr$ Then Exit Do
                    idxPtr = idxPtr + 1
                Loop
            EndIf

            ' Loop from the start point to the end
            For i = idxPtr To db_Record_Count-1
                ' FETCH the full record using the ID from the index
                Load_Row_Fast(DB_Index(i).RecNo)

                ' PRINT
                Print_Result(DB_Index(i).RecNo, q$, displayFolds$)
                matchCount = matchCount + 1
            Next i

        ' --- CASE B: Less Than (<) or Less Equal (<=) ---
        ElseIf op$ = "<" Or op$ = "<=" Then
            For i = 0 To db_Record_Count-1
                ' CHECK condition using the Index Key
                If (op$ = "<" And DB_Index(i).Key < valStr$) Or (op$ = "<=" And DB_Index(i).Key <= valStr$) Then
                    ' FETCH and PRINT
                    Load_Row_Fast(DB_Index(i).RecNo)
                    Print_Result(DB_Index(i).RecNo, q$, displayFolds$)
                    matchCount = matchCount + 1
                Else
                    ' OPTIMIZATION: Stop as soon as we exceed the value!
                    Exit For
                EndIf
            Next i

        ' --- CASE C: Equals (=) ---
        ElseIf op$ = "=" Then
            Do While idxPtr < db_Record_Count
                If DB_Index(idxPtr).Key <> valStr$ Then Exit Do

                ' FETCH and PRINT
                Load_Row_Fast(DB_Index(idxPtr).RecNo)
                Print_Result(DB_Index(idxPtr).RecNo, q$, displayFolds$)

                matchCount = matchCount + 1
                idxPtr = idxPtr + 1
            Loop
        EndIf

    ' =========================================================
    ' STRATEGY: LINEAR / COMPLEX SCAN
    ' =========================================================
    ' --- PATH B: SCAN EXECUTION ---
    Else
        For i = 1 To totalRecs
            ' 1. Load the record from disk
            Load_Row_Fast(i) 
            
            ' 2. GHOST RECORD FIX:
            ' Check Table 1's buffer. If it's empty, skip it.
            If Trim$(DB_RowBuf$(1)) = "" Then Continue For
            
            isMatch = 0
            If q$ = "" Then isMatch = 1 
            
            If useComplex Then
                If Check_Condition(q$) = 1 Then isMatch = 1
            Else
                If Evaluate_Primitive(q$) = 1 Then isMatch = 1
            EndIf

            If isMatch = 1 Then
                matchCount = matchCount + 1
                If Aggr_Mode > 0 Then
                    If Aggr_Fld_ID > 0 Then rowVal = Val(Get_Value$(Aggr_Fld_ID))
                    If Aggr_Mode = 1 Then Aggr_Val = Aggr_Val + 1

                    If Aggr_Mode = 2 Then Aggr_Val = Aggr_Val + rowVal

                    If Aggr_Mode = 3 Then
                        Aggr_Val = Aggr_Val + rowVal
                        Aggr_Count = Aggr_Count + 1
                    EndIf

                    If Aggr_Mode = 4 Then
                        If rowVal < Aggr_Val Then Aggr_Val = rowVal
                    EndIf

                    If Aggr_Mode = 5 Then
                        If rowVal > Aggr_Val Then Aggr_Val = rowVal
                    EndIf                ElseIf Action_Mode > 0 Then
                    If Action_Mode = 1 Then Print "Record " + Str$(i) + " marked for DELETION."
                    If Action_Mode = 2 Then Print "Record " + Str$(i) + " marked for RESTORE."
                Else
                    If Q_OrderFld$ <> "" Then
                        Result_Count = matchCount
                        Result_List(matchCount).RecNo = i
                        Result_List(matchCount).Key = Get_Value$(sortFldIdx)
                    Else
                        Print_Result(i, q$, displayFolds$)
                        If Q_Limit > 0 And matchCount >= Q_Limit Then Exit For
                    EndIf
                EndIf
            EndIf
        Next i
    EndIf

    ' --- POST-PROCESS ---
    If Aggr_Mode > 0 Then
        If Aggr_Mode = 3 And Aggr_Count > 0 Then Aggr_Val = Aggr_Val / Aggr_Count
        Print "Result: " + Str$(Aggr_Val)
    ElseIf Action_Mode > 0 Then
        Print "Action Complete. " + Str$(matchCount) + " records affected."
    ElseIf Q_OrderFld$ <> "" And Result_Count > 0 Then
        Sort_Results
        For i = 1 To Result_Count
            If Q_Limit > 0 And i > Q_Limit Then Exit For
            Load_Row_Fast(Result_List(i).RecNo)
            Print_Result(Result_List(i).RecNo, q$, displayFolds$)
        Next i
        Print "------------------------------------------------"
        Print Str$(matchCount) + " matches found."
    ElseIf Aggr_Mode = 0 Then
        Print "------------------------------------------------"
        Print Str$(matchCount) + " matches found."
    EndIf

    ' CLOSE TABLES
    If Table_Count >= 1 Then Close #1
    If Table_Count >= 2 Then Close #2
    If Table_Count >= 3 Then Close #3
    If Table_Count >= 4 Then Close #4
    If Table_Count >= 5 Then Close #5
End Sub

' Helper to centralize printing (Add this if you haven't already)
Sub Print_Result(recID, q$, dFolds$)
    ' Logic: Only use dFolds$ (calculated in Execute_Query)
    ' Fallback: If no format, print the match criteria.

    If dFolds$ <> "" Then
        Print_Record_Dynamic(dFolds$)
    Else
        Print "[Match] " + q$
    EndIf
End Sub

Sub Execute_Advanced_Query
    ' Wrapper: Connects the new Parser to the existing Engine

    ' 1. Handle Basic Filtering
    ' We pass the cleaned-up filter string (e.g., "city=Abilene") to the engine
    If Q_Filter$ <> "" Or Aggr_Mode > 0 Then
        Execute_Query(Q_Filter$)
    Else
        Print "Error: No filter specified."
    EndIf

    ' Note: Sorting (Q_OrderFld) and Limits (Q_Limit)
    ' will be implemented here in the next steps.
End Sub

' Helper to print list of fields
Sub Print_Record_Dynamic(fieldList$)
    Local STRING fToken$, valx$, buffer$
    Local INTEGER fIdx, w, commaPos

    buffer$ = fieldList$ + ","

    Do While buffer$ <> ""
        commaPos = Instr(buffer$, ",")
        If commaPos = 0 Then Exit Do

        fToken$ = Trim$(Left$(buffer$, commaPos - 1))
        buffer$ = Mid$(buffer$, commaPos + 1)

        If fToken$ = "" Then Continue Do

        fIdx = Get_Field_ID(fToken$)
        If fIdx > 0 Then
            valx$ = Get_Value$(fIdx)
            w = DB_Fld_Width(fIdx)
            ' Print value padded to width
            Print Left$(valx$ + Space$(w), w) + " ";
        EndIf
    Loop
    Print "" ' New Line
End Sub

' --- LOGIC: SINGLE CONDITION CHECK ---
' --- LOGIC: SINGLE CONDITION CHECK ---
Function Evaluate_Primitive(cond$)
    Local STRING c$, fName$, op$, valx$, dbVal$
    Local INTEGER opIdx, fIdx, isMatch
    Local FLOAT n1, n2

    c$ = cond$

    ' Detect Operator (Order matters: >= before >)
    op$ = "" : opIdx = 0
    opIdx = Instr(c$, ">=") : If opIdx > 0 Then op$ = ">="
    If opIdx = 0 Then opIdx = Instr(c$, "<=") : If opIdx > 0 Then op$ = "<="
    If opIdx = 0 Then opIdx = Instr(c$, "<>") : If opIdx > 0 Then op$ = "<>"
    If opIdx = 0 Then opIdx = Instr(c$, "=") : If opIdx > 0 Then op$ = "="
    If opIdx = 0 Then opIdx = Instr(c$, ">") : If opIdx > 0 Then op$ = ">"
    If opIdx = 0 Then opIdx = Instr(c$, "<") : If opIdx > 0 Then op$ = "<"
    If opIdx = 0 Then opIdx = Instr(c$, "~") : If opIdx > 0 Then op$ = "~" 

    If opIdx = 0 Then Evaluate_Primitive = 0 : Exit Function

    fName$ = Trim$(Left$(c$, opIdx - 1))
    valx$ = Trim$(Mid$(c$, opIdx + Len(op$)))
    If Left$(valx$, 1) = Chr$(34) Then valx$ = Mid$(valx$, 2, Len(valx$)-2)

    fIdx = Get_Field_ID(fName$)
    If fIdx = 0 Then Evaluate_Primitive = 0 : Exit Function

    dbVal$ = Get_Value$(fIdx)
    isMatch = 0

    ' STRING COMPARISON
    If DB_Fld_Type(fIdx) = 1 Then
         ' Standard String Ops
         If op$ = "=" And UCase$(dbVal$) = UCase$(valx$) Then isMatch = 1
         If op$ = "<>" And UCase$(dbVal$) <> UCase$(valx$) Then isMatch = 1
         
         ' FIX: Add missing Inequality Operators for Strings
         If op$ = ">" And UCase$(dbVal$) > UCase$(valx$) Then isMatch = 1
         If op$ = "<" And UCase$(dbVal$) < UCase$(valx$) Then isMatch = 1
         If op$ = ">=" And UCase$(dbVal$) >= UCase$(valx$) Then isMatch = 1
         If op$ = "<=" And UCase$(dbVal$) <= UCase$(valx$) Then isMatch = 1

         ' LIKE Operator
         If op$ = "~" Then
             If Instr(valx$, "%") > 0 Or Instr(valx$, "_") > 0 Then
                 If Like_Match(UCase$(dbVal$), UCase$(valx$)) = 1 Then isMatch = 1
             Else
                 If Instr(UCase$(dbVal$), UCase$(valx$)) > 0 Then isMatch = 1
             EndIf
         EndIf
         
    ' NUMERIC COMPARISON
    Else
         n1 = Val(dbVal$) : n2 = Val(valx$)
         If op$ = "=" And n1 = n2 Then isMatch = 1
         If op$ = ">" And n1 > n2 Then isMatch = 1
         If op$ = "<" And n1 < n2 Then isMatch = 1
         If op$ = ">=" And n1 >= n2 Then isMatch = 1
         If op$ = "<=" And n1 <= n2 Then isMatch = 1
         If op$ = "<>" And n1 <> n2 Then isMatch = 1
    EndIf

    Evaluate_Primitive = isMatch
End Function

' --- LOGIC: RECURSIVE EVALUATOR ---
Function Check_Condition(e$)
    Local INTEGER i, pLevel, opPos, opType ' 1=&, 2=|
    Local STRING c$, L$, R$, inner$

    ' 1. Handle Outer Parentheses: (A&B) -> A&B
    If Left$(e$, 1) = "(" And Right$(e$, 1) = ")" Then
         pLevel = 0 : opPos = 0
         ' Check if they are truly balancing outer parens
         For i = 1 To Len(e$) - 1
             If Mid$(e$, i, 1) = "(" Then pLevel = pLevel + 1
             If Mid$(e$, i, 1) = ")" Then pLevel = pLevel - 1
             If pLevel = 0 Then opPos = 1 ' Closed early
         Next i
         If opPos = 0 Then
              inner$ = Mid$(e$, 2, Len(e$) - 2)
              Check_Condition = Check_Condition(inner$)
              Exit Function
         EndIf
    EndIf

    ' 2. Find Split Operator (Lowest Precedence: | then &)
    pLevel = 0 : opPos = 0 : opType = 0
    For i = 1 To Len(e$)
        c$ = Mid$(e$, i, 1)
        If c$ = "(" Then pLevel = pLevel + 1
        If c$ = ")" Then pLevel = pLevel - 1

        If pLevel = 0 Then
            ' Found a top-level operator
            If c$ = "|" Then
                opPos = i : opType = 2 : Exit For ' Split on OR immediately
            ElseIf c$ = "&" And opType = 0 Then
                opPos = i : opType = 1 ' Mark AND, but keep looking for OR
            EndIf
        EndIf
    Next i

    ' 3. Recursive Split
    If opPos > 0 Then
        L$ = Left$(e$, opPos - 1)
        R$ = Mid$(e$, opPos + 1)
        If opType = 2 Then ' OR logic
            If Check_Condition(L$) = 1 Then Check_Condition = 1 : Exit Function
            If Check_Condition(R$) = 1 Then Check_Condition = 1 : Exit Function
            Check_Condition = 0
        Else ' AND logic
            If Check_Condition(L$) = 0 Then Check_Condition = 0 : Exit Function
            If Check_Condition(R$) = 0 Then Check_Condition = 0 : Exit Function
            Check_Condition = 1
        EndIf
        Exit Function
    EndIf

    ' 4. Base Case: No operators left
    Check_Condition = Evaluate_Primitive(e$)
End Function

Sub Sort_Results
    Local INTEGER i, j, gap, n, isNumeric, fIdx
    Local STRING tempKey$
    Local INTEGER tempRec
    Local FLOAT v1, v2
    Local INTEGER swapNeeded

    n = Result_Count
    If n < 2 Then Exit Sub

    ' Determine if we are sorting Numbers or Strings
    fIdx = Get_Field_ID(Q_OrderFld$)
    isNumeric = 0
    ' FIX: Check if Type is 2 (Int) or 3 (Float)
    If DB_Fld_Type(fIdx) > 1 Then isNumeric = 1

    Print "Sorting " + Str$(n) + " records..."

    ' --- SHELL SORT ALGORITHM ---
    gap = n \ 2
    Do While gap > 0
        For i = gap + 1 To n
            tempKey$ = Result_List(i).Key
            tempRec = Result_List(i).RecNo
            j = i

            Do While j > gap
                swapNeeded = 0

                If isNumeric Then
                    v1 = Val(Result_List(j - gap).Key)
                    v2 = Val(tempKey$)
                    If Q_OrderDesc = 0 Then ' Ascending
                        If v1 > v2 Then swapNeeded = 1
                    Else ' Descending
                        If v1 < v2 Then swapNeeded = 1
                    EndIf
                Else
                    ' String Compare
                    If Q_OrderDesc = 0 Then ' Ascending
                        If Result_List(j - gap).Key > tempKey$ Then swapNeeded = 1
                    Else ' Descending
                        If Result_List(j - gap).Key < tempKey$ Then swapNeeded = 1
                    EndIf
                EndIf

                If swapNeeded Then
                    Result_List(j) = Result_List(j - gap)
                    j = j - gap
                Else
                    Exit Do
                EndIf
            Loop

            Result_List(j).Key = tempKey$
            Result_List(j).RecNo = tempRec
        Next i
        gap = gap \ 2
    Loop
End Sub

Function Find_First_GE(searchVal$, maxIdx)
    Local INTEGER low, high, mid

    low = 0
    high = maxIdx -1

    Do While low <= high
        mid = (low + high) \ 2

        ' FIX: Use the DB_Index structure you already defined
        If DB_Index(mid).Key >= searchVal$ Then
            high = mid - 1
        Else
            low = mid + 1
        EndIf
    Loop

    Find_First_GE = low
End Function

' Function to read the definition file and count records in the main table
' Usage: db_Record_Count = Init_Record_Count("us500")
Function Init_Record_Count()
    Local STRING dataFile$, dummy$
    Local INTEGER count

    dataFile$=DB_Tbl_File$(1)
    If Not MM.Info(Exists File dataFile$) Then
        Print "Error: Data file " + dataFile$ + " not found!"
        Init_Record_Count = 0
        Exit Function
    EndIf

    ' Use #8 (Utility Handle)
    Open dataFile$ For Input As #8
    count = 0
    Do While Not Eof(#8)
        Line Input #8, dummy$
        If Trim$(dummy$) <> "" Then count = count + 1
    Loop
    Close #8
    Init_Record_Count = count
End Function

' Pattern Matching Function (SQL Style)
' Returns 1 (True) if subject$ matches pattern$
' Supports: % (multiple chars) and _ (single char)
Function Like_Match(subject$, pattern$)
    Local INTEGER sIdx, pIdx, sLen, pLen
    Local INTEGER starIdx, sTmpIdx

    sIdx = 1 : pIdx = 1
    starIdx = 0 : sTmpIdx = 0
    sLen = Len(subject$) : pLen = Len(pattern$)

    Do While sIdx <= sLen
        ' Case 1: Characters match or pattern has underscore (_)
        If pIdx <= pLen And (Mid$(pattern$, pIdx, 1) = "?" Or UCase$(Mid$(pattern$, pIdx, 1)) = UCase$(Mid$(subject$, sIdx, 1))) Then
             sIdx = sIdx + 1
             pIdx = pIdx + 1

        ' Case 2: Pattern has Wildcard (%) - Record position and advance pattern
        ElseIf pIdx <= pLen And Mid$(pattern$, pIdx, 1) = "%" Then
             starIdx = pIdx
             sTmpIdx = sIdx
             pIdx = pIdx + 1

        ' Case 3: Mismatch, but we saw a % earlier - Backtrack/Retry
        ElseIf starIdx > 0 Then
             pIdx = starIdx + 1
             sTmpIdx = sTmpIdx + 1
             sIdx = sTmpIdx

        ' Case 4: Mismatch and no wildcard - Fail
        Else
             Like_Match = 0
             Exit Function
        EndIf
    Loop

    ' Check for trailing wildcards (e.g., pattern "Bob%")
    Do While pIdx <= pLen And Mid$(pattern$, pIdx, 1) = "%"
        pIdx = pIdx + 1
    Loop

    ' If we reached the end of the pattern, it's a match
    If pIdx > pLen Then Like_Match = 1 Else Like_Match = 0
End Function


' =============================================================
' DATABASE BUILDER TOOLS
' =============================================================

Sub Cmd_MakeDB(argStr$)
    Local STRING dbName$, fileList$, currentFile$, rawList$
    Local INTEGER spcPos, commaPos
    
    rawList$ = Trim$(argStr$)
    spcPos = Instr(rawList$, " ")
    
    If spcPos = 0 Then Print "Usage: make-db..." : Exit Sub
    
    dbName$ = Left$(rawList$, spcPos - 1)
    fileList$ = Trim$(Mid$(rawList$, spcPos + 1))
    
    Print "Creating Schema: " + dbName$ + ".def"
    ' Use #8 for Definition File
    Open dbName$ + ".def" For Output As #8
    Print #8, "# Database Definition for " + dbName$
    Print #8, "# Generated by make-db on " + Date$
    
    Do
        commaPos = Instr(fileList$, ",")
        If commaPos > 0 Then
            currentFile$ = Trim$(Left$(fileList$, commaPos - 1))
            fileList$ = Mid$(fileList$, commaPos + 1)
        Else
            currentFile$ = Trim$(fileList$)
            fileList$ = ""
        EndIf
        ' Pass #8 as the open definition file handle
        If currentFile$ <> "" Then Convert_CSV_To_DB(currentFile$, dbName$, 8)
    Loop Until fileList$ = ""
    
    Close #8
    Print "Done. Load '" + dbName$ + "'"
End Sub

Sub Convert_CSV_To_DB(csvFile$, dbName$, defNum)
    ' Uses #9 (CSV In) and #10 (DAT Out)
    Local INTEGER i, fields, rowCount, n
    Local STRING line$, tblName$, datFile$, token$
    Local STRING headers$(50), rowVals$(50)
    Local INTEGER colWidth(50), colType(50), colStart, totalLen
    Const T_STR = 1, T_INT = 2, T_FLOAT = 3
    
    Print "Processing " + csvFile$ + "..."
    If Not MM.Info(EXISTS FILE csvFile$) Then Print "Error: " + csvFile$ : Exit Sub
    
    tblName$ = Left$(csvFile$, Instr(csvFile$, ".") - 1)
    datFile$ = tblName$ + ".dat"
    
    ' PASS 1: ANALYSIS (#9)
    Open csvFile$ For Input As #9
    Line Input #9, line$
    fields = Parse_CSV_Line%(line$, headers$())
    
    For i = 1 To fields : colWidth(i) = Len(headers$(i)) : colType(i) = T_INT : Next i
    
    rowCount = 0
    Do While Not Eof(#9)
        Line Input #9, line$
        If line$ = "" Then Continue Do
        n = Parse_CSV_Line%(line$, rowVals$())
        For i = 1 To fields
            If Len(rowVals$(i)) > colWidth(i) Then colWidth(i) = Len(rowVals$(i))
            If colType(i) <> T_STR Then
                If Is_Number(rowVals$(i)) = 0 Then
                    colType(i) = T_STR
                ElseIf colType(i) = T_INT And Instr(rowVals$(i), ".") > 0 Then
                    colType(i) = T_FLOAT
                EndIf
            EndIf
        Next i
        rowCount = rowCount + 1
        If rowCount Mod 100 = 0 Then Print ".";
    Loop
    Close #9
    Print " Analyzed " + Str$(rowCount) + " rows."
    
    ' WRITE DEF to #8 (passed as defNum)
    totalLen = 0
    For i = 1 To fields : totalLen = totalLen + colWidth(i) : Next i
    totalLen = totalLen + 2
    Print #defNum, "TABLE," + tblName$ + "," + datFile$ + "," + Str$(totalLen)
    colStart = 1
    For i = 1 To fields
        Print #defNum, "FIELD," + tblName$ + "," + headers$(i) + "," + Str$(colType(i)) + "," + Str$(colStart) + "," + Str$(colWidth(i))
        colStart = colStart + colWidth(i)
    Next i
    
    ' PASS 2: CONVERSION (#9 In, #10 Out)
    Open csvFile$ For Input As #9
    Open datFile$ For Output As #10
    Line Input #9, line$ ' Skip Header
    Do While Not Eof(#9)
        Line Input #9, line$
        If line$ = "" Then Continue Do
        n = Parse_CSV_Line%(line$, rowVals$())
        For i = 1 To fields
            token$ = rowVals$(i)
            If Len(token$) > colWidth(i) Then token$ = Left$(token$, colWidth(i))
            Print #10, token$ + Space$(colWidth(i) - Len(token$));
        Next i
        Print #10, ""
    Loop
    Close #9 : Close #10
    Print " Created " + datFile$
End Sub

' Parses a CSV line, handling quotes correctly.
' Uses % suffix to define return type as Integer explicitly
Function Parse_CSV_Line%(l$, parsed$())
    Local INTEGER i, inQuote, fIdx
    Local STRING c$, token$
    
    fIdx = 1
    token$ = ""
    inQuote = 0
    
    For i = 1 To Len(l$)
        c$ = Mid$(l$, i, 1)
        
        If c$ = Chr$(34) Then
            inQuote = Not inQuote ' Toggle Quote
        ElseIf c$ = "," And inQuote = 0 Then
            ' Field Separator
            parsed$(fIdx) = token$
            fIdx = fIdx + 1
            token$ = ""
        Else
            token$ = token$ + c$
        EndIf
    Next i
    
    ' Add last field
    parsed$(fIdx) = token$
    Parse_CSV_Line% = fIdx
End Function

' Simple check if string is numeric
Function Is_Number(s$)
    Local Float v
    Local Integer i
    ' MMBasic's Val() is permissive, but let's trust it for now.
    ' A robust check would scan for non-digit chars.
    ' For this import, assuming non-empty Val means number is risky for "123 Main St"
    ' So we scan for digits.
    If s$ = "" Or UCase$(s$) = "NULL" Then Is_Number = 1 : Exit Function ' Treat NULL as 0/Empty
    
    For i = 1 To Len(s$)
        If Instr("0123456789.-", Mid$(s$, i, 1)) = 0 Then 
            Is_Number = 0 : Exit Function
        EndIf
    Next i
    Is_Number = 1
End Function

Function Parse_SQL$(rawSql$)
    Local STRING s$, native$, verb$, fields$, table$, cond$, order$, limit$
    Local INTEGER i, pWhere, pOrder, pLimit, pFrom
    
    s$ = Trim$(rawSql$)
    If Right$(s$, 1) = ";" Then s$ = Left$(s$, Len(s$) - 1)
    
    ' Pre-clean tight parentheses so " WHERE " finder works
    s$ = Replace_Str$(s$, "WHERE(", "WHERE (")
    
    ' --- CASE 1: DELETE / UNDELETE ---
    ' SQL: DELETE FROM table WHERE condition
    ' Native: condition DELETE FROM table
    If UCase$(Left$(s$, 7)) = "DELETE " Or UCase$(Left$(s$, 9)) = "UNDELETE " Then
        i = Instr(UCase$(s$), " FROM ")
        If i = 0 Then Print "Error: Missing FROM" : Exit Function
        
        ' Extract Action and Table
        verb$ = Trim$(Left$(s$, i - 1)) ' DELETE or UNDELETE
        s$ = Mid$(s$, i + 6)            ' table WHERE...
        
        pWhere = Instr(UCase$(s$), " WHERE ")
        If pWhere > 0 Then
            table$ = Trim$(Left$(s$, pWhere - 1))
            cond$ = Trim$(Mid$(s$, pWhere + 7))
            cond$ = Translate_SQL_Condition$(cond$)
            
            ' Result: filter VERB FROM table
            Parse_SQL$ = cond$ + " " + verb$ + " FROM " + table$
            Exit Function
        EndIf
    EndIf
    
    ' --- CASE 2: SELECT ---
    ' SQL: SELECT [fields] FROM [table] [WHERE...] [ORDER...] [LIMIT...]
    ' Native: [filter] [ORDER...] [SHOW fields] [LIMIT...]
    If UCase$(Left$(s$, 7)) = "SELECT " Then
        s$ = Mid$(s$, 8) ' Strip SELECT
        
        ' 1. Extract LIMIT (Right to Left)
        pLimit = Instr(UCase$(s$), " LIMIT ")
        If pLimit > 0 Then
            limit$ = " LIMIT " + Trim$(Mid$(s$, pLimit + 7))
            s$ = Left$(s$, pLimit - 1)
        EndIf
        
        ' 2. Extract ORDER BY
        pOrder = Instr(UCase$(s$), " ORDER BY ")
        If pOrder > 0 Then
            order$ = " ORDER BY " + Trim$(Mid$(s$, pOrder + 10))
            s$ = Left$(s$, pOrder - 1)
        EndIf
        
        ' 3. Extract WHERE
        pWhere = Instr(UCase$(s$), " WHERE ")
        If pWhere > 0 Then
            cond$ = Trim$(Mid$(s$, pWhere + 7))
            cond$ = Translate_SQL_Condition$(cond$)
            s$ = Left$(s$, pWhere - 1)
        EndIf
        
        ' 4. Extract FROM (Split fields and table)
        pFrom = Instr(UCase$(s$), " FROM ")
        If pFrom = 0 Then Print "Error: Missing FROM" : Exit Function
        
        fields$ = Trim$(Left$(s$, pFrom - 1))
        
        ' 5. Construct Native Command
        ' CHECK: Is it an Aggregate? (SELECT COUNT(f) ...)
        If Instr(fields$, "(") > 0 Then
            ' Parse "COUNT(fld)" -> "COUNT fld"
            i = Instr(fields$, "(")
            verb$ = Trim$(Left$(fields$, i - 1))
            fields$ = Mid$(fields$, i + 1)
            i = Instr(fields$, ")")
            If i > 0 Then fields$ = Left$(fields$, i - 1)
            
            ' Native Aggregate: VERB field [WHERE cond]
            native$ = verb$ + " " + fields$
            If cond$ <> "" Then native$ = native$ + " WHERE " + cond$
        Else
            ' Standard Select
            ' Native order: Filter -> Order -> Show -> Limit
            native$ = cond$
            If order$ <> "" Then native$ = native$ + " " + order$
            If fields$ <> "*" Then native$ = native$ + " SHOW " + fields$
            If limit$ <> "" Then native$ = native$ + " " + limit$
        EndIf
        
        Parse_SQL$ = native$
    EndIf
End Function

' Helper: Converts SQL Syntax (AND/OR/'Val') to Native (&/|/"Val")
Function Translate_SQL_Condition$(sqlWhere$)
    Local STRING s$, c$, outx$
    Local INTEGER i, inQuote
    
    s$ = sqlWhere$
    
    ' 1. Map Operators (Replace SQL keywords with Native operators)
    s$ = Replace_Str$(s$, " AND ", " & ")
    s$ = Replace_Str$(s$, " OR ", " | ")
    
    ' 2. Handle Quotes (Convert ' to ")
    outx$ = ""
    inQuote = 0
    For i = 1 To Len(s$)
        c$ = Mid$(s$, i, 1)
        If c$ = "'" Then 
            outx$ = outx$ + Chr$(34) ' Convert ' to "
        Else
            outx$ = outx$ + c$
        EndIf
    Next i
    
    ' 3. HAND OFF TO NATIVE NORMALIZER
    ' This handles the stripping of spaces (e.g. city = "Abilene" -> city="Abilene")
    Translate_SQL_Condition$ = Normalize_Filter$(outx$)
End Function

' Helper: Simple Replace function (if your version doesn't have one)
Function Replace_Str$(src$, find$, rep$)
    Local INTEGER p
    Local STRING temp$
    temp$ = src$
    p = Instr(UCase$(temp$), find$)
    Do While p > 0
        temp$ = Left$(temp$, p - 1) + rep$ + Mid$(temp$, p + Len(find$))
        p = Instr(UCase$(temp$), find$, p + Len(rep$))
    Loop
    Replace_Str$ = temp$
End Function

Sub Show_Help
    Local STRING hLine$
    If Not MM.Info(EXISTS FILE "native.hlp") Then
        Print "Error: native.hlp not found."
        Exit Sub
    EndIf
    
    Open "native.hlp" For Input As #8
    Do While Not Eof(#8)
        Line Input #8, hLine$
        Print hLine$
    Loop
    Close #8
End Sub

Sub Show_Help_SQL
    Local STRING hLine$
    If Not MM.Info(EXISTS FILE "sql.hlp") Then
        Print "Error: sql.hlp not found."
        Exit Sub
    EndIf
    
    Open "sql.hlp" For Input As #8
    Do While Not Eof(#8)
        Line Input #8, hLine$
        Print hLine$
    Loop
    Close #8
End Sub

