/*******************************************************************************
 *
 * Driver for ILI9163 Display written as CFunctions
 *
 * (c) Peter Mather 2015 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1963_V44
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *
 * V1.0     2015-09-23 Peter Mather
 * 
 ******************************************************************************/
#include <stdarg.h>

#define Version 100     //Version 1.00
#define _SUPPRESS_PLIB_WARNING                                      // required for XC1.33  Later compiler versions will need PLIB to be installed
#include <plib.h>                                                   // the pre Harmony peripheral libraries


#include "../cfunctions.h"

#define ILI9163_NOP             0x00//Non operation
#define ILI9163_SWRESET 	0x01//Soft Reset
#define ILI9163_SLPIN   	0x10//Sleep ON
#define ILI9163_SLPOUT  	0x11//Sleep OFF
#define ILI9163_PTLON   	0x12//Partial Mode ON
#define ILI9163_NORML   	0x13//Normal Display ON
#define ILI9163_DINVOF  	0x20//Display Inversion OFF
#define ILI9163_DINVON   	0x21//Display Inversion ON
#define ILI9163_GAMMASET 	0x26//Gamma Set (0x01[1],0x02[2],0x04[3],0x08[4])
#define ILI9163_DISPOFF 	0x28//Display OFF
#define ILI9163_DISPON  	0x29//Display ON
#define ILI9163_IDLEON  	0x39//Idle Mode ON
#define ILI9163_IDLEOF  	0x38//Idle Mode OFF
#define ILI9163_CLMADRS   	0x2A//Column Address Set
#define ILI9163_PGEADRS   	0x2B//Page Address Set
#define ILI9163_RAMWR   	0x2C//Memory Write
#define ILI9163_RAMRD   	0x2E//Memory Read
#define ILI9163_CLRSPACE   	0x2D//Color Space : 4K/65K/262K
#define ILI9163_PARTAREA	0x30//Partial Area
#define ILI9163_VSCLLDEF	0x33//Vertical Scroll Definition
#define ILI9163_TEFXLON		0x35//Tearing Effect Line ON
#define ILI9163_TEFXLOF		0x34//Tearing Effect Line OFF
#define ILI9163_MADCTL  	0x36//Memory Access Control
#define ILI9163_VSSTADRS	0x37//Vertical Scrolling Start address
#define ILI9163_PIXFMT  	0x3A//Interface Pixel Format
#define ILI9163_FRMCTR1 	0xB1//Frame Rate Control (In normal mode/Full colors)
#define ILI9163_FRMCTR2 	0xB2//Frame Rate Control(In Idle mode/8-colors)
#define ILI9163_FRMCTR3 	0xB3//Frame Rate Control(In Partial mode/full colors)
#define ILI9163_DINVCTR		0xB4//Display Inversion Control
#define ILI9163_RGBBLK		0xB5//RGB Interface Blanking Porch setting
#define ILI9163_DFUNCTR 	0xB6//Display Fuction set 5
#define ILI9163_SDRVDIR 	0xB7//Source Driver Direction Control
#define ILI9163_GDRVDIR 	0xB8//Gate Driver Direction Control
#define ILI9163_PWCTR1  	0xC0//Power_Control1
#define ILI9163_PWCTR2  	0xC1//Power_Control2
#define ILI9163_PWCTR3  	0xC2//Power_Control3
#define ILI9163_PWCTR4  	0xC3//Power_Control4
#define ILI9163_PWCTR5  	0xC4//Power_Control5
#define ILI9163_VCOMCTR1  	0xC5//VCOM_Control 1
#define ILI9163_VCOMCTR2  	0xC6//VCOM_Control 2
#define ILI9163_VCOMOFFS  	0xC7//VCOM Offset Control
#define ILI9163_PGAMMAC		0xE0//Positive Gamma Correction Setting
#define ILI9163_NGAMMAC		0xE1//Negative Gamma Correction Setting
#define ILI9163_GAMRSEL		0xF2//GAM_R_SEL
#define ILI9163_Portrait        0b00001000
#define ILI9163_Portrait180     0b11001000
#define ILI9163_Landscape       0b01101000
#define ILI9163_Landscape180    0b10101000

#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4

// SPI pin numbers and registers
#define SPI_INP_PIN         (HAS_44PINS ? 41 : 14)
#define SPI_OUT_PIN         (HAS_44PINS ? 20 :  3)
#define SPI_CLK_PIN         (HAS_44PINS ? 14 : 25)
#define SPI_PPS_OPEN        PPSInput(2, SDI1, RPB5); PPSOutput(2, RPA1, SDO1)
#define SPI_PPS_CLOSE       PPSOutput(2, RPA1, NULL)
#define SPICON *(volatile unsigned int *)(0xbf805800)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805810)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805820)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805830)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805840)              //SPI status register


#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}
// set the chip select for the SPI to low (enabled)
// if the SPI is currently set to a different mode or baudrate this will change it accordingly
// also, it checks if the chip select pin needs to be changed
// set the chip select for SPI1 to high (disabled)
void spi_write_data(unsigned char data){
    PinSetBit(Option->LCD_CD, LATSET);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_command(unsigned char data){
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_cd(unsigned char command, int data, ...){
   int i;
   va_list ap;
   va_start(ap, data);
   spi_write_command(command);
   for(i = 0; i < data; i++) spi_write_data((char)va_arg(ap, int));
   va_end(ap);
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to SSD1963
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void DefineRegion(int xstart, int ystart, int xend, int yend) {
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(ILI9163_CLMADRS);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(xstart >> 8);
    SPIsend(xstart);
    SPIsend(xend >> 8);
    SPIsend(xend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9163_PGEADRS);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(ystart >> 8);
    SPIsend(ystart);
    SPIsend(yend >> 8);
    SPIsend(yend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9163_RAMWR);
    PinSetBit(Option->LCD_CS, LATSET);
}


//Print the bitmap of a char on the video output
//    x, y - the top left of the char
//    width, height - size of the char's bitmap
//    scale - how much to scale the bitmap
//	  fc, bc - foreground and background colour
//    bitmap - pointer to the butmap
void DrawBitmapSPI(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap){
    int i, j, k, m;
    unsigned char fhb, flb, bhb, blb;
    unsigned int consave=0,brgsave=0,con2save;
    int vertCoord, horizCoord, XStart, XEnd, YEnd;

    // adjust when part of the bitmap is outside the displayable coordinates
    vertCoord = y1; if(y1 < 0) y1 = 0;                                 // the y coord is above the top of the screen
    XStart = x1; if(XStart < 0) XStart = 0;                            // the x coord is to the left of the left marginn
    XEnd = x1 + (width * scale) - 1; if(XEnd >= HRes) XEnd = HRes - 1; // the width of the bitmap will extend beyond the right margin
    YEnd = y1 + (height * scale) - 1; if(YEnd >= VRes) YEnd = VRes - 1;// the height of the bitmap will extend beyond the bottom margin
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x8060;
    SPIBRG=1;
    SPICON2=0xC00;
     // convert the colours to 565 format
    fhb = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111);
    flb = ((fc >>  5) & 0b11100000) | ((fc >>  3) & 0b00011111);
    bhb = ((bc >> 16) & 0b11111000) | ((bc >> 13) & 0b00000111);
    blb = ((bc >>  5) & 0b11100000) | ((bc >>  3) & 0b00011111);

    DefineRegion(x1, y1, x1 + (width * scale) - 1, y1 + (height * scale) -1);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            if(vertCoord++ < 0) continue;                           // we are above the top of the screen
            if(vertCoord > VRes) return;                            // we have extended beyond the bottom of the screen
            horizCoord = x1;
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(horizCoord++ < 0) continue;                  // we have not reached the left margin
                    if(horizCoord > HRes) continue;                 // we are beyond the right margin
                    if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                        SPIsend(fhb); SPIsend(flb);
                    } else {
                        SPIsend(bhb); SPIsend(blb);
                    }
                }
            }
        }
    }
    PinSetBit(Option->LCD_CS, LATSET);
    SPIBRG=brgsave;  //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
}

// Draw a rectangle
// this is the basic drawing primitive used by most drawing routines
//    x1, y1, x2, y2 - the coordinates
//    c - the colour
void DrawRectangleSPI(int x1, int y1, int x2, int y2, int c){
    unsigned int consave=0,brgsave=0,con2save;
	int i, t;
    unsigned char hb, lb;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0x8060;
    SPIBRG=1;
    SPICON2=0xC00;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;

    // convert the colours to 565 format
    hb = ((c >> 16) & 0b11111000) | ((c >> 13) & 0b00000111);
    lb = ((c >> 5) & 0b11100000) | ((c >> 3) & 0b00011111);

    DefineRegion(x1, y1, x2, y2);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
	i = x2 - x1 + 1;
	i *= (y2 - y1 + 1);
    while(i--){
        SPIsend(hb);
        SPIsend(lb);
     }
    PinSetBit(Option->LCD_CS, LATSET);
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;

}

__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}

/*******************************************************************************
 *
 * ILI9163 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * ILI9163 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangleSPI and
 * DrawBitmapSPI respectively
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *                                        long long *size)
 * 
 ******************************************************************************/
//CFunction Driver_ILI9163
void main(long long *CD, long long *RST, long long *CS,long long *orientation){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    int HorizontalRes=128,VerticalRes=128;
    unsigned int consave=0,brgsave=0,con2save;
    Option->LCD_Reset=*RST;
    Option->LCD_CD=*CD;
    Option->LCD_CS=*CS;
    Option->DISPLAY_ORIENTATION=*orientation;
 
    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);
    ExtCfg(Option->LCD_CD,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CD,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CD, LATSET);
    ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CS, LATSET);
    if(ExtCurrentConfig[SPI_OUT_PIN] == EXT_RESERVED) { //already open
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }
    ExtCfg(SPI_OUT_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_OUT_PIN, EXT_BOOT_RESERVED, 0);
    ExtCfg(SPI_INP_PIN, EXT_DIG_IN, 0); ExtCfg(SPI_INP_PIN, EXT_BOOT_RESERVED, 0);
    ExtCfg(SPI_CLK_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_CLK_PIN, EXT_BOOT_RESERVED, 0);
    SPI_PPS_OPEN; 
    SPICON=0x8060;
    SPIBRG=1;
    SPICON2=0xC00;// this is defined in IOPorts.h
    if(!brgsave){ //save my settings
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }    


  //Reset the SSD1963
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);

	spi_write_command(ILI9163_SLPOUT);                      //exit sleep
  	uSec(5000);
  	spi_write_cd(ILI9163_PIXFMT,1,0x05);
  	uSec(5000);
  	spi_write_cd(ILI9163_GAMMASET,1,0x04);                  //0x04
  	uSec(1000);
  	spi_write_cd(ILI9163_GAMRSEL,1,0x01);
  	uSec(1000);
  	spi_write_command(ILI9163_NORML);
  	spi_write_cd(ILI9163_DFUNCTR,2,0b11111111,0b00000110);  //
  	spi_write_cd(ILI9163_PGAMMAC,15,0x36,0x29,0x12,0x22,0x1C,0x15,0x42,0xB7,0x2F,0x13,0x12,0x0A,0x11,0x0B,0x06);//Positive Gamma Correction Setting
  	spi_write_cd(ILI9163_NGAMMAC,15,0x09,0x16,0x2D,0x0D,0x13,0x15,0x40,0x48,0x53,0x0C,0x1D,0x25,0x2E,0x34,0x39);//Negative Gamma Correction Setting
  	spi_write_cd(ILI9163_FRMCTR1,2,0x08,0x02);              //0x0C//0x08
 	uSec(1000);
  	spi_write_cd(ILI9163_DINVCTR,1,0x07);
  	uSec(1000);
  	spi_write_cd(ILI9163_PWCTR1,2,0x0A,0x02);               //4.30 - 0x0A
  	uSec(1000);
  	spi_write_cd(ILI9163_PWCTR2,1,0x02);
  	uSec(1000);
  	spi_write_cd(ILI9163_VCOMCTR1,2,0x50,99);               //0x50
  	uSec(1000);
  	spi_write_cd(ILI9163_VCOMOFFS,1,0);                     //0x40
  	uSec(1000);
  	spi_write_cd(ILI9163_VSCLLDEF,5,0,0,VerticalRes,0,0);
 	spi_write_command(ILI9163_DISPON);                      //display ON
  	uSec(1000);
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE)     spi_write_cd(ILI9163_MADCTL,1,ILI9163_Landscape);
    if(Option->DISPLAY_ORIENTATION==PORTRAIT)      spi_write_cd(ILI9163_MADCTL,1,ILI9163_Portrait); 
    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE)    spi_write_cd(ILI9163_MADCTL,1,ILI9163_Landscape180);
    if(Option->DISPLAY_ORIENTATION==RPORTRAIT)     spi_write_cd(ILI9163_MADCTL,1,ILI9163_Portrait180);

    if(Option->DISPLAY_ORIENTATION&1){
        VRes=HorizontalRes;
        HRes=VerticalRes;
    } else {
        HRes=HorizontalRes;
        VRes=VerticalRes;
    }

    //Set the DrawRectangle vector to point to our function
    DrawRectangleVector= (unsigned int)&DrawRectangleSPI + libAddr;

    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=(unsigned int)&DrawBitmapSPI + libAddr;

    //CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    static const char startup[]="ILI9163 driver loaded\r\n";
    pstring(startup);
}
