/*******************************************************************************************
 *
 *  API Definitions used when calling MMBasic Interpreter API Functions from CFunctions
 *
 *  The MMBasic CFunction Tutorial, MMBasic API Description, associated tool(s), this file,
 *  and example source code may be downloaded from http://www.mmbasic.org/
 *
 *  Documentation specific to the MMBasic Interpreter API may be found at
 *  http://www.mmbasic.org/calltable
 *
 *  This file may be downloaded from http://www.mmbasic.org/downloads/APIDefs.h
 *
 *  Copyright 2015 Peter Carnegie, Geoff Graham, Peter Mather
 *
 *  Version 2.7.18  2015-07-17 Peter Carnegie
 *      1)  Added macro to hook DrawRectangle vector
 *      2)  Added macro to hook DrawBitmap vector
 * 
 *  Version 2.7.17  2015-07-17 Peter Carnegie
 *      0)  Support 4.7 B23
 *      1)  Revert DrawLine to be just a pointer to the DrawLine function
 *      2)  Replaced DisplayMetrics with HRes, and VRes
 *
 *  Version 2.7.16  2015-07-16 Peter Carnegie
 *      1)  Added macro NBRPINS which returns the number of pins on the chip
 *
 *  Version 2.7.15 2015-07-08 Peter Carnegie
 *      1)  Changed DrawRectangle, DrawBitmap, and DrawLine to use a pointer to a
 *          pointer, ie the pointer to the function is now held in RAM
 *      2)  Added macro to enable CFunctions to access the MMBasic Options
 *      3)  Added macro to enable CFunctions to access the DisplayMetrics
 *
 ******************************************************************************************/

// PIC32 Fixed Address Mode
#if defined(__PIC32MX__)
    #define TableAddr (int*)0x9D000000
#else
    #error "This table is for the PIC32 version of MMBasic"
#endif

//Index into the API Table for each function

#define __CurrentCpuSpeedIDX    0       //CurrentCpuSpeed				http://www.mmbasic.org/calltable/CurrentCpuSpeed.html
#define __uSecIDX               1       //void uSec(unsigned int us)                    http://www.mmbasic.org/calltable/uSec.html
#define __putConsoleIDX         2       //void putConsole(int c)                        http://www.mmbasic.org/calltable/putConsole.html
#define __getConsoleIDX         3       //int getConsole(void)                          http://www.mmbasic.org/calltable/getConsole.html
#define __ExtCfgIDX             4       //void ExtCfg(int pin, int cfg, int option)	http://www.mmbasic.org/calltable/ExtCfg.html
#define __ExtSetIDX             5       //void ExtSet(int pin, int val)                 http://www.mmbasic.org/calltable/ExtSet.html
#define __ExtInpIDX             6       //int ExtInp(int pin)                           http://www.mmbasic.org/calltable/ExtInp.html
#define __PinSetBitIDX          7       //void PinSetBit(int pin, unsigned int offset)  http://www.mmbasic.org/calltable/PinSetBit.html
#define __PinReadIDX            8       //int PinRead(int pin)                          http://www.mmbasic.org/calltable/PinRead.html
#define __GetPortAddrIDX        9       //unsigned int *GetPortAddr(int pin, unsigned int offset)   http://www.mmbasic.org/calltable/GetPortAddr.html
#define __GetPinBitIDX          10      //int GetPinBit(int pin)                        http://www.mmbasic.org/calltable/GetPinBit.html
#define __MMPrintStringIDX      11      //void MMPrintString(char* s)                   http://www.mmbasic.org/calltable/MMPrintString.html
#define __IntToStrIDX           12      //void IntToStr(char *strr, long long int nbr, unsigned int base)   http://www.mmbasic.org/calltable/IntToStr.html
#define __FloatToStrIDX         13      //void FloatToStr(char *p, float f, int m, int n, char ch)  http://www.mmbasic.org/calltable/FloatToStr.html
#define __CheckAbortIDX         14      //void CheckAbort(void)                         http://www.mmbasic.org/calltable/CheckAbort.html
#define __GetMemoryIDX          15      //void *GetMemory(size_t msize);                http://www.mmbasic.org/calltable/GetMemory.html
#define __GetTempMemoryIDX      16      //void *GetTempMemory(int NbrBytes)             http://www.mmbasic.org/calltable/GetTempMemory.html
#define __FreeMemoryIDX         17      //void FreeMemory(void *addr)                   http://www.mmbasic.org/calltable/FreeMemory.html
#define __DrawRectangleIDX      18      //void DrawRectangle(int x1, int y1, int x2, int y2, int c) http://www.mmbasic.org/calltable/DrawRectangle.html
#define __DrawBitmapIDX         19      //void DrawBitmap(int x1, int y1, int width, int height, int scale, int fg, int bg, unsigned char *bitmap ) http://www.mmbasic.org/calltable/DrawBitMap.html
#define __DrawLineIDX           20      //void DrawLine(int x1, int y1, int x2, int y2, int w, int c)  http://www.mmbasic.org/calltable/DrawLine.html
#define __FontTableIDX          21      //const unsigned char *FontTable[FONT_NBR]      http://www.mmbasic.org/calltable/FontTable.html
#define __FMulIDX               22      //float FMul(float a, float b)			http://www.mmbasic.org/calltable/FMul.html 
#define __FAddIDX               23      //float FAdd(float a, float b)                  http://www.mmbasic.org/calltable/FAdd.html
#define __FSubIDX               24      //float FSub(float a, float b)                  http://www.mmbasic.org/calltable/FSub.html
#define __FDivIDX               25      //float FDiv(float a, float b)                  http://www.mmbasic.org/calltable/FDiv.html
#define __FCmpIDX               26      //int   FCmp(float a,float b)                   http://www.mmbasic.org/calltable/FCmp.html
#define __FSinIDX               27      //float sinf(float)                             http://www.mmbasic.org/calltable/sinfExample.html
#define __FLogIDX               28      //float logf(float)                             http://www.mmbasic.org/calltable/logfExample.html
#define __FPowIDX               29      //float powf(float, float)                      http://www.mmbasic.org/calltable/powfExample.html
#define __atanfIDX              30      //float atanf(float)                            http://www.mmbasic.org/calltable/atanfExample.html
#define __FloatToIntIDX         31      //long long int FloatToInt(float)               http://www.mmbasic.org/calltable/FloatToInt.html
#define __IntToFloatIDX         32      //float IntToFloat(long long int a)             http://www.mmbasic.org/calltable/IntToFloat.html
#define __CFuncmSecIDX          33      //CFuncmSec                                     http://www.mmbasic.org/calltable/CFuncmSec.html
#define __ExtCurrentConfigIDX   34      //int ExtCurrentConfig[NBR_PINS_44CHIP + 1];    http://www.mmbasic.org/calltable/ExtCurrentCfg.html
#define __StartOfCFuncRAMIDX    35      //StartOfCFuncRAM                               http://www.mmbasic.org/calltable/StartOfCFuncRam.html

#define __OptionIDX             36      //Option
//#define __DisplayMetricsIDX     37
#define __HResIDX               37      //HRes
#define __VResIDX               38      //VRes



//The macros for each function.  Refer to http://www.mmbasic.org/calltable for documentation of each function
//and/or the MMBasic source code for a definition of the function and arguments

#define CurrentCpuSpeed                 (*(unsigned int *)(*(TableAddr + __CurrentCpuSpeedIDX)))
#define uSec(a) 			((void (*)(unsigned int )) (*(TableAddr + __uSecIDX))) (a)
#define putConsole(a) 			((void (*)(char)) (*(TableAddr + __putConsoleIDX))) (a)
#define getConsole()  			((int (*)(void)) (*(TableAddr + __getConsoleIDX))) ()
#define ExtCfg(a,b,c)			((void (*)(int, int, int)) (*(TableAddr + __ExtCfgIDX))) (a,b,c)
#define ExtSet(a,b)			((void(*)(int, int)) (*(TableAddr + __ExtSetIDX))) (a,b)
#define ExtInp(a)			((int(*)(int)) (*(TableAddr + __ExtInpIDX))) (a)
#define PinSetBit(a,b)			((void(*)(int, int)) (*(TableAddr + __PinSetBitIDX))) (a,b)
#define PinRead(a)			((int(*)(int)) (*(TableAddr + __PinReadIDX))) (a)
#define GetPortAddr(a,b) 		((volatile unsigned int * (*)(int,int)) (*(TableAddr + __GetPortAddrIDX))) (a,b)
#define GetPinBit(a) 			((int (*)(int)) (*(TableAddr + __GetPinBitIDX))) (a)
#define MMPrintString(a) 		((void (*)(char*)) (*(TableAddr + __MMPrintStringIDX))) (a)
#define IntToStr(a,b,c) 		((void (*)(char *, long long int, unsigned int)) (*(TableAddr + __IntToStrIDX))) (a,b,c)
#define FloatToStr(a,b,c,d,e)           ((void (*)(char *, float, int, int, char)) (*(TableAddr + __FloatToStrIDX))) (a,b,c,d,e)
#define CheckAbort() 			((void (*)(void)) (*(TableAddr + __CheckAbortIDX))) ()
#define GetMemory(a) 			((void* (*)(int)) (*(TableAddr + __GetMemoryIDX))) (a)
#define GetTempMemory(a) 		((void* (*)(int)) (*(TableAddr + __GetTempMemoryIDX))) (a)
#define FreeMemory(a) 			((void (*)(void *)) (*(TableAddr + __FreeMemoryIDX))) (a)
//#define DrawRectangle(a,b,c,d,e)      ((void (*)(int,int,int,int,int)) (*(TableAddr + __DrawRectangleIDX))) (a,b,c,d,e)
#define DrawRectangle(a,b,c,d,e)        ((void (*)(int,int,int,int,int)) (*(unsigned int *)*(unsigned int *)(TableAddr + __DrawRectangleIDX))) (a,b,c,d,e)
#define DrawRectangleVector             ((*(unsigned int *)*(unsigned int *)(TableAddr + __DrawRectangleIDX)))

//#define DrawBitmap(a,b,c,d,e,f,g,h) 	((void (*)(int,int,int,int,int,int,int, char*)) (*(TableAddr + __DrawBitmapIDX))) (a,b,c,d,e,f,g,h)
#define DrawBitmap(a,b,c,d,e,f,g,h) 	((void (*)(int,int,int,int,int,int,int, char*)) (*(unsigned int *)*(unsigned int *)(TableAddr + __DrawBitmapIDX))) (a,b,c,d,e,f,g,h)
#define DrawBitmapVector                ((*(unsigned int *)*(unsigned int *)(TableAddr + __DrawBitmapIDX)))

#define DrawLine(a,b,c,d,e,f)           ((void (*)(int,int,int,int,int,int)) (*(TableAddr + __DrawLineIDX))) (a,b,c,d,e,f)
//#define DrawLine(a,b,c,d,e,f)           ((void  (*)(int,int,int,int,int,int)) ((*(unsigned int *)*(unsigned int*)(TableAddr + __DrawLineIDX)))) (a,b,c,d,e,f)
#define FontTable                       (void*)((int*)((*(TableAddr + __FontTableIDX))))
#define FMul(a,b)                       ((float (*)(float, float)) (*(TableAddr + __FMulIDX))) (a,b)
#define FAdd(a,b)                       ((float (*)(float, float)) (*(TableAddr + __FAddIDX))) (a,b)
#define FSub(a,b)                       ((float (*)(float, float)) (*(TableAddr + __FSubIDX))) (a,b)
#define FDiv(a,b)                       ((float (*)(float, float)) (*(TableAddr + __FDivIDX))) (a,b)
#define FCmp(a,b)                       ((int (*)(float, float)) (*(TableAddr + __FCmpIDX))) (a,b)
#define FSin(a)                         ((float (*)(float))(*(TableAddr + __FSinIDX))) (a)
#define FLog(a)                         ((float (*)(float))(*(TableAddr + __FLogIDX))) (a)
#define FPow(a,b)                       ((float (*)(float, float))(*(TableAddr + __FPowIDX))) (a,b)
#define atanf(a)                        ((float (*)(float))(*(TableAddr + __atanfIDX))) (a)
#define FloatToInt(a)                   ((long long (*)(float)) (*(TableAddr + __FloatToIntIDX))) (a)
#define IntToFloat(a)                   ((float (*)(long long)) (*(TableAddr + __IntToFloatIDX))) (a)
#define CFuncmSec                       (*(unsigned int *)(*((TableAddr + __CFuncmSecIDX))))
#define ExtCurrentConfig                ((int *)((*(TableAddr + __ExtCurrentConfigIDX))))

//Returns the address of the pointer which stores the start of the Cfunction static ram area
#define StartOfCFuncRam                 (*(unsigned int *)(*((TableAddr + __StartOfCFuncRAMIDX))))

//Return a record from the Option structure
#define Option                          ({struct Option_s *__op; __op=(void *)(unsigned int)((*(TableAddr + __OptionIDX)));})

//Return a record from the DisplayMetrics structure
//#define DisplayMetrics                  ({struct DisplayMetrics_s *__dm; __dm=(void *)(unsigned int)((*(TableAddr + __DisplayMetricsIDX)));})

//Return Horizontal Resolution of TFT display
#define HRes                            (*(unsigned int *)(*(TableAddr + __HResIDX)))

//Return Vertical Resolution of TFT display
#define VRes                            (*(unsigned int *)(*(TableAddr + __VResIDX)))

//http://www.mmbasic.org/calltable/DrawPixel.html
#define DrawPixel(x, y, c)              DrawRectangle(x, y, x, y, c)

//http://www.mmbasic.org/calltable/PrintInt.html
#define PrintInt(a,b)                   ({char __PrintIntS[64];IntToStr(__PrintIntS,a,b);MMPrintString(__PrintIntS);})

//http://www.mmbasic.org/calltable/PrintFloat.html
#define PrintFloat(a,b,c,d)             ({char __PrintFloatS[c+b+1];FloatToStr(__PrintFloatS,a,b,c,d);MMPrintString(__PrintFloatS);})

//http://www.mmbasic.org/calltable/PrintCrLf.html
#define PrintCrLf                       ({putConsole('\r');putConsole('\n');})

//http://www.mmbasic.org/calltable/GetMemory.html
#define MALLOC(a)                       GetMemory(a)

//http://www.mmbasic.org/calltable/FreeMemory.html
#define FREE(a)                         FreeMemory(a)

//Some useful MACROs to make dealing with pointers a bit easier
//http://www.mmbasic.org/calltable/MiscMacros.html

//Cast a 64 bit Integer to a 32 bit pointer
#define INT64TOPTR(a)                   (void *)(unsigned int)(a)
//Cast a pointer to a 64 bit integer
#define PTRTOINT64(a)                   (long long)(unsigned int)(a)
//Cast an INT to a Pointer
#define INTTOPTR(a)                     (void *)(unsigned int)(a)
//Cast a pointer to an unsigned int
#define PTRTOINT(a)                    (unsigned int)(a)


//Mathematical Constants M_XXX
//usage : variable=constantname will assign the constant to the variable
//eg float p;p=M_Log2E will result in p containing the value 1.442695
//eg FloatToString(str,M_Log2E,3,6,10), ie there is no need to assign the constant to a variable

//http://www.mmbasic.org/calltable/M_ANY.html
#define M_ANY(a) ({ float __value;__asm__ volatile ("li $t0,%[constant]\r\n\t" "move %[result],$t0" :[result]"=r"(__value):[constant]"n"(a):"$8");__value;})

#define	M_E             M_ANY(0x402df854) /*2.7182818284590452354     e */
#define M_LOG2E         M_ANY(0x3fb8aa3b) /*1.4426950408889634074     log 2e */
#define M_LOG10E        M_ANY(0x3ede5bd9) /*0.43429448190325182765    log 10e */
#define	M_LN2           M_ANY(0x3f317218) /*0.69314718055994530942    log e2 */
#define	M_LN10          M_ANY(0x40135d8e) /*2.30258509299404568402    log e10 */
#define	M_PI            M_ANY(0x40490fdb) /*3.14159265358979323846    pi */
#define	M_PI_2          M_ANY(0x3fc90fdb) /*1.57079632679489661923    pi/2 */
#define	M_PI_4          M_ANY(0x3f490fdb) /*0.78539816339744830962    pi/4 */
#define	M_1_PI          M_ANY(0x3ea2f983) /*0.31830988618379067154    1/pi */
#define	M_2_PI          M_ANY(0x3f22f983) /*0.63661977236758134308    2/pi */
#define	M_2_SQRTPI      M_ANY(0x3f906ebb) /*1.12837916709551257390    2/sqrt(pi) */
#define	M_SQRT2         M_ANY(0x3fb504f3) /*1.41421356237309504880    sqrt(2) */
#define	M_SQRT1_2       M_ANY(0x3f3504f3) /*0.70710678118654752440    1/sqrt(2) */
#define M_RAD2DEG       M_ANY(0x42652ee1) /*57.2957795                180/Pi*/
#define M_DEG2RAD       M_ANY(0x3c8efa35) /*0.0174532925              Pi/180*/

//Useful numerical constants
#define M_0             M_ANY(0x00000000) /*0.00000                        */
#define M_EIGHTH        M_ANY(0x3e000000) /*0.125                     1/8) */
#define M_QUART         M_ANY(0x3e800000) /*0.25                      1/4) */
#define M_THIRD         M_ANY(0x3EAAAAAB) /*0.33                      1/3  */
#define M_HALF          M_ANY(0x3f000000) /*0.5                       1/2) */
#define M_3QUART        M_ANY(0x3f400000) /*0.75                      3/4/) */
#define M_1             M_ANY(0x3f800000) /*1.0                       1.0 */
#define M_2             M_ANY(0x40000000) /*2.0                       2.0 */
#define M_3             M_ANY(0x40400000) /*3.0                       3.0 */
#define M_4             M_ANY(0x40800000) /*4.0                       4.0 */
#define M_5             M_ANY(0x40a00000) /*5.0                       5.0 */
#define M_6             M_ANY(0x40C00000) /*6.0                       6.0 */
#define M_7             M_ANY(0x40e00000) /*7.0                       7.0 */
#define M_8             M_ANY(0x41000000) /*8.0                       8.0 */
#define M_9             M_ANY(0x41100000) /*9.0                       9.0 */
#define M_11            M_ANY(0x41300000) /*11.0                      11.0 */
#define M_TEN           M_ANY(0x41200000) /*10.0                      10.0 */
#define M_HUNDRED       M_ANY(0x42c80000) /*100.0                     100.0 */
#define M_THOUSAND      M_ANY(0x447a0000) /*1000.0                    1000.0 */
#define M_MILLION       M_ANY(0x49742400) /*1000000.0                 1000000.0 */

//Useful Trigonometric values
#define M_30Deg         M_ANY(0x41F00000)   /*30*/
#define M_45Deg         M_ANY(0x42340000)   /*45*/
#define M_60Deg         M_ANY(0x42700000)   /*60*/
#define M_90Deg         M_ANY(0x42B40000)   /*90*/

#define M_30Rad         M_ANY(0x3F060A96)   /*30*/
#define M_45Rad         M_ANY(0x3F490FD8)   /*45*/
#define M_60Rad         M_ANY(0x3F860AA6)   /*60*/
#define M_90Rad         M_ANY(0x3FC90FF9)   /*90*/

//Mathematical functions modeled after math.h
#define sinf(a)		FSin(a)                                                         //http://www.mmbasic.org/calltable/sinfExample.html
#define cosf(a)		({float __cosfvalue,__cosftemp=FSub(M_90Rad,a);__cosfvalue=sinf(__cosftemp);__cosfvalue;})  //http://www.mmbasic.org/calltable/cosfExample.html
#define tanf(a)		({float __tanfvalue=FDiv(sinf(a),cosf(a));__tanfvalue;})        //http://www.mmbasic.org/calltable/tanfExample.html
#define powf(a,b)	({float __powfvalue = FPow(a,b);__powfvalue;})                  //http://www.mmbasic.org/calltable/powfExample.html
#define sqrtf(a)	({float __sqrtfvalue=FPow(a,M_HALF);__sqrtfvalue;})             //http://www.mmbasic.org/calltable/sqrtfExample.html
#define FSqrt(a)        sqrtf(a)
#define cbrtf(a)	({float __cbrtfvalue=FPow(a,M_THIRD);__cbrtfvalue;})            //http://www.mmbasic.org/calltable/cbrtfExample.html

#define absf(a)         ({ float __absfvalue;__asm__ volatile("move $t0,%[varx]\r\n\t""li $t1,0x7fffffff\r\n\t" "and $t0,$t0,$t1\r\n\t" "move %[result],$t0" \
                            :[result]"=r"(__absfvalue):[varx]"r"(a):"$8","$9");__absfvalue;})  //http://www.mmbasic.org/calltable/absfExample.html

//a % n = a - (n * int(a/n))
#define fmodf(a,n)	({float __fmodfvalue;__fmodfvalue=FSub(a,FMul(IntToFloat(FloatToInt(FSub( FDiv(a,n), M_HALF) )),n));})  //http://www.mmbasic.org/calltable/fmodfExample.html

#define negf(a)         ({ float __negfvalue;__asm__ volatile("move $t0,%[varx]\r\n\t" "li $t1,0x80000000\r\n\t" "xor $t0,$t0,$t1\r\n\t" "move %[result],$t0"\
                        :[result]"=r"(__negfvalue):[varx]"r"(a):"$8","$9");__negfvalue;})  //http://www.mmbasic.org/calltable/negfExample.html

#define logf(a)		FLog(a)                                                         //http://www.mmbasic.org/calltable/logfExample.html
#define log2f(a)        FDiv(logf(a), M_ANY(0x3F317218))                                //http://www.mmbasic.org/calltable/log2fExample.html
#define log10f(a)       FDiv(logf(a), M_ANY(0x40135D8E))                                //http://www.mmbasic.org/calltable/log10fExample.html

//radians=degrees * (PI/180)
#define DEGTORAD(a)     FMul(a,M_ANY(0x3C8EFA35))                                       //http://www.mmbasic.org/calltable/DegToRadExample.html

//degrees=radians * (180/PI)
#define RADTODEG(a)     FMul(a,M_ANY(0x42652EE1))                                       //http://www.mmbasic.org/calltable/RadToDegExample.html


////Structure of DisplayMetrics
//struct DisplayMetrics_s{
//    int DisplayHRes;
//    int DisplayVRes;
//};

//The Option structure
struct Option_s {
    char Autorun;
    char Tab;
    char Invert;
    char Listcase;
    char Height;
    char Width;
    int  PIN;
    int  Baudrate;
    char DISPLAY_TYPE;
    char DISPLAY_ORIENTATION;
    char LCD_CD;
    char LCD_CS;
    char LCD_Reset;
    char TOUCH_CS;
    char TOUCH_IRQ;
    int  TOUCH_XZERO;
    int  TOUCH_YZERO;
    float TOUCH_XSCALE;
    float TOUCH_YSCALE;
};

#if defined(__PIC32MX__)

#define GetDevId (*(volatile unsigned int *)0xBF80F220)
#define PIC32MX470F512H_DEVID  0x0580A053
#define PIC32MX470F512L_DEVID  0x0580B053
#define PIC32MX170F256B_DEVID  0x06610053
#define PIC32MX270F256B_DEVID  0x06600053
#define PIC32MX170F256D_DEVID  0x0661A053
#define PIC32MX270F256D_DEVID  0x0660A053

//Return the number of pins on the chip or -1 if unknow chip
#define NBRPINS     ({int __NbrPins=-1;unsigned int DevIdSFR=(GetDevId) & 0xfffffff;\
                        if ((DevIdSFR==PIC32MX170F256B_DEVID) || (DevIdSFR==PIC32MX270F256B_DEVID))__NbrPins=28;if ((DevIdSFR==PIC32MX170F256D_DEVID) || (DevIdSFR==PIC32MX270F256D_DEVID))__NbrPins=44;\
                        if (DevIdSFR==PIC32MX470F512H_DEVID)__NbrPins=64;if (DevIdSFR==PIC32MX470F512L_DEVID)__NbrPins=100;__NbrPins;})

// Structure that defines the SFR, bit number and mode for each I/O pin
struct s_PinDef {
    volatile unsigned int *sfr;
    unsigned char bitnbr;
    unsigned char mode;
};

// Define the offsets from the PORT address
// these are used by GetPortAddr(a,b)
#define ANSEL               -8
#define ANSELCLR            -7
#define ANSELSET            -6
#define ANSELINV            -5
#define TRIS                -4
#define TRISCLR             -3
#define TRISSET             -2
#define TRISINV             -1
#define PORT                0
#define PORTCLR             1
#define PORTSET             2
#define PORTINV             3
#define LAT                 4
#define LATCLR              5
#define LATSET              6
#define LATINV              7
#define ODC                 8
#define ODCCLR              9
#define ODCSET              10
#define ODCINV              11
#define CNPU                12
#define CNPUCLR             13
#define CNPUSET             14
#define CNPUINV             15
#define CNPD                16
#define CNPDCLR             17
#define CNPDSET             18
#define CNPDINV             19
#define CNCON               20
#define CNCONCLR            21
#define CNCONSET            22
#define CNCONINV            23
#define CNEN                24
#define CNENCLR             25
#define CNENSET             26
#define CNENINV             27
#define CNSTAT              28
#define CNSTATCLR           29
#define CNSTATSET           30
#define CNSTATINV           31
#endif

// configurations for an I/O pin
// these are used by ExtCfg(a,b,c)
#define EXT_NOT_CONFIG                  0       //Not Configured
#define EXT_ANA_IN			1       //Analogue Input
#define EXT_DIG_IN			2       //Digital Input
#define EXT_FREQ_IN			3       //Frequency Measurement Input
#define EXT_PER_IN			4       //Period measurement Input
#define EXT_CNT_IN			5       //Pulse Counting Input
#define EXT_INT_HI			6       //Generate Interrupt on RISING edge
#define EXT_INT_LO			7       //Generate Interrupt on FALLING edge
#define EXT_DIG_OUT			8       //Digital Output
#define EXT_OC_OUT			9       //Digital Output with Open Collector
#define EXT_INT_BOTH                    10      //Generate Interrupt on BOTH Falling and Rising edges
#define EXT_RESERVED                    100     // this pin is reserved and SETPIN and PIN cannot be used
#define EXT_BOOT_RESERVED               101     // this pin is reserved at boot


