/*******************************************************************************
 *
 * Driver for ILI9481 Display written as CFunctions
 *
 * (c) Peter Mather 2017 with acknowledgements to Peter Carnegie & Geoff Graham
 * 
 *
 * This CFunction MUST be compiled with Optimization Level 1, -O1
 * -O2,-O3,-Os will compile successfully, but generate exceptions at runtime.
 *
 * When Generating the CFunction, use MERGE CFunction mode, and name the CFunction
 * SSD1963_V44
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *
 * V1.0     2017-10-02 Peter Mather
 * 
 ******************************************************************************/
#include <stdarg.h>

#define Version 100     //Version 1.00
#define _SUPPRESS_PLIB_WARNING                                      // required for XC1.33  Later compiler versions will need PLIB to be installed
#include <plib.h>                                                   // the pre Harmony peripheral libraries
#define MX170
//#define MX470

#include "../cfunctions.h"
#define ILI9481_SOFTRESET       0x01
#define ILI9481_SLEEPIN         0x10
#define ILI9481_SLEEPOUT        0x11
#define ILI9481_NORMALDISP      0x13
#define ILI9481_INVERTOFF       0x20
#define ILI9481_INVERTON        0x21
#define ILI9481_GAMMASET        0x26
#define ILI9481_DISPLAYOFF      0x28
#define ILI9481_DISPLAYON       0x29
#define ILI9481_COLADDRSET      0x2A
#define ILI9481_PAGEADDRSET     0x2B
#define ILI9481_MEMORYWRITE     0x2C
#define ILI9481_RAMRD           0x2E
#define ILI9481_PIXELFORMAT     0x3A
#define ILI9481_FRAMECONTROL    0xB1
#define ILI9481_DISPLAYFUNC     0xB6
#define ILI9481_ENTRYMODE       0xB7
#define ILI9481_POWERCONTROL1   0xC0
#define ILI9481_POWERCONTROL2   0xC1
#define ILI9481_VCOMCONTROL1    0xC5
#define ILI9481_VCOMCONTROL2    0xC7
#define ILI9481_MEMCONTROL 	0x36
#define ILI9481_MADCTL_MY  	0x80
#define ILI9481_MADCTL_MX  	0x40
#define ILI9481_MADCTL_MV  	0x20
#define ILI9481_MADCTL_ML  	0x10
#define ILI9481_MADCTL_RGB 	0x00
#define ILI9481_MADCTL_BGR 	0x08
#define ILI9481_MADCTL_MH  	0x04

#define ILI9481_Portrait        ILI9481_MADCTL_MX | ILI9481_MADCTL_BGR
#define ILI9481_Portrait180     ILI9481_MADCTL_MY | ILI9481_MADCTL_BGR
#define ILI9481_Landscape       ILI9481_MADCTL_MV | ILI9481_MADCTL_BGR
#define ILI9481_Landscape180    ILI9481_MADCTL_MY | ILI9481_MADCTL_MX | ILI9481_MADCTL_MV | ILI9481_MADCTL_BGR


#define LANDSCAPE       1
#define PORTRAIT        2
#define RLANDSCAPE      3
#define RPORTRAIT       4
#if defined(MX170)
// SPI pin numbers and registers
#define SPI_INP_PIN         (HAS_44PINS ? 41 : 14)
#define SPI_OUT_PIN         (HAS_44PINS ? 20 :  3)
#define SPI_CLK_PIN         (HAS_44PINS ? 14 : 25)
#define SPI_PPS_OPEN        PPSInput(2, SDI1, RPB5); PPSOutput(2, RPA1, SDO1)
#define SPI_PPS_CLOSE       PPSOutput(2, RPA1, NULL)
#define SPICON *(volatile unsigned int *)(0xbf805800)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805810)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805820)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805830)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805840)              //SPI status register
#define SPISTATCLR *(volatile unsigned int *)(0xbf805814)           //SPI status clear register
#define ILImode 0x18520
#define ILIbrg 0
#elif defined(MX470)
// SPI pin numbers and registers
#define SPI_INP_PIN        (HAS_100PINS ?  11 : 47)
#define SPI_OUT_PIN        (HAS_100PINS ? 12 :  5)
#define SPI_CLK_PIN        (HAS_100PINS ? 10 :  4)
#define SPI_PPS_OPEN        { if(HAS_100PINS) {PPSInput(2, SDI2, RPG7); PPSOutput(1, RPG8, SDO2); } else {PPSInput(2, SDI2, RPC13); PPSOutput(2, RPG7, SDO2);}}
#define SPI_PPS_CLOSE       { if(HAS_100PINS) PPSOutput(1, RPG8, NULL); else PPSOutput(2, RPG7, NULL); }
#define SPICON *(volatile unsigned int *)(0xbf805A00)               //SPI status register
#define SPISTAT *(volatile unsigned int *)(0xbf805A10)              //SPI status register
#define SPIBUF *(volatile unsigned int *)(0xbf805A20)               //SPI output buffer
#define SPIBRG *(volatile unsigned int *)(0xbf805A30)               //SPI output buffer
#define SPICON2 *(volatile unsigned int *)(0xbf805A40)              //SPI status register
#define SPISTATCLR *(volatile unsigned int *)(0xbf805A14)           //SPI status clear register
#define ILImode 0x18520
#define ILIbrg 1
#endif
#define SPIsend(a) {int j;SPIBUF=a; while((SPISTAT & 0x80)==0); j=SPIBUF;}
#define SPIqueue(a) {while(SPISTAT & 0x02){};SPIBUF=a;}

// set the chip select for the SPI to low (enabled)
// if the SPI is currently set to a different mode or baudrate this will change it accordingly
// also, it checks if the chip select pin needs to be changed
// set the chip select for SPI1 to high (disabled)
void spi_write_data(unsigned char data){
    PinSetBit(Option->LCD_CD, LATSET);
    PinSetBit(Option->LCD_CS, LATCLR);
//    SPIsend(0);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_command(unsigned char data){
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
//    SPIsend(0);
    SPIsend(data);
    PinSetBit(Option->LCD_CS, LATSET);
}
void spi_write_cd(unsigned char command, int data, ...){
   int i;
   va_list ap;
   va_start(ap, data);
   spi_write_command(command);
   for(i = 0; i < data; i++) spi_write_data((char)va_arg(ap, int));
   va_end(ap);
}
/*******************************************************************************
 *
 * defines start/end coordinates for memory access from host to SSD1963
 * also maps the start and end points to suit the orientation
 *
 * This function is a modified version of the function inside the MMBasic Interpreter
 * for MM+ on 'MX470 chips
 *
*******************************************************************************/
void DefineRegion(int xstart, int ystart, int xend, int yend) {
    PinSetBit(Option->LCD_CD, LATCLR);
    PinSetBit(Option->LCD_CS, LATCLR);
    SPIsend(ILI9481_COLADDRSET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(xstart >> 8);
    SPIsend(xstart);
    SPIsend(xend >> 8);
    SPIsend(xend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9481_PAGEADDRSET);
    PinSetBit(Option->LCD_CD, LATSET);
    SPIsend(ystart >> 8);
    SPIsend(ystart);
    SPIsend(yend >> 8);
    SPIsend(yend);
    PinSetBit(Option->LCD_CD, LATCLR);
    SPIsend(ILI9481_MEMORYWRITE);
    PinSetBit(Option->LCD_CS, LATSET);
}


//Print the bitmap of a char on the video output
//    x, y - the top left of the char
//    width, height - size of the char's bitmap
//    scale - how much to scale the bitmap
//	  fc, bc - foreground and background colour
//    bitmap - pointer to the butmap
void DrawBitmapSPI(int x1, int y1, int width, int height, int scale, int fc, int bc, unsigned char *bitmap){
    int i, j, k, m, fh, bh;
    unsigned char fhb, flb, bhb, blb;
    unsigned int consave=0,brgsave=0,con2save;
    int vertCoord, horizCoord, XStart, XEnd, YEnd;

    // adjust when part of the bitmap is outside the displayable coordinates
    vertCoord = y1; if(y1 < 0) y1 = 0;                                 // the y coord is above the top of the screen
    XStart = x1; if(XStart < 0) XStart = 0;                            // the x coord is to the left of the left marginn
    XEnd = x1 + (width * scale) - 1; if(XEnd >= HRes) XEnd = HRes - 1; // the width of the bitmap will extend beyond the right margin
    YEnd = y1 + (height * scale) - 1; if(YEnd >= VRes) YEnd = VRes - 1;// the height of the bitmap will extend beyond the bottom margin
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0;
    SPIBRG=ILIbrg;
    SPICON=ILImode;
    SPICON2=0xC00;
     // convert the colours to 565 format
    fhb = ((fc >> 16) & 0b11111000) | ((fc >> 13) & 0b00000111);
    flb = ((fc >>  5) & 0b11100000) | ((fc >>  3) & 0b00011111);
    bhb = ((bc >> 16) & 0b11111000) | ((bc >> 13) & 0b00000111);
    blb = ((bc >>  5) & 0b11100000) | ((bc >>  3) & 0b00011111);
    fh=(fhb<<8) | flb;
    bh=(bhb<<8) | blb;
    DefineRegion(x1, y1, x1 + (width * scale) - 1, y1 + (height * scale) -1);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
    for(i = 0; i < height; i++) {                                   // step thru the font scan line by line
        for(j = 0; j < scale; j++) {                                // repeat lines to scale the font
            if(vertCoord++ < 0) continue;                           // we are above the top of the screen
            if(vertCoord > VRes) return;                            // we have extended beyond the bottom of the screen
            horizCoord = x1;
            for(k = 0; k < width; k++) {                            // step through each bit in a scan line
                for(m = 0; m < scale; m++) {                        // repeat pixels to scale in the x axis
                    if(horizCoord++ < 0) continue;                  // we have not reached the left margin
                    if(horizCoord > HRes) continue;                 // we are beyond the right margin
                    if((bitmap[((i * width) + k)/8] >> (((height * width) - ((i * width) + k) - 1) %8)) & 1) {
                        SPIqueue(fh);
                    } else {
                        SPIqueue(bh);
                    }
                }
            }
        }
    }
    while((SPISTAT & 0x80) == 0);                                   // wait for all writes to complete
    while(!(SPISTAT & 0x20)) {i = SPIBUF;}                          // empty the rx fifo
    SPISTATCLR = 0x40;                                              //clear the overflow bit
    PinSetBit(Option->LCD_CS, LATSET);
    SPICON=0x0;
    SPIBRG=brgsave;  //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
}

// Draw a rectangle
// this is the basic drawing primitive used by most drawing routines
//    x1, y1, x2, y2 - the coordinates
//    c - the colour
void DrawRectangleSPI(int x1, int y1, int x2, int y2, int c){
    unsigned int consave=0,brgsave=0,con2save;
	int i, t;
    unsigned char hb, lb;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0;
    SPIBRG=ILIbrg;
    SPICON=ILImode;
    SPICON2=0xC00;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;

    // convert the colours to 565 format
    hb = ((c >> 16) & 0b11111000) | ((c >> 13) & 0b00000111);
    lb = ((c >> 5) & 0b11100000) | ((c >> 3) & 0b00011111);
    t=(hb<<8) | lb;
    DefineRegion(x1, y1, x2, y2);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);
	i = x2 - x1 + 1;
	i *= (y2 - y1 + 1);
    while(i--){
//        SPIsend(hb);
        SPIqueue(t);
     }
    while((SPISTAT & 0x80) == 0);                                   // wait for all writes to complete
    while(!(SPISTAT & 0x20)) {i = SPIBUF;}                          // empty the rx fifo
    SPISTATCLR = 0x40;                                              //clear the overflow bit

    PinSetBit(Option->LCD_CS, LATSET);
    SPICON=0x0;
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;

}
#if defined(MX470)
void DrawBufferSPI(int x1, int y1, int x2, int y2, char* p) {
    unsigned int i,t, consave=0,brgsave=0,con2save;
    unsigned char hb, lb;
    union colourmap
    {
    char rgbbytes[4];
    unsigned int rgb;
    } c;
    brgsave=SPIBRG; //save any user SPI setup
    consave=SPICON;
    con2save=SPICON2;
    SPICON=0;
    SPIBRG=ILIbrg;
    SPICON=ILImode;
    SPICON2=0xC00;
    // make sure the coordinates are kept within the display area
    if(x2 <= x1) { t = x1; x1 = x2; x2 = t; }
    if(y2 <= y1) { t = y1; y1 = y2; y2 = t; }
    if(x1 < 0) x1 = 0; if(x1 >= HRes) x1 = HRes - 1;
    if(x2 < 0) x2 = 0; if(x2 >= HRes) x2 = HRes - 1;
    if(y1 < 0) y1 = 0; if(y1 >= VRes) y1 = VRes - 1;
    if(y2 < 0) y2 = 0; if(y2 >= VRes) y2 = VRes - 1;


    DefineRegion(x1, y1, x2, y2);
    PinSetBit(Option->LCD_CD, LATSET);                               //set CD high
    PinSetBit(Option->LCD_CS, LATCLR);

    // switch to SPI enhanced mode for the bulk transfer

    for(i = (x2 - x1 + 1) * (y2 - y1 + 1); i > 0; i--){
        c.rgbbytes[0]=*p++; //this order swaps the bytes to match the .BMP file
        c.rgbbytes[1]=*p++;
        c.rgbbytes[2]=*p++;
    // convert the colours to 565 format
        hb = ((c.rgb >> 16) & 0b11111000) | ((c.rgb >> 13) & 0b00000111);
        lb = ((c.rgb >> 5) & 0b11100000) | ((c.rgb >> 3) & 0b00011111);
        t=(hb<<8) | lb;
        SPIqueue(t);
    }      

    while((SPISTAT & 0x80) == 0);                                   // wait for all writes to complete
    while(!(SPISTAT & 0x20)) {i = SPIBUF;}                          // empty the rx fifo
    SPISTATCLR = 0x40;                                              //clear the overflow bit

    PinSetBit(Option->LCD_CS, LATSET);
    SPICON=0x0;
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;

}
#endif
__attribute__((noinline)) void getFPC(void *a, void *b, volatile unsigned int *c) 
     { 
         *c = (unsigned int) (__builtin_return_address (0) - (b -a)) ;      
     } 
void pstring(const char *s){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    unsigned char  * testData    = (unsigned char *)((void *)s + libAddr );
    MMPrintString(testData);
}

/*******************************************************************************
 *
 * ILI9163 : Initialise the CFunction Driver Sub-System
 *
 * Function called to initialise the driver SubSystem
 *
 * ILI9163 is ALWAYS called from an MMBasic program
 * On exit, vectors DrawRectangleVector, and DrawBitmapVector will
 * be set to point to the CFunctions DrawRectangleSPI and
 * DrawBitmapSPI respectively
 *
 * Entry point is function long long main(long long *MyAddress,
 *                                        long long *DC,
 *                                        long long *RST,
 *                                        long long *CS
 *                                        long long *orientation)
 *                                        long long *size)
 * 
 ******************************************************************************/
//CFunction Driver_ILI9163
void main(long long *CD, long long *RST, long long *CS,long long *orientation){
    volatile unsigned int libAddr ; 
    getFPC(NULL,&&getFPCLab,&libAddr) ; // warning can be ignored, stupid editor 
    getFPCLab: { } 
    int HorizontalRes=480,VerticalRes=320;
    unsigned int consave=0,brgsave=0,con2save=0;
    Option->LCD_Reset=*RST;
    Option->LCD_CD=*CD;
    Option->LCD_CS=*CS;
    Option->DISPLAY_ORIENTATION=*orientation;
 
    ExtCfg(Option->LCD_Reset,EXT_DIG_OUT,0);ExtCfg(Option->LCD_Reset,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_Reset, LATSET);
    ExtCfg(Option->LCD_CD,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CD,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CD, LATSET);
    ExtCfg(Option->LCD_CS,EXT_DIG_OUT,0);ExtCfg(Option->LCD_CS,EXT_BOOT_RESERVED,0);
    PinSetBit(Option->LCD_CS, LATSET);
    if(ExtCurrentConfig[SPI_OUT_PIN] == EXT_RESERVED) { //already open
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }
    if(ExtCurrentConfig[SPI_OUT_PIN] != EXT_BOOT_RESERVED){ExtCfg(SPI_OUT_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_OUT_PIN, EXT_BOOT_RESERVED, 0);}
    if(ExtCurrentConfig[SPI_INP_PIN] != EXT_BOOT_RESERVED){ExtCfg(SPI_INP_PIN, EXT_DIG_IN, 0); ExtCfg(SPI_INP_PIN, EXT_BOOT_RESERVED, 0);}
    if(ExtCurrentConfig[SPI_CLK_PIN] != EXT_BOOT_RESERVED){ExtCfg(SPI_CLK_PIN, EXT_DIG_OUT, 0); ExtCfg(SPI_CLK_PIN, EXT_BOOT_RESERVED, 0);}
    SPI_PPS_OPEN; 
    SPICON=ILImode;
    SPIBRG=ILIbrg+1;
    SPICON2=0xC00;// this is defined in IOPorts.h
    if(!brgsave){ //save my settings
        brgsave=SPIBRG;
        consave=SPICON;
        con2save=SPICON2;
    }    


  //Reset the ILI9481
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATCLR);
    uSec(10000);
    PinSetBit(Option->LCD_Reset,LATSET);
    uSec(10000);

    spi_write_command(0x11);
    uSec(20000);
    spi_write_cd(0xD0,3,0x07,0x42,0x18);
    spi_write_cd(0xD1,3,0x00,0x07,0x10);
    spi_write_cd(0xD2,2,0x01,0x02);
    spi_write_cd(0xC0,5,0x10,0x3B,0x00,0x02,0x11);
    spi_write_cd(0xC5,1,0x03);
    spi_write_cd(0xB3,4,0x00,0x00,0x00,0x10);
    spi_write_cd(0xC8,12,0x00,0x32,0x36,0x45,0x06,0x16,0x37,0x75,0x77,0x54,0x0C,0x00);
    spi_write_cd(0xE0,15,0x0f,0x24,0x1c,0x0a,0x0f,0x08,0x43,0x88,0x03,0x0f,0x10,0x06,0x0f,0x07,0x00);
    spi_write_cd(0xE1,15,0x0F,0x38,0x30,0x09,0x0f,0x0f,0x4e,0x77,0x3c,0x07,0x10,0x05,0x23,0x1b,0x00); 
    spi_write_cd(0x36,0x0A);
    spi_write_cd(0x3A,1,0x55);
    spi_write_cd(0x2A,4,0x00,0x00,0x01,0x3F);
    spi_write_cd(0x2B,4,0x00,0x00,0x01,0xE0);
    uSec(120000);
    spi_write_command(0x29);
    if(Option->DISPLAY_ORIENTATION==LANDSCAPE)     spi_write_cd(ILI9481_MEMCONTROL,1,ILI9481_Landscape);
    if(Option->DISPLAY_ORIENTATION==PORTRAIT)      spi_write_cd(ILI9481_MEMCONTROL,1,ILI9481_Portrait); 
    if(Option->DISPLAY_ORIENTATION==RLANDSCAPE)    spi_write_cd(ILI9481_MEMCONTROL,1,ILI9481_Landscape180);
    if(Option->DISPLAY_ORIENTATION==RPORTRAIT)     spi_write_cd(ILI9481_MEMCONTROL,1,ILI9481_Portrait180); 

    if(Option->DISPLAY_ORIENTATION&1){
        HRes=HorizontalRes;
        VRes=VerticalRes;
    } else {
        VRes=HorizontalRes;
        HRes=VerticalRes;
    }

    //Set the DrawRectangle vector to point to our function
    DrawRectangleVector= (unsigned int)&DrawRectangleSPI + libAddr;

    //Set the DrawBitmap vector to point to our function
    DrawBitmapVector=(unsigned int)&DrawBitmapSPI + libAddr;
#if defined(MX470)
    DrawBufferVector = (unsigned int)&DrawBufferSPI + libAddr;
#endif

    //CLS
    DrawRectangle(0,0,HRes-1,VRes-1,0x000000);
    SPIBRG=brgsave; //restore user (or my) setup
    SPICON=consave;
    SPICON2=con2save;
    static const char startup[]="ILI9481 driver loaded\r\n";
    pstring(startup);
}
