/*
 * ascii ICSP implementation for the PIC16F1455
 * additions to m-stack source (c) Peter Mather  2016
 * based on USB CDC-ACM Demo
 * licenced as per the original licensing of m-stack, see below.
 *
 * ICSP extensions adapted from:
 * ascii ICSP implementation for the Arduino NANO
 * arduino code (c) Robert Rozee  2015
 *
 * ascii ICSP extension developed for use with PIC32PROG by Serge Vakulenko
 * see:
 * https://github.com/sergev/pic32prog
 * 
 * Version 1.00, 27/7/2016
 * Version 1.01, 28/7/2016
 *   add break handling to reset uP
 *   Performance tuning
 *   increased clock wait states
 *  Version 1.02, 29/7/2016
 *   Change PID/VID
 *  Version 1.03, 7/8/2016
 *   Added "!" command to return version number
 *  Version 1.04, 12/8/2016
 *   Minor bug fixes to non-essential areas
 *  Version 1.05, 21/8/2016
 *   Removed light LED on framing error
 *   Set USB1.1 
 *  Version 1.06, 21/8/2016
 *   Implement firmware reset if switch pressed > 2 seconds
 *   "-" command removed
 *  Version 1.07, 21/8/2016
 *   Flash LED after switch pressed > 2 seconds
 *  Version 1.08, 21/8/2016 
 *   Exit programming mode with > 2seconds push
 *  Version 1.09
 *   DS30 Bootloader compatible version
 *  Version 1.10, 03/09/2016
 *    Implement commands 6, 7, 9
 *    Tidy up programmer exit mechanism
 *  Version 1.11, 04/09/2016
 *    Tidy up some logic
 *    Change exit routine to increase reset time
 * Version 1.12, 05/09/2016
 *    Further tidy up
 * Version 1.14, 10/09/2016
 *    Performance tuning, Don't drive RESET high
 * Version 1.15, 12/09/2016
 *    Further tidy up
 * Version 1.17, 12/09/2016
 *    LED flashes with serial I/O
 * Version 1.18, 02/02/2017
 *    Improve tidy up on break. Remove weak pullups unless in programming mode
 * 
 * 
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commercial license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
 */

#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif
#define pulsePGC {asm ("nop"); ICSP_PGC_LAT=PIN_ON ; asm ("nop"); ICSP_PGC_LAT=PIN_OFF ; }
//#define pulsePGC {ICSP_PGC_LAT=PIN_ON ; ICSP_PGC_LAT=PIN_OFF; } //save 2 seconds programming MX170
#define RBUF_SIZE    576

typedef struct  ringBufS
{
    unsigned char buf[RBUF_SIZE];
    int head;
    int tail;
    int count;
} ringBufS;

//Global variables and buffers
char in_buf[56];
char rBuf[sizeof(ringBufS)];
unsigned char in_count=0,just_programmed=0,break_received=0,VPP_flag=0,VCC_flag=0,VOUT_enabled=0;
unsigned char Programming_active=0,led_status=LED_OFF;
volatile unsigned char h_millis=0; // counter that increments every 1/2 millisecond
volatile uint16_t resetcount=0; // counter that increments every 1/2 millisecond
volatile uint16_t ledcount=0; 
volatile unsigned char ledpoke=0;
volatile unsigned char ledflash=0;


void ringBufS_flush (ringBufS *_this)
{
  int i;
  _this->count  = 0;
  _this->head   = 0;
  _this->tail   = 0;
  for(i=0;i<RBUF_SIZE;i++)_this->buf[i]=0;
}
void clock1(unsigned char D) //clock out a single bit
{
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=D;
    pulsePGC
}
unsigned char clock1r(void) //read in a single bit
{
    unsigned char TDO='0';
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    asm ("nop"); 
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO='1';
    return TDO;
}
void clock4(unsigned char TDI, unsigned char TMS) // 4 phase clock without read
{
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=TDI;
    pulsePGC
    ICSP_PGD_LAT=TMS;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    pulsePGC
}
unsigned char clock4r(unsigned char TDI, unsigned char TMS) //4 phase clock with read
{
    unsigned char TDO='0';
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=TDI;
    pulsePGC
    ICSP_PGD_LAT=TMS;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO='1';
    pulsePGC
    return TDO;
}

void clock4x4(unsigned char TDI) //4 x 4phase clocks without read
{
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 1);
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 2)>>1;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 4)>>2;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 8)>>3;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    pulsePGC
}
unsigned char clock4x4r(unsigned char TDI)//4 x 4phase clocks with read
{
    unsigned char TDO=0;
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 1);
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO=1;
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 2)>>1;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=2;
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 4)>>2;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=4;
    pulsePGC
    ICSP_PGD_TRIS=PIN_OUTPUT;
    ICSP_PGD_LAT=(TDI & 8)>>3;
    pulsePGC
    ICSP_PGD_LAT=0;
    pulsePGC
    ICSP_PGD_TRIS=PIN_INPUT;
    pulsePGC
    if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=8;
    pulsePGC
    return TDO+='I';
}

void putConsole(unsigned char a){
    in_buf[in_count]=a;
    in_count++;
}
void checkConsole(unsigned char time, unsigned char number){
    if(h_millis>=time || in_count>=number){ //output every time/2 msecs or if number characters buffered
        if(in_count){
            while (usb_in_endpoint_busy(2))	;
            memcpy(usb_get_in_buffer(2),in_buf,in_count);
            usb_send_in_buffer(2,in_count);
            in_count=0;
        }
        h_millis=0;
        TMR2=0; //make sure the timer starts  at zero
    }
}
void test_exit(unsigned char a){
    if((VOUT_enabled && VPP_flag==0 && VCC_flag==0) || (VOUT_enabled==0 && a=='4') || (a=='X')){
        led_status=LED_OFF;
        if(a!='X'){
            just_programmed=1;
            ICSP_PGD_LAT=PIN_OFF;
            ICSP_PGC_LAT=PIN_OFF;
        }
        asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");
        asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");
        ICSP_RESET_TRIS=PIN_INPUT;
        Programming_active=0; 
        ringBufS *rb;
        rb=(void*)rBuf;
        ringBufS_flush(rb);
        h_millis=0;
        while(h_millis<5);
        ICSP_PGD_WPUA=0;
        ICSP_PGC_WPUA=0;
        ICSP_PGD_TRIS=PIN_INPUT;
        ICSP_PGC_TRIS=PIN_INPUT;
        VPP_TRIS=PIN_INPUT;
        VCC_TRIS=PIN_INPUT;
        VPP_flag=0;
        VCC_flag=0;
        VOUT_enabled=0;
    }
}

int main(void)
{
    ringBufS *rb;
    rb=(void*)rBuf;
    unsigned char ch, Letter, Uppercase, I, PrAcc, ptest=0;
    const unsigned char *out_buf;
    size_t out_buf_len;
    
    ringBufS_flush(rb);
    hardware_init(); //sets PGD, PGC, and MCLR connections to PIC32 as inputs
    timer_init(); //set up 0.5 millisecond interrupts
	usb_init();
    ICSP_RESET_TRIS=PIN_OUTPUT; 
    ICSP_RESET_LAT=PIN_OFF; 
    led_status=LED_OFF;
    while(h_millis<250); //wait for weak pullup to work
    h_millis=0;
    while(h_millis<250); //wait for weak pullup to work
    h_millis=0;
    while(h_millis<250); //wait for weak pullup to work
    h_millis=0;
    while(h_millis<250); //wait for weak pullup to work
    h_millis=0;
    ICSP_RESET_TRIS=PIN_INPUT;
    
	while (1) {
        if(!Programming_active){ //not currently programming
            if(MUP_SWITCG_PORT==BUTTON_PRESSED){
                if(ptest==0){
                    ptest=1; //set that the switch has been pressed
                    resetcount=0;
                } else {
                    if(resetcount>=400){ //put into reset after pressed for 100msec
                        ICSP_RESET_TRIS=PIN_OUTPUT;
                        ICSP_RESET_LAT=PIN_OFF; //put into reset so we never have two outputs connected
                    } 
                    if(resetcount> 4000 && resetcount<4400){ //debounce
                        led_status=LED_ON;
                    }
                    if(resetcount>= 4400){ //debounce
                        led_status=LED_OFF;
                    } 
                }
            } else {
                if(ptest==1){
                    if(resetcount< 400){ //debounce
                        ptest=0;
                    }
                    if(resetcount< 4000){ //programming mode
                        ptest=0;
                        led_status=LED_ON; 
                        Programming_active=1;
                        ICSP_RESET_TRIS=PIN_OUTPUT; 
                        ICSP_RESET_LAT=PIN_OFF; 
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        ICSP_PGD_LAT=PIN_OFF; 
                        ICSP_PGC_TRIS=PIN_OUTPUT;
                        ICSP_PGC_LAT=PIN_OFF;
                        ICSP_PGD_WPUA=1;
                        ICSP_PGC_WPUA=1;
                        PrAcc='1';
                    } 
                    if(resetcount>= 4000){ //reset MM
                        ptest=0;
                        test_exit('X');
                    }    
                }            
            }


        }
		/* Handle data received from the host */
		if (usb_is_configured()){ 
		    if(!usb_out_endpoint_halted(2) && usb_out_endpoint_has_data(2) && (RBUF_SIZE - rb->count > 64)){

                /* Check for an empty transaction. */
                out_buf_len = usb_get_out_buffer(2, &out_buf);
                if (out_buf_len <= 0)
                    goto empty;
                else {
                    /* Copy contents of OUT buffer to ring buffer */
                    for(I=0;I<out_buf_len;I++){
                          if (rb->count < RBUF_SIZE)
                            {
                                rb->buf[rb->head] = out_buf[I];
                                rb->head += 1;
                                if (rb->head >= RBUF_SIZE) rb->head = 0;
                                ++rb->count;
                            }
                           
                    }                    /* Send a zero-length packet if the transaction
                    * length was the same as the endpoint
                    * length. This is for demo purposes. In real
                    * life, you only need to do this if the data
                    * you're transferring ends on a multiple of
                    * the endpoint length. */
                    if (out_buf_len == EP_2_LEN) {
                        /* Wait until the IN endpoint can accept it */
                        while (usb_in_endpoint_busy(2))	;
                        usb_send_in_buffer(2, 0);
                    }
                }
empty:
                usb_arm_out_endpoint(2);
            }
            if(Programming_active){
                if(MUP_SWITCG_PORT==BUTTON_PRESSED){
                    if(ptest==0){
                        ptest=1; //set that the switch has been pressed
                        resetcount=0;
                    } else {
                        if(resetcount>= 4000){ //debounce
                            led_status=LED_OFF;
                        }
                    }
                } else {
                    if(ptest==1){
                        if(resetcount< 4000){ //need long press
                            ptest=0;
                        } else {//programming mode cancel
                            ptest=0;
                            test_exit('X');
                        } 
                    }
                }

                unsigned char counter=0;
                while(rb->count!=0 && counter<=35){
                    counter++;
                    I= rb->buf[rb->tail];
                    rb->tail += 1;
                    if (rb->tail >= RBUF_SIZE) rb->tail = 0;
                    --rb->count;
                   if(I=='i'){ //optimise the most used character
                        ICSP_PGD_TRIS=PIN_OUTPUT;//clock4x4(0,0))
                        ICSP_PGD_LAT=0;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC
                        goto skipit;
                    } 
                    if(I=='e' || I=='z' || I=='a'){ //optimise other high usage characters
                        ICSP_PGD_TRIS=PIN_OUTPUT; //clock4r(0,1))
                        ICSP_PGD_LAT=0;
                        pulsePGC
                        ICSP_PGD_LAT=1;
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC
                        if(I=='e')goto skipit;        
                        ICSP_PGD_TRIS=PIN_OUTPUT; //clock4r(0,0))
                        ICSP_PGD_LAT=0;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC                            
                        if(I=='z')goto skipit;
                        ICSP_PGD_TRIS=PIN_OUTPUT;//clock4r(0,0))
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        pulsePGC                            
                        goto skipit;
                    }
                    if(I=='I'){ //optimise most used read
                        unsigned char TDO=0; //clock4x4r(0,0))
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        ICSP_PGD_LAT=0;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO=1;
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=2;
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=4;
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_OUTPUT;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) TDO+=8;
                        pulsePGC
                        putConsole(TDO+='I');
                        goto skipit;
                    }
                    if(I=='D'){ //optimise second most used read
                        ICSP_PGD_TRIS=PIN_OUTPUT;//clock4r(0,0))
                        ICSP_PGD_LAT=0;
                        pulsePGC
                        pulsePGC
                        ICSP_PGD_TRIS=PIN_INPUT;
                        pulsePGC
                        if(ICSP_PGD_PORT & ICSP_PGD_MASK != 0) putConsole('1');
                        else putConsole('0');
                        pulsePGC
                        goto skipit;
                    }
                    // Now process other characters
                    if((I>='a' && I<='z') || (I>='A' && I<='Z')){ //is it a letter
                        Letter=1; //found a char
                        Uppercase=1; //assume uppercase
                        if(I>='a'){ //if lowercase
                            Uppercase=0;
                        }
                    } else Letter=0;
                    
                    if(Letter){ //letter found
                        if(Uppercase){
                            ch='#'; //code character to avoid echoing invalid inputs
                            if ((I >= 'I') && (I <= 'X')){
                                ch = clock4x4r(I-'I');
                            } else {
                                switch(I){
                                    case 'E':
                                        ch = clock4r(0, 1);        
                                        break;
                                    case 'F':
                                        ch = clock4r(1, 0);
                                        break;
                                    case 'G':
                                        ch = clock4r(1, 1);
                                        break;
                                    case 'A':
                                        clock4r(0, 1);                           // (data header)
                                        clock4r(0, 0);
                                        ch = clock4r(0, 0);
                                        break;
                                }
                            }   
                            if(ch!='#')putConsole(ch);
                        } else { //lower case
                            if ((I >= 'i') && (I <= 'x')){
                                clock4x4(I-'i');
                            } else {
                                switch(I){
                                    case 'd':
                                        clock4(0, 0);
                                        break;
                                    case 'f':
                                        clock4(1, 0);
                                        break;
                                    case 'g':
                                        clock4(1, 1);
                                        break;
                                }
                            }                               
                        }
                    } else { //not a letter
                        switch(I){
                            case '0':
                                clock1(0);
                                break;
                            case '1':
                                clock1(1);
                                break;
                            case '+':
                                if(clock4r(0, 0)=='0')PrAcc='0'; 
                                break;
                            case '>':
                                putConsole('<');
                                h_millis=2; //force usb send
                                break;
                            case '=':
                                putConsole(PrAcc);
                                PrAcc = '1'; 
                                break;
                            case '2':
                                ICSP_RESET_TRIS=PIN_OUTPUT;
                                ICSP_RESET_LAT=PIN_OFF;  
                                break;
                            case '3':
                                ICSP_RESET_TRIS=PIN_INPUT;
                                break;
                            case '4':
                                VCC_TRIS=PIN_INPUT;
                                VCC_flag=0;
                                test_exit(I);
                                break;
                            case '5':
                                if(VOUT_enabled){
                                    VCC_TRIS=PIN_OUTPUT;
                                    VCC_LAT=PIN_OFF; 
                                    VCC_flag=1;
                                }
                                break;
                            case '6':
                                VPP_TRIS=PIN_INPUT;
                                VPP_flag=0;
                                test_exit(I);
                               break;
                            case '7':
                                if(VOUT_enabled){
                                    VPP_TRIS=PIN_OUTPUT;
                                    VPP_LAT=PIN_OFF; 
                                    VPP_flag=1;
                                }
                                break;
                            case '8':
                                h_millis=0;
                                while(h_millis<20);
                                break;
                            case '9':
                                VOUT_enabled=1;
                                break;
                            case '-':
                                putConsole(clock1r());
                                break;
                            case '!':
                                putConsole('V');
                                putConsole('1');
                                putConsole('.');                                
                                putConsole('1');
                                putConsole('8');
                                putConsole(13);
                                putConsole(10);
                                break;
                            case '?':
                                putConsole('a');
                                putConsole('s');
                                putConsole('c');
                                putConsole('i');
                                putConsole('i');
                                putConsole(' ');
                                putConsole('I');
                                putConsole('C');
                                putConsole('S');
                                putConsole('P');
                                putConsole(' ');
                                putConsole('v');
                                putConsole('1');
                                putConsole('N');
                        }
                    }
skipit:; 
                }
                checkConsole(1,10);
            } else {
                if(mDataRdyUSART())putConsole(USART_getcUSART()); //if any characters received then buffer them
                checkConsole(40,60);
                if(rb->count!=0 && mTxRdyUSART()) {
                    I = rb->buf[rb->tail];
                    rb->tail += 1;
                    if (rb->tail >= RBUF_SIZE) rb->tail = 0;
                    --rb->count;
                    USART_putcUSART(I);
                }
            }
        }
	}
}

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return 0;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
    
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
//    *idle_setting=0;
//    *data_multiplexed_state=0;
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
    if(just_programmed){
        ringBufS *rb;
        rb=(void*)rBuf;
        ringBufS_flush(rb);
        ICSP_RESET_TRIS=PIN_OUTPUT;
        ICSP_RESET_LAT=PIN_OFF; 
    }
    switch(coding->dwDTERate){
        case 300:
            USART_Initialize(0x3F,0x9C);
            break;
        case 600:
            USART_Initialize(0x1F,0x4E);
            break;
        case 1200:
            USART_Initialize(0x0F,0x27);
            break;
        case 2400:
            USART_Initialize(0x87,0x13);
            break;
        case 4800:
            USART_Initialize(0xC3,0x09);
            break;
        case 9600:
            USART_Initialize(0xE1,0x04);
            break;
        case 19200:
            USART_Initialize(0x71,0x02);
            break;
        case 38400:
            USART_Initialize(0x37,0x01);
            break;
        case 57600:
            USART_Initialize(0xD0,0x00);
            break;
        case 76800:
            USART_Initialize(0x9B,0x00);
            break;
        case 115200:
            USART_Initialize(0x68,0x00);
            break;
        case 230400:
            USART_Initialize(0x33,0x00);
            break;
        default:
            USART_Initialize(0x37,0x01);
    }
        if(just_programmed){
        just_programmed=0;
        ICSP_RESET_TRIS=PIN_INPUT;
    }
    return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
//    coding->dwDTERate   = 38400;      // baud rate
//    coding->bCharFormat = 0x00;             // 1 stop bit
//    coding->bParityType = 0x00;             // None
//    coding->bDataBits = 0x08;               // 5,6,7,8, or 16	/* This is where baud rate, data, stop, and parity bits are set. */
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	return -1;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
    if(!Programming_active){
        if(!break_received) {
            ICSP_RESET_TRIS=PIN_OUTPUT;
            ICSP_RESET_LAT=PIN_OFF; 
            asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");asm("nop");
            ICSP_RESET_TRIS=PIN_INPUT;
            break_received=1;
        } else break_received=0;
    }
    return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
    if (PIR1bits.TMR2IF) { // timer 2 interrupt at 2000 Hz
        h_millis++;
        resetcount++;

        if(led_status==LED_ON){
            LED_LAT=LED_ON;
        } else {
            if(ledpoke){
                ledpoke=0;
                if(ledflash==0){
                    ledcount=0;
                    ledflash=1;
                }
            }
            if(ledflash){
                if(ledcount==0)LED_LAT=LED_ON;
                if(ledcount==50)LED_LAT=LED_OFF;
                if(ledcount==150)ledflash=0;
                ledcount++;
            } else {
                LED_LAT=LED_OFF;
            }
        }
        PIR1bits.TMR2IF = 0; // reset flag
        return;
    } 
    usb_service();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif
