  '**************************************************************************
  ' Camper Power Management System                                          *
  ' VARANUS - The Water Monitor                                             *
  ' Ver 1.03 26 December 2019                                               *
  ' File name:vcpms-esp-103rc1.bas (also vcpms1.bas on sd card)             *
  ' - integrated V64 mini  - tcp version                                    *
  ' - includes web interface using ESP8266                                  *
  ' last known good version  -  vcpmsm-18.bas  - non-esp                    *
  ' Copyright D. & T. Pankhurst 2018                                        *
  ' MMBasic Copyright Geoff Graham.                                         *
  ' Epoch time and CFunctions from Peter Mather.                            *
  ' Assitance and code suggestions from TheBackShed.com members             *
  ' greatfully acknowleged.                                                 *
  '**************************************************************************
  '-------------------------------------------------------------------------*
  ' For Varanus V65-mini with 5" LCD and Relay board                        *
  ' OPTION AUTORUN ON                                                       *
  ' OPTION LCDPANEL SSD1963_5, LANDSCAPE                                    *
  ' OPTION TOUCH 50, 33                                                     *
  ' GUI CALIBRATE 1, 3993, 3869, -2044, -1301                               *
  ' OPTION SDCARD 31                                                        *
  ' OPTION RTC 43, 44                                                       *
  ' OPTION CONTROLS 180                                                     *
  '**************************************************************************
  ' Name       E100         V64-5  V64-Mini                                 *
  ' ------------------------------------------------------------------------*
  'Ignition    25 (GPIO29)   29     29      Used to sense a Charging source *
  'IR          78 (GPIO7)    51     51      Used to Wakeup VCPSM            *
  'RlySR-      59 (GPIO40)   59     59      E100 LRlySel0 - COM1 on 470-64  *
  'RlyAddr0    60 (GPIO38)   55     55      E100 LRlySel1                   *
  'RlyAddr1    61 (GPIO36)   54     54      E100 LRlySel2                   *
  'RlyLtch0    68 (GPIO30)   52     53      E100 LRB0                       *
  'RlyLtch1    74 (GPIO22)   49     49      E100 LRB1                       * 
  'RlyLtch2    ?? (GPIO  )   46     46      E100 LRB2 - Rly1 on mini        * 
  'RlyLtch3    ?? (GPIO  )   32     32      E100 LRB3 - Rly2 on mini        *
  'RlySense0   77 (GPIO9)    53     52                                      *
  'RlySense1   ?? (GPIO )    48     48                                      *
  'RlySense2   ?? (GPIO )    42     42                                      *
  'RlySense3   ?? (GPIO )    31      x      Not used on mini                *
  'Input1      44 (GPIO13)   30     30                                      *
  'Input2      43 (GPIO15)   45     45      Digital only for 470-64         *
  'Input3      41 (GPIO17)    8      8                                      *
  'A_Input4    35 (GPIO19)   12     12      FreshWaterPin                   *
  'Input5      34 (GPIO21)   14     14                                      *
  'Input6      33 (GPIO23)   15     23                                      *
  'Input7      32 (GPIO25)   21      x      Not on Mini                     *
  'Input8      21 (GPIO35)   22      x         "  "                         *
  'Input9      14 (GPIO39)   23     23      Count-moved to logical 6 on mini*
  'ExtPwr_R    97 (GPIO2)     x     46      On board relay on mini-RlyLtch2 *
  'GenKill_R   96 (GPIO4)     x     32              "  "           RlyLtch3 *
  'OBRly1_R    x              x     46      On board relay on mini-RlyLtch2 *
  'OBRly2 _R   x              x     32              "  "           RlyLtch3 *
  'Notes: on the V64-5 and the Mini, pins 52 & 53 swap.                     *
  ' This is to allow better future use of PWM,                              *
  '                                                                         *
  'Input1 on both boards have a connection designed for the Dallas Temp chip*
  '   - that could be a feature.                                            *
  '**************************************************************************  
  cpu 100 ' in case we stopped in power save mode previously
  Option explicit
'---------------------
'************ V64-5 unlikely to be used for camper *****************
'  const Ignition       = 29 '
'  const FreshWaterPin  = 12 ' AIN
'  ' Constants for driving relay boards, pac board etc.
'  Const LRlySel0       = 59 ' DOUT  base MM pin for latch relay set/reset sel
'  Const LRlySel1       = 55 ' DOUT  use MM pins 60 and 61 to select relay
'  Const LRlySel2       = 54 ' DOUT
'  ' latch relay board selection and pulse to set/reset relay
'  Const LRB0           = 52 ' DOUT  latch relay board 1
'  Const LRB1           = 49 ' DOUT  latch relay board 2
'  Const LRB2           = 46 ' DOUT  latch relay board 3
'  Const LRB3           = 32 ' DOUT  latch relay board 4
'  Const GenKill_R      = 32 ' temporary
'  Const ExtPwr_R       = 46 ' temporary
'  const RlySense0      = 53 ' DIN   temp - relay state input
'  const RlySense1      = 48 ' AIN   temp - relay state input
'  const RlySense2      = 42 ' AIN   temp - relay state input
'  const RlySense3      = 31 ' AIN   temp - relay state input
'  ' end of V64-5 pin constants and relay variables
'--------------------------
' For V64-mini platform
  dim Platform% = 1 ' 
  const Ignition       = 29 '
  const FreshWaterPin  = 12 ' AIN
  ' Constants for driving relay boards, pac board etc.
  Const LRlySel0       = 59 ' DOUT  base MM pin for latch relay set/reset sel
  Const LRlySel1       = 55 ' DOUT  use MM pins 55 and 54 to select relay
  Const LRlySel2       = 54 ' DOUT
  ' latch relay board selection and pulse to set/reset relay
  Const LRB0           = 53 ' DOUT  latch relay board 1
  Const LRB1           = 49 ' DOUT  latch relay board 2
  Const LRB2           = 46 ' DOUT  onboard relay board
  ' Note: If normal relay, pin needs to be held high to activate relay
  '  ... if latching relay, pulse to turn on
  '  ... also note if latching relay used in position 1, no relay in position 2
  Const LRB3           = 32 ' DOUT  onboard relay board
  ' Note: If normal relay, pin needs to be held high to activate relay
  '  ... if latching relay in position 1, pulse to turn off
  Const GenKill_R      = 46 ' onboard relay 1
  Const ExtPwr_R       = 32 ' onboard relay 2
  CONST OBRly1_R       = 46 ' alt name for above
  const OBRly2_R       = 32 '    "    "
  const RlySense0      = 53 ' DIN   temp - relay state input
  const RlySense1      = 48 ' AIN   temp - relay state input
  const RlySense2      = 42 ' AIN   temp - relay state input
'  const RlySense3      = 31 ' AIN relay state input-share with SD card - jumper select
  ' end of V64-5 pin constants and relay variables
  ' end of platform conditional code
  ' --------------------------------
  Dim F_Ver$ = "1.03 26 December 2019"
  '---------------------------------
  '   GUI Constants
  ' ***** Note: Requires 180 controls - use OPTION CONTROLS 180 from cmd line
  '   GUI touch areas - CTRL 1 -14
  Const Sockets_A      = 1 ' these areas are all switches
  Const Fridge_A       = 2
  Const Lights_A       = 3
  Const Pump_A         = 4
  const GenKill_A      = 5  
  const Charge_A       = 6
  const LCD_Off_A      = 7
  Const Setup_A       = 8
  const ExSetup_A      = 9 ' exit setup switch
  ' GUI Frames CTRL 15 -24
  Const Setup_F       = 19 ' setup options frame
  ' GUI text and number boxes
  Const SetupOption_T = 20
  Const SetupOption_N = 21

  ' GUI Display Boxes - CTRL - 26-59
  const LVB_D      = 27 ' large vehicle battery box
  const LCB_D      = 28 ' large camper battery box
  Const VehicleB_D = 30 ' graph box
  Const CamperB_D  = 31 ' graph box
  Const Startup_D  = 32 ' splash screen at start
  const Setup_D    = 33 ' switch to invoke setup
'  Const Setup_D    = 34
  const ExSetup_D  = 35 ' exit setup switch
  Const Help_D     = 48 ' bottom box for messages, time etc.
  Const FWater_D   = 51 ' water graph box
  Const Sockets_D  = 52 ' switch
  Const Fridge_D   = 53 ' switch
  Const Lights_D   = 54 ' switch
  Const Pump_D     = 55 ' switch
  const GenKill_D  = 56 ' switch
  const ExtPwr_D   = 57 ' switch
  const LCD_Off_D  = 58 ' switch
  const Charge_D   = 59 ' switch
  ' GUI Gauges & LEDS- CTRL 60-79
  Const FWater_G       = 65
  Const ExtPwr_Volts_G = 67
  Const ExtPwr_Amps_G  = 68
  Const CB_SOC_G       = 70
  Const CB_Volts_G     = 71  
  Const CB_AmpsP_G     = 72
  Const CB_AmpsN_G     = 73
  const CB_PwrP_G      = 74
  const CB_PwrN_G      = 75
  ' GUI Switches & Momentary Push Buttons- CTRL 80-119
  '  - switches are created with display boxes and touch areas
  ' GUI  Captions - these must come after display boxes - CTRL 120 -159
  Const Startup1_C     = 120
  Const Startup2_C     = 121
  Const Startup3_C     = 122
  Const LVB_C          = 123
  Const LVBV_C         = 124
  const LVBA_C         = 125
  const LCB_C          = 126
  const LCBV_C         = 127
  const LCBA_C         = 128
  Const FWaterLow_C    = 129
  const F_Water_C      = 130
  Const SetupOption_C  = 131
  Const SetupHelp_C    = 132
  ' gauge battery layout
  Const ExtPwr_C        = 148
  Const ExtPwr_Volts_C  = 149
  Const ExtPwr_Amps_C   = 150
  Const CB_SOC_C        = 151
  Const CB_Volts_C      = 152
  Const CB_Amps_C       = 153
  const CB_Pwr_C        = 154
  ' LED's and other GUI
  const H_Beat_L        = 160
  ' ----End GUI constant definitions
  '*********************************************************************
  ' ----General constants
  '---- Epoch time variables and constants
  const EPOCH_YR% = 1970
  CONST SECS_DAY% = 86400
  CONST SECS_HOUR% = 3600
  CONST SECS_MIN% = 60
  dim ytab1%(11)=(31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31)
  dim monoff%(11) = (0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334)
  dim daysofweek$(6) length 9  =("Sunday","Monday","Tuesday","Wednesday","Thursday","Friday", "Saturday")
  dim my_year%, my_month%, my_day%, my_hour%, my_min%, my_sec%, my_dayno%, my_dayinyear%, my_timestamp%
  dim monthsinyear$(11) length 9 = ("January", "February", "March", "April", "May", "June" , "July", "August", "September", "October", "November", "December")
  dim th$(30) length 2 = ("st","nd","rd","th","th","th","th","th","th","th","th","th","th","th","th","th","th","th","th","th","st","nd","rd","th" ,"th","th","th","th","th","th","st")
  '---- end epoch time definitions
  '-----------------------------------
  ' PAC1934 constants and variable definitions
  ' PAC1934 I2C 7 bit address is &B0010000 (bits 7 thru 1)
  '   - MM adds bit 0 as read (1) or write(0)
  ' PAC Channels used 
  ' Ch 1 - bi-directional current into (charge) or
  '        out of (discharge) camper battery
  ' Ch 2 - unidirectional current - fridge and lights
  ' Ch 3 - unidirectional current - pump and sockets
  ' example PAC constants
  CONST PACAddr% = &B0010000      ' (dec 16) - MM adds bit 0 as read or write
  ' PAC Registers:-
  CONST PAC_Refresh% = &B00000000 ' write 0 bytes to reg 0 to start refresh
  CONST PAC_CtrlReg% = &B00000001 ' we use only bits 7, 6 (&B10000000= 64s/s)
  CONST PAC_Samples% = &B10000000  ' (dec 128) is 64 samples per sec
  const PAC_Ch1BiDi% = &B00100000  ' (dec 128) set ch 3 to bidirectional
  CONST PAC_Refresh_V% = &H1F     ' refresh without clearing accumulators
  CONST PAC_ACC_Count% = &H02
  CONST PAC_VBase% = &H07
  CONST PAC_ABase% = &H0B
  CONST PAC_PBase% = &H17 ' instantanious power based on V and A above
  CONST PAC_VAvgBase% = &H0F ' average voltage
  CONST PAC_AAvgBase% = &H13 ' average current
  const PAC_PAccBase% = &H03 ' Accumulated power 
  CONST PAC_NegPwr% = &H1D ' bit 7 thru 4 select channel 1 thru 4 for neg 
  const VScale=(32/65536)        'scales 16 bit voltage from 32V FSR
  ' shunt values - default Chipkit PAC board - update in setup 
  dim Ch1Shunt = 0.007
  dim Ch2Shunt = 0.007
  dim Ch3Shunt = 0.007
  dim Ch4Shunt = 0.007
  ' Note: Ch1 & 2 are uni dir load amps, ch3 is bi dir battery amps
  const AScaleCh1=((0.1/Ch1Shunt)/65536)' 
  const AScaleCh2=((0.1/Ch2Shunt)/65536)' 
  const AScaleCh3=((0.1/Ch3Shunt)/32768)' setup as bi dir
  const AScaleCh4=((0.1/Ch4Shunt)/65536)' 
  const PScale=VScale*AScaleCh3
  dim PAC_Channel%           'PAC Channel to access, 1-4
  DIM PAC_Regs%(38)    ' array to hold all PAC regs from read all
  dim PAC_AccFlag% = 0
  '-----------end of PAC definitions
  ' latching relay variables, arrays - constants declared in platform code
  Dim VSwitch% = 1 ' switch number to select latch relay
  Dim SwSR% = 0    ' 0 = reset, 1 = set for latch relays
  '-------------------------------------
  ' -----General Variables
  Dim ExtPwr_Volts = 12.67 ' external source charge volts - vehicle,solar,gen
  Dim CB_Volts = 12.73 ' camper battery volts
  dim CB_VoltsCal = 12/CB_Volts ' calibration factor to apply to anything 
  ' camper battery volts as reference and assumes 12V
  dim F_L_Amps = 0 ' fridge/lights on PAC Ch1
  dim P_S_Amps = 0 ' pump/sockets on PAC Ch2
  dim Sys_Amps = 0.5 ' current used by Varanus
  DIM Load_Amps = F_L_Amps + P_S_Amps + Sys_Amps
  Dim CB_Amps = 0 ' camper battery amps - can be + for charge or - for disch
  Dim ExtPwr_Amps = -(Load_Amps)+(CB_Amps) ' calculated
  dim CB_LoadPwr = 0   ' an accumulating value from PAC - reset on SOC = 100%
  dim CB_Pwr = 0  ' an accumulating value from PAC - reset on SOC = 100%
  dim CB_SOC% = 100 ' camper battery state of charge
  ' NB. This is a calculated value based on charge power minus load power
  ' giving actual battery in/out power. An efficiency factor accounted for.
  ' The power in and power out numbers are calculated every
  ' 2 seconds and reset when soc equals 100%
  Dim FWater = 100
  dim ExtPwr = 0 ' 0 to 3.3V - external power into ignition pin
  '                 - could also be solar or genset
  dim SolPwrDly% = 0 ' delay counter used in external charge relay code
  dim TmpExt_Volts = 12.8 ' temp ext volts - lags real ext volts by 2 sec
  dim TmpCB_Volts = 12.8 ' old battery volts - lags real by 2 sec
  dim ExtPwrMode% = 0 ' 
  Dim iLoop      = 0  ' general loop counter
  dim S_Bright = 100
  Dim Startup1$
  Dim Startup2$
  Dim Startup3$
  dim TchRef = 0  ' temp storage for GUI touch reference
  dim ESPRef% = 0  ' temp storage for ESP touch reference
  dim OBRly1Type = 0 ' 0 for std relay, 1 for latching relay
  dim sw$ = ""  ' temp bucket for console input
  '----------------------------------
  ' flags for various operations
  Dim TwoSecFlag% ' set by settick 2, cleared at end of 2 sec process
  Dim TwentySecFlag% ' red-paint time and date in help box
  Dim FiveMinFlag%' set by settick 2, cleared at end of min process
  Dim HrFlag%' set by settick 2, cleared at end of hour process
  Dim EODFlag%' set by settick 2, cleared at end of EOD process
  Dim NewDataFlag = 0 ' set when new external values read, cleared by
  Dim StartupFlag = 1 ' used to check about re-init of tacho
  Dim SetupFlag%  = 0 ' 
  Dim SetupAskFlag% = 0 ' set to 1 if setup textbox touched
  Dim SetupAnsFlag% = 0  ' set to 1 when textbox enter pressed
  dim ExSetupFlag%  = 0 ' used to exit from User setup
  dim InsideSetupFlag%   = 0
  dim UpdateSuspendFlag% = 0
  Dim SetupDoneFlag% = 1 ' set to 1 when setup complete
  Dim TouchDownFlag% = 0 ' set to 1 on touch down
  dim BatChgFlag% = 0 ' 0 is discharge, 1 is charge
  dim CBA_NegFlag% = 0 ' to show CB_Amps as neg or pos
  dim CBPwrNegFlag% = 0 ' to show WHrs as discharge or charge
  ' these are the switch flags
  dim SocketsFlag% = 0
  dim FridgeFlag%  = 0
  dim LightsFlag%  = 0
  dim PumpFlag%    = 0
  dim ChargeFlag%  = 0
  '
  dim ChargeMode%	=	1 ' sets automatic mode or manual mode
  dim GenKillFlag% = 0 ' set to send kill signal to generator
  dim LCD_OffFlag% = 1 ' initially set to 1 - display on, awake
  dim lcd_on = 0
  '
  Dim ExtPwrFlag% = 0 ' set when ext power available
  dim OBRly1Flag% = 0 ' 0 for standard, 1 for latching
  '-------------------------------
  ' ESP specific variables
  dim ESP_Rx$ = "" ' holder for ESP msgs
  dim ESP_in_flag% = 0 ' indicates msg rxd from ESP
  '-------------------------------
  ' Initialisation code
  '--------------------
  ' Initialise I/O pins - platform (pin numbers) determined at start
  '    - see pin chart at start of code
  SetPin LRlySel0,DOUT ' determines if latch set (1) or reset (0)
  SetPin LRlySel1,dout ' one of 4 relays selected per board
  SetPin LRlySel2,dout
  setpin RlySense0,DIN ' if > than ~1V then relay active
  ' relay board select pins - all are active low
  pin(LRB0) = 1 ' preset high as relay logic negative
  SetPin LRB0,dout ' latch relay board 1 - pulse pin to set/reset
  ' other inputs and outputs
  setpin Ignition,AIN ' 
  setpin FreshWaterPin,AIN '
  pin(ExtPwr_R) = 0 ' preset in case OBRly1 is a latching relay
  pin(GenKill_R) = 0 ' preset in case OBRly1 is a latching relay
  setpin ExtPwr_R,DOUT  ' charging relay (also known as OBRly1)
  setpin GenKill_R,dout ' generator kill relay (also known as OBRly2)
  ' Note: if OB relay 1 is a latching relay, relay 2 position not used
  '  and OB relay 2 line is used to turn off the OB 1 latching relay
  InitPAC1934  ' set I2C speed for PAC1934
  ' setup for ESP8266 web interface
'  open "COM3:115200,512,ESP_COM3_Int,1" as #1
  open "COM3:9600" as #1
  '------------------------------
  rtc gettime
  ' Touch Interrupts
  GUI Interrupt TouchDown,TouchUp
  '------------------------------
  ' Enable timer tick interrupts to start data collection at 00 seconds
  SetTick 1000,OneSecInt,1 'Call every sec, update time,battery volts,current
  '----------------------------------------
  ' preset startup display
  Startup1$ = "Welcome to VARANUS~(The Water Monitor)~~Camper Power Monitor System"
  Startup2$ = "Version "+F_Ver$
  Startup3$ = "Copyright Alan T. Bunbury 2018~MMbasic Copyright Geoff Graham 2018"
  '
  CLS ' start with clear screen on lcd
  BackLight 100
  ' Initialise all GUI functions
  InitGUI
  ' pre setup colours on switch panel
  GUI bcolour RGB(0,80,0),Sockets_D,Lights_D,Pump_D,Fridge_D
  gui bcolour rgb(70,30,25),GenKill_D,Charge_D
  gui bcolour rgb(180,100,70),LCD_Off_D
  ' end of initialisation routines
  '**************************************************************************
  ' Display initial GUI screen with startup welcome
  ' - hang in that screen until any screen touch then fall through to
  '   the main GUI screen
  '*** Actual program entry point ****
  ' First off, welcome screen for 10 secs unless screen touched
  Page 1,2
  Ctrlval(Help_D) = "Fresh startup - welcome!"
  TouchDownFlag% = 0 ' so we can display startup screen
  For iLoop = 1 To 30000
    If TouchDownFlag% = 1 Then
          Exit For
    EndIf
  Next iLoop
  ' send some info out console port
  print " Starting Varanus Camper Control Program"
  print 
  print "For manual relay control, enter console mode"
  print " and suspend GUI operation until finished."
  print
  print "Press C to start console mode"
  
  Page 2,3,4   ' help box, switches, water, gauges
  TouchDownFlag% = 0 ' clear touch flag
  ' end of startup code
  '-----------------------------------
  ' main program loop
  Do
    If ESP_in_flag% = 1 Then
      pause 10  ' let things settle
      VSwitch% = ESPRef%
      Sw_Rly_Service ' decode switch, operate relay and update GUI and web displays
      ESP_in_flag% = 0
    endif
    '
    If TouchDownFlag% = 1 Then
      pause 10  ' let things settle
      ' check to see what was touched
      ' - note -switches(GUI AREA) start at CTRLVAL 1
      VSwitch% = TchRef ' convert ctrl const to switch number
      ' Note: VSwitch% 1 thru 4
      ' Note also: switch flag value is pos logic, 1 = on, 0 = off
      if TchRef > 0 and TchRef < 5 then
        Sw_Rly_Service ' decode switch, operate relay and update GUI and web displays
        TchRef = 0
      endif
      '
      select case TchRef  ' touch was not a switched relay so ?
        case Charge_A 'Charge (enable ext pwr)switch has special treatment
          ExtPwrMode% = ExtPwrMode% + 1
          if ExtPwrMode% > 5 then
            ExtPwrMode% = 0
          endif
          pin(ExtPwr_R) = 0 ' jam ext pwr relay off
          select case ExtPwrMode%
            case 0 ' relay off unconditional
              GUI bcolour RGB(255,0,0),Charge_D
              ctrlval (Charge_D) = "Ext~OFF"
              ExtPwrFlag% = 0 ' reset
              pin(ExtPwr_R) = 0 ' jam ext pwr relay off
            case 1 ' if ext power >12.8 and < 15, allow relay on
              GUI bcolour RGB(0,255,0),Charge_D
              ctrlval(Charge_D) = "AUTO"
            case 2 ' allow relay on unconditional
              GUI bcolour RGB(0,0,255),Charge_D
              ctrlval (Charge_D) = "MANUAL"
            case 3 ' allow relay on, wait for solar to stabilise
              GUI bcolour RGB(255,255,0),Charge_D
             ctrlval (Charge_D) = "HUNT"
            case 4 ' allow relay on, wait for solar to stabilise
              GUI bcolour RGB(255,255,0),Charge_D
              ctrlval (Charge_D) = "SOLAR"
            case 5 ' turn relay off, solar not available
              GUI bcolour RGB(255,255,0),Charge_D
              ctrlval (Charge_D) = "LOST"
            case 6  ' nothing to change
            end select
        case GenKill_A ' send background pulse of 6 secs
          Toggle GenKillFlag%
          If GenKillFlag% = 0 Then
            GUI bcolour RGB(80,30,25),GenKill_D
          Else
            GUI bcolour RGB(190,100,70),GenKill_D
            GenKillFlag% = 1
            pin(GenKill_R) = 0 'preset so pulse below operates rly for 6 secs          
            pulse GenKill_R,6000 ' disable generator
          EndIf
        case LCD_Off_A ' starts off at 1 and switch bright
          Toggle LCD_OffFlag%
          GUI bcolour RGB(80,30,25),LCD_Off_D
          S_Bright = S_Bright + 10
          if S_Bright > 100 then
            S_Bright = 50
          endif
          ctrlval(LCD_Off_D) = str$(S_Bright)
          Backlight S_Bright
        case Setup_A  'Config - go do setup
          Toggle SetupFlag%
          If SetupFlag% = 0 Then
            GUI bcolour RGB(0,0,80),Setup_D
          Else
            GUI bcolour RGB(80,80,200),Setup_D
            UserSetup
            SetupFlag% = 0 ' reset ready for another touch
            page 2,3,4 ' help,switches,gauges
            GUI bcolour RGB(0,0,80),Setup_D
          EndIf
        ' end of switch/relay detects
        case else
      End Select
      ' end of touch selection so clear the touch areas
      TouchDownFlag% = 0
      TchRef = 0
    endif
    ' jump straight down here if no touch but check in case something
    ' else has set or cleared a relay or other function (IR etc)
    ' ** IR to be implimented later **
    If TwoSecFlag% = 1 Then
      ' update all data values 
      TwoSecProcess
    ElseIf TwentySecFlag% = 1 Then
      TwentySecProcess ' hint messages
    ElseIf FiveMinFlag% = 1 Then
      FiveMinProcess ' calculate averages of 2 sec data
     ElseIf HrFlag% = 1 Then
      HrProcess ' handles PAC board accumulating power data, reset accums
    ElseIf EODFlag% = 1 Then
      EODProcess ' tidy , create new log file name
    EndIf
    '*****************************************
    ' check for console mode request
    ' NOTE: If we go into this mode, we will loop there forever
    ' until an X is entered at the console!!!!!!!!
    sw$ = input$(8,#0)
    if left$(sw$,1)=  "C" then 
    ' enter console mode to control switches
      Con_Mode ' go do manual switch control
    endif
    '***************************************** 
    NewDataFlag = 0
  Loop
  ' end of main program loop'
  '**************************************************************************
  ' Subroutines to gather and display all boat data
Sub TwoSecProcess
  Local i% ' loop counters for averaging array
  ' update all non-realtime display items
  if pin(GenKill_R) = 0 then ' check gen kill timeout and reset switch
    GenKillFlag% = 0
    GUI bcolour RGB(80,30,25),GenKill_D
  endif
  UpdateExtVals ' get external data (batteries, etc.)
  ' check ignition - if on, external power available so LCD on, full speed
  if ExtPwr_Volts > 12.8 then ' turn on LCD, disable sleep, full speed
    LCD_OffFlag% = 1 ' may as well run full speed, bright
    GUI bcolour RGB(190,100,70),LCD_Off_D
    ctrlval(LCD_Off_D) = "LCD~ON"
    cpu 100
    Backlight 100 '
    
    'test for state of charge, if less than 100%
' *** code needed here ****
    '   turn on charge relay
  endif
  GUIUpdateVals ' update GUI values and display
  TwoSecFlag% = 0
End Sub ' end of 2 sec process
  '---------------------------
Sub TwentySecProcess ' overwrite any help message with date time
  Local t$,d$ length 64
  local VWMstat$ ' contains all the switch and battery values to update ESP
  Local i%,j% ' loop counters to fill log array
  ' grab time into component variables and update GUI display
  convert_time()
  t$ = MyAMPM_Time$(my_hour%,my_min%)
  d$ = t$ + " "+daysofweek$(my_dayno%)+"~"
  d$ = d$ +str$(my_day%,2)+th$(my_day%-1)+" "+monthsinyear$(my_month%-1)
  d$ = d$ +" "+str$(my_year%,4)
  GUI bcolour RGB(100,100,150)),Help_D
  GUI fcolour RGB(white),Help_D
  CtrlVal(Help_D) = d$
  Update_ESP  ' send out switch status and data values to ESP
  TwentySecFlag% = 0  
End Sub
  '-------------------------------
Sub FiveMinProcess ' hint box update
  local i%,j%,bat_va%,f_l_va%,p_s_va%,load_va%, charge_va%,acc_count%,CBPwr_Tmp%
  ' load in the PAC accumulated power for the last hour for each channel
'  bat_va% = PAC_AccPwr%(3)
  load_va% = PAC_AccPwr%(4)   '
  charge_va% = load_va% + bat_va%
'  '---
'  ' now do the power thing for CB_Pwr
'print "raw tmp = ",bat_va%
'      
'  CB_Pwr = ((bat_va%/PAC_ACCC%())* PScale)/12
'  ClearAccum
'print "pwr = ",CB_Pwr
''**** now what do I do with it ????
  FiveMinFlag% = 0
End Sub ' end of 5 min process
'--------------------------------
Sub HrProcess ' PAC power handling

  HrFlag% = 0
End Sub ' end of 5 min process
'--------------------------------
Sub EODProcess
'  LogFile$ = right$(Date$,4)+mid$(date$,4,2)+left$(date$,2)+".log"
  EODFlag% = 0
End Sub ' end of eod process
'****************************************************************************
' Interrupt handling subroutines - Time tick, touch
' 1 Second SETTICK timer 1 Interrupt Routine Handling
'  - test/set the time flags, 
Sub OneSecInt      ' SetTick Timer 1 Interrupt handler - every second
  Local D$
  ' grab time into component variables and update GUI display
  ' now set time flags
  If (val(right$(time$,2)) Mod 2) = 0 Then 'set every even sec, eg. 0,2,4 etc
    TwoSecFlag% = 1
  EndIf
  If (val(right$(time$,2)) Mod 20) = 0 Then
    TwentySecFlag% = 1
  EndIf
  If (val(right$(time$,2))=0) And ((val(mid$(time$,4,2)) Mod 5)=0) Then ' 
    FiveMinFlag% = 1
  EndIf
  If (val(right$(time$,2)) = 1) And (val(mid$(time$,4,2)) = 0) Then ' 
    HrFlag% = 1
  EndIf
  ' reset clock at midnight
  If (val(right$(time$,2)) = 3) And (val(mid$(time$,4,2)) = 0) And (val(left$(time$,2)) = 0) Then ' 3 secs after midnight
    EODFlag% = 1
  EndIf    
End Sub ' end of 1 sec tick interrupt  
  '  
  '**************************************************************************
  ' Touch interrupt handling routines
  ' screen touch routines for GUI interface
  ' - just set TouchDownFlag
Sub TouchDown
  TouchDownFlag% = 1 ' actual touch area or switch detected in main loop
  TchRef = touch(REF)
  select case TchRef
    case ExSetup_A  'needed to allow exit from setup
      Toggle ExSetupFlag%
      If ExSetupFlag% = 0 Then
        GUI bcolour RGB(0,0,80),ExSetup_D
      Else
        GUI bcolour RGB(80,80,200),ExSetup_D
      EndIf
  end select
  if lcd_on = 0 then
    cpu 100
    backlight S_Bright
    lcd_on = 1
  endif
End Sub ' end of touch down subroutine
  '----------------------------------
  ' Touch up subroutine
Sub TouchUp ' used for User Setup textbox and also
  ' and also to operate any switches
  Select Case Touch(LASTREF)
    Case SetupOption_T ' textbox enter key pressed
      SetupAnsFlag% = 1
      TouchDownFlag% = 0
    Case SetupOption_N ' numberbox enter key pressed
      SetupAnsFlag% = 1
      TouchDownFlag% = 0
  End Select
End Sub ' end of touch up interrupt
  '----------------------------------
  'ESP8266 COM3 Interrupt
  '  received should be a string like this
  ' Sn, Fn, Ln or Pn  where n is 0=OFF, 1=ON
  sub ESP_COM3_Int ' from COM3 after a character received
    local tmp$,ESP_sw_stat$,ESP_sw$
    tmp$ = ""
    do while loc(#1) > 0 ' keep reading till empty
      tmp$ = tmp$ + Input$(1,#1)
    loop
    ESP_sw$ = left$(tmp$,1) ' switch touched
    ESP_sw_stat$ = mid$(tmp$,2,1) ' on or off ?
    select case ESP_sw$ ' select on switch
      case "S" ' switch data
        ESPRef% = 1
        SocketsFlag%=val(ESP_sw_stat$) ' should be 0 or 1
        Toggle SocketsFlag%
      case "F"
        ESPRef% = 2
        FridgeFlag%=val(ESP_sw_stat$) ' should be 0 or 1
        Toggle FridgeFlag%
      case "L"
        ESPRef% = 3
        LightsFlag%=val(ESP_sw_stat$) ' should be 0 or 1
        Toggle LightsFlag%
      Case "P"
        ESPRef% = 4
        PumpFlag%=val(ESP_sw_stat$) ' should be 0 or 1
        Toggle PumpFlag%
      Case "?"
        Update_ESP
    end select
    ESP_in_flag% = 1    
    tmp$ = Input$(8,#1) ' flush any junk
  end sub  'End of Interrupt handling routines
  '**************************************************************************
  ' console mode routine to service switch/relay control
  sub Con_Mode
    print "Console Mode"
    print "Enter switch letter followed by 0 (off) or 1 (on) - then ENTER key"
    print "Eg. For fridge, F1 will turn on relay"
    print "Enter ? to see switch status or X to exit console mode."
    print "ConMode> "
    do
      line input sw$
      select case left$(sw$,1)
        case "S"
          VSwitch% = 1
          if mid$(sw$,2,1) = "0" then
            SWSR% = 0
            PRINT "Turn sockets relay off"
            SocketsFlag% = 0
          else
            PRINT "Turn sockets relay on"
            SWSR% = 1
            SocketsFlag% = 0
          endif
          RlySetReset(VSwitch%,SwSR%)
          print "ConMode> "
        case "F"
          VSwitch% = 2      
          if mid$(sw$,2,1) = "0" then        
            SWSR% = 0
            PRINT "Turn fridge relay off"
            FridgeFlag% = 0
          else
            PRINT "Turn fridge relay on"
            SWSR% = 1
            FridgeFlag% = 1
          endif
          RlySetReset(VSwitch%,SwSR%)
          print "ConMode> "
        case "L"
          VSwitch% = 3
          if mid$(sw$,2,1) = "0" then
            PRINT "Turn lights relay off"
            LightsFlag% = 0
            SWSR% = 0
          else
            PRINT "Turn lights relay on"
            LightsFlag% = 1
            SWSR% = 1
          endif
          RlySetReset(VSwitch%,SwSR%)
          print "ConMode> "
        case "P"
          VSwitch% = 4
          if mid$(sw$,2,1) = "0" then
            PRINT "Turn pump relay off"
            PumpFlag% = 0
            SWSR% = 0
          else
            PRINT "Turn pump relay on"
            PumpFlag% = 1
            SWSR% = 1
          endif
          RlySetReset(VSwitch%,SwSR%)
          print "ConMode> "
        CASE "?"
          print "Current relays ON are -"
          if SocketsFlag% = 1 then print "Sockets ON",;
          if FridgeFlag% = 1 then  print "Fridge ON",;
          if LightsFlag% = 1 then  print "Lights ON",;
          if PumpFlag% = 1 then print "Pump ON"
          print "ConMode> "
        case "X"
          print "Exiting console mode"      
          exit sub
      end select
    loop
  end sub 
  '-------------------------------------------------
  ' Code to gather external data
  ' - in live situation, scan external data sources
Sub UpdateExtVals
    local CB_Tmp%,CBPwr_Tmp
    ' check ranges and adjust if necessary
    ' Fresh Water - nominal analog voltage 0V to 500mV at 12V supply
    ' 500mV(100) - full, 250mV(50) - half, 0V(0) - empty
    FWater = (CB_VoltsCal*Pin(FreshWaterPin))*460
    If FWater > 100 Then
      FWater = 100
    ElseIf FWater < 0 Then
      FWater = 0
    EndIf      
    ' code here to scan external data sources
    ' scan PAC1934 channels and assign volts, amps and power for each of 4 ch
    TmpCB_Volts =  CB_Volts ' save old  battery to temporary
    CB_Volts = int(VScale*PAC_AV%(3)*99.8)/100 ' average volts for camper battery
    CB_Tmp% = PAC_AA%(3)
    if CB_Tmp% > 32767 then
      CB_Tmp% = CB_Tmp% XOR &HFFFF
      CB_Tmp% = CB_Tmp% and &HFFFF
      CB_Tmp% = CB_Tmp% + 1
      CBA_NegFlag% = 1
    else      
      CBA_NegFlag% = 0 ' to tell display routine that CB_Amps are neg
    endif
    CB_Amps =  int(AScaleCh3*CB_Tmp%*100)/100 'amps to/from camper bat
    if CBA_NegFlag% = 1 then
      CB_Amps = -CB_Amps
    endif
    if FridgeFlag% = 1 or LightsFlag% = 1 then
      F_L_Amps =  int(AScaleCh2*PAC_AA%(2)*1000)/1000 'av ampsd fridge/lights
    else
      F_L_Amps = 0
    endif
    If PumpFlag% =1 or SocketsFlag% = 1 then
      P_S_Amps =  int(AScaleCh1*PAC_AA%(1)*1000)/1000 ' av amps pump/sockets
    else
      P_S_Amps = 0
    endif
    Load_Amps = (F_L_Amps+P_S_Amps) ' MM pwr included with pumps/sockets
    ExtPwr_Amps = (Load_Amps)+(CB_Amps)
    if ExtPwr_Amps < 0.1 then ' must be noise
      ExtPwr_Amps = 0
    endif
    ' do WHr calculations direct - not with PAC
    CBPwr_Tmp = CB_Volts * CB_Amps / 1800 ' battery charge in WHrs
    If CBPwr_Tmp > 0 then
      CBPwr_Tmp = CBPwr_Tmp * 0.95  ' assume 5% loss
    endif
    CB_Pwr = FixDP(CB_Pwr + CBPwr_Tmp,3) ' keep accumulating
    if CB_Pwr > 1200 then ' but limit to 12V * 100AHrs
      CB_Pwr = 1200  ' something seriously wrong if we get here
    endif
    if CB_Pwr < 0 then
      CBPwrNegFlag% = 1
    else
      CBPwrNegFlag% = 0
    endif
    CB_SOC% = CamperB_SOC%(CB_Volts)

    ' scan for external power, and action as per Ext Pwr switch
    TmpExt_Volts = ExtPwr_Volts ' save the old ext volts
    ExtPwr_Volts = int(pin(Ignition)*520)/100 ' average volts vehicle battery
    If ExtPwr_Volts > 12.6 and ExtPwr_Volts < 15 then ' must be receiving external power
      ExtPwrFlag% = 1 'power avilable
    else
      ExtPwrFlag% = 0 ' either none or out of acceptable voltage range
    endif
      
 	  select case ExtPwrMode%
      case 0 ' turn off regardless
        if pin(ExtPwr_R) = 1 then ' if on, turn off else leave off
          Pin(ExtPwr_R) = 0 ' 
        endif
      
      case 1 ' auto - turn on if valid ext pwr
        if ExtPwrFlag% = 1 then ' if ext power available, automatically turn on
          if pin(ExtPwr_R) = 0 then ' if off, turn on else leave on
            Pin(ExtPwr_R) = 1 ' 
          endif
        endif
      case 2 ' manual - turn on regardless
          if pin(ExtPwr_R) = 0 then ' if off, turn on else leave on
            Pin(ExtPwr_R) = 1 ' 
          endif
      
      case 3' solar
        ' turn on regardless, then test if solar kicks in
        if pin(ExtPwr_R) = 0 then ' if off, turn on else leave on
          Pin(ExtPwr_R) = 1 ' 
        endif
        ' solar needs to "see" the battery volts so try a few times
        ' to let the solar stabilise so compare the previous 2 sec battery volts
        ' with current solar volts
        if TmpCB_Volts > ExtPwr_Volts then ' solar hasn't kicked in yet
          SolPwrDly% = SolPwrDly% + 1 ' waiting for solar to "see" battery and stabilise
          ctrlval (Charge_D) = "HUNT"
          if SolPwrDly% > 10 then ' trying for 20 secs
          ' if not found, will go to LOST for 2 secs then try again next time
          ' thru 2 sec loop
            SolPwrDly% = 0
            ctrlval (Charge_D) = "LOST"
          endif
        else ' solar is good and greater than battery so
          SolPwrDly% = 0 ' all good
          ctrlval (Charge_D) = "SOLAR"
        endif
      end select
           
    NewDataFlag = 1 
End Sub

function CamperB_SOC% (CB_Volts)
  ' Calculate battery charge as percentage
  ' 12.1V equates to 30% charge - note this is a rough calculation
  '  - it should be done under no load.
  ' *** work needed here ***
  '  - this code attemps to simulate a discharge curve for the purposes
  '    of calculating state of charge under load
  local RemainBC,AverageBDChg,BatCharge
  local range = 29       ' shape of discharge curve
  local offset = 10.9   ' offset to position range of values
  local curve = 6       ' shape determined by discharge current
  ' ... some rule of thumb values to position values in the
  ' correct range and with a discharge curve approximating a discharge
  ' rate of between 4 and 10 amps
  '  curve = 10-(AverageLC/2.1)
  BatCharge = (((CB_Volts-offset)^2)*range)+ curve
  ' now set up to display charge graphically
  BatCharge = Int(BatCharge)
  If BatCharge < 1 Then BatCharge = 1
  If BatCharge > 100 Then BatCharge = 100  
  CamperB_SOC% = BatCharge
'  ' Calculate time to 30% discharge of battery
'  RemainBC = BatCharge - 30 ' rough convert from 30% - 100%
'  ' to 0Ahrs to 70AHrs remaining charge
'  If RemainBC < 1 Then
'    RemainBC = 1
'  EndIf
'  AverageBDChg = 10 ' assume an average discharge rate of 10AH
'  Time2Discharge = (RemainBC/AverageBDChg)*60  'if we have 70 AHrs and we
'  ' discharging at 10AHrs
'  ' then we have 7 hours of usage till 30%
'  ' - again these are only rough approximations
End function
  '------------------------------------
  ' GUI Update all values
  ' - calculate if alerts are needed
Sub GUIUpdateVals
  CtrlVal(FWater_G) = FWater
  If FWater < 20 Then
    GUI fcolour RGB(red),FWaterLow_C
  Else
    GUI fcolour RGB(0,180,0),FWaterLow_C
  EndIf        
  '------------------------------------
  ' now check battery volt and amp limits, alert if needed
  If ExtPwrFlag% = 1 Then
    CtrlVal(ExtPwr_Volts_G) = ExtPwr_Volts
    CtrlVal(CB_Volts_G) = CB_Volts
    CtrlVal(ExtPwr_Amps_G)  = ExtPwr_Amps
  Else 
    ' graphing battery display
    CtrlVal(ExtPwr_Volts_G) = 0
    CtrlVal(ExtPwr_Amps_G) = 0
  endif
  CtrlVal(CB_Volts_G) = CB_Volts
  CtrlVal(CB_SOC_G) = CB_SOC%
  if CBA_NegFlag% = 0 Then
    CtrlVal(CB_AmpsP_G) = CB_Amps
    CtrlVal(CB_AmpsN_G) = 0
  Else
    CtrlVal(CB_AmpsN_G) = CB_Amps    
    CtrlVal(CB_AmpsP_G) = 0
  EndIf
  'update running whrs
  if CBPwrNegFlag% = 0 Then
    CtrlVal(CB_PwrP_G) = CB_Pwr
    CtrlVal(CB_PwrN_G) = 0
  Else
    CtrlVal(CB_PwrN_G) = CB_Pwr    
    CtrlVal(CB_PwrP_G) = 0
  EndIf
  
  ' update the large value display
  ctrlval(LVBV_C) = str$(ExtPwr_Volts,2,2)+"V"
  ctrlval(LVBA_C) = str$(ExtPwr_Amps,2,2)+"A"
  if CB_Volts < 12.0 then
    gui fcolour rgb(red), LCBV_C
  elseif CB_Volts > 12.4 then
    gui fcolour rgb(green), LCBV_C
  else
   gui fcolour rgb(yellow),LCBV_C
  endif
  ctrlval(LCBV_C) = str$(CB_Volts,2,2)+"V"
  if CB_Amps < 0 then
    GUI FCOLOUR rgb(red),LCBA_C
    gui bcolour rgb(100,100,150),LCBA_C
  else
    GUI FCOLOUR rgb(green),LCBA_C
    gui bcolour rgb(100,100,150),LCBA_C
  endif  
  ctrlval(LCBA_C) = str$(CB_Amps,-2,2)+"A"   
  ' finally, heartbeat every 2 sec - no heartbeat no new vals
  if ctrlval(H_Beat_L) = 0 then
    ctrlval(H_Beat_L) = 1
  else
    ctrlval(H_Beat_L) = 0
  endif
End Sub
'  
'****************************************************************************
' ESP8266 related routines
sub Update_ESP
  ' send out switch status and data values to ESP
  local VWMstat$
  VWMstat$ = ""
  VWMstat$ = VWMstat$ +"S"+str$(SocketsFlag%)+","
  VWMstat$ = VWMstat$ +"L"+str$(LightsFlag%)+","
  VWMstat$ = VWMstat$ +"F"+str$(FridgeFlag%)+","
  VWMstat$ = VWMstat$ +"P"+str$(PumpFlag%)+","
  VWMstat$ = VWMstat$ +"V"+str$(CB_Volts+(rnd()+12),2,2)+","
  VWMstat$ = VWMstat$ +"C"+str$(ExtPwr_Amps+(rnd()+6),2,2)+","
  VWMstat$ = VWMstat$ +"D"+str$(Load_Amps+((rnd()-0.5)+6),2,2)+","
  VWMstat$ = VWMstat$ +"B"+str$(CB_Amps+((rnd()-0.5)*10),2,2)
  print #1,VWMstat$
end sub
'--------------------------------
' I/O Related routines - latch relays and PAC1934 actions
  '------------------------------
  ' Switch/Relay service routine
  ' - will set/reset relays based on GUI or web switch touch
  ' - enter with Sw_No% 1 thru 4 representing
  ' 1 - sockets
  ' 2 - fridge
  ' 3 - lights
  ' 4 - pump
sub Sw_Rly_Service
  select case VSwitch%  
    case 1
      Toggle SocketsFlag%
      If SocketsFlag% = 0 Then
        GUI bcolour RGB(0,90,0),Sockets_D
        SwSR% = 0
      Else
        GUI bcolour RGB(0,255,0),Sockets_D
        SwSR% = 1
      EndIf
      RlySetReset(VSwitch%,SwSR%)
    Case 2
      Toggle FridgeFlag%
      If FridgeFlag% = 0 Then
        GUI bcolour RGB(0,90,0),Fridge_D
        SwSR% = 0
      Else
        GUI bcolour RGB(0,255,0),Fridge_D
        SwSR% = 1
      EndIf
      RlySetReset(VSwitch%,SwSR%)
    Case 3
      Toggle LightsFlag%
      If LightsFlag% = 0 Then
        GUI bcolour RGB(0,90,0),Lights_D
        SwSR% = 0
      Else
        GUI bcolour RGB(0,255,0),Lights_D
        SwSR% = 1
      EndIf
      RlySetReset(VSwitch%,SwSR%)
    Case 4
      Toggle PumpFlag%
      If PumpFlag% = 0 Then
        GUI bcolour RGB(0,90,0),Pump_D
        SwSR% = 0
      Else
        GUI bcolour RGB(0,255,0),Pump_D
        SwSR% = 1
      EndIf
      RlySetReset(VSwitch%,SwSR%)
  end select
  Update_ESP ' for MM touch change
end sub
  '------------------------------
  ' Relay set and reset routines
Sub RlySetReset(VSwitch%,SwSR%) ' vswitch is the physical sw 1 thru 4
  ' swsr is 1 for set, 0 for reset
  '  switches numbered from 1 to 4, left, top to bottom
  ' **Note: switches ExtPwr, Gen Kill and LCD Off are managed differently.
  ' if switch is on (ctrlval of 0),  set relay off, pulse latch relay off
  Local RlyBrd%, Rly% '  
  ' now convert relay number to board number, relay number plus set/reset
  Rly% = (VSwitch% - 1) mod 4  ' now 0 through 3 to select relay
  Rly% = &B111 AND (((VSwitch% -1) mod 4) * 2 + SwSR%) ' add in set/reset
  ' not used currently - only 1 relay board in use
  Port(LRlySel0,1,LRlySel1,1,LRlySel2,1) = Rly% ' MM 59, 55 and 54
  Pulse LRB0,10 ' 10 mS active low pulse to set or reset latch pin 52
End Sub
  ' end of relay control routines
'--------------------------------
' PAC1934 functions to read voltage, current and power
  ' CONST I2COpt% = &B00000000
  ' I2C instructions:-
  ' I2C OPEN speed, timeout ' default 100kHz, 100mSec
  ' I2C WRITE addr, option, send len, send data[,send data, ...]
  ' I2C READ addr, option,recv len,recv buff  - recv buff can be an array
  ' I2C CLOSE
  '   - in read or write above, option is always 0
  '  
sub InitPAC1934  ' initialise PAC1934
  i2c open 100,100  'speed = 100kHz, timeout 100mSec
  i2c write PACAddr%,0,1,PAC_Refresh%
  pause 10
  I2C write PACAddr%,0,2,PAC_CtrlReg%,PAC_Samples% 'set rate at 64 samps/s
  pause 10
  I2C write PACAddr%,0,2,PAC_NegPwr%,PAC_Ch1BiDi% ' set Ch 1 to bidirectional
  pause 10
  i2c close
end sub
'
sub ClearAccum ' clears acummulator
  i2c open 100,100  'speed = 100kHz, timeout 100mSec
  i2c write PACAddr%,0,1,PAC_Refresh%
  pause 10
  i2c close
end sub  
'
function PAC_AccPwr%(PAC_Channel)  ' return accumulated from PAC1934
' Note: each channel should be done straight after each other then 
' read accumulator count followed by a ClearAccum to reset value and
' restart accum count
  local a%(6) ' returns a 48 bit number (6 bytes)
  i2c open 100,100
  i2c write PACAddr%,0,1,PAC_Refresh_V%
  pause 10
  I2C write PACAddr%,0,1,PAC_PAccBase% +(PAC_Channel-1)
  I2C read PACAddr%,0,6,a%()
  i2c close
  PAC_AccPwr%=(((a%(0)<<56)+(a%(1)<<48)+(a%(2)<<40)+(a%(3)<<32)+(a%(4)<<24)+a%(5)<<16))\2^16
end function

function PAC_ACCC%(arg1%)  ' return accumulator count from PAC1934
  local a%(3),tmp%
  i2c open 100,100
  i2c write PACAddr%,0,1,PAC_Refresh_V%
  pause 10
  I2C write PACAddr%,0,1,PAC_ACC_Count%
  I2C read PACAddr%,0,3,a%()
  i2c close
  tmp% = (a%(0)<<16)+(a%(1)<<8)+a%(2) 
  PAC_ACCC% = tmp%
end function


function PAC_AV%(PAC_Channel) 'return voltage from PAC1934 PAC_Channel
  local a(2)
  i2c open 100,100
  i2c write PACAddr%,0,1,PAC_Refresh_V%
  pause 10
  I2C write PACAddr%,0,1,PAC_VAvgBase%+(PAC_Channel-1)
  I2C read PACAddr%,0,2,a()
  i2c close
  PAC_AV% = (a(0)<<8)+a(1) 
end function
  '
function PAC_AA%(PAC_Channel)  ' return current from PAC1934  PAC_Channel
  local a(2)
  i2c open 100,100
  i2c write PACAddr%,0,1,PAC_Refresh_V%
  pause 10
  I2C write PACAddr%,0,1,PAC_AAvgBase%+(PAC_Channel-1)
  I2C read PACAddr%,0,2,a()
  i2c close
  PAC_AA% = (a(0)<<8)+a(1)
end function
  '
function PAC_AP%(PAC_Channel) ' return power from PAC1934 channel PAC_Channel
  local a(3)
  local i
  i2c open 100,100
  i2c write PACAddr%,0,1,PAC_Refresh_V%
  pause 10
  I2C write PACAddr%,0,1,PAC_PAvgBase%+(PAC_Channel-1)
  I2C read PACAddr%,0,4,a()
  i2c close
  PAC_AP%=(a(0)<<16)+(a(1)<<8)+a(2)
end function
' end of PAC routines
'****************************************************************************
' GUI Initialisation
' - this is only called once but more manageable in a subroutine
Sub InitGUI  ' initialise graphics
GUI setup 1 ' initial startup and settings screen
  GUI displaybox Startup_D,0,0,799,400,RGB(white),RGB(0,0,150)
  Font 5
  GUI caption Startup1_C,Startup1$,400,80,C,RGB(white),RGB(0,0,150)
  Font 1
  GUI caption Startup2_C,Startup2$,400,170,C,RGB(white),RGB(0,0,150)
  Font 2
  GUI caption Startup3_C,Startup3$,400,270,C,RGB(white),RGB(0,0,150)
  '  
GUI setup 2 ' help box - bottom center (date/time)
  Font 3
  GUI displaybox Help_D,135,410,530,69,RGB(white),RGB(100,100,150)
  '
GUI setup 3 ' main display page - switches and water
  Font 2
'  GUI displaybox FWater_D,684,130,115,265,RGB(white),RGB(60,60,60)
  GUI displaybox FWater_D,684,195,115,205,RGB(white),RGB(60,60,60)
  CtrlVal(FWater_D) = "~~100%   ~~~~~25%   ~~~~0%   ~~"
  font 4
  gui caption FWaterLow_C,"WATER",740,200,C,rgb(white),rgb(60,60,60)
  GUI BARGAUGE FWater_G,760,240,20,155,RGB(white),RGB(black),0,100,RGB(red),25,RGB(100,100,150)
  GUI displaybox Sockets_D,0,5,120,70,RGB(white),RGB(0,80,0)' switches
  GUI displaybox Fridge_D,0,100,120,70,RGB(white),RGB(0,80,0)
  GUI displaybox Lights_D,0,195,120,70,RGB(white),RGB(0,80,0)
  GUI displaybox Pump_D,0,290,120,70,RGB(white),RGB(0,80,0)
  GUI displaybox GenKill_D,680,410,119,69,RGB(white),RGB(80,30,25)
  gui displaybox Charge_D,0,410,120,69,rgb(white),rgb(200,0,0)
  gui displaybox LCD_Off_D,680,0,119,70,rgb(white),rgb(80,30,25)
  GUI displaybox Setup_D,680,100,120,70,RGB(white),RGB(0,0,80)
  GUI AREA Sockets_A,0,5,120,70
  GUI AREA Fridge_A,0,100,120,70
  GUI AREA Lights_A,0,195,120,70
  GUI AREA Pump_A,0,290,120,70
  GUI AREA GenKill_A,680,410,119,69
  gui AREA Charge_A,0,410,120,69
  gui AREA LCD_Off_A,680,5,119,70
  GUI AREA Setup_A,680,100,120,70
  ctrlval(Sockets_D) = "Sockets"  ' switches
  ctrlval(Fridge_D) = "Fridge"
  ctrlval(Lights_D) = "Lights"
  ctrlval(Pump_D) = "Pump"
  ctrlval(GenKill_D) = "Gen~Kill"  
  ctrlval(Charge_D) = "Charge~Enable"
  ctrlval(LCD_Off_D) = "LCD~ON"
  ctrlval(Setup_D) = "Setup"
'
gui setup 4 ' large battery values display and gauges,heartbeat led
  Font 4
  gui displaybox LCB_D,135,0,255,170,rgb(white),rgb(100,100,150)
  gui caption LCB_C,"Camper Battery",260,10,C,rgb(white),rgb(100,100,150)
  gui displaybox LVB_D,410,0,255,170,rgb(white),rgb(100,100,150)
  gui caption LVB_C,"External Power",540,10,C,rgb(white),rgb(100,100,150)
  font 4,2
  gui caption LCBV_C,"",260,40,C,rgb(white),rgb(100,100,150)
  gui caption LCBA_C,"",260,100,C,rgb(white),rgb(100,100,150)
  gui caption LVBV_C,"",540,40,C,rgb(white),rgb(100,100,150)
  gui caption LVBA_C,"",540,100,C,rgb(green),rgb(100,100,150)
' battery graphs
  font 4
  GUI displaybox CamperB_D,135,195,255,205,RGB(white),RGB(60,60,60)
  CtrlVal(CamperB_D) = "Camper Battery~~~~~~~"
  GUI displaybox ExtPwr_D,410,195,255,205,RGB(white),RGB(60,60,60)
  CtrlVal(ExtPwr_D) = "External Power~~~~~~~"
  Font 1
  GUI caption CB_Volts_C,"11.5V   .    13.0V   .    14.5V",263,240,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE CB_Volts_G,136,260,254,20,RGB(white),RGB(black),11.5,14.5,RGB(red),12.2,RGB(220,220,0),12.5,RGB(0,220,0)
  GUI caption CB_Amps_C,"-10A    .     0A     .    +10A",263,300,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE CB_AmpsP_G,263,320,127,20,RGB(white),RGB(black),0,10,RGB(green)
  GUI BARGAUGE CB_AmpsN_G,136,320,127,20,RGB(white),RGB(red),-10,0,RGB(black)
  GUI caption CB_SOC_C,"0%     25%   50%   75%   100%",263,350,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE CB_SOC_G,136,370,254,20,RGB(white),RGB(black),0,100,RGB(red),50,RGB(220,220,0),65,RGB(0,220,0)
  GUI caption ExtPwr_Volts_C,"11.5V   .    13.0V    .   14.5V",540,240,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE ExtPwr_Volts_G,410,260,255,20,RGB(white),RGB(black),11.5,14.5,RGB(red),12.2,RGB(220,220,0),12.5,RGB(0,220,0)
  GUI caption ExtPwr_Amps_C,"0A     .     10A     .    +20A",540,300,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE ExtPwr_Amps_G,410,320,255,20,RGB(white),RGB(black),0,20,RGB(green)
  GUI caption CB_Pwr_C,"-10    -5      0      5      10",540,350,C,RGB(white),RGB(60,60,60)
  GUI BARGAUGE CB_PwrP_G,537,370,127,20,RGB(white),RGB(black),0,10,RGB(green)
  GUI BARGAUGE CB_PwrN_G,410,370,127,20,RGB(white),RGB(red),-10,0,RGB(black)
  gui led H_Beat_L,"",400,182,10,rgb(green) 
  '  
GUI setup 5 'configuration screen
  Font 3
  GUI frame Setup_F," Setup User Options ",135,10,530,395,RGB(white)
  GUI bcolour RGB(100,100,100),Setup_F
  Font 2
  GUI caption SetupHelp_C,"",400,100,c,RGB(white),RGB(black)
  GUI caption SetupOption_C,"",400,270,c,RGB(white),RGB(black)
  Font 2,2
  GUI textbox SetupOption_T,140,345,300,57,RGB(white),RGB(50,50,100)
  GUI numberbox SetupOption_N,140,345,300,57,RGB(white),RGB(50,50,100)
  Font 4
  GUI displaybox ExSetup_D,546,335,119,70,RGB(white),RGB(0,0,80)
  GUI AREA ExSetup_A,546,335,119,70
  ctrlval(ExSetup_D) = "Exit~SetUp"

gui setup 6 ' dummy to end setups
  ' end of GUI setups
End Sub
  '
  ' End of GUI subroutines
'****************************************************************************
Sub UserSetup  ' - got here after setup button pressed
  ' handle setup questions and answers
  Local sq$,t$,t_valid%,d$,d_valid%,lr_valid%,OBR1$
  Page 2,5 ' change to setup gui page
  TouchDownFlag% = 0
  ExSetupFlag% = 0 ' set by if exit switch touch
  sq$ = "Touch the box below to enter setup option"
  sq$ = sq$ + "~- if value is already correct,"
  sq$ = sq$ +"~just touch Enter to continue"
  sq$ = sq$ +"~Software version: "+F_ver$
  CtrlVal(SetupHelp_C) = sq$
  gui disable SetupOption_N ' hide numberbox for now
  gui enable SetupOption_T  ' show the text box
  do
    t_valid% = 1
    SetupAnsFlag% = 0
    t$ = time$
    CtrlVal(SetupOption_C) = String$(39," ")
    CtrlVal(SetupOption_C) = "The current time is "+t$
    CtrlVal(SetupOption_T) = "##"+"HH:MM:SS"
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    t$ = ctrlval(SetupOption_T)
    if t$ = "" then
      t$ = time$
    endif
    on error skip 1
    time$ = t$
    if mm.errno <> 0 then t_valid% = 0
  loop until t_valid% = 1
  do
    d_valid% = 1
    SetupAnsFlag% = 0
    d$ = date$
    CtrlVal(SetupOption_C) = String$(40," ")
    CtrlVal(SetupOption_C) = "The current date is "+d$
    CtrlVal(SetupOption_T) = "##"+"DD-MM-YY"
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    d$ = ctrlval(SetupOption_T)    
    if d$ = "" then
      d$ = date$
      endif
    on error skip
    date$ = d$
    if mm.errno <> 0 then d_valid% = 0
  loop until d_valid% = 1
  gui disable SetupOption_T ' done with setup textbox
  ' update RTC
  convert_time
  rtc settime my_year%,my_month%,my_day%,my_hour%,my_min%,my_sec%
  gui enable SetupOption_N ' now use numberbox
  CtrlVal(SetupOption_C) = "There are 4 shunts, 1 for each PAC channel"
  pause 2000
  do
    d_valid% = 1
    SetupAnsFlag% = 0
    CtrlVal(SetupOption_C) = String$(40," ")
    CtrlVal(SetupOption_C) = "PAC Ch1 shunt is "+str$(Ch1Shunt,1,3)+"Ohms"
    CtrlVal(SetupOption_N) = "##"+str$(Ch1Shunt,1,3)
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    if ctrlval(SetupOption_N) <> 0 then
      Ch1Shunt = FixDP(ctrlval(SetupOption_N),3)
    endif
    if Ch1Shunt < 0.004 or Ch1Shunt > 0.010 then
      d_valid% = 0
    else
      d_valid% = 1
    endif
  loop until d_valid% = 1
  do
    d_valid% = 1
    SetupAnsFlag% = 0
    CtrlVal(SetupOption_C) = String$(40," ")
    CtrlVal(SetupOption_C) = "PAC Ch2 shunt is "+str$(Ch2Shunt,1,3)+"Ohms"
    CtrlVal(SetupOption_N) = "##"+str$(Ch2Shunt,1,3)
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    if ctrlval(SetupOption_N) <> 0 then
      Ch2Shunt = FixDP(ctrlval(SetupOption_N),3)
    endif
    if Ch2Shunt < 0.004 or Ch2Shunt > 0.010 then
      d_valid% = 0
    else
      d_valid% = 1
    endif
  loop until d_valid% = 1
  do
    d_valid% = 1
    SetupAnsFlag% = 0
    CtrlVal(SetupOption_C) = String$(40," ")
    CtrlVal(SetupOption_C) = "PAC Ch3 shunt is "+str$(Ch3Shunt,1,3)+"Ohms"
    CtrlVal(SetupOption_N) = "##"+str$(Ch3Shunt,1,3)
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    if ctrlval(SetupOption_N) <> 0 then
      Ch3Shunt = FixDP(ctrlval(SetupOption_N),3)
    endif
    if Ch3Shunt < 0.004 or Ch3Shunt > 0.010 then
      d_valid% = 0
    else
      d_valid% = 1
    endif
  loop until d_valid% = 1
  do
    d_valid% = 1
    SetupAnsFlag% = 0
    CtrlVal(SetupOption_C) = String$(40," ")
    CtrlVal(SetupOption_C) = "PAC Ch4 shunt is "+str$(Ch4Shunt,1,3)+"Ohms"
    CtrlVal(SetupOption_N) = "##"+str$(Ch4Shunt,1,3)
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    if ctrlval(SetupOption_N) <> 0 then
      Ch4Shunt = FixDP(ctrlval(SetupOption_N),3)
    endif
    if Ch4Shunt < 0.004 or Ch4Shunt > 0.010 then
      d_valid% = 0
    else
      d_valid% = 1
    endif
  loop until d_valid% = 1
  do ' code here to select onboard relay as latching or non latching
    lr_valid% = 1
    SetupAnsFlag% = 0
    CtrlVal(SetupOption_C) = String$(39," ")
    CtrlVal(SetupOption_C) = "On board relay 1~1 for latching, 0 for standard "
    CtrlVal(SetupOption_T) = "##"+"1 or 0"
    Do While SetupAnsFlag% = 0 ' wait on textbox touch
      If ExSetupFlag% = 1 Then
        Exit sub 'Do
      EndIf
    Loop
    if ctrlval(SetupOption_N) = 1 then
      OBRly1Flag% = 1 ' onboard relay 1 is latching
    else
      OBRly1Flag% = 0
    endif
  loop until lr_valid% = 1
  gui disable SetupOption_N ' hide numberbox
End Sub ' end of user setup
'*******************************************************************
  ' General Functions and Subroutines
  '------------------------------------
Function Field$(s As String, n As Integer) ' Geoff's field$ function returns
  ' the field number n (starts at 1) in the string s as a substring - eg.
  '  B$ = "abc,def,ghi"
  '  A$ = Field$(B$,2)
  ' will return A$ as "def"
  Local delim$ = ","           ' set this to the field delimiter
  Local Integer i, StartIdx = 1, EndIdx
  ' get the start of the field in StartIdx
  For i = 2 To n
    StartIdx = Instr(StartIdx, s, delim$) + 1
    If StartIdx = 1 Or StartIdx > Len(s) Then Exit Function
  Next i
  ' get the end of the field in EndIdx and extract the field
  EndIdx = Instr(StartIdx, s, delim$)
  If EndIdx = 0 Then EndIdx = 255
  Field$ = Mid$(s, StartIdx, EndIdx - StartIdx)
  ' trim leading and trailing spaces
  Do While Left$(Field$, 1) = " "
    Field$ = Mid$(Field$, 2,Len(Field$)-1)
  Loop
  Do While Right$(Field$, 1) = " "
    Field$ = Mid$(Field$, 1, Len(Field$) - 1)
  Loop
End Function ' end field function
'-------------------------
  ' toggle arg - if zero set arg to not zero, if not 0, set arg to zero
Sub Toggle(arg1%)
  If arg1% = 0 Then
    arg1% = Not arg1%
  Else
    arg1% = 0
  EndIf
End Sub ' end toggle
  '--------------------
  ' fix number of decimal places
Function FixDP(num2fix,dplaces)
  num2fix = Int(num2fix *(10^dplaces))
  FixDP = num2fix/(10^dplaces)
End Function
'----------------------
  ' Date conversion routines EPOCH time to human time and vice versa
  ' - courtesy Peter Mather
  ' demo -------------
  'dim nowtime%
  'convert_time() 'convert time$ and date$ to Unix epoch time
  'print format_time$()
  'print my_timestamp%
  'nowtime% = my_timestamp% +240
  'gmtime(nowtime%) ' re-build date time from epoch time
  'print format_time$()
  'print format_date$()
  ' demo -------------
  '
sub convert_time() 'loads global time variables from time$ and date$
  my_year%= val(right$(date$,4))
  my_month%=val(mid$(date$,4,2))  
  my_day%=val(left$(date$,2))
  my_hour%=val(left$(time$,2))
  my_min%=val(mid$(time$,4,2))
  my_sec%=val(right$(time$,2))
  my_timestamp%=timegm%(my_year%, my_month%, my_day%, my_hour%, my_min%, my_sec%)
  daycalcs(my_timestamp%)
end sub
  '
function is_leap_year%(year%)
  is_leap_year% = 0
  if year% mod 4 = 0 then is_leap_year%=1
  if year% mod 100 = 0 then is_leap_year%=0
  if year% mod 400 = 0 then is_leap_year%=1
end function
  '
function leap_days%(yy1%, yy2%)
  local integer y1%=yy1%, y2%=yy2%
  y1%=y1%-1
  y2%=y2%-1
  leap_days%= (y2%\4 -y1%\4) - (y2%\100 -y1%\100) + (y2%\400 -y1%\400)
end function
  '
function timegm%(tm_year%, tm_mon%, tm_mday%, tm_hour%, tm_min%, tm_sec%)
  local year%, days%, hours%, minutes%
  year% = tm_year%
  days% = 365 * (year%-1970) + leap_days%(1970,year%)
  days%=days% + monoff%(tm_mon%-1)
  if tm_mon% >1 and is_leap_year%(year%) then days%=days%+1
  days% = days% + tm_mday% -1
  hours% = days% * 24 + tm_hour%
  minutes% = hours% * 60 + tm_min%
  timegm% = minutes% * 60 + tm_sec%
end function
  '
sub daycalcs(tm_seconds% ) 'calculate day of week and day in year
  local  year% = EPOCH_YR%, dayno% = tm_seconds%\SECS_DAY%
  my_dayno% = (dayno% + 4) mod 7
  do while dayno% >= yearsize%(year%)
    dayno% = dayno%-yearsize%(year%)
    year% = year%+1
  loop
  my_dayinyear% = dayno%
end sub
  '
sub gmtime(tm_seconds%) 'calculate all parameters from the epoch time
  local year% = EPOCH_YR%, dayclock% = tm_seconds% mod SECS_DAY%, dayno% = tm_seconds%\SECS_DAY%, tm_mon%=0
  my_sec% = dayclock% mod 60
  my_min% = (dayclock% mod 3600) \60
  my_hour% = dayclock% \ 3600
  my_dayno% = (dayno% + 4) mod 7
  do while dayno% >= yearsize%(year%)
    dayno% = dayno% - yearsize%(year%)
    year% = year%+1
  loop
  my_year% = year%
  my_dayinyear% = dayno%
  do while dayno% >= ytab1%(tm_mon%)
    dayno% = dayno% - ytab1%(tm_mon%)
    if tm_mon% = 1 and is_leap_year%(year%) then dayno% = dayno%-1
    tm_mon% = tm_mon%+1
  loop
  my_month% = tm_mon%+1
  my_day% = dayno%+1
end sub
  '
function yearsize%(year%)
  yearsize%=365
  if is_leap_year%(year%) then yearsize%=366
end function
  '
function format_time$()
  format_time$=str$(my_hour%,2)+":"+right$(str$(my_min%+100),2)+":"+right$(str$(my_sec%+100),2)
end function
  '
function format_date$()
  format_date$ = daysofweek$(my_dayno%)+" "+str$(my_day%,2)+th$(my_day%-1)+", "+monthsinyear$(my_month%-1)+" "+str$(my_year%,4)
end function
  '
Function MyAMPM_Time$(my_hour%,my_min%) 'convert time to am or pm
  Local h%
  h% = my_hour%
  If my_hour% > 12 Then h%=h%-12
  MyAMPM_Time$ = Str$(h%)+":"+Str$(my_min%,2,0,"0")
  If my_hour% =< 12 Then
    MyAMPM_Time$ = MyAMPM_Time$+" AM"
  Else
    MyAMPM_Time$ = MyAMPM_Time$+" PM"
  EndIf
End Function ' end myampm_time function
  ' end of EPOCH date and time subs/functions
'END OF PROGRAM *************************************************************