/*
  inverter program, bring your own 1/2 bridge gate drivers.
  Outputs are pin 9 and pin 10
  Gate drive enable pin 5, logic HIGH
  Fan control, pin 12
*/

#include <avr/pgmspace.h>
const char m_0[] PROGMEM = "0 - unused";  
const char m_1[] PROGMEM = "1 - unused";
const char m_2[] PROGMEM = "2 - cal.  DC volts      ";
const char m_3[] PROGMEM = "3 - unused";
const char m_4[] PROGMEM = "4 - cal.  DC current    ";
const char m_5[] PROGMEM = "5 - unused";
const char m_6[] PROGMEM = "6 - cal.  AC output     ";  
const char m_7[] PROGMEM = "7 - set AC set point    ";
const char m_8[] PROGMEM = "8 - unused";
const char m_9[] PROGMEM = "9 - set AC output freq. ";
const char m_A[] PROGMEM = "A - Restart voltage     ";
const char m_B[] PROGMEM = "B - fan ON temp         ";
const char m_C[] PROGMEM = "C - fan OFF temp        ";  
const char m_D[] PROGMEM = "D - shutdown temp       ";
const char m_E[] PROGMEM = "E - AC output low limit ";
const char m_F[] PROGMEM = "F - LV cutoff voltage   ";
const char m_G[] PROGMEM = "G - over current limit  ";
const char m_H[] PROGMEM = "H - enable/disable DC LV cutoff         ";
const char m_I[] PROGMEM = "I - enable/disable over current cutoff  ";  
const char m_J[] PROGMEM = "J - enable/disable AC under volt cutoff ";
const char m_K[] PROGMEM = "K - enable/disable over temp cutoff     ";
const char m_L[] PROGMEM = "Z - zero E2PROM";
const char m_M[] PROGMEM = "Y - put defaults into E2PROM";

const char s1[] PROGMEM = "";
const char s2[] PROGMEM = "DC cal. Enter volts";
const char s3[] PROGMEM = "";
const char s4[] PROGMEM = "DC current cal. Enter amps";
const char s5[] PROGMEM = "";
const char s6[] PROGMEM = "AC output cal. Enter volts";
const char s7[] PROGMEM = "AC setpoint  Enter volts";
const char s8[] PROGMEM = "";
const char s9[] PROGMEM = "AC output frequency (integer, 50 or 60)";
const char sa[] PROGMEM = "Restart after LV stop. Enter Volts";
const char sb[] PROGMEM = "Fan ON temperature (degC)";
const char sc[] PROGMEM = "Fan OFF temperature (degC)";
const char sd[] PROGMEM = "Shut down temperature (degC)";
const char se[] PROGMEM = "AC output Low Limit (percent, e.g. 90)";
const char sf[] PROGMEM = "LV shut down voltage. Enter volts";
const char sg[] PROGMEM = "Over current limit. Enter Amps";
const char sh[] PROGMEM = "DC LV cutoff (1 = enable,0 = disable)";
const char si[] PROGMEM = "Overcurrent cutoff (1 = enable,0 = disable)";
const char sj[] PROGMEM = "AC under volt cutoff (1 = enable,0 = disable)";
const char sk[] PROGMEM = "over temp cutoff (1 = enable,0 = disable)";

char pm_buf[60];  // above progmem ms can not exceed the size of this buffer.

const char * const s_t[] PROGMEM =    
{   
  m_0,  m_1,  m_2,  m_3,  m_4,  m_5,  m_6,  m_7,  m_8,  m_9,  m_A,  m_B,
  m_C,  m_D,  m_E,  m_F,  m_G,  m_H,  m_I,  m_J,  m_K,  m_L,  m_M,  s1,
  s2,  s3,  s4,  s5,  s6,  s7,  s8,  s9,  sa,  sb,  sc,  sd,  se,  sf,
  sg,  sh,  si,  sj,  sk  
 };

#include <EEPROM.h>
#include <Wire.h>
#include <LiquidCrystal_PCF8574.h>
LiquidCrystal_PCF8574 lcd(0x27);

#define ADC_OUT  0          // channel for AC output sample
#define ADC_DCI 3           // DC current sense
#define ADC_HS  2           // Heat sink temp sensor
#define ADC_DCV 1           // DC supply volts sample


#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))

#define NPWM  200         // 200 for this and 800 for PPWM for 50Hz and 20kHz PWM. Both must be integers.
//#define PPWM  665       // 60.062 Hz - choose one or the other.
//#define PPWM  799       // 50.002 Hz

// these are produced from ADC counts multiplied by scale factors. See loop() , parse_it(), show_menu()  
float ac_setpoint, ac_output, dcv, dci, hs_temp;

typedef struct  {
float dc_v_cal,
      dc_i_cal,
      ac_v_cal,
      ac_setpoint;
float fan_on_temp,
      fan_off_temp,over_temp,
      low_limit_pc,lv_cutoff,
      overcurrent, lv_restart;
uint16_t freq;
uint8_t lv_enabled,
      oi_enabled,
      uv_enabled,
      ot_enabled,
      mains_sync_enabled,
      e2prom_ok;
} _nvdata;
      
_nvdata nvd;
      
uint8_t uf,oen,v1low, serial_enabled, lv_stopped;
uint16_t pcount,ppwm;
uint32_t vpwr;
int sst;
uint16_t l[NPWM+1];
float pwr,ch2,ch0;
uint8_t dbounce;
double Input, Output, Setpoint;
double errSum, lastErr,error,dErr;
double kp, ki, kd;
float xv[5],yv[5];

char lcdbuf[82],*lbp,bufc;
String spstr = "";
uint8_t got_data;

float get_float(void);
int get_int(void);
float get_temp(int);
void init_vars(void);

void setup()
  {
  int i;
  float t,u;

  for(t=0.0,i=0; i <= NPWM; i++,t +=   3.14159/ (float) (NPWM+1))
    {   // 1/2 wave sine lookup table, scaled to (16 bits - 1)
    u = 65535.0 * sin(t);
    l[i] =  (int)u;  
    }
 
  pinMode(9, OUTPUT);     // V1 drive
  pinMode(10, OUTPUT);    // V2 drive
  pinMode(12,OUTPUT);     // fan
  pinMode(8,INPUT_PULLUP);// on/off button 
  DDRD = 0xff;   // can NOT let arduino system alter any pins from D0 to D7 via calls to pinMode() etc.. It buggers d5 which I need to pull down low-Z upon boot
  noInterrupts();

  TCCR1A = _BV(COM1A1) | _BV(COM1B0) | _BV(COM1B1) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
  OCR1A = 10;
  ICR1 = ppwm;
  TIMSK1 |= (1 << TOIE1);
  
  cbi(PORTD,5);   // disable gate drive output. data direction flag set by above statement.
  oen = 0;      // output enable. = 1 to run inverter, = 0 to stop inverter
  init_vars();
  
  interrupts();             // enable all interrupts
  TIMSK0 = 0;               // except arduino clock

  sbi(ADCSRA,ADPS2) ;   // ADC clock prescale = 16 now. this saves 100us in main loop with little cost to loop stability
  cbi(ADCSRA,ADPS1) ;
  cbi(ADCSRA,ADPS0) ;
  
  lcdinit();
  Serial.begin(115200);
  }


ISR(TIMER1_OVF_vect)        
 {    // 20Khz SPWM, code has only 50uS to run, takes about 10uS
  long c;
  
  c = (l[pcount] * vpwr) >> 16;   // scale sin wave by vpwr 32 bit integer calcs.
  if (v1low == 1)
    OCR1B = c;
  else
    OCR1A = c;

  pcount++;
  if(pcount >= NPWM)
    {
    pcount = 0;
    uf=1;
    if (v1low == 1)
      {
       v1low = 0; 
       TCCR1A =  _BV(COM1A1) |  _BV(WGM11);
       sbi(PORTD,7);  // DSO sync pulse
       cbi(PORTD,7);  
      }
    else
      {
       v1low = 1;
       TCCR1A =   _BV(COM1B1) | _BV(WGM11);
      }
    }
}


void init_vars()      // called prior to setting oen = 1, as well as from setup()
  {
  ppwm = 799;   // in case freq is not yet set in E2PROM. It must be something reasonable  
  if (nvd.freq > 40 && nvd.freq < 101)
    ppwm = 40000/nvd.freq;    // it really is that simple..
  ICR1 = ppwm;
  
  lv_stopped = 0;  
  sst = 0;      // slow start counter 0 - 251
  v1low = 1;    // half wave output flag, 0 or 1
  pcount = 0;   // counter for sine wave lookup table
  uf = 0;       // set to 1 to enable AC output voltage feedback code to execute
  pwr = 0.0;    // PWM duty cycle analog, 0.0 to 1.0
  vpwr= 0;      // integer equivalent of pwr, 0 to PPWM
  dbounce = 0;    // debounce counter for input switch 
  serial_enabled = 0;  // = 1 if port open, = 0 if closed
  got_data = 0;
  get_nvdata();
  // set up PID 
  kp = 0.1/1000.0;   
  ki = 0.0;
  kd = 0.01/1000.0;
  errSum = 0.0;
  lastErr = 0.0;
  lbp = &lcdbuf[1];   // point local pointer to lcd buffer, used by strncpy(), sprintf()
  // clear error if needed
  strncpy(lbp+40,    "                    ",20);
  }


void do_pid()
{   
   double timeChange = 0.01;    // always called at 100Hz, so dT will be 10msec..
   error = Setpoint - Input;
   errSum += (error * timeChange);
   dErr = (error - lastErr) / timeChange;
   Output = kp * error + ki * errSum + kd * dErr;
   lastErr = error;
}


float get_temp(int Vo) 
 {    // https://circuitdigest.com/microcontroller-projects/arduino-thermistor-interfacing-code-circuit
 float a = 1.009249522e-03, b = 2.378405444e-04, c = 2.019202697e-07;
 float T,logRt,Tc;
 logRt = log(10000.0*((1024.0/Vo-1))); 
 T = (1.0 / (a + b*logRt + c*logRt*logRt*logRt)); // We get the temperature value in Kelvin from this Stein-Hart equation
 Tc = T - 273.15;                                 // Convert Kelvin to Celsius
 return Tc;
}
 
void loop()
{
  float dcw,fch0;
  if(uf == 1  )    // this will execute at 100Hz
      {            // it runs at near zero volt crossing of AC output     
      sbi(PORTD,2);   
      uf=0;
      
      ac_output = (float)analogRead(ADC_OUT) * nvd.ac_v_cal;  // obtain working vars. All floats
      ac_setpoint = nvd.ac_setpoint;  
     
      if (oen == 1) sst++;   
      if (sst > 251)sst = 251;  
      if (oen == 0) sst--; 
      if (sst <= 0)
        {
          sst = 0;
          cbi(PORTD,5);   // pull IR2184 shutdown LOW to disable gate drive output
          if (serial_enabled == 0)
            {
              Serial.begin(115200);
              serial_enabled = 1;
            }
          if (got_data == 1)
            parse_it();
        }
      else
        {
        sbi(PORTD,5);     // pull it HIGH, to enable output
        if (serial_enabled == 1)
          {
            serial_enabled = 0;
            Serial.end();
          }
        }

      if(sst > 0 && sst < 250)     // slow start timer counts to 250 with gentle ramp up, then switches over to PID control. (gulp!)
        {                           // sst can increase or decrease, giving the slow start and slow stop.
        if (oen == 1)             // slow stop? I think of it as de-gauss
          { 
          if(ac_output < ac_setpoint)
            pwr += 1.0/250.0;     // bang-bang control, should AC output reach setpoint prior to PID
          else
            pwr -= 1.0/250.0;
          }
        else
          pwr -= 1.0 /250.0;
        }
        
      if (sst == 251)       // once sst = 251, change to PID. sst will not be incremented further
        {         
        Setpoint = ac_setpoint;     
        Input = ac_output;  
        do_pid();
        if (Output < -0.05) Output = -0.05;   // ensure some stability during huge transients.
        if (Output > 0.05) Output = 0.05;
        pwr = pwr + Output;
        }
        
      if (pwr > 1.0) pwr = 1.0;   // clamp. Do math in floating point to prevent any integer overflow
      if (pwr < 0.01) pwr = 0.01; // it's slower. We could do clamp after conversion to int but it's your mosfets...
      vpwr = (int)((float)ppwm * pwr);     

      //
      // LCD data update. Write to local buffer, then send out one byte at a time, each 100Hz
      //
      //
      if (oen == 1)
        strncpy(lbp,"ON ",3);
      else
        strncpy(lbp,"OFF",3);
      sprintf(lbp+5,"%3d",nvd.freq); strncpy(lbp+8," Hz",3);
      dcw = dci*dcv;
      //sprintf(lbp+40,"%3d",(int)ac_setpoint);*(lbp+43)=' ';
      sprintf(lbp+24,"%3d",(int)ac_output);*(lbp+27)=' ';
      sprintf(lbp+35,"%3d",(int)hs_temp);*(lbp+38)=' ';
      sprintf(lbp+62,"%3d",(int)dci);*(lbp+65)=' ';
      sprintf(lbp+68,"%3d",(int)dcv);*(lbp+71)=' ';
      sprintf(lbp+74,"%5d",(int)dcw);*(lbp+79)='W';
      bufc++;
      poke(lcdbuf[bufc],1);
      if (bufc > 79)
        {
        lcdxy(0,0);
        bufc = 0;
        }

      dci = (float)analogRead(ADC_DCI) * nvd.dc_i_cal;
      dcv = (float)analogRead(ADC_DCV) * nvd.dc_v_cal;
      hs_temp = get_temp(analogRead(ADC_HS));
      //
      // do limit checking. Checks occur at 100Hz. Probably need some timeouts too. The inputs will be noisy.
      //
      //
      if (nvd.oi_enabled == 1 && dci > nvd.overcurrent)    
        {
          oen = 0;
          strncpy(lbp+40,"Over current",12);
        }

      if (nvd.lv_enabled == 1 && dcv < nvd.lv_cutoff)
        {
          oen = 0;
          strncpy(lbp+40,"DC U.V. stop",12);
          lv_stopped = 1;
        }

      if (lv_stopped == 1 && (dcv > nvd.lv_restart) && oen == 0 && sst == 0)
        {
          lv_stopped = 0;
          init_vars();
          oen = 1;    // enable it.
        }

      if (nvd.ot_enabled == 1 && hs_temp > nvd.over_temp)
        {
          oen = 0;
          strncpy(lbp+40,"Over Temp.",10);
        }
      if (nvd.uv_enabled == 1 && sst == 251 && ((ac_output * nvd.low_limit_pc * 1e-2) < ac_setpoint))
        {
        oen = 0;
        strncpy(lbp+40,"AC under volts",14);  
        }
      if ( hs_temp > nvd.fan_on_temp)
        {
        strncpy(lbp+14,"Fan ON",6);
        sbi(PORTB,4);  
        }
      if ( hs_temp < nvd.fan_off_temp)
        {
        strncpy(lbp+14,"      ",6);
        cbi(PORTB,4);  
        }
 
      check_switch();
      cbi(PORTD,2);
      }
   
}

void check_switch()
{
  dbounce++;  
  if (dbounce > 100) dbounce = 100;
  if ((PINB & 1) == 0)     // momentary button pressed, pin 8, pulling internal pullup low
    {
    if (dbounce > 80)
      {  
      if(oen == 0)
        {
          init_vars();
          oen = 1;    // enable it.
        }
      else
        oen = 0;   // stop inverter
      dbounce=0;
      }       
    }
}


/*
 *  lcd support, custom code for speed. Uses some of the library too
 */
void lcdinit()
  { 
  lcd.begin(20,4);
  lcd.clear();
  strncpy(lbp,    "---                 ",20);
  strncpy(lbp+40, "                    ",20);
  strncpy(lbp+20, "AC      V  Temp    C",20);
  strncpy(lbp+60, "DC    A,    V,     W",20);
  bufc=0;
  }

void lcdxy( uint8_t col, uint8_t row)
  {
  int row_offsets[] = { 0x00, 0x40, 0x14, 0x54   }; 
  poke(0x80 | (col + row_offsets[row]),0);
  }

void poke(uint8_t c,uint8_t cmd)
  {
  uint8_t v2 = c    & 0x0F;
  uint8_t v1 = c >> 4 & 0x0F;
  uint8_t b;
  Wire.beginTransmission(0x27);
  b = (v1 << 4) + 4 + cmd;
  Wire.write(b);
  _delay_us(2);
  b = (v1 << 4) + cmd;
  Wire.write(b);
  _delay_us(37);
  b = (v2 << 4) + 4 + cmd;
  Wire.write(b);
  _delay_us(2);
  b = (v2 << 4) + cmd;
  Wire.write(b);
   _delay_us(37);
  Wire.endTransmission(); 
  }

/*
 *  config menu and serial port stuff. Only active when oen = 0
 */
void show_menu()
  {
  get_nvdata();
  dci = (float)analogRead(ADC_DCI) * nvd.dc_i_cal;
  dcv = (float)analogRead(ADC_DCV) * nvd.dc_v_cal;
  hs_temp = get_temp(analogRead(ADC_HS));
  ac_output = (float)analogRead(ADC_OUT) * nvd.ac_v_cal;
  ac_setpoint = nvd.ac_setpoint;  
  spstr = "";
  got_data=0;
  Serial.println("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
  Serial.print("heat sink temp = ");Serial.print(hs_temp,0);Serial.print(" degC    ch.");Serial.println(ADC_HS);
  pm_str(2); Serial.print(nvd.dc_v_cal);Serial.print(" -> ");Serial.print(dcv,1);Serial.print(" V    ch.");Serial.println(ADC_DCV);
  pm_str(4); Serial.print(nvd.dc_i_cal);Serial.print(" -> ");Serial.print(dci,1);Serial.print(" A    ch.");Serial.println(ADC_DCI);
  pm_str(6); Serial.print(nvd.ac_v_cal);Serial.print(" -> ");Serial.print(ac_output,0);Serial.print(" V AC   ch.");Serial.println(ADC_OUT);
  pm_str(7); Serial.print(nvd.ac_setpoint,0);Serial.println(" V AC");
  pm_str(9); Serial.print(nvd.freq);Serial.println(" Hz");
  pm_str(11); Serial.print(nvd.fan_on_temp,0);Serial.println(" degC");
  pm_str(12); Serial.print(nvd.fan_off_temp,0);Serial.println(" degC");
  pm_str(13); Serial.print(nvd.over_temp,0);Serial.println(" degC");
  pm_str(14); Serial.print(nvd.low_limit_pc,0);Serial.println(" %");
  pm_str(15); Serial.print(nvd.lv_cutoff,1);Serial.println(" V");
  pm_str(10); Serial.print(nvd.lv_restart,1);Serial.println(" V");
  pm_str(16); Serial.print(nvd.overcurrent,1);Serial.println(" A");
  pm_str(17);  if (nvd.lv_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(18);  if (nvd.oi_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(19);  if (nvd.uv_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(20);  if (nvd.ot_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(22); Serial.println();
  pm_str(21); Serial.println();
  Serial.println();
  }

void pm_str(int i)
  {
   strcpy_P(pm_buf, (char*)pgm_read_word(&(s_t[i]))); 
   Serial.print(pm_buf);   
  }

void serialEvent() 
{
  char c;
  while (Serial.available()) 
    {
    c = (char)Serial.read();
    if (got_data == 0)    // dont store while still working on previous string
      {
      spstr += c;
      if (c == '\n')
        got_data = 1;
      }
    }
}


void parse_it()
  {
  int i;
  float f;
  char c;
  spstr.toUpperCase();
  c = spstr.charAt(0);
  switch (c)
    {
    case '?':
      show_menu();
      break;
    case 'Z':      // clear e2prom
      Serial.println("clearing e2prom..");
      for ( i = 0 ; i < EEPROM.length() ; i++) 
        EEPROM.write(i, 0);
      Serial.println("done");
      nv();
      break;
    case 'Y':
      do_nvdefaults();
      nv();
      break;
    case '2': // cal DC voltage
      pm_str(24);
      f = get_float();
      i = analogRead(ADC_DCV);
      nvd.dc_v_cal = f / (float)i;
      nv();
      break;   
    case '4': // cal DC current
      pm_str(26);
      f = get_float();
      i = analogRead(ADC_DCI);
      nvd.dc_i_cal = f/ (float)i;
      nv();
      break;       
    case '6': 
      pm_str(28);
      f = get_float();
      i = analogRead(ADC_OUT);
      nvd.ac_v_cal = f/(float)i;
      nv();
      break;         
    case '7': //  get AC setpoint in AC volts
      pm_str(29);
      nvd.ac_setpoint = get_float();
      nv();
      break;             
    case '9': //  get freq
      pm_str(31);
      nvd.freq = get_int();
      nv();
      break;       
    case 'A': //  restart volts
      pm_str(32);
      nvd.lv_restart = get_float();
      nv();
      break;  
    case 'B': //  fan ON temp
      pm_str(33);
      nvd.fan_on_temp = get_float();
      nv();
      break;    
    case 'C': //  fan OFF temp
      pm_str(34);
      nvd.fan_off_temp = get_float();
      nv();
      break;   
    case 'D': //  cut off temp
      pm_str(35);
      nvd.over_temp = get_float();
      nv();
      break;     
    case 'E': //  AC output low
      pm_str(36);
      nvd.low_limit_pc = get_float();
      nv();
      break; 
    case 'F': //  DC input low
      pm_str(37);
      nvd.lv_cutoff = get_float();
      nv();
      break; 
    case 'G': //  over current
      pm_str(38);
      nvd.overcurrent = get_float();
      nv();
      break;
    case 'H': //  enable DC LV
      pm_str(39);
      nvd.lv_enabled = (get_int() > 0);
      nv();
      break; 
    case 'I': //  enable overcurrent
      pm_str(40);
      nvd.oi_enabled = (get_int() > 0);
      nv();
      break; 
    case 'J': //  enable AC UV
      pm_str(41);
      nvd.uv_enabled = (get_int() > 0);
      nv();
      break;     
    case 'K': //  enable over temp
      pm_str(42);
      nvd.ot_enabled = (get_int() > 0);
      nv();
      break;                                                             
    default:   
      spstr="";
      got_data = 0;
  }
 }

void nv()
  {
    EEPROM.put(10,nvd);
    get_nvdata();
    show_menu();
  }

float get_float()
  {
    char c;
    got_data = 0;
    spstr = "";
    while (got_data == 0)
      {
      if (Serial.available()) 
        {
        c = (char)Serial.read();
        spstr += c;
        if (c == '\n')
          got_data = 1;
        }
      }
  return spstr.toFloat();
  }

int get_int()
  {
    char c;
    got_data = 0;
    spstr = "";
    while (got_data == 0)
      {
      if (Serial.available()) 
        {
        c = (char)Serial.read();
        spstr += c;
        if (c == '\n')
          got_data = 1;
        }
      }
  return spstr.toInt();
  }

void get_nvdata()
{
 EEPROM.get(10,nvd);
}

void do_nvdefaults()
{
 EEPROM.put(10,nvd);
}

