/*
  Fan control, D12
  inverter on/off, D7
*/

#include <avr/pgmspace.h>
const char m_0[] PROGMEM = "0 - unused";  
const char m_1[] PROGMEM = "1 - unused";
const char m_2[] PROGMEM = "2 - cal.  DC volts      ";
const char m_3[] PROGMEM = "3 - unused";
const char m_4[] PROGMEM = "4 - cal.  DC current    ";
const char m_5[] PROGMEM = "5 - unused";
const char m_6[] PROGMEM = "6 - cal.  AC output     ";  
const char m_7[] PROGMEM = "7 - Toroid fan ON       ";
const char m_8[] PROGMEM = "8 - Toroid fan OFF      ";
const char m_9[] PROGMEM = "9 - Toroid shutdown     ";
const char m_A[] PROGMEM = "A - Restart voltage     ";
const char m_B[] PROGMEM = "B - H.S. fan ON         ";
const char m_C[] PROGMEM = "C - H.S  fan OFF        ";  
const char m_D[] PROGMEM = "D - H.S. shutdown       ";
const char m_E[] PROGMEM = "E - AC output low limit ";
const char m_F[] PROGMEM = "F - LV cutoff voltage   ";
const char m_G[] PROGMEM = "G - over current limit  ";
const char m_H[] PROGMEM = "H - enable/disable DC LV cutoff            ";
const char m_I[] PROGMEM = "I - enable/disable over current cutoff     ";  
const char m_J[] PROGMEM = "J - enable/disable AC under volt cutoff    ";
const char m_K[] PROGMEM = "K - enable/disable H.S. over temp cutoff   ";
const char m_L[] PROGMEM = "Z - zero E2PROM";
const char m_M[] PROGMEM = "Y - put defaults into E2PROM";
const char m_N[] PROGMEM = "L - enable/disable Toroid over temp cutoff ";

const char s1[] PROGMEM = "";
const char s2[] PROGMEM = "DC cal. Enter volts";
const char s3[] PROGMEM = "";
const char s4[] PROGMEM = "DC current cal. Enter amps";
const char s5[] PROGMEM = "";
const char s6[] PROGMEM = "AC output cal. Enter volts";
const char s7[] PROGMEM = "Toroid Fan ON temperature (degC)";
const char s8[] PROGMEM = "Toroid Fan OFF temperature (degC)";
const char s9[] PROGMEM = "Toroid Shut down temperature (degC)";
const char sa[] PROGMEM = "Restart after LV stop. Enter Volts";
const char sb[] PROGMEM = "H.S. Fan ON temperature (degC)";
const char sc[] PROGMEM = "H.S. Fan OFF temperature (degC)";
const char sd[] PROGMEM = "H.S. Shut down temperature (degC)";
const char se[] PROGMEM = "AC output Low Limit. Enter Volts";
const char sf[] PROGMEM = "LV shut down voltage. Enter volts";
const char sg[] PROGMEM = "Over current limit. Enter Amps";
const char sh[] PROGMEM = "DC LV cutoff (1 = enable,0 = disable)";
const char si[] PROGMEM = "Overcurrent cutoff (1 = enable,0 = disable)";
const char sj[] PROGMEM = "AC under volt cutoff (1 = enable,0 = disable)";
const char sk[] PROGMEM = "H.S. over temp cutoff (1 = enable,0 = disable)";
const char sl[] PROGMEM = "Toroid over temp cutoff (1 = enable,0 = disable)";

char pm_buf[60];  // above progmem ms can not exceed the size of this buffer.

const char * const s_t[] PROGMEM =    
{   
  m_0,  m_1,  m_2,  m_3,  m_4,  m_5,  m_6,  m_7,  m_8,  m_9,  m_A,  m_B,
  m_C,  m_D,  m_E,  m_F,  m_G,  m_H,  m_I,  m_J,  m_K,  m_L,  m_M,  s1,
  s2,  s3,  s4,  s5,  s6,  s7,  s8,  s9,  sa,  sb,  sc,  sd,  se,  sf,
  sg,  sh,  si,  sj,  sk, sl, m_N  
 };

#include <EEPROM.h>
#include <Wire.h>
#include <LiquidCrystal_PCF8574.h>
LiquidCrystal_PCF8574 lcd(0x27);

#define ADC_OUT  0          // channel for AC output sample
#define ADC_DCI 3           // DC current sense
#define ADC_HS  2           // Heat sink temp sensor
#define ADC_DCV 1           // DC supply volts sample
#define ADC_TOR 7           // toroid temp sensor

#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))



// these are produced from ADC counts multiplied by scale factors. See loop() , parse_it(), show_menu()  
float ac_setpoint, ac_output, dcv, dci, hs_temp,tor_temp;

typedef struct  {
float dc_v_cal,
      dc_i_cal,
      ac_v_cal,
      ac_setpoint;
float fan_on_temp,fan_off_temp,over_temp,
      tor_fan_on_temp,tor_fan_off_temp,tor_over_temp,
      low_limit,lv_cutoff,
      overcurrent, lv_restart;
uint16_t freq;
uint8_t lv_enabled,
      oi_enabled,
      uv_enabled,
      ot_enabled,
      tor_ot_enabled,
      e2prom_ok;
} _nvdata;
      
_nvdata nvd;
      
uint8_t lv_stopped,dbounce,oen,last_oen, stop_reason;

double kp, ki, kd;
float xv[5],yv[5];

char lcdbuf[82],*lbp,bufc;
String spstr = "";
uint8_t got_data;

float get_float(void);
int get_int(void);
float get_temp(int);
void init_vars(void);
void do_20chars(char *);

void setup()
  {
  Serial.begin(115200);  
  
  lcd.begin(20,4);      // LCD stuff
  lcd.clear();
  lcd.print("Nanoverter");
  delay(2000);
  lcd.clear();
  lbp = &lcdbuf[1];   // point local pointer to lcd buffer, used by strncpy(), sprintf()
  strncpy(lbp,    "---                 ",20);
  strncpy(lbp+40, "H.S.     C TOR     C",20);
  strncpy(lbp+20, "AC      V           ",20);
  strncpy(lbp+60, "DC    A,    V,     W",20);
  bufc=0;

  pinMode(12,OUTPUT);     // heat sink fan
  pinMode(11,OUTPUT);     // toroid fan
  pinMode(8,INPUT_PULLUP);// momentary on/off button, normally open 
  DDRD = 0xff;  
  pinMode(6,INPUT);       // let SCR_SD_N drive this, low impedence input 0 or 5V
  cbi(PORTD,7);           // on boot, pulled low for some reason, starting inverter. so pull high.
  lv_stopped = 2;         // prevent auto start on boot up.
  init_vars();
  sbi(ADCSRA,ADPS2) ;   // ADC clock prescale = 16 now. this saves 100us in main loop with little cost to loop stability
  cbi(ADCSRA,ADPS1) ;
  cbi(ADCSRA,ADPS0) ;
  }



void init_vars()      
  {
  dbounce = 0;    
  got_data = 0;
  get_nvdata();
  stop_reason = 0;
  // clear error if needed
  strncpy(lbp,"                    ",20);
  oen = 0;
  last_oen = 0;
  }


float get_temp(int Vo) 
 {    // https://circuitdigest.com/microcontroller-projects/arduino-thermistor-interfacing-code-circuit
 float a = 1.009249522e-03, b = 2.378405444e-04, c = 2.019202697e-07;
 float T,logRt,Tc;
 //return Vo; 
 logRt = log(10000.0*((1024.0/Vo-1))); 
 T = (1.0 / (a + b*logRt + c*logRt*logRt*logRt)); // We get the temperature value in Kelvin from this Stein-Hart equation
 Tc = T - 273.15;                                 // Convert Kelvin to Celsius
 return Tc;
}
 
void loop()
{
  float dcw;
  //
  // read inputs
  // oen = 1 if inverter output enabled, = 0 if stopped
  //
  check_switch();   
  ac_output = (float)analogRead(ADC_OUT) * nvd.ac_v_cal; 
  dci = (float)analogRead(ADC_DCI) * nvd.dc_i_cal;
  dcv = (float)analogRead(ADC_DCV) * nvd.dc_v_cal;
  hs_temp = get_temp(analogRead(ADC_HS)); 
  tor_temp = get_temp(analogRead(ADC_TOR)); 
  //
  // update LCD
  //
  if (oen == 1)
    strncpy(lbp,"ON ",3);
  else
    strncpy(lbp,"OFF",3);
  dcw = dci*dcv;  // calc watts

  sprintf(lbp+24,"%3d",(int)ac_output);*(lbp+27)=' ';
  sprintf(lbp+45,"%3d",(int)hs_temp);*(lbp+48)=' ';
  sprintf(lbp+55,"%3d",(int)tor_temp);*(lbp+58)=' ';
  sprintf(lbp+62,"%3d",(int)dci);*(lbp+65)=' ';
  sprintf(lbp+68,"%3d",(int)dcv);*(lbp+71)=' ';
  sprintf(lbp+74,"%5d",(int)dcw);*(lbp+79)='W';

  //
  // do limit checking. Probably need some timeouts too. The inputs will be noisy.
  //
  if (nvd.oi_enabled == 1 && dci > nvd.overcurrent)    // if overcurrent limit enabled check if over current
    {
    oen = 0;
    strncpy(lbp+5,"O.C. SD     ",12);
    stop_reason = 1;
    }
  if (lv_stopped == 0 && nvd.lv_enabled == 1 && dcv < nvd.lv_cutoff)
    {
    oen = 0;
    strncpy(lbp+5,"DC U.V. stop",12);
    lv_stopped = 1;
    stop_reason = 2;
    }
  if (lv_stopped == 1 && (dcv > nvd.lv_restart) && oen == 0 )
    {
    lv_stopped = 0;
    init_vars();
    oen = 1;    // enable it.
    }
  if (nvd.ot_enabled == 1 && hs_temp > nvd.over_temp)
    {
    oen = 0;
    strncpy(lbp+5,"H.S. OT SD",10);
    stop_reason = 3;
    }
  if (nvd.tor_ot_enabled == 1 && tor_temp > nvd.tor_over_temp)
    {
    oen = 0;
    strncpy(lbp+5,"Tor OT SD ",10);
    stop_reason = 6;
    }
    //
    // need a delay timers after eon = 1 for the following, else it will always be under voltage
    //
  if (nvd.uv_enabled == 1 && (ac_output < nvd.low_limit))
    {
    oen = 0;
    strncpy(lbp+5,"AC U.V. SD    ",14);  
    stop_reason = 4;
    }
  if ((PIND & 0x40) == 0x40)      // D6 goes high
    {
    oen = 0;
    strncpy(lbp+5,"SCR OC SD     ",14);  
    stop_reason = 5;
    }  
  if (hs_temp > nvd.fan_on_temp)
    {
    lbp[44] = '*';
    sbi(PORTB,4);  
    }
  if ( hs_temp < nvd.fan_off_temp)
    {
    lbp[44] = ' ';
    cbi(PORTB,4);  
    }
  if (tor_temp > nvd.tor_fan_on_temp)
    {
    lbp[54] = '*';
    sbi(PORTB,3);  
    }
  if (tor_temp < nvd.tor_fan_off_temp)
    {
    lbp[54] = ' ';
    cbi(PORTB,3);  
    }  
  //
  // drive nano1 on/off control
  //
  if (oen == 1)
    sbi(PORTD,7);
  else
    cbi(PORTD,7);
   //
   // update LCD
   // 
  lcd.setCursor(0, 0);
  lcdbuf[81]=0;
  lcd.print(lbp);
  if (got_data == 1)
    parse_it();
  if (oen == 1)
    stop_reason = 0;
  delay(10);  
}

void check_switch()
{
  dbounce++;  
  if (dbounce > 100) dbounce = 100;
  if ((PINB & 1) == 0)     // momentary button pressed, pin 8, pulling internal pullup low
    {
    if (dbounce > 20)
      { 
      lv_stopped = 0;   
      if(oen == 0)
        {
        init_vars();
        oen = 1;    // enable it.
        }
      else
        oen = 0;   // stop inverter
      dbounce=0;
      }       
    }
}


/*
 *  lcd support, custom code for speed. Uses some of the library too
 */
void lcdinit()
  { 
  lcd.begin(20,4);
  lcd.clear();
  strncpy(lbp,    "---                 ",20);
  strncpy(lbp+40, "                    ",20);
  strncpy(lbp+20, "AC      V  Temp    C",20);
  strncpy(lbp+60, "DC    A,    V,     W",20);
  bufc=0;
  }


/*
 *  config menu and serial port stuff. Only active when oen = 0
 */
void show_menu()
  {
  get_nvdata();
  ac_output = (float)analogRead(ADC_OUT) * nvd.ac_v_cal; 
  dci = (float)analogRead(ADC_DCI) * nvd.dc_i_cal;
  dcv = (float)analogRead(ADC_DCV) * nvd.dc_v_cal;
  hs_temp = get_temp(analogRead(ADC_HS)); 
  tor_temp = get_temp(analogRead(ADC_TOR)); 
 
  spstr = "";
  got_data=0;
  Serial.println("\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n\n");
  Serial.print("heat sink temp = ");Serial.print(hs_temp,0);Serial.print(" degC    ch.");Serial.println(ADC_HS);
  Serial.print("toroid temp = ");Serial.print(tor_temp,0);Serial.print(" degC    ch.");Serial.println(ADC_TOR);
  pm_str(2); Serial.print(nvd.dc_v_cal);Serial.print(" -> ");Serial.print(dcv,1);Serial.print(" V    ch.");Serial.println(ADC_DCV);
  pm_str(4); Serial.print(nvd.dc_i_cal);Serial.print(" -> ");Serial.print(dci,1);Serial.print(" A    ch.");Serial.println(ADC_DCI);
  pm_str(6); Serial.print(nvd.ac_v_cal);Serial.print(" -> ");Serial.print(ac_output,0);Serial.print(" V AC   ch.");Serial.println(ADC_OUT);
  pm_str(7); Serial.print(nvd.tor_fan_on_temp,0);Serial.println(" degC");
  pm_str(8); Serial.print(nvd.tor_fan_off_temp,0);Serial.println(" degC");
  pm_str(9); Serial.print(nvd.tor_over_temp,0);Serial.println(" degC");
  pm_str(11); Serial.print(nvd.fan_on_temp,0);Serial.println(" degC");
  pm_str(12); Serial.print(nvd.fan_off_temp,0);Serial.println(" degC");
  pm_str(13); Serial.print(nvd.over_temp,0);Serial.println(" degC");
  pm_str(14); Serial.print(nvd.low_limit,0);Serial.println(" V");
  pm_str(15); Serial.print(nvd.lv_cutoff,1);Serial.println(" V");
  pm_str(10); Serial.print(nvd.lv_restart,1);Serial.println(" V");
  pm_str(16); Serial.print(nvd.overcurrent,1);Serial.println(" A");
  pm_str(17);  if (nvd.lv_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(18);  if (nvd.oi_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(19);  if (nvd.uv_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(20);  if (nvd.ot_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(44);  if (nvd.tor_ot_enabled == 1)  Serial.println("ON"); else Serial.println("OFF");  
  pm_str(22); Serial.println();
  pm_str(21); Serial.println();
  Serial.println();
  }

void pm_str(int i)
  {
   strcpy_P(pm_buf, (char*)pgm_read_word(&(s_t[i]))); 
   Serial.print(pm_buf);   
  }

void serialEvent() 
{
  char c;
  while (Serial.available()) 
    {
    c = (char)Serial.read();
    if (got_data == 0)    // dont store while still working on previous string
      {
      spstr += c;
      if (c == '\n')
        got_data = 1;
      }
    }
}

void do_20chars(char *p)
  {
    int i;
    for(i=0; i < 20; i++)
      Serial.print(p[i]);
    Serial.println();
  }

void parse_it()
  {
  int i;
  float f;
  char c;
  spstr.toUpperCase();
  c = spstr.charAt(0);
  switch (c)
    { 
    case '^':         // this probably used by myself only, to allow remote control of nanverter via the raspberry pi & USB serial
      spstr.setCharAt(0,' ');                           // remove ^, so as to allow conversion to int
      i = spstr.toInt();                                // send ^nn<newline>
      if (i == 42)        Serial.println("the answer to life, the universe and everything");
      if (i == 10)        oen = 0;                      // 
      if (i == 11)        oen = 1;                      // eg 13<newline> to obtain DC volts
      if (i == 12)        Serial.println(ac_output);    // or 11<newline> to start inverter, etc.
      if (i == 13)        Serial.println(dcv);
      if (i == 14)        Serial.println(dci);
      if (i == 15)        Serial.println(hs_temp);
      if (i == 16)        Serial.println(nvd.overcurrent);
      if (i == 17)        Serial.println(nvd.over_temp);
      if (i == 18)        Serial.println(nvd.lv_cutoff);
      if (i == 19)        Serial.println(nvd.lv_restart);
      if (i == 20)        Serial.println(nvd.oi_enabled);
      if (i == 21)        Serial.println(nvd.ot_enabled);
      if (i == 22)        Serial.println(nvd.lv_enabled);
      if (i == 23)        Serial.println(tor_temp);
      if (i == 24)        Serial.println(nvd.tor_ot_enabled);
      if (i == 30)        Serial.println(oen);
      if (i == 31)        Serial.println(stop_reason);   
      if (i == 32)        Serial.println((PINB));         // D11,D12 is fan control. look for bit 3,4
      if (i == 33)        // dump LCD buffer
        {
          do_20chars(lbp);
          do_20chars(lbp + 40);
          do_20chars(lbp + 20);
          do_20chars(lbp + 60);
        }
      break;      
    case '?':
      show_menu();
      break;
    case 'Z':      // clear e2prom
      Serial.println("clearing e2prom..");
      for ( i = 0 ; i < EEPROM.length() ; i++) 
        EEPROM.write(i, 0);
      Serial.println("done");
      nv();
      break;
    case 'Y':
      do_nvdefaults();
      nv();
      break;
    case '2': // cal DC voltage
      pm_str(24);
      f = get_float();
      i = analogRead(ADC_DCV);
      nvd.dc_v_cal = f / (float)i;
      nv();
      break;   
    case '4': // cal DC current
      pm_str(26);
      f = get_float();
      i = analogRead(ADC_DCI);
      nvd.dc_i_cal = f/ (float)i;
      nv();
      break;       
    case '6': 
      pm_str(28);
      f = get_float();
      i = analogRead(ADC_OUT);
      nvd.ac_v_cal = f/(float)i;
      nv();
      break;     
    case '7': //  fan ON temp
      pm_str(29);
      nvd.tor_fan_on_temp = get_float();
      nv();
      break;    
    case '8': //  fan OFF temp
      pm_str(30);
      nvd.tor_fan_off_temp = get_float();
      nv();
      break;   
    case '9': //  cut off temp
      pm_str(31);
      nvd.tor_over_temp = get_float();
      nv();
      break;                   
    case 'A': //  restart volts
      pm_str(32);
      nvd.lv_restart = get_float();
      nv();
      break;  
    case 'B': //  fan ON temp
      pm_str(33);
      nvd.fan_on_temp = get_float();
      nv();
      break;    
    case 'C': //  fan OFF temp
      pm_str(34);
      nvd.fan_off_temp = get_float();
      nv();
      break;   
    case 'D': //  cut off temp
      pm_str(35);
      nvd.over_temp = get_float();
      nv();
      break;     
    case 'E': //  AC output low
      pm_str(36);
      nvd.low_limit = get_float();
      nv();
      break; 
    case 'F': //  DC input low
      pm_str(37);
      nvd.lv_cutoff = get_float();
      nv();
      break; 
    case 'G': //  over current
      pm_str(38);
      nvd.overcurrent = get_float();
      nv();
      break;
    case 'H': //  enable DC LV
      pm_str(39);
      nvd.lv_enabled = (get_int() > 0);
      nv();
      break; 
    case 'I': //  enable overcurrent
      pm_str(40);
      nvd.oi_enabled = (get_int() > 0);
      nv();
      break; 
    case 'J': //  enable AC UV
      pm_str(41);
      nvd.uv_enabled = (get_int() > 0);
      nv();
      break;     
    case 'K': //  enable HS over temp
      pm_str(42);
      nvd.ot_enabled = (get_int() > 0);
      nv();
      break;    
    case 'L': //  enable toroid over temp
      pm_str(43);
      nvd.tor_ot_enabled = (get_int() > 0);
      nv();
      break;                                                             
    default:   
      spstr="";
      got_data = 0;
  }
  spstr="";
  got_data = 0;
 }

void nv()
  {
    EEPROM.put(10,nvd);
    EEPROM.get(10,nvd);
    get_nvdata();
    show_menu();
  }

float get_float()
  {
    char c;
    got_data = 0;
    spstr = "";
    while (got_data == 0)
      {
      if (Serial.available()) 
        {
        c = (char)Serial.read();
        spstr += c;
        if (c == '\n')
          got_data = 1;
        }
      }
  return spstr.toFloat();
  }

int get_int()
  {
    char c;
    got_data = 0;
    spstr = "";
    while (got_data == 0)
      {
      if (Serial.available()) 
        {
        c = (char)Serial.read();
        spstr += c;
        if (c == '\n')
          got_data = 1;
        }
      }
  return spstr.toInt();
  }

void get_nvdata()
{
 EEPROM.get(10,nvd);
}

void do_nvdefaults()
{
 EEPROM.put(10,nvd);
}

