

#define ADC_OUT  0          // channel for AC output sample

#define cbi(sfr, bit) (_SFR_BYTE(sfr) &= ~_BV(bit))
#define sbi(sfr, bit) (_SFR_BYTE(sfr) |= _BV(bit))

#define NPWM  200         // 200 for this and 800 for PPWM for 50Hz and 20kHz PWM. Both must be integers.
//#define PPWM  665       // 60.062 Hz - choose one or the other.
#define PPWM  799         // 50.002 Hz 799  49.939 Hz for 800

      
uint8_t uf,oen,v1low, lv_stopped,int_f;
uint16_t pcount;
uint32_t vpwr;
int sst;
uint16_t l[NPWM+1];
float pwr;
uint8_t dbounce;
double Input, Output, Setpoint;
double errSum, lastErr,error,dErr;
double kp, ki, kd;
float xv[5],yv[5];
float ac_output, ac_setpoint;

void init_vars(void);

long pcount_acc;
int pcount_delta, phase_error, old_phase_error;
int pcint,new_delta;

uint8_t init_done; 

void setup()
  {
  int i;
  float t,u;

  for(t=0.0,i=0; i <= NPWM; i++,t +=   3.14159/ (float) (NPWM+1))
    {   
    u = 65535.0 * sin(t);   // 1/2 wave sine lookup table, scaled to (16 bits - 1)
    l[i] =  (int)u;  
    }
 
  pinMode(9, OUTPUT);     // V1 drive
  pinMode(10, OUTPUT);    // V2 drive
  pinMode(8,INPUT);       // continuous on/off pin, driven by nano2
  
  // can NOT let arduino system alter any pins from D0 to D7 via calls to pinMode() etc.. It buggers d5 which I need to pull down low-Z upon boot
  DDRD = 0xfb; // and let D2 be input for INT0 
  pinMode(2,INPUT_PULLUP);
  noInterrupts();

  TCCR1A = _BV(COM1A1) | _BV(COM1B0) | _BV(COM1B1) | _BV(WGM11);
  TCCR1B = _BV(WGM13) | _BV(WGM12) | _BV(CS10);
  OCR1A = 10;
  ICR1 = PPWM;
  TIMSK1 |= (1 << TOIE1);

  EICRA = 3;               // interrupt on falling edge from pin D2
  EIMSK |= bit (INT0);     // enable it  
  cbi(PORTD,4);   // no sync
  cbi(PORTD,5);   // disable gate drive output. data direction flag set by above statement.
  oen = 0;        // output enable. = 1 to run inverter, = 0 to stop inverter
  init_vars();
  init_done = 1;
  
  interrupts();             // enable all interrupts
  TIMSK0 = 0;               // except arduino clock

  sbi(ADCSRA,ADPS2) ;   // ADC clock prescale = 16 now. this saves 100us in main loop with little cost to loop stability
  cbi(ADCSRA,ADPS1) ;
  cbi(ADCSRA,ADPS0) ;  

  Serial.begin(115200);
  }

//
// complex sync code. chicken entrail future telling stuff. No, just PID
// this block needs to run fast, it might interrupt INT1 overflow block and I don't want that.
// one recent version takes 6uS
// It runs at the start of a half cycle, where the PWM duty% is always going to be low
// i.e. near zero crossing of AC voltage. The 6uS disturbance of PWM will be minor, if any.
// int_f set to 1 to enable this interrupt AFTER PWM interrupt code completes processing
// so it occurs in some random time shortly after INT1 (PWM) (which runs at 20kHz, implying 50uS between each time it's run)
// DSO captures confirm this is no issue.
//
ISR (INT0_vect)
{
if  (int_f == 0) 
{
  sbi(PORTD,7);
  cbi(PORTD,7);
  return;  // only once per 50 Hz cycle. Noise! much noise...
}
int_f = 0;
//sbi(PORTD,7);
//cbi(PORTD,7);
pcint = pcount +  (v1low == 1?200:0) - 200;   // get position in 50Hz waveform, -200 to 200 counts
                                              // pcint is position in output waveform when this interrupt occurs
phase_error = (pcint - 0);    // subtract setpoint, 0 for my setup, in counts
                              // setpoint = 0, will be different, effected by RC LP filter delay at least, other influences?               
new_delta =  (phase_error/4 + (phase_error - old_phase_error)); // PID, no I. P = 0.25, D = 1.0.  the divide by 4 is compiled as 2 x looped arithmetic shifts
if (new_delta > 10) 
  {
    
    new_delta = 10;       // Keeps things safe. When 20 or higher is used, my inverter spasms. Safe thanks to current limited power supply..
  }
if (new_delta < -10) 
  {
    
    new_delta = -10;     // even this small a range of changes still lets it sync to mains is less than 1/2 second.    
  }
pcount_delta = pcount_delta - new_delta;  // apply PID output to the process, i.e. modify AC output frequency
if(pcount_delta < 200) pcount_delta = 200;  // clamp change to output frequency to reasonable limits
if(pcount_delta > 300) pcount_delta = 300;  // this range is approx 40 - 58 Hz, with 256 = 50Hz mid point
old_phase_error = phase_error;
if (phase_error < 3)
   sbi(PORTD,4);              // tell world I think I'm in sync.
}  

//
// 20Khz SPWM, code has only 50uS to run, takes about 10uS
//
ISR(TIMER1_OVF_vect)        
 {   
  long c;
  c = (l[pcount] * vpwr) >> 16;   // scale sine wave by vpwr 32 bit integer calcs.
  if (v1low == 1)                 // alternate between 2 output compare pins. Get a full 50Hz waveform
    OCR1B = c;                    //  1/2 out this pin
  else
    OCR1A = c;                    // and then 1/2 out this pin. It alternates.
  pcount_acc += pcount_delta; // count through 1/2 wave sine table, 200 entries. At 20KHz, = 50Hz output 
  pcount = (pcount_acc >> 8); // this fractional math allows varying freq +/- 8Hz or so.
  if(pcount >= NPWM)          // pcount  will = NPWM, at each 1/2 wave, at the start
    {
    pcount = 0;               // reset counter
    pcount_acc = 0;           // and the fractional math part of the counter
    uf=1;                     // enable one PID control loop execution, in loop()
    if (v1low == 1)           // if first 1/2 wave..
      {
       v1low = 0;             // toggle to 2nd half wave
       TCCR1A =  _BV(COM1A1) | _BV(WGM11); // config output compare to suit
       //sbi(PORTD,7);          // DSO sync pulse   
       //cbi(PORTD,7); 
       if (int_f == 1)    // no AC sync signal at all, pull sync lock output LOW
        {
          cbi(PORTD,4);         // tell world, no sync
          pcount_delta = 256;   // after sync loss, also when/if no sync lock ever attained, ensure AC output is 50Hz
        }
       int_f = 1;               // this is here, last to execute, to ensure this PWM code runs first and is not delayed by
      }                         // subsequent AC sync interrupt processing. 
    else
      {
       v1low = 1;             // toggle to 1st half wave
       TCCR1A =  _BV(COM1B1) | _BV(WGM11);   // config O.C. to suit
      }
    }
}


void init_vars()      // called prior to setting oen = 1, as well as from setup()
  {
  ICR1 = PPWM;
  lv_stopped = 0;  
  sst = 0;      // slow start counter 0 - 251
  v1low = 1;    // half wave output flag, 0 or 1
  pcount = 0;   // counter for sine wave lookup table
  uf = 0;       // set to 1 to enable AC output voltage feedback code to execute
  pwr = 0.0;    // PWM duty cycle analog, 0.0 to 1.0
  vpwr= 0;      // integer equivalent of pwr, 0 to PPWM
  dbounce = 0;  // debounce counter for input switch 
  kp = 0.1;     // set up PID for AC output voltage control
  ki = 0.0;
  kd = 0.01;
  errSum = 0.0;
  lastErr = 0.0;
  int_f = 0;          //used in AC sync, one shot per 50Hz
  pcount_acc = 0;     // fractional integer math. Each PWM, pcount_delta is added to pcount_acc and divided by 256
  pcount_delta = 256; // in effect we are getting an addition of 1. But pcount_delta can be modified, a little bigger or smaller..
  old_phase_error = 0;// used in PID, to obtain derivative
  cbi(PORTD,4);  
  }


void do_pid()
{   
   double timeChange = 0.01;    // always called at 100Hz, so dT will be 10msec..
   error = Setpoint - Input;
   errSum += (error * timeChange);
   dErr = (error - lastErr) / timeChange;
   Output = kp * error + ki * errSum + kd * dErr;
   lastErr = error;
}


void loop()
{
  float ch0;
  ch0 = (float)analogRead(ADC_OUT) / 1024.0; // scale 0 - 5V to 0.0 to 1.0
  // filter AC output 1000 sample rate, Fc = 10, 4 pole Bessel LP
  // thanks to  https://www-users.cs.york.ac.uk/~fisher/mkfilter/trad.html
  xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3]; xv[3] = xv[4];
  xv[4] = ch0 / 2.259747797e+05;
  yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3]; yv[3] = yv[4];
  yv[4] =   (xv[0] + xv[4]) + 4 * (xv[1] + xv[3]) + 6 * xv[2]
                     + ( -0.7428578687 * yv[0]) + (  3.1954036268 * yv[1])
                     + ( -5.1599230905 * yv[2]) + (  3.7073065280 * yv[3]);
  ac_output = yv[4];

  if(uf == 1  )    // this will execute at 100Hz
      {            // it runs at near zero volt crossing of AC output       
      uf=0; 
      ac_setpoint = 0.56;       // for same Vfb as EG8010. change AC output voltage via Vfb voltage divider trimpot
      if (oen == 1) sst++;      // slow start. If output is enabled, keep starting up. sst = output pwm% or approx. voltage.
      if (sst > 251)sst = 251;  // stop increasing when sst has got to the end of slow start. when = 251, this enables PID. See below..
      if (oen == 0) sst--;      // if stopping, slow stop
      if (sst <= 0)             // once fully stopped, no more changes to sst needed
        {
        sst = 0;
        cbi(PORTD,5);     // pull IR2184 shutdown LOW to disable gate drive output
        }
      else
        sbi(PORTD,5);     // pull it HIGH, to enable output

      if(sst > 0 && sst < 250)     // slow start timer counts to 250 with gentle ramp up, then switches over to PID control. (gulp!)
        {                          // sst can increase or decrease, giving the slow start and slow stop.
        if (oen == 1)              // slow stop? I think of it as de-gauss
          { 
          if(ac_output < ac_setpoint)
            pwr += 1.0/250.0;     // bang-bang control, should AC output reach setpoint prior to PID.
          else                    // During slow start need to ensure output is tracking setpoint
            pwr -= 1.0/250.0;     // should output get close enough to matter.
          }
        else
          pwr -= 1.0 /250.0;      // this will execute in slow stop
        }
        
      if (sst == 251)       // once sst = 251, run PID. the above bang-bang control will NOT have executed, because sst = 251
        {                   
        Setpoint = ac_setpoint;     
        Input = ac_output;  
        do_pid();
        pwr = pwr + Output;
        }
        
      if (pwr > 0.99) pwr = 0.99;   // clamp. Do math in floating point to prevent any integer overflow
      if (pwr < 0.01) pwr = 0.01; // it's slower. We could do clamp after conversion to int but it's your mosfets...
      vpwr = (int)((float)PPWM * pwr);     

      check_switch_cont();
      
      Serial.print((pcount_delta-256)*10);      // used during AC sync code dev. 
      Serial.print(" -150 150 ");
      Serial.println(pcint*10);
      
      }
}

void check_switch_cont()
{   // called at 100 Hz
  if ((PINB & 0x01) == 0x01)     
    {
    if (init_done == 0)
      {
      init_vars();
      init_done = 1;
      }  
    oen = 1;    // slow start
    }
      
  if (((PINB & 0x01) == 0) || ((PIND & 0x40) == 0x40))     
    {
    init_done = 0;
    oen = 0;    // slow stop
    }
          
}

