' Transpiled on 09-11-2022 12:13:55

' Copyright (c) 2022 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
' For PicoGAME VGA 1.4 running PicoMiteVGA MMBasic 5.07.05

Option Base 0
Option Default None
Option Explicit On

' BEGIN:     #Include "ctrl.ipp" -----------------------------------------------
' Copyright (c) 2022 Thomas Hugo Williams
' License MIT <https://opensource.org/licenses/MIT>
'
' MMBasic Controller Library
'
' Preprocessor flag PICOMITE defined

Const ctrl.VERSION = 901  ' 0.9.1

' Button values as returned by controller read functions.
Const ctrl.R      = &h01
Const ctrl.START  = &h02
Const ctrl.HOME   = &h04
Const ctrl.SELECT = &h08
Const ctrl.L      = &h10
Const ctrl.DOWN   = &h20
Const ctrl.RIGHT  = &h40
Const ctrl.UP     = &h80
Const ctrl.LEFT   = &h100
Const ctrl.ZR     = &h200
Const ctrl.X      = &h400
Const ctrl.A      = &h800
Const ctrl.Y      = &h1000
Const ctrl.B      = &h2000
Const ctrl.ZL     = &h4000

Const ctrl.OPEN  = -1
Const ctrl.CLOSE = -2
Const ctrl.SOFT_CLOSE = -3

' The NES standard specifies a 12 micro-second pulse, but all the controllers
' I've tested work with 1 micro-second, and possibly less.
Const ctrl.PULSE = 0.001 ' 1 micro-second

' When a key is pressed ctrl.on_key() sets the corresponding byte of this
' 256-byte map to 1. When ctrl.keydown(i%) is called the corresponding
' byte is read and set to 0. Note that a 256-bit map could be used but would
' be slower.
Dim ctrl.key_map%(31 + Mm.Info(Option Base))

' Reads the keyboard as if it were a controller.
'
' Note that the PicoMite has no KEYDOWN function so we are limited to
' reading a single keypress from the input buffer and cannot handle multiple
' simultaneous keys or properly handle a key being pressed and not released.
Sub keys_cursor(x%)
  If x% < 0 Then Exit Sub
  x% =    ctrl.keydown%(32)  * ctrl.A
  Inc x%, ctrl.keydown%(128) * ctrl.UP
  Inc x%, ctrl.keydown%(129) * ctrl.DOWN
  Inc x%, ctrl.keydown%(130) * ctrl.LEFT
  Inc x%, ctrl.keydown%(131) * ctrl.RIGHT
End Sub

' Initialises keyboard reading.
Sub ctrl.init_keys()
  ctrl.term_keys()
  On Key ctrl.on_key()
End Sub

Sub ctrl.on_key()
  Local ch$ = Inkey$
  If ch$ <> "" Then Poke Var ctrl.key_map%(), Asc(ch$), 1
End Sub

' Terminates keyboard reading.
Sub ctrl.term_keys()
  On Key 0
  Memory Set Peek(VarAddr ctrl.key_map%()), 0, 256
  Do While Inkey$ <> "" : Loop
End Sub

Function ctrl.keydown%(i%)
  ctrl.keydown% = Peek(Var ctrl.key_map%(), i%)
  If ctrl.keydown% Then Poke Var ctrl.key_map%(), i%, 0
End Function

Function ctrl.poll$(duration%, ctrls$())
  Local expires% = Choice(duration%, Timer + duration%, &h7FFFFFFFFFFFFFFF), i%
  Do
    For i% = Bound(ctrls$(), 0) To Bound(ctrls$(), 1)
      If ctrl.poll_single%(ctrls$(i%), ctrl.A Or ctrl.START) Then
        ctrl.poll$ = ctrls$(i%)
        Exit Function
      EndIf
    Next
  Loop While Timer < expires%
End Function

' Opens, polls (for a maximum of 5ms) and closes a controller.
'
' @param  ctrl$  controller driver function.
' @param  mask%  bit mask to match against.
' @return        1 if any of the bits in the mask match what is read from the
'                controller, otherwise 0.
Function ctrl.poll_single%(ctrl$, mask%)
  On Error Ignore
  Call ctrl$, ctrl.OPEN
  ' TODO: What if CLASSIC controller is already open ?
  If Mm.ErrNo = 0 Then
    Local key%, t% = Timer + 5
    Do
      Call ctrl$, key%
      If key% And mask% Then
        ctrl.poll_single% = 1
        ' Wait for user to release key.
        Do While key% <> 0 : Call ctrl$, key% : Loop
        Exit Do
      EndIf
    Loop While Timer < t%
    Call ctrl$, ctrl.SOFT_CLOSE
  EndIf
  On Error Abort
End Function

' Atari joystick on PicoGAME Port A.
Sub atari_a(x%)
  Select Case x%
    Case Is >= 0
      x% =    Not Pin(GP14) * ctrl.A
      Inc x%, Not Pin(GP0)  * ctrl.UP
      Inc x%, Not Pin(GP1)  * ctrl.DOWN
      Inc x%, Not Pin(GP2)  * ctrl.LEFT
      Inc x%, Not Pin(GP3)  * ctrl.RIGHT
      Exit Sub
    Case ctrl.OPEN
      SetPin GP14, DIn
      SetPin GP0, DIn
      SetPin GP1, DIn
      SetPin GP2, DIn
      SetPin GP3, DIn
  End Select
End Sub

' Atari joystick on PicoGAME Port B.
Sub atari_b(x%)
  Select Case x%
    Case Is >= 0
      x% =    Not Pin(GP15) * ctrl.A
      Inc x%, Not Pin(GP28) * ctrl.UP
      Inc x%, Not Pin(GP4)  * ctrl.DOWN
      Inc x%, Not Pin(GP5)  * ctrl.LEFT
      Inc x%, Not Pin(GP22) * ctrl.RIGHT
      Exit Sub
    Case ctrl.OPEN
      SetPin GP15, DIn
      SetPin GP28, DIn
      SetPin GP4, DIn
      SetPin GP5, DIn
      SetPin GP22, DIn
    End Select
End Sub

' SNES gamepad on PicoGAME Port A.
Sub snes_a(x%)
  Select Case x%
    Case Is >= 0
      Pulse GP2, ctrl.PULSE
      x% =    Not Pin(GP1) * ctrl.B      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.Y      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.SELECT : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.START  : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.UP     : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.DOWN   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.LEFT   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.RIGHT  : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.A      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.X      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.L      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.R      : Pulse GP3, ctrl.PULSE
      Exit Sub
    Case ctrl.OPEN
      nes_a(ctrl.OPEN)
  End Select
End Sub

' SNES gamepad on PicoGAME Port B.
Sub snes_b(x%)
  Select Case x%
    Case Is >= 0
      Pulse GP5, ctrl.PULSE
      x% =    Not Pin(GP4) * ctrl.B      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.Y      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.SELECT : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.START  : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.UP     : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.DOWN   : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.LEFT   : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.RIGHT  : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.A      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.X      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.L      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.R      : Pulse GP22, ctrl.PULSE
      Exit Sub
    Case ctrl.OPEN
      nes_b(ctrl.OPEN)
  End Select
End Sub

' Reads port A connected to a NES gamepad.
'
' Note that the extra pulse after reading bit 7 (Right) should not be necessary,
' but in practice some NES clone controllers require it to behave correctly.
Sub nes_a(x%)
  Select Case x%
    Case Is >= 0
      Pulse GP2, ctrl.PULSE
      x% =    Not Pin(GP1) * ctrl.A      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.B      : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.SELECT : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.START  : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.UP     : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.DOWN   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.LEFT   : Pulse GP3, ctrl.PULSE
      Inc x%, Not Pin(GP1) * ctrl.RIGHT  : Pulse GP3, ctrl.PULSE
      Exit Sub
    Case ctrl.OPEN
      SetPin GP2, Dout ' Latch
      SetPin GP3, Dout ' Clock
      SetPin GP1, Din  ' Data
      Pin(GP2) = 0
      Pin(GP3) = 0
  End Select
End Sub

' NES gamepad on PicoGAME Port B.
Sub nes_b(x%)
  Select Case x%
    Case Is >= 0
      Pulse GP5, ctrl.PULSE
      x% =    Not Pin(GP4) * ctrl.A      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.B      : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.SELECT : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.START  : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.UP     : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.DOWN   : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.LEFT   : Pulse GP22, ctrl.PULSE
      Inc x%, Not Pin(GP4) * ctrl.RIGHT  : Pulse GP22, ctrl.PULSE
      Exit Sub
    Case ctrl.OPEN
      SetPin GP5,  Dout ' Latch
      SetPin GP22, Dout ' Clock
      SetPin GP4,  Din  ' Data
      Pin(GP5) = 0
      Pin(GP22) = 0
  End Select
End Sub

' END:       #Include "ctrl.ipp" -----------------------------------------------
Mode 1

If Mm.Device$ = "MMB4L" Then
  Dim WIDTH%, HEIGHT%
  Console GetSize WIDTH%, HEIGHT%
  Console HideCursor
Else
  Const WIDTH% = Mm.Hres \ Mm.Info(FontWidth)
EndIf

Dim ctrl$ = "no_ctrl", err$

Option Break 4
On Key 3, on_break

read_controller_data()
show_menu()
main_loop()
end_program()

Sub on_break()
  Option Break 3
  end_program()
End Sub

Sub end_program()
  On Error Ignore
  Call ctrl$, ctrl.CLOSE
  On Error Abort
  If Mm.Device$ = "MMB4L" Then Console ShowCursor
  End
End SUb

Sub read_controller_data()
  Local i%, s1$, s2$
  restore_controller_data()
  Dim NUM_CTRL% = 0
  Do
    Read s1$, s2$
    If s1$ = "" Then Exit Do
    Inc NUM_CTRL%
  Loop

  Dim CTRL_IDS$(NUM_CTRL%)
  Dim CTRL_NAMES$(NUM_CTRL%)
  restore_controller_data()
  For i% = 1 To NUM_CTRL%
    Read CTRL_IDS$(i%), CTRL_NAMES$(i%)
  Next
End Sub

Sub restore_controller_data()
  Select Case Mm.Device$
    Case "PicoMite", "PicoMiteVGA"
      Restore controller_data_picomite
    Case "Colour Maximite 2", "Colour Maximite 2 G2"
      Restore controller_data_cmm2
    Case "MMB4L"
      Restore controller_data_mmb4l
    Case Else
      Error "Unsupported device: " + Mm.Device$
  End Select
End Sub

Sub show_menu()
  Cls
  print_at(0, 0, "MMBasic Controller Test " + format_version$(ctrl.VERSION))
  print_at(0, 1, Mm.Device$ + " " + Str$(Mm.Info(Version)))
  print_at(0, 3,  "Select controller using the function keys")
  print_at(0, 4,  "Then 'play' with controller to test response")
  Local i%
  For i% = 1 To NUM_CTRL%
    print_ctrl_option(i%, 0)
  Next
  print_at(2, i% + 5, "[Esc] Quit")
End Sub

Function format_version$(version%)
  Local major% = version% \ 10000
  Local minor% = (version% - major% * 10000) \ 100
  Local micro% = version% - major% * 10000 - minor% * 100
  format_version$ = Str$(major%) + "." + Str$(minor%) + "." + Str$(micro%)
End Function

Sub main_loop()
  Local bits%, current%, i%, s$

  ctrl.init_keys();

  Do
    If ctrl.keydown%(27) Then Exit Do ' Escape pressed.

    For i% = 1 To 12
      If ctrl.keydown%(i% + 144) Then ' Fn key i% pressed.
        print_ctrl_option(current%, 0)
        On Error Ignore
        Call ctrl$, ctrl.CLOSE
        err$ = Choice(Mm.ErrNo = 0, "", Mm.ErrMsg$)
        On Error Abort
        current% = Choice(is_valid%(i%), i%, 0)
        print_ctrl_option(current%, 1)
        ctrl$ = Choice(current% = 0, "no_ctrl", CTRL_IDS$(i%))
        On Error Ignore
        Call ctrl$, ctrl.OPEN
        err$ = Choice(Mm.ErrNo = 0, "", Mm.ErrMsg$)
        On Error Abort
        Do While ctrl.keydown%(i% + 144) : Loop ' Wait for key to be released.
        Exit For
      EndIf
    Next

    If err$ = "" Then
      Call ctrl$, bits%
      s$ = rpad$("Currently reading: " + ctrl_bits_to_string$(bits%), WIDTH%)
    Else
      s$ = rpad$(Mid$(err$, InStr(err$, ":") + 1), WIDTH%)
    EndIf
    print_at(0, 8 + NUM_CTRL%, s$)

    ' Compensate for Inkey$ not being the ideal way to read the keyboard.
    If Not InStr(Mm.Device$, "Colour Maximite 2") And InStr(ctrl$, "keys") Then Pause 100
  Loop
End Sub

Sub no_ctrl(x%)
  x% = 0
End Sub

Function is_valid%(fn_key%)
  is_valid% = fn_key% >= 1 And fn_key% <= NUM_CTRL%
End Function

' Prints text on the VGA screen at the given column and row.
'
' @param col%      the column, from 0.
' @param row%      the row, from 0.
' @param s$        the text.
' @param inverse%  print black on white instead of white on black.
Sub print_at(col%, row%, s$, inverse%)
  If Mm.Device$ = "MMB4L" Then
' TODO: Why does Print @ not work as expected in MMB4L
'    Print @(col%, row%), s$
    Console SetCursor col%, row%
    Console Inverse inverse%
    Print s$
    Console Inverse 0
  Else
    Local x% = col% * Mm.Info(FontWidth)
    Local y% = row% * Mm.Info(FontHeight) * 1.5
    Local fg% = Choice(inverse%, Rgb(Black), Rgb(White))
    Local bg% = Choice(inverse%, Rgb(White), Rgb(Black))
    Text x%, y%, s$, LT, 1, 1, fg%, bg%
  EndIf
End Sub

Sub print_ctrl_option(fn_key%, selected%)
  If Not is_valid%(fn_key%) Then Exit Sub
  Local s$ = rpad$("[F" + Str$(fn_key%) + "] ", 6) + CTRL_NAMES$(fn_key%)
  print_at(2, fn_key% + 5, s$, selected%)
End Sub

' Gets a string representation of bits read from a controller.
'
' @param bits%  controller state returned by controller read function.
Function ctrl_bits_to_string$(bits%)
  Static BUTTONS$(14) = ("R","Start","Home","Select","L","Down","Right","Up","Left","ZR","X","A","Y","B","ZL")

  If bits% = 0 Then
    ctrl_bits_to_string$ = "None"
    Exit Function
  EndIf

  ctrl_bits_to_string$ = Str$(bits%) + " = "
  Local count%, i%, s$
  For i% = 0 To Bound(BUTTONS$(), 1)
    If bits% And 2^i% Then
      s$ = BUTTONS$(i%)
      If count% > 0 Then Cat ctrl_bits_to_string$, ", "
      Cat ctrl_bits_to_string$, s$
      Inc count%
    EndIf
  Next
End Function

' Gets a string padded to a given width with spaces to the right.
'
' @param s$  the string.
' @param w%  the width.
' @return    the padded string.
'            If Len(s$) > w% then returns the unpadded string.
Function rpad$(s$, w%)
  rpad$ = s$
  If Len(s$) < w% Then rpad$ = s$ + Space$(w% - Len(s$))
End Function

controller_data_picomite:

Data "keys_cursor", "Keyboard: Cursor keys & Space"
Data "atari_a",     "Port A: Atari VCS joystick"
Data "atari_b",     "Port B: Atari VCS joystick"
Data "nes_a",       "Port A: NES gamepad"
Data "nes_b",       "Port B: NES gamepad"
Data "snes_a",      "Port A: SNES gamepad"
Data "snes_b",      "Port B: SNES gamepad"
Data "", ""

controller_data_cmm2:

Data "keys_cursor",   "Keyboard: Cursor keys & Space"
Data "atari_dx",      "CMM2 DX Atari VCS joystick"
Data "nes_dx",        "CMM2 DX NES gamepad (attached to Atari port with adapter)"
Data "wii_classic_3", "Port 1: Wii Classic Controller (I2C3)"
Data "wii_classic_1", "Port 2: Wii Classic Controller (I2C1)"
Data "wii_classic_2", "Port 3: Wii Classic Controller (I2C2)"
Data "", ""

controller_data_mmb4l:

Data "keys_cursor", "Keyboard: Cursor keys & Space"
Data "", ""
