'
'===========================================
' MaxiCalc 0.1
'
' A basic spreadsheet program for the Colour Maximite 2 computer
' by vegipete, December 2020
'
' Why? Well, why not?
' Kinda pointless considering the power of spreadsheets on a modern computer.
'
' Commands:
' (Note: cell references are in square brackets. Ex: [A1])
' Ctrl-Q            Quit
'       =           start entering an equation
'       "           start entering a string
' 0123456789+-.     start entering a numerical value
' Ctrl-K            recalculate
'
' Arrow keys        move around
' Shift-Arrows      select cells for Cut/Copy/Paste/Delete
' Alt-Arrows        select rows or columns
' Ctrl-Arrows       change height/width of current row/column
' PageUp/Down       move up and down in bigger steps
' Ctrl-PageUp/Down  move left and right in bigger steps
' Home              jump to cell A1
'
' Alt-Enter         edit current cell
'
' Generally, when entering or editing, left/right arrows, backspace and delete work as expected.
'   [Enter] finishes input
'   [Esc] cancels input
' When entering or editing numbers or strings, up/down arrows finish input and move.
' When entering equations, up/down arrows switch to cell select mode.
' When in cell select mode, arrow keys move, Esc cancels.
' When in cell select mode, Enter, Space or a math function {one of: )*+-/<=>\^} selects.
'
'===========================================
mode 9,8  '1024 x 768

option break 25  ' Ctrl-Y to stop program running - so that Ctrl-C can be used to copy

page write 3 : cls
page write 2 : cls
page write 1 : cls
page write 0 : cls

const MAX_X = 63
const MAX_Y = 255
const CELL_SZ = 64 ' max cell size in characters
const MIN_V = 12   ' min cell height
const MAX_V = 100  ' max cell height
const MIN_H = 12   ' min cell width
const MAX_H = 405  ' max cell width

font 1
const FH = MM.INFO(FONTHEIGHT)
const FW = MM.INFO(FONTWIDTH)
const CXV_BASE = MM.INFO(PAGE ADDRESS 2)  ' copy/cut/paste buffer

'===========================================
'#Include "GetFileDialog.inc"
' change dialog colours
d_shadow = &h404040   ' dark grey
d_frame  = &hB0B0B0   ' light grey
d_back   = &h101010   ' really dark grey
h_width  = MM.HRES * .75  ' help box dimensions
h_height = MM.VRES * .75
h_x      = (MM.HRES - h_width ) / 2
h_y      = (MM.VRES - h_height) / 2
ytop = 40 ' top margin - for menus or whatever

fc = &h606060   ' cell frame colour
hc = &hF0F000   ' active cell colour

' highlight colours - black/red/orange/yellow/green/cyan/blue/purple
dim hilicol(7) = (0,&hF04040,&hF0A040,&hF0F040,&h40F040,&h40F0F0,&h4040F0,&hF040F0)

dim selrange(4)   ' top left x, y, bottom right x, y of selection
dim sheetasc(MAX_X,MAX_Y) as string length CELL_SZ
dim sheetval(MAX_X,MAX_Y) as float

ClearSheet

'VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
' demo data at startup
restore DemoSheet
j = 3
sheetasc(3,1) = chr$(34) + "Colour Maximite 2 Parts List"
do
  read i
  if i <> 0 then
    if i > 0 then sheetasc(2,j) = str$(i)   ' number field
    read d$
    sheetasc(3,j) = chr$(34) + d$   ' text field
    k = len(sheetasc(3,j)) * FW
    sheetasc(3,0) = str$(min(MAX_H,max(val(sheetasc(3,0)),k)))
    read d$
    sheetasc(4,j) = chr$(34) + d$   ' text field
    k = len(sheetasc(4,j)) * FW
    sheetasc(4,0) = str$(min(MAX_H,max(val(sheetasc(4,0)),k)))
    j = j + 1
  else
    exit do
  endif
loop
sheetasc(3,j+1) = chr$(34) + "Total Number of Parts"
sheetasc(2,j+1) = "=sum([B3] to [B" + str$(j) + "])"
'AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA

print @(0,0) "MaxiCalc  by vegipete"
print @(MM.HRES-18*FW,0) "Press F1 for help"

ShowSheet
HighLight(3,posx,posy)
do
  if arrowpending then
    k = arrowpending
    arrowpending = 0
  else
    do : k = asc(inkey$) : loop until k <> 0
'? @(850,FH) k, bin$(keydown(7),8) "  "
  endif
  select case k
    case 17     ' "Ctrl-Q"   quit
      exit do   ' break out of menu loop
    case 8,127  ' backspace, delete
      sheetasc(posx,posy) = ""
      HighLight(3,posx,posy)
    case 13    ' ALT-Enter to edit current cell
      if keydown(7) and 17 then  ' Alt pressed (either one)
        if sheetasc(posx,posy) <> "" then   ' is something there?
          if left$(sheetasc(posx,posy),1) = "=" then
            n$ = InputEquat(sheetasc(posx,posy))
            if n$ <> "" then
              sheetasc(posx,posy) = n$
              sheetval(posx,posy) = val(CalcCellAsc(posx,posy))
            endif
            HighLight(3,posx,posy)
          elseif left$(sheetasc(posx,posy),1) = chr$(34) then  ' quote
            n$ = InputString(sheetasc(posx,posy))
            if n$ <> "" then
              sheetasc(posx,posy) = n$
              sheetval(posx,posy) = val(CalcCellAsc(posx,posy))
            endif
            HighLight(3,posx,posy)
          else
            n$ = InputNumber(sheetasc(posx,posy))
            if n$ <> "" then
              sheetasc(posx,posy) = n$
              sheetval(posx,posy) = val(n$)
            endif
            HighLight(3,posx,posy)
          endif
        endif
      endif
    case 11    ' Ctrl-K  rekalkulate
      ShowSheet
      HighLight(3,posx,posy)
    case  14   ' Ctrl-N - new sheet
      NewSheet
      HighLight(3,posx,posy)
    case  3    ' Ctrl-C - copy single cell
      selrange(0) = 0
      cxv_single$ = sheetasc(posx,posy)
      cxv_x = posx ' record source column (x-coord)
      cxv_y = posy ' record source row (y-coord)
    case 22    ' Ctrl-V - paste
      PasteRange
      ShowSheet
      HighLight(3,posx,posy)
    case 24    ' Ctrl-X - cut single cell
      selrange(0) = 0
      cxv_single$ = sheetasc(posx,posy)
      sheetasc(posx,posy) = ""
      HighLight(3,posx,posy)
      cxv_x = posx ' record source column (x-coord)
      cxv_y = posy ' record source row (y-coord)
    case 34    ' strings must start with a double quote "
      n$ = InputString(chr$(34))
      if n$ <> "" then sheetasc(posx,posy) = n$
      HighLight(3,posx,posy)
    case 61    ' equations must start with an equal sign =
      n$ = InputEquat("=")
      if n$ <> "" then
        sheetasc(posx,posy) = n$
        sheetval(posx,posy) = val(CalcCellAsc(posx,posy))
      endif
      HighLight(3,posx,posy)
    case 134     ' Home
      posx = 1 : posy = 1
      topx = 1 : topy = 1
      ShowSheet
      HighLight(3,posx,posy)
    case 145     ' F1  Help
      ShowHelp
    case 48 to 57, 43, 45, 46   ' numerals and +-.
      n$ = InputNumber(chr$(k))
      if n$ <> "" then
        sheetasc(posx,posy) = n$
        sheetval(posx,posy) = val(n$)
      endif
      HighLight(3,posx,posy)
    case 128,129,130,131,136,137,161,163  ' up,down,left,right,Page Up,Page Down
      HandleArrow(k)
  end select
loop
option break 3  ' back to normal
end

'===========================================
' turn highlight on or off for global cell x,y
' s = 1-7  ON, highlight colour
' s = 0    OFF
sub HighLight(s,x,y)
  local i,sx,sy

  ' calculate cell position - depends on cell widths/heights and which ones are visible
  sx = 30
  for i = topx to x-1
    sx = sx + val(sheetasc(i,0)) - 1
  next i
  sy = ytop + 20
  for i = topy to y-1
    sy = sy + val(sheetasc(0,i)) - 1
  next i

  ' show cell on sheet
  box sx-1,sy-1,val(sheetasc(x,0)),val(sheetasc(0,y)),1,fc,hilicol(s)
  t$ = left$(CalcCellAsc(x,y),fix(val(sheetasc(x,0))/FW))  ' limit string length
  text sx+1,sy+1,t$,"LT",1,1,rgb(white)*(s=0),hilicol(s)

  ' show full cell contents on edit row
  box  0,ytop-FH-1,     40,12,0,0,hilicol(s) ' erase previous cell address
  box 50,ytop-FH-1,MM.HRES,12,0,0,hilicol(s) ' erase previous cell contents
  if s then
    text  0,ytop-FH-1,X2Letter(x)+str$(y),"LT",1,1,0,hilicol(s)
    text 50,ytop-FH-1,sheetasc(x,y),"LT",1,1,0,hilicol(s)
  endif
end sub

'===========================================
' Highlight cell to select
' used to select a single cell other than the active one
sub HighSel(s,x,y)
  local i,sx,sy

  if (y <> posy) or (x <> posx) or selrange(0) then  ' only if not current cell
    sx = 30
    for i = topx to x-1
      sx = sx + val(sheetasc(i,0)) - 1
    next i
    sy = ytop + 20
    for i = topy to y-1
      sy = sy + val(sheetasc(0,i)) - 1
    next i

    box sx-1,sy-1,val(sheetasc(x,0)),val(sheetasc(0,y)),1,fc,hilicol(s)
    t$ = left$(CalcCellAsc(x,y),fix(val(sheetasc(x,0))/FW))  ' limit string length
    text sx+1,sy+1,t$,"LT",1,1,rgb(white)*(s=0),hilicol(s)
  endif
end sub

'===========================================
' highlight the range in the selrange() array
sub HighLightRange(col)
  local i,j

  for i = selrange(1) to selrange(3)
    for j = selrange(2) to selrange(4)
      if (i >= topx) and (i <= botx) and (j >= topy) and (j <= boty) then
        HighSel(col,i,j)
      endif
    next j
  next i
end sub

'===========================================
' Shift-arrows to select a range of cells
' Ends with [Esc], Ctrl-C, Ctrl-X or Ctrl-V, [Enter], [Del], [Backspace]
sub SelectRange(ak)
  local shift = 1  ' Shift was pressed when routine was called
  local i,j

  selrange(1) = posx : selrange(2) = posy
  selrange(3) = posx : selrange(4) = posy
  selrange(0) = 1

  do
    if shift then   ' Shift pressed so extend selection
      select case ak
        case  27      ' Esc
          exit do
        case 128      ' up
          if posy > 1 then
            posy = posy - 1
            selrange(2) = min(selrange(2), posy)
            if topy > posy then
              topy = posy
              ShowSheet
            endif
            HighLightRange(2)
          endif
        case 129,161  ' down
          if posy < MAX_Y then
            posy = posy + 1
            selrange(4) = max(selrange(4), posy)
            if boty < posy then
              topy = topy + 1
              ShowSheet
            endif
            HighLightRange(2)
          endif
        case 130      ' left
          if posx > 1 then
            posx = posx - 1
            selrange(1) = min(selrange(1), posx)
            if topx > posx then
              topx = posx
              ShowSheet
            endif
            HighLightRange(2)
          endif
        case 131,163  ' right
          if posx < MAX_X then
            posx = posx + 1
            selrange(3) = max(selrange(3), posx)
            if botx < posx then
              topx = topx + 1
              ShowSheet
            endif
            HighLightRange(2)
          endif
      end select
    else    ' Shift not pressed
      select case ak
        case 8,127  ' backspace, delete - erase selected range
          DeleteRange
          exit do
        case  128,129,130,131   ' arrow key, no shift
          exit do
        case  27    ' Esc
          exit do
        case   3    ' Ctrl-C - copy
          CopyRange
          exit do
        case  22    ' Ctrl-V - paste
          PasteRange
          ShowSheet
          HighLight(3,posx,posy)
        case  24    ' Ctrl-X - cut
          CopyRange
          DeleteRange
          exit do
      end select
    endif

    ' wait for next keypress
    do : ak = asc(inkey$) : loop until ak <> 0
    shift = keydown(7) and 136  '  set if either Shift pressed
'? @(850,FH) ak, bin$(keydown(7),8) "  "
  loop

  selrange(0) = 0   ' end select range mode
  ShowSheet
  HighLight(3,posx,posy)

end sub

'===========================================
' select complete rows or columns
sub SelectRowCol(ak)
  local alt = 1  ' Alt was pressed on entry
  local rcmode = 0  ' assume row select to start

  if (ak = 130) or (ak = 131) then rcmode = 1   ' column mode

  selrange(0) = 1
  if rcmode then  ' columns
    selrange(1) = posx : selrange(2) = 1
    selrange(3) = posx : selrange(4) = MAX_Y
  else    ' rows
    selrange(1) = 1 : selrange(2) = posy
    selrange(3) = MAX_X : selrange(4) = posy
  endif
  HighLightRange(7)   ' highlight row/column in purple

  do
    ' wait for next keypress
    do : ak = asc(inkey$) : loop until ak <> 0
    alt = keydown(7) and 17  ' set if either Alt pressed

    if alt then   ' alt pressed so extend selection
      select case ak
        case  27      ' Esc
          exit do
        case 128      ' up
          if (posy > 1) and (rcmode = 0) then
            posy = posy - 1
            selrange(2) = min(selrange(2), posy)
            if topy > posy then
              topy = posy
              ShowSheet
            endif
            HighLightRange(7)
          endif
        case 129      ' down
          if (posy < MAX_Y) and (rcmode = 0) then
            posy = posy + 1
            selrange(4) = max(selrange(4), posy)
            if boty < posy then
              topy = topy + 1
              ShowSheet
            endif
            HighLightRange(7)
          endif
        case 130      ' left
          if (posx > 1) and (rcmode = 1) then
            posx = posx - 1
            selrange(1) = min(selrange(1), posx)
            if topx > posx then
              topx = posx
              ShowSheet
            endif
            HighLightRange(7)
          endif
        case 131      ' right
          if (posx < MAX_X) and (rcmode = 1) then
            posx = posx + 1
            selrange(3) = max(selrange(3), posx)
            if botx < posx then
              topx = topx + 1
              ShowSheet
            endif
            HighLightRange(7)
          endif
      end select
    else    ' alt not pressed
      select case ak
        case 8,127  ' backspace, delete - erase selected range
          DeleteRange
          exit do
        case  128,129,130,131   ' arrow key, no alt
          exit do
        case  27    ' Esc
          exit do
        case   3    ' Ctrl-C - copy
          CopyRange
          if rcmode then
            posy = 1    ' column mode so move to top of column
          else
            posx = 1    ' row mode so move to left of row
          endif
          exit do
        case  22    ' Ctrl-V - paste
          PasteRange
          ShowSheet
          HighLight(3,posx,posy)
        case  24    ' Ctrl-X - cut
          CopyRange
          DeleteRange
          if rcmode then
            posy = 1
          else
            posx = 1
          endif
          exit do
      end select
    endif
  loop

  selrange(0) = 0   ' end select row/column mode
  ShowSheet
  HighLight(3,posx,posy)

end sub

'===========================================
' general arrow key processing
' move cell selection or change cell size if right-ctrl down
' parameter d holds arrow key ascii code
sub HandleArrow(d)
  local alt   = keydown(7) and 17  ' set if either Alt pressed
  local ctrl  = keydown(7) and 34  ' set if either Ctrl pressed
  local shift = keydown(7) and 136 ' set if either Shift pressed

  if shift then
    SelectRange(d)
    exit sub
  endif

  if alt then
    SelectRowCol(d)
    exit sub
  endif

  select case d
    case 128    ' up
      if ctrl then ' shorter
        if val(sheetasc(0,posy)) > MIN_V then
          sheetasc(0,posy) = str$(val(sheetasc(0,posy)) - 2)
          ShowSheet
          HighLight(3,posx,posy)
        endif
      else      ' move focus up
        if posy > 1 then
          HighLight(0,posx,posy)
          posy = posy - 1
          if topy > posy then
            topy = posy
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
    case 129    ' down
      if ctrl then ' taller
        if val(sheetasc(0,posy)) < MAX_V then
          sheetasc(0,posy) = str$(val(sheetasc(0,posy)) + 2)
          ShowSheet
          HighLight(3,posx,posy)
        endif
      else      ' move focus down
        if posy < MAX_Y then
          HighLight(0,posx,posy)
          posy = posy + 1
          if boty < posy then
            topy = topy + 1
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
    case 130    ' left
      if ctrl then ' narrower
        if val(sheetasc(posx,0)) > MIN_H then
          sheetasc(posx,0) = str$(val(sheetasc(posx,0)) - FW)
          ShowSheet
          HighLight(3,posx,posy)
        endif
      else      ' move focus left
        if posx > 1 then
          HighLight(0,posx,posy)
          posx = posx - 1
          if topx > posx then
            topx = posx
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
    case 131    ' right
      if ctrl then ' wider
        if val(sheetasc(posx,0)) < MAX_H then
          sheetasc(posx,0) = str$(val(sheetasc(posx,0)) + FW)
          ShowSheet
          HighLight(3,posx,posy)
        endif
      else      ' move focus right
        if posx < MAX_X then
          HighLight(0,posx,posy)
          posx = posx + 1
          if botx < posx then
            topx = topx + 1
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
    case 136    ' Page Up Key
      if ctrl then ' Page Left
        if posx > 1 then
          HighLight(0,posx,posy)
          posx = max(1, posx - 10)
          if topx > posx then
            topx = posx
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      else      ' Page Up
        if posy > 1 then
          HighLight(0,posx,posy)
          posy = max(1, posy - 25)
          if topy > posy then
            topy = posy
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
    case 137    ' Page Down Key
      if ctrl then ' Page Right
        if posx < MAX_X then
          HighLight(0,posx,posy)
          posx = min(MAX_X, posx + 10)
          if botx < posx then
            topx = topx + 10
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      else      ' Page Down
        if posy < MAX_Y then
          HighLight(0,posx,posy)
          posy = min(MAX_Y, posy + 25)
          if boty < posy then
            topy = topy + 25
            ShowSheet
          endif
          HighLight(3,posx,posy)
        endif
      endif
  end select

end sub

'===========================================
sub ShowSheet
  local x,y,xr,yr

  ' empty box top left
  box 0,ytop,30,20,1,fc,fc

  ' left edge index boxes
  yr = ytop + 20
  y = topy
  do
    box 0,yr-1,30,val(sheetasc(0,y)),1,0,fc
    text 2,yr+1,str$(y),"LT",1,1,0,fc
    yr = yr + val(sheetasc(0,y)) - 1
    y = y + 1
  loop until (yr > MM.VRES) or (y > MAX_Y)
  boty = y - 2
  if yr < MM.VRES then
    box 0,yr,MM.HRES,MM.VRES-yr,0,0,0   ' erase stuff below sheet
  endif

  ' top edge index boxes
  xr = 30
  x = topx
  do
    box xr,ytop-1,val(sheetasc(x,0)),21,1,0,fc
    text xr+2,ytop + 4,X2Letter(x),"LT",1,1,0,fc
    xr = xr + val(sheetasc(x,0)) - 1
    x = x + 1
  loop until (xr > MM.HRES) or (x > MAX_X)
  botx = x - 2
  if xr < MM.HRES then
    box xr,ytop,MM.HRES-xr,MM.VRES,0,0,0  ' erase stuff to right of sheet
  endif

  ' body cells
  x = topx
  xr = 30
  do
    y = topy
    yr = ytop + 20
    do
      box xr-1,yr-1,val(sheetasc(x,0)),val(sheetasc(0,y)),1,fc,0
      t$ = left$(CalcCellAsc(x,y),fix(val(sheetasc(x,0))/FW))  ' limit string length
      text xr+1,yr+1,t$,"LT",1  ' show cell contents
      yr = yr + val(sheetasc(0,y)) - 1 ' next line down
      y = y + 1
    loop until (yr > MM.VRES) or (y > MAX_Y)
    xr = xr + val(sheetasc(x,0)) - 1
    x = x + 1
  loop until (xr > MM.HRES) or (x > MAX_X)
end sub

'===========================================
sub NewSheet

'  msg$ = "Erase entire sheet and start a new blank one." + chr$(13)
'  msg$ = msg$ + "Are you sure?" + chr$(13) + chr$(13)
'  msg$ = msg$ + "[Esc] = No     [Y] = Yes"

  msg$ = "Erase entire sheet and start a new blank one.\nAre you sure?\n\n"
  msg$ = msg$ + "\[Esc\] = No     \[Y\] = Yes"
  if AlertBox(50,5,msg$) then
    ClearSheet
    ShowSheet
  endif
end sub

'===========================================
sub ClearSheet
  local integer i,j

  posx  = 1 : posy  = 1 ' hilighted cell
  selx = posx : sely = posy   ' individual selected cell
  topx = posx : topy = posy   ' top left corner of view
  botx = 0 : boty = 0   ' bottom right corner of view - calculated by ShowSheet
  arrowpending = 0      ' =128to131 when arrow key response needed
  selrange(0) = 0   ' not select mode
  cxv_single$ = ""  ' single cell to copy
  cxv_x = 0         ' copy/cut/paste block top left x
  cxv_y = 0         ' copy/cut/paste block top left y
  cxv_w = 0         ' copy/cut/paste block width
  cxv_h = 0         ' copy/cut/paste block height

  for i = 1 to MAX_X
    sheetasc(i,0) = "60"   ' initial column width (pixels)
  next i
  for j = 1 to MAX_Y
    sheetasc(0,j) = "16"   ' initial row height
  next j
  for i = 1 to MAX_X
    for j = 1 to MAX_Y
      sheetasc(i,j) = ""
    next j
  next i
  math set 0, sheetval()

end sub

'===========================================
' Return a string representation of the contents of cell
' Put numerical result into sheetval(x,y) also
function CalcCellAsc(x,y) as string
  local c$,t$

  c$ = left$(sheetasc(x,y),1)    ' examine first character
  if c$ = "=" then      ' equation so evaluate
    ' handle sum function here
    c$ = mid$(sheetasc(x,y),2)    ' remove equal sign from start
    if instr(ucase$(c$),"SUM(") then CalcSumFunc(c$)
    t$ = DeReferenceCells(c$)
    on error skip
    sheetval(x,y) = eval(t$)
    if MM.ERRNO then
      c$ = "ERR"
      sheetval(x,y) = 0
    else
      c$ = str$(sheetval(x,y))
    endif
  elseif c$ = chr$(34) then   ' string
    c$ = right$(sheetasc(x,y),len(sheetasc(x,y))-1)    ' chop off quote
  else                  ' number or empty cell
    sheetval(x,y) = val(sheetasc(x,y))
    c$ = sheetasc(x,y)
  endif
  CalcCellAsc = c$
end function

'===========================================
' Replace occurrences of SUM([refcell] to [refcell]) in string
'   with numerical value - ie, calculate the SUM
sub CalcSumFunc(c$)
  local integer m, n, i, j, xs, ys, xe, ye
  local sumval
  local r$

  sumval = 0
  m = instr(ucase$(c$),"SUM(")    ' find first SUM
  do while m
    n = instr(m,ucase$(c$),")")   ' find closing bracket
    r$ = mid$(c$,m+4,n-m)
    ExtractRange(r$,xs,ys,xe,ye)  ' extract number range to be summed
    for i = xs to xe
      for j = ys to ye
        r$ = CalcCellAsc(i,j)
        sumval = sumval + sheetval(i,j)
      next j
    next i
    c$ = left$(c$,m-1) + str$(sumval) + mid$(c$,n+1)  ' replace SUM() with numerical value
    m = instr(ucase$(c$),"SUM(")    ' find next SUM, if it exists
  loop
end sub

'===========================================
' Convert cell references to numerical values
' Return modified string
' cell reference starts with [
function DeReferenceCells(s$) as string
  local ref$, c$ = s$
  local i,j,x,y

  do
    i = instr(c$,"[")   ' look for start of cell reference
    if i then
      j = instr(c$,"]") ' find end of cell reference
      if j then
        ref$ = mid$(c$,i,j-i+1)   ' extract full reference
        x = Refer2X(ref$)
        y = Refer2Y(ref$)
        ' check for ERR
        c$ = left$(c$,i-1) + str$(sheetval(x,y)) + mid$(c$,j+1) ' replace cell ref with value
      else
        exit do  'error
      endif
    else
      exit do  ' no (more) cell reference found
    endif
  loop
  DeReferenceCells = c$
end function

'===========================================
' Return numerical value of reference row
function Refer2X(s$) as float
  local i,res

  res = 0
  for i = 1 to len(s$)
    select case mid$(s$,i,1)
      case "A" to "Z"   ' only capital letters, all others ignored
        res = res * 26 + asc(mid$(s$,i,1)) - 64
    end select
  next i
  Refer2X = res
end function

'===========================================
' Return numerical value of reference column
function Refer2Y(s$) as float
  local i,res

  res = 0
  for i = 1 to len(s$)
    select case mid$(s$,i,1)
      case "0" to "9"   ' only capital letters, all others ignored
        res = res * 10 + asc(mid$(s$,i,1)) - 48
    end select
  next i
  Refer2Y = res
end function

'===========================================
' Convert horizontal number to 1 or 2 letter string
function X2Letter(x) as string
  if x > 26 then
    X2Letter = chr$(64 + (x-1)\26) + chr$(65 + (x-1) mod 26)
  else
    X2Letter = chr$(64 + x)
  endif
end function

'===========================================
' Extract number range
' s$ should contain something in the form: "FUNC([ref1] to [ref2])"
' note: parameters are specified as integer so that values can be returned
sub ExtractRange(s$,x1 as integer,y1 as integer,x2 as integer,y2 as integer)
  local i,j

  i = instr(s$,"[")   ' look for start of cell reference
  if i then
    j = instr(s$,"]") ' find end of cell reference
    if j then
      x1 = Refer2X(mid$(s$,i,j-i+1))
      y1 = Refer2Y(mid$(s$,i,j-i+1))
      ' check for ERR
    else
      exit sub  'error
    endif
  else
    exit sub  ' error
  endif
  s$ = mid$(s$,j+1)   ' rip off first cell reference
  i = instr(s$,"[")   ' look for start of cell reference
  if i then
    j = instr(s$,"]") ' find end of cell reference
    if j then
      x2 = Refer2X(mid$(s$,i,j-i+1))
      y2 = Refer2Y(mid$(s$,i,j-i+1))
      ' check for ERR
    else
      exit sub  'error
    endif
  else
    exit sub  ' error
  endif
end sub

'===========================================
' Read a user input number
' first digit in s$
' 'Esc' to cancel
' 'Enter' or up/dn arrows to accept
function InputNumber(s$) as string
  local r$ = s$
  local k$
  local curp    ' cursor position

  curp = len(r$)
  arrowpending = 0
  do
    box 50,ytop-FH-1,MM.HRES,FH,0,0,0 ' erase previous
    text 50,ytop-FH-1,r$,"LT",1   ' show string so far
    do  ' loop waiting for user input
      do
        k$ = inkey$
        line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,&hFFFFFF*((timer/500) and 1)
      loop until k$ <> ""
      line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,0
      select case k$
        case chr$(8)      ' Backspace,
          if curp > 0 then
            r$ = left$(r$,curp-1) + mid$(r$,curp+1)
            curp = curp - 1
            exit do
          endif
        case chr$(13)     ' Enter
          if len(r$) = 0 then
            InputNumber = ""
          else
            InputNumber = str$(val(r$))
          endif
          exit function
        case chr$(27)     ' Esc = cancel
          InputNumber = ""
          exit function
        case chr$(127)    ' Delete
          if curp < len(r$) then
            r$ = left$(r$,curp) + mid$(r$,curp+2)
            exit do
          endif
        case chr$(128),chr$(129)  ' up/down arrow key
          if len(r$) = 0 then
            InputNumber = ""
          else
            InputNumber = str$(val(r$))
          endif
          arrowpending = asc(k$)
          exit function
        case chr$(130)     ' Arrow key left  - edit contents
          if curp > 0 then
            curp = curp - 1
            exit do
          endif
        case chr$(131)    ' Arrow key right - edit contents
          if curp < len(r$) then
            curp = curp + 1
            exit do
          endif
        case "0" to "9", "."   ' add/insert new digit
          r$ = left$(r$,curp) + k$ + mid$(r$,curp+1)
          curp = curp + 1
          exit do

        case "+", "-"
          if r$ = "" then
            r$ = k$   ' + and - must be first character
            curp = 1
            exit do
          endif
        'case
      end select
    loop
  loop

end function

'===========================================
' Read a user input string
' 'Esc' to cancel
' 'Enter' or arrows to accept
function InputString(s$) as string
  local r$ = s$
  local k$
  local curp    ' cursor position

  if r$ = "" then r$ = chr$(34)  ' start with a quote for display purposes
  curp = len(r$)
  arrowpending = 0
  do
    box 50,ytop-FH-1,MM.HRES,FH,0,0,0 ' erase previous
    text 50,ytop-FH-1,r$,"LT",1   ' show string so far
    do  ' loop waiting for user input
      do
        k$ = inkey$
        line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,&hFFFFFF*((timer/500) and 1)
      loop until k$ <> ""
      line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,0
      select case k$
        case chr$(8)      ' Backspace,
          if curp > 1 then
            r$ = left$(r$,curp-1) + mid$(r$,curp+1)
            'r$ = left$(r$,len(r$)-1)
            curp = curp - 1
            exit do
          endif
        case chr$(13)   ' Enter
          InputString = r$
          exit function
        case chr$(27)   ' Esc = cancel
          InputString = ""
          exit function
        case chr$(127)    ' Delete
          if curp < len(r$) then
            r$ = left$(r$,curp) + mid$(r$,curp+2)
            exit do
          endif
        case chr$(128),chr$(129)  ' up/down arrow key
          InputString = r$
          arrowpending = asc(k$)
          exit function
        case chr$(130)    ' Arrow key left  - edit contents
          if curp > 1 then
            curp = curp - 1
            exit do
          endif
        case chr$(131)    ' Arrow key right - edit contents
          if curp < len(r$) then
            curp = curp + 1
            exit do
          endif
        case > chr$(31)   ' add/insert new character
          r$ = left$(r$,curp) + k$ + mid$(r$,curp+1)
          curp = curp + 1
          exit do
      end select
    loop
  loop

end function

'===========================================
' Read a user input equation
' 'Esc' to cancel
' 'Enter' or arrows to accept
' cell references are in square brackets []
function InputEquat(s$) as string
  local r$ = s$
  local k$
  local curp    ' cursor position

  if r$ = "" then r$ = "="
  curp = len(r$)
  do
    box 50,ytop-FH-1,MM.HRES,FH,0,0,0 ' erase previous
    text 50,ytop-FH-1,r$,"LT",1   ' show string so far
    do  ' loop waiting for user input
      do
        k$ = inkey$
        line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,&hFFFFFF*((timer/500) and 1)
      loop until k$ <> ""
      line 49+curp*FW,ytop-FH-1,49+curp*FW,ytop-1,2,0
      select case k$
        case chr$(27)     ' Esc = cancel
          InputEquat = ""
          exit function
        case chr$(13)     ' Enter
          InputEquat = r$
          exit function
        case chr$(128),chr$(129)  ' Arrow key up/down - select cell
          selx = posx : sely = posy
          k$ = SelectCell(asc(k$))   ' gone into cell select mode
          r$ = left$(r$,curp) + k$ + mid$(r$,curp+1)
          curp = curp + len(k$)
          exit do
        case chr$(130)    ' Arrow key left  - edit contents
          if curp > 1 then
            curp = curp - 1
            exit do
          endif
        case chr$(131)    ' Arrow key right - edit contents
          if curp < len(r$) then
            curp = curp + 1
            exit do
          endif
        case chr$(8)      ' Backspace,
          if curp > 1 then
            r$ = left$(r$,curp-1) + mid$(r$,curp+1)
            curp = curp - 1
            exit do
          endif
        case chr$(127)    ' Delete
          if curp < len(r$) then
            r$ = left$(r$,curp) + mid$(r$,curp+2)
            exit do
          endif
        case > chr$(31)   ' add/insert new character
          r$ = left$(r$,curp) + k$ + mid$(r$,curp+1)
          curp = curp + 1
          exit do
      end select
    loop
  loop

end function

'===========================================
' Move around the sheet with the arrow keys
' Select cell with [Enter] or math symbol or [space]
' Cancel with [Esc]
function SelectCell(k) as string

  do
    select case k
      case 27     ' [Esc]  Cancel
        HighSel(0,selx,sely)
        SelectCell = ""
        exit function  ' no change
      case 128    ' up
        if sely > 1 then
          HighSel(0,selx,sely)
          sely = sely - 1
          HighSel(4,selx,sely)
        endif
      case 129    ' down
        if sely < MAX_Y then
          HighSel(0,selx,sely)
          sely = sely + 1
          HighSel(4,selx,sely)
        endif
      case 130    ' left
        if selx > 1 then
          HighSel(0,selx,sely)
          selx = selx - 1
          HighSel(4,selx,sely)
        endif
      case 131    ' right
        if selx < MAX_X then
          HighSel(0,selx,sely)
          selx = selx + 1
          HighSel(4,selx,sely)
        endif
      case 13     ' [Enter]
        HighSel(0,selx,sely)
        SelectCell = "[" + X2Letter(selx) + str$(sely) + "]"
        exit function
      case 32,41,42,43,45,47,60,61,62,92,94  ' [Space])*+-/<=>\^
        HighSel(0,selx,sely)
        SelectCell = "[" + X2Letter(selx) + str$(sely) + "]" + chr$(k)
        exit function
      'case
      'case
    end select
    do : k = asc(inkey$) : loop until k <> 0
  loop

end function

'===========================================
' Copy the contents of the block of cells to a buffer
' Might want to check size!
sub CopyRange
  local i,j,k,m
  local c$

  cxv_single$ = ""
  cxv_x = selrange(1) ' record source x-coord
  cxv_y = selrange(2) ' record source y-coord
  cxv_w = selrange(3) - selrange(1) + 1
  cxv_h = selrange(4) - selrange(2) + 1
  m = CXV_BASE    ' point to copy buffer

  for i = selrange(1) to selrange(3)
    for j = selrange(2) to selrange(4)
      c$ = sheetasc(i,j)
      for k = 1 to len(c$)
        poke byte m,asc(mid$(c$,k,1))
        m = m + 1
      next k
      poke byte m,0   ' flag end of cell contents (cell delimiter)
      m = m + 1
    next j
  next i

end sub

'===========================================
' Paste the contents of the buffer to a block of cells at posx,posy
' Need to paste single cell into a range
sub PasteRange
  local i,j,k,m,n
  local c$

  if cxv_single$ = "" then
    if cxv_w * cxv_h = 0 then exit sub  ' paste buffer is empty

    m = CXV_BASE    ' point to copy buffer

    for i = 1 to cxv_w
      for j = 1 to cxv_h
        c$ = ""
        do while peek(byte m)
          c$ = c$ + chr$(peek(byte m))
          m = m + 1   ' next character from buffer
        loop
        m = m + 1     ' move past null (cell delimiter)
        sheetasc(posx+i-1,posy+j-1) = OffsetRelative(c$,posx,posy) '+i-1,posy+j-1)
      next j
    next i
  else  ' paste a single cell
    if selrange(0) then   ' destination is a range
      for i = selrange(1) to selrange(3)
        for j = selrange(2) to selrange(4)
          sheetasc(i,j) = OffsetRelative(cxv_single$,i,j)
        next j
      next i
    else  ' destination is a single cell
      sheetasc(posx,posy) = OffsetRelative(cxv_single$,posx,posy)
    endif
  endif

end sub

'===========================================
sub DeleteRange
  local integer i,j

  for i = selrange(1) to selrange(3)
    for j = selrange(2) to selrange(4)
      sheetasc(i,j) = ""
    next j
  next i

end sub

'===========================================
' change relative cell references in c$
' global variables cxv_x & cxv_y hold cell source
' parameter dstx & dsty are destination cell coordinates
function OffsetRelative(src$,dstx,dsty) as string
  local integer k,n,coord
  local c$ = src$

'? @(100,0) ">>>" c$ "<<<",
  k = 1
  do while k < len(c$)
    if mid$(c$,k,1) = "[" then
      k = k + 1

      if mid$(c$,k,1) <> "$" then   ' relative reference so we need to change it
        'k pointing to (first) letter
        n = 1 : if mid$(c$,k+1,1) > "9" then n = 2   ' 2 letter column 'number'
        coord = Refer2X(mid$(c$,k,n)) + dstx - cxv_x ' adjust x (column) coordinate
        if coord < 1 then coord = 1 'error
        if coord > MAX_X then coord = MAX_X 'error
        c$ = left$(c$,k-1) + X2Letter(coord) + mid$(c$,k+n)
        k = k + n
      else
        do
          k = k + 1
        loop until mid$(c$,k,1) <= "9"
      endif

      if mid$(c$,k,1) <> "$" then   ' relative reference so we need to change it
        'k pointing to (first) digit
        n = 1  ' how many digits long?
        if mid$(c$,k+1,1) <> "]" then ' at least 2 digit row number
          n = 2
          if mid$(c$,k+2,1) <> "]" then n = 3 ' 3 digit row number
        endif
        coord = Refer2Y(mid$(c$,k,n)) + dsty - cxv_y ' adjust y (row) coordinate
        if coord < 1 then coord = 1 'error
        if coord > MAX_Y then coord = MAX_Y 'error
        c$ = left$(c$,k-1) + str$(coord) + mid$(c$,k+n)
        k = k + n
      endif

      do while mid$(c$,k,1) <> "]"
        k = k + 1
      loop
    endif
    k = k + 1
'? k,
  loop
'? @(100,14) ">>>" c$ "<<<"

  OffsetRelative = c$
end function

'===========================================
' Parse and display a string that may contain ESC codes.
' x,y are location on screen, y gets adjusted down screen for each line by FONTHEIGHT
' txt$ holds string
'
' recognized ESC codes:
'     \nnn      nnn is 3 digit ascii code of desired character
'                 - useful for special characters with high bit set
'     \\        display single back slash \
'     \'        display double quote "
'     \[        start a box (nesting will likely make a mess)
'     \]        end box - use these 2 to put characters in a box, best with a space each side
'     \n or \r  new line, use for multi-line in a single string
'
sub EscText(x,y,txt$)
  local integer bs, bxn, escpos, i
  local boxbeg(15), boxend(15)
  local te$ = txt$
  local t$ = ""

  do
    t$ = te$
    te$ = ""
    bs = 1
    bxn = 0
    escpos = instr(bs,t$,"\")    ' search for ESC sequences
    do while escpos
      select case mid$(t$,escpos+1,1)
        case "\"    ' double \\ means single one
          t$ = left$(t$,escpos-1) + mid$(t$,escpos+1)
          bs = escpos + 1   ' ignore this \ next time
        case "0","1","2","3","4","5","6","7","8","9"
          ascval = min(255,max(32,val(mid$(t$,escpos+1,3))))
          t$ = left$(t$,escpos-1) + chr$(ascval) + mid$(t$,escpos+4)
        case "["    ' start box
          bxn = min(bxn + 1,15)
          boxbeg(bxn) = (escpos-1)*MM.INFO(FONTWIDTH) - 2
          boxend(bxn) = (escpos)*MM.INFO(FONTWIDTH) - boxbeg(bxn) - 9 ' first guess, end later
          t$ = left$(t$,escpos-1) + mid$(t$,escpos+2)
        case "]"    ' end box
          boxend(bxn) = (escpos)*MM.INFO(FONTWIDTH) - boxbeg(bxn) - 9
          t$ = left$(t$,escpos-1) + mid$(t$,escpos+2)
        case "'"    ' change single to double quote
          t$ = left$(t$,escpos-1) + chr$(34) + mid$(t$,escpos+2)
        case "n","r"    ' line feed or carriage return
          te$ = mid$(t$,escpos+2)   ' split string, save tail for next loop
          t$ = left$(t$,escpos-1)
      end select
      escpos = instr(bs,t$,"\")     ' search for more ESC sequences
    loop
    text x,y,t$
    for i = 1 to bxn
      box x + boxbeg(i),y,boxend(i),MM.INFO(FONTHEIGHT)
    next i
    y = y + MM.INFO(FONTHEIGHT) + 2
  loop until te$ = ""

end sub

'===========================================
' Make an Alert Box, centered on screen, saving background image
' msg$ hold entire text to display, escape codes accepted
' wide and high are the size in characters, plus some boundary
' Y/y returns TRUE, anything else FALSE
function AlertBox(wide,high,msg$)
  local integer a_kp,a_width,a_height,a_x,a_y

  font 4    ' slightly larger with extended characters

  a_width  = (wide + 4) * MM.INFO(FONTWIDTH)  ' alert box dimensions
  a_height = (high + 2) * MM.INFO(FONTHEIGHT)
  a_x      = (MM.HRES - a_width ) / 2
  a_y      = (MM.VRES - a_height) / 2

  ' save underlying screen image to page 1
  page write 1
  blit a_x, a_y, 0, 0, a_width, a_height, 0
  page write 0

  ' draw Alert box
  rbox a_x + 7, a_y + 7, a_width -  8, a_height -  8, 10, d_shadow, d_shadow
  rbox a_x    , a_y    , a_width -  8, a_height -  8, 10,  d_frame, d_frame
  rbox a_x + 5, a_y + 5, a_width - 18, a_height - 17,  5,   d_back, d_back  ' text area

  EscText(a_x+30,a_y+12,msg$)
  font 1

  do : loop until inkey$  = ""  ' make sure no key is pressed
  do : a_kp = asc(inkey$) : loop until a_kp <> 0  ' wait for key press
  do : loop until inkey$ = ""  ' wait for key release
  ' restore original screen image
  blit 0, 0, a_x, a_y, a_width, a_height, 1
  AlertBox = 0
  if (a_kp = 89) or (a_kp = 121) then AlertBox = 1  ' only Y/y returns TRUE

end function

'===========================================
' Display the Help page
' Text to display is stored in DATA statements, escape codes accepted
' Help page dimensions are defined at start of program
' Underlying image is saved and restored
sub ShowHelp
  local x,y
  ' save underlying screen image to page 1
  page write 1
  blit h_x, h_y, 0, 0, h_width, h_height, 0
  page write 0

  ' draw help box
  rbox h_x + 7, h_y +  7, h_width -  8, h_height -  8, 10, d_shadow, d_shadow
  rbox h_x    , h_y     , h_width -  8, h_height -  8, 10,  d_frame, d_frame
  rbox h_x + 5, h_y + 30, h_width - 18, h_height - 42,  5,   d_back, d_back  ' text area

  text h_x+50,h_y+4,"MaxiCalc Help",,2,1,0,-1
  text h_x+51,h_y+4,"MaxiCalc Help",,2,1,0,-1
  text h_x+50,h_y+5,"MaxiCalc Help",,2,1,0,-1
  text h_x+51,h_y+5,"MaxiCalc Help",,2,1,0,-1

  restore Help01
  x = h_x+25
  y = h_y+45
  font 4    ' slightly larger with extended characters
  do
    read t$
    if t$ = "END" then
      exit do
    elseif t$ = "" then
      y = y + MM.INFO(FONTHEIGHT)/2 + 1   ' turn blank lines into half lines
    elseif t$ = "G" then  ' grayed out - not available
      colour &h303030
      read t$
      EscText(x,y,t$)
      colour rgb(white)
    else
      EscText(x,y,t$)
    endif
  loop
  font 1

  do : loop until inkey$ <> ""  ' wait for key press
  do : loop until inkey$  = ""  ' wait for key release
  ' restore original screen image
  blit 0, 0, h_x, h_y, h_width, h_height, 1
end sub

Help01:
data "\[\149\] \[\146\] \[\147\] \[\148\] to move."
data "    Hold \[Shift\] to select, \[Ctrl\] to change cell size."
data "    Hold \[Alt\] to select rows or columns."
data "\[PgUp\] \[PgDn\] to move faster, hold \[Ctrl\] to move sideways faster."
data "\[Home\] to move to cell [A1]"
data ""
data "In general, \[Esc\] cancels the current operation."
data ""
data "Cells can hold 3 types of info: numbers, text or equations."
data "    To enter text, start with a quote character \[\'\]"
data "    To enter an equation, start with an equals sign \[=\]"
data "    A number starts with \[0\] \[1\] \[2\] \[3\] \[4\] \[5\] \[6\] \[7\] \[8\] \[9\] \[.\] \[+\] or \[-\]"
data "Press \[Alt\]-\[Enter\] to edit an existing cell"
data ""
data "When entering an equation, use \[\146\] or \[\147\] to switch to cell select mode."
data "    Choose cell with \[Enter\] \[ Space \] \[)\] \[*\] \[+\] \[-\] \[/\] \[<\] \[=\] \[>\] \[\\\] or \[^\]"
data "Cells are in [] brackets. Use $ for absolute reference."
data "In general, any MMBasic function can be used."
data "The function SUM([refcell] to [refcell]) is also available."
data "\[Ctrl\]-\[K\] to recalculate entire spreadsheet."
data ""
data "\[Del\] and \[Backspace\] clear the highlighted cell or range."
data ""
data "For Cut, Copy and Paste, use \[Ctrl\]-\[X\], \[Ctrl\]-\[C\] and \[Ctrl\]-\[V\]."
data ""
data "Press \[Ctrl\]-\[Q\] to quit."
data ""
data "Press \[Ctrl\]-\[N\] to start a new file."
data "G","Press \[Ctrl\]-\[O\] to open a file."
data "G","Press \[Ctrl\]-\[S\] to save the current file."
data "END"

' list of functions (unused as such by program, although usable in equations)
data "*", "+", "-", "/", "<<", ">>", "\", "^", "=", "<", ">", "<=", ">="
data "Abs(", "ACos(", "AND", "Asc(", "ASin(", "Atan2(", "Atn("
data "Chr$(", "Cint(", "Cos(", "Date$", "DateTime$(", "Day$(", "Deg("
data "Dir$(", "Distance(", "Epoch(", "Exp(", "Fix(", "Format$("
data "Hex$(", "Instr(", "Int(", "INV", "LCase$(", "Left$(", "Len("
data "Log(", "Math(", "Max(", "Mid$(", "Min(", "MOD", "NOT", "Oct$(", "OR"
data "Pi", "Port(", "Pulsin(", "Rad(", "RGB(", "Right$(", "Rnd", "Rnd("
data "Sgn(", "Sin(", "Space$(", , "Sqr(", "Str$(", "Str2bin(", "String$("
data "Tan(", "TEMPR(", "Time$", "Timer", "UCase$(", "Val(", "XOR"

DemoSheet:
data 1, "PC board", "103x130mm."
data 1, "15 Pin D-Sub Socket", "AMP 1-1734530-1"
data 1, "3.5mm Stereo Socket", "Switchcraft 35RASMT4BHNTRX"
data 1, "3mm Dual LED Assembly", "Dialight 553-0112F"
data 1, "USB Type-A Upright PCB Socket RA", "Amphenol FCI 73725-0110BLF"
data 1, "USB V2.0 Type B Connector", "Amphenol FCI 61729-0010BLF"
data 1, "SPDT toggle switch", "EG2364"
data 1, "SD Card Socket", "Hirose DM1AA-SF-PEJ"
data 2, "PCB Header Socket, 2mm Pitch", "SAMTEC MMS-140-01-L-DV"
data 1, "Coin Cell Holder", "HARWIN S8411-45R"
data 1, "Battery", "CR1220"
data 1, "IC Socket", "14-pin"
data 1, "Instrument Case 140x110x35", "Multicomp Pro G738"
data 1, "MCU core board", "Waveshare CoreH743I STM32H743IIT6"
data 1, "USB to Serial Bridge", "Microchip MCP2221A-I/P"
data 2, "Capacitor Multilayer Ceramic (MLCC)", "100nF 50VW"
data 1, "Capacitor Tantalum", "1µF 16VW"
data 2, "Capacitor Tantalum", "10µF 16VW"
data 3, "Resistor (metal film, 0.25W, 1%)", "75 ohm"
data 13, "Resistor (metal film, 0.25W, 1%)", "120 ohm"
data 19, "Resistor (metal film, 0.25W, 1%)", "240 ohm"
data 1, "Resistor (carbon, 0.25W, 10%)", "2.2 ohm"
data 1, "Resistor (carbon, 0.25W, 10%)", "10 ohm"
data 2, "Resistor (carbon, 0.25W, 10%)", "1K ohm"
data 6, "Resistor (carbon, 0.25W, 10%)", "10K ohm"
data -1, "Optional", "special features"
data 1, "temperature sensor TO92 package", "Dallas DS18B20+"
data 1, "IR Remote Receiver 38KHz", "Vishay TSOP4838"
data 1, "Resistor for temp sensor", "4.7K ohm"
data 0
