  '
  OPTION EXPLICIT
  OPTION DEFAULT NONE
  option escape
  
  dim string uars, uars1
  '
  'Program to display the radar data in list form or on graphical display
  'Radar sensor used is LD2450LD
  'Note: zone filtering is currently not implemented
  '
  '
  '************ HW dependencies, change as required ****************
  '
  '++++++ definition of serial line: RX/TX pins, com-number ++++++
  '       baudrate, buffersize, ISR name, buffersize for interrupt
  
  'config 1
  'setpin gp0,gp1,com1
  'const s_comspec = "com1:256000, 256, ser_int, 30"
  
  'config 2
  setpin gp20,gp21,com2
  const s_comspec = "com2:256000, 256, ser_int, 30"
  '****************** end dependencies *****************************
  
  '**********+ LD2450LD parameter definitions ***************
  
  'definition of command and reply strings for the sensor
  'the definition conforms to the operations manual of the LD2450LD sensor
  'in the actual sent/received packets data value low/high bytes are corrected to higb byte first
  'this is done in the send/receive routines
  
  '+++++++ definition of synch/ACK characters +++++++++++++++
  'IMPORTANT NOTE: the name of "HT_arry()" is fixed !!!
  dim string ht_arry(6)
  ht_arry(0) = "AAFF0300" : ht_arry(1) = "55CC"         'data header / trailer
  ht_arry(2) = "FDFCFBFA" : ht_arry(3) = "04030201"     'ACK header / trailer
  const data_offs = 0, ACK_offs = 2                    'define offsets into the arrays
  '-------- end synch chars def -----------------------------
  
  '+++++++ definition of command characters +++++++++++++++++
  dim string cmd_head = "FDFCFBFA", cmd_trail = "04030201"      'cmd header/trailer
  dim string cmd_m_track = "0090", cmd_s_track = "0080"         'cmd code to track multi/single object
  dim string cmd_beg_conf = "00FF0001", cmd_end_conf = "00FE"   'cmd begin/end configuration
  '-------- end command chars def ---------------------------
  
  '+++++++ LD2450LD field definition tool +++++++++++++++++++
  dim integer std_d_arry(32,4), fld_idx                             'array storing definition data and return value
  dim string std_d_arry_s(bound(std_d_arry(),1))                        'string return array
  'decode_data definitions
  const l_end=0, b_end=1, bin_code=0, hex_code=1, str_code=2
  const fldsize=0, endian=1, coding=2, result=3
  const err_not_hex=1, err_strlen=2
  '---------- end field definition tool ---------------------
  '********** end LD2450LD parameter definitions ************
  
  '+++++++++ variables section ++++++++++++++++++++++++++++++
  dim string inchar, cons_char, out_str
  dim integer idx, stat, synch, targ
  dim float t_tmr, alpha, dist
  dim integer param_arry(3,4)                 'contains the radar measurements for 3 targets
  dim integer rcv_mode                        '0 ... sync on header, 1 ... get data, 2 ... sync on trailer
  '--------- end variables ----------------------------------
  
  '++++++++++ eventflag definitions +++++++++++++++++++++++++
  dim integer  ef_clu                             '64 bit eventflag cluster
  const ser_ef = 0, radar_ef = 1, ser_lock = 2, cons_ef = 3, data_ef = 4    'eventflag numbers
  const ovr_ef = 5                  'serial line input-string overrun (>200 bytes)
  const keep_ef = 0, clear_ef = 1                 'used in ef sub/functions
  '---------- end eventflags --------------------------------
  
  '+++++++++ general purpose flag definitions +++++++++++++++
  dim integer gp_clu                                'GP flags cluster, 64 bits
  const int_dis = 1                                 'if set serial line interrupts are ignored
  const dsp_enab = 2, prt_enab = 3, stat_enab = 4   'if set display, print and statistics are enabled
  const trk_mode = 5                                'clear ... single target, set ... multi target
  const z_mode = 6                                  '0 ... suppress / 1 ... eanble zero x-axis tuples
  
  '+++++++++ graphics definitions +++++++++++++++++++++++++++
  'the values returned by the sensor should be in mm but accuracy leaves a lot to desired.
  'Maybe some calibration against real measured distances would help....
  
  'the below values to get the correction factors are chosen for conveniance only
  const v_fact = mm.vres/6000 'pixel/mm (3000 mm left and 3000 mm right)
  const h_fact = mm.hres/3000 'pixel/mm (3000 mm forward)
  const x_0 = mm.hres/2, y_0 = mm.vres    'x0 and y0 on the display (lower middle)
  
  'colors for target #1 to #3 on LCD display
  dim integer x_pos(3), y_pos(3), x_draw(3), y_draw(3)
  dim integer col(3)
  col(0) = rgb(yellow)    'target 0
  col(1) = rgb(green)     'target 1
  col(2) = rgb(magenta)   'target 2
  '--------- end graphics -----------------------------------
  
  '+++++++++++ serial channel definitions ++++++++++++++++++++
  const s_chan = 1                        'serial channel number
  '------------ end serial channel ---------------------------
  
  '--------- end variables/constants ------------------------
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++ Begin of executable code +++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '++++++++ init section ++++++++++++++++++++++++++++++++++++
  
  '+++++ define LD2450LD header/trailer and data fields +++++
  '+++++ as defined in the manual                       +++++
  
  'standard data record definition (position output from sensor)
  
  'data lines 1 ... n-1
  'field values in setup_field_data() for default data input:
  'field #1 ... length of data in bytes (must not be 0 !!)
  'field #2 ... datatype: bin=binary byte, hex=hex char (2 chars for a byte), str=string (byte by byte)
  'field #3 ... big/little endian: b= big endian, l=little endian, valid for all 3 datatypes
  'field #4 ... generates this variable name for result
  ' results are in:
  '   d_arry(field-name,3) or
  '   s_arry(field-name)
  ' depending on the datatype
  
  'data line n (last line) is the array definition line
  'field #1 ... must be 0
  'field #2 and 3 ... don't care
  'field #4 ... definition/result array for integers
  'field #5 ... result array for strings (the indexes are the same)
  
  setup_field_data(2, "bin", "l","t1_x")  'target 1 x-data
  setup_field_data(2, "bin", "l","t1_y")  'target 1 y-data
  setup_field_data(2, "bin", "l","t1_speed")  'target 1 speed data
  setup_field_data(2, "bin", "l","t1_res")  'target 1 resolution
  setup_field_data(2, "bin", "l","t2_x")  'target 2 x-data
  setup_field_data(2, "bin", "l","t2_y")  'target 2 y-data
  setup_field_data(2, "bin", "l","t2_speed")  'target 2 speed data
  setup_field_data(2, "bin", "l","t2_res")  'target 2 resolution
  setup_field_data(2, "bin", "l","t3_x")  'target 3 x-data
  setup_field_data(2, "bin", "l","t3_y")  'target 3 y-data
  setup_field_data(2, "bin", "l","t3_speed")  'target 3 speed data
  setup_field_data(2, "bin", "l","t3_res")  'target 3 resolution
  setup_field_data(0,,, "d_arry", "s_arry")   'define target arrays d_arry = integer, s_arry = string
  '------------- end definition of fields ---------------------
  
  set_ef(gp_clu,int_dis)       'serial data are handled synchronous not by interrupt for setting up the radar
  open s_comspec as s_chan    'open serial port to sensor
  
  'setup and initialize sensor
  set_sync(data_offs)     'setup header/trailer for sensor data processing
  stat = init_sensor (s_chan, cmd_m_track)      'send multi-target command to sensor
  if stat then
    print "%Radar-F-InitErr, error at init in startup ... exit ";stat
end
  end if
  set_ef(gp_clu,trk_mode)               'tracking mode is multi
  
  clr_ef(gp_clu,int_dis)  'enable interrupt processing on serial line to drive the processing
  
  on key char_isr             'enable character interrupts from console
  
  'say hello to the outside world
  print
  print "%Radar-I-Start, measurement starts"
  print
  
  'simulate console input to display help text
  cons_char = "H"
  cons_menue
  
  '----------------------------------------------------------
  '--------------- end init section -------------------------
  '----------------------------------------------------------
  
  'processing of sensor data is as follows:
  ' serial interrupr sets ser_ef
  ' if ser_ef is set, the data are parsed/checked and computed into param_arry()
  ' if parsing is complete, ser_ef is reset and radar_ef is set to signal new data is available
  'if radar_ef is set, the data are just printed out
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++++++ main code ++++++++++++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '+++++++++++ Main loop ++++++++++++++++++++++++++++++++++++
  ' this loop is executed as main program
  ' first part is the processing of event-flags and statistics related stuff
  ' 1) enter the idel-loop if no event-flag is set
  '    do statistics related stuff
  '    individual user code should be inserted here
  '
  ' 2) process the ef-handler if any ebent-flag is set
  '    do some statistics stuff
  '    if ser_ef is set, process a data record from the sensor
  '    if data_ef is set, decode, compute and process the radar data
  '    if cons_ef is set, process the console input
  
  do
    '++++++++++++++++ begin idle loop +++++++++++++++++++++++
    '********************************************************
    
    ' wait for an eventflag to become set else loop => NULL PROCESS
    do while not test_ef(ef_clu, -1)         'idle loop, wait for any eventflag to be set
      
      'idle time statstics
      set_exec_timer(0)       'start timer for idle loop
      
      'time statistics of radar data processing
      t_tmr = get_exec_timer(1)
      if t_tmr <> 0 and test_ef(gp_clu, stat_enab) = 1 then print "radar tmr "; t_tmr 'print timer for radar processing
      
      '+++++++++++++++++++ user code ++++++++++++++++++++++++++
      '++++++++++++++++++++++++++++++++++++++++++++++++++++++++
      
      '----> insert your user code here but do not exceed the available time for execution
      '----> time available is "idle tmr" from statistics menue
      '----> otherwise you may lose some measurements
      
      '----- end user code ------------------------------------
      '--------------------------------------------------------
    loop
    '---------------- end idle loop ---------------------------
    '----------------------------------------------------------
    
    
    '++++++++++++ ef-handler if any eventflag set  ++++++++++++
    
    '************ sensor data should be here ******************
    'print "ef ";bin$(ef_clu,16)
    'handle statistics for idle loop
    if test_ef(gp_clu, stat_enab) then print "idle tmr "; get_exec_timer(0)      'print timer for idle loop
    set_exec_timer(1)          'start timer for radar processing
    
    if test_ef(ef_clu,ovr_ef) then
      clr_ef(ef_clu,ovr_ef)
      print "%I-restart, Buffer overrun ... resyncing"
      set_sync(data_offs)     'setup header/trailer for sensor data processing
      stat = init_sensor (s_chan, cmd_m_track)      'send multi-target command to sensor
      if stat then
        print "%Radar-F-init-fail, failed to re-init sensor ... exit"
end
      end if
    end if
    
    'arrive here if sensor data received and thus ser_ef was set
    if test_ef(ef_clu, ser_ef) then         'ser_ef set
      clr_ef(ef_clu, ser_ef)                'reset serial eventflag
      do
        'print "proc rec, __ar_offs ";__ar_offs
        stat = proc_in_record(inchar, out_str, ef_clu, data_ef)  'sget data, data_ef is set if payload available
        if stat <> 0 then exit do       'read all avalable data, skipping older data
      loop
      'print "proc rec ok"
    end if
    
    if test_ef(ef_clu, data_ef) then                         'data are in out_str to be processed
      clr_ef(ef_clu, data_ef)                                 'reset data ef
      stat = decode_data(out_str, d_arry(), s_arry())        'decode serial buffer
      if stat then print "%decode-W-daterr, data error"
      
      for targ = 0 to 2           'calculate radar data for each target
        calc_val(param_arry(), targ)      'compute values
      next targ
      
      proc_data                   'compute distance and angle and print/display if required
      'set_ef (ef_clu, radar_ef)        'signal new radar data available
    end if
    
    'do_exec_ef(ef_clu, cons_ef, "cons_menue", 1, clear_ef)       'execute console menue if ef is set
    if test_ef(ef_clu, cons_ef) then
      clr_ef(ef_clu,cons_ef)
      cons_menue
    end if
  loop
  '----------------------------------------------
  '-------------- end main loop -----------------
  '----------------------------------------------
  
  '+++++++++++ DEV AREA +++++++++++++++++++
  
sub do_exec_ef(ef_clstr as integer, ef_no as integer, uar as string, tst_val as integer, bit_reset as integer)
  local integer bitval
  '***************** Warning *******************
  ' if used to call commands like 'print "Test"' this results in nested execute
  ' which is not supported but works for strings < 6 chars
  '**************** Warning end ****************
  
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  'tst_val ... value to test ef for (0 or 1)
  bitval = test_ef(ef_clstr, ef_no)
  if bit_reset = 1 then clr_ef(ef_clstr, ef_no)    'reset ef if requried
  
  if  bitval = tst_val then                'if ef has the right value
    if(uar <> "") then                     'and if there is something to execute
      execute uar                          'execute user-action-routine if ef is set
    end if
  end if
  
end sub
  '----------------------------------------------------
function get_ack_stat(in_str as string) as integer
  'get ACK return status value (byte 5 & 6 in the ACK string)
  get_ack_stat = 1
  if hex$(asc(mid$(in_str,5,2)),4) = "0000" then get_ack_stat = 0
end function
  
  '---------- END DEV AREA ----------------
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++
  '++++++++++ SUBs and FUNCTIONs ++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '++++++++++ app specific functions and subs +++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '+++++++ display console menue ++++++++++++++++++++++
sub cons_menue
  if asc(cons_char) < 32 then exit sub    'discard control chars
  select case cons_char
    case "D"
      toggle_ef(gp_clu,dsp_enab)       'toggle flag
      do_exec_ef(gp_clu, dsp_enab, "print \qD on\q", 1, keep_ef)
      do_exec_ef(gp_clu, dsp_enab, "print \qD off\q", 0, keep_ef)
      cls
    case "P"
      toggle_ef(gp_clu,prt_enab)       'toggle flag
      do_exec_ef(gp_clu, prt_enab, "print \qP on\q", 1, keep_ef)
      do_exec_ef(gp_clu, prt_enab, "print \qP off\q", 0, keep_ef)
    case "S"
      toggle_ef(gp_clu,stat_enab)       'toggle flag
      do_exec_ef(gp_clu, stat_enab, "print \qS on\q", 1, keep_ef)
      do_exec_ef(gp_clu, stat_enab, "print \qS off\q", 0, keep_ef)
    case "X"
end
    case "1"
      stat = init_sensor (s_chan, cmd_s_track)
      if not stat then
        print "%Radar-I-mode, mode set to single target"
      else
        print "%Radar-W-mode_fail, failed to change mode to single target ";stat
        set_sync(data_offs)     'set to default
      end if
      clr_ef(gp_clu,trk_mode)        'tracking mode is single
    case "M"
      stat = init_sensor (s_chan, cmd_m_track)
      if not stat then
        print "%Radar-I-mode, mode set to multi target"
      else
        print "%Radar-W-mode_fail, failed to change mode to multi target "; stat
        set_sync(data_offs)     'set to default
      end if
      set_ef(gp_clu,trk_mode)        'tracking mode is multi
    case "Z"
      toggle_ef(gp_clu,z_mode)       'toggle flag
      do_exec_ef(gp_clu, z_mode, "print \qZ on\q", 1, keep_ef)
      do_exec_ef(gp_clu, z_mode, "print \qZ off\q", 0, keep_ef)
    case "R"
      print "%I-restart, ... resyncing"
      set_sync(data_offs)     'setup header/trailer for sensor data processing
      stat = init_sensor (s_chan, cmd_m_track)      'send multi-target command to sensor
      if stat then
        print "%Radar-F-init-fail, failed to re-init sensor"
        set_sync(data_offs)     'set to default
      end if
    case "H"
      print "+++++++++++"
      print "D ... enable display"
      print "P ... print data on console"
      print "S ... print timer statistics on console"
      print "1 ... single target tracking"
      print "M ... multi-target tracking"
      print "Z ... enable/suppress zero x-axis values"
      print "R ... restart"
      print "X ... Exit"
      print "H ... this help text"
      print "-----------"
    case else
      print cons_char;" is not a valid command ...."
  end select
  cons_char = chr$(0)
end sub
  '----------------------------------------------------
  
sub set_sync(array_offset as integer) as integer
  'set the header/trailer arrays and init some variables
  on error skip 1   'define some variables needed in the next line
  dim integer  __ar_offs, substate, trl_c_match, c_chr_ptr, c_hd_ptr, c_tr_ptr
  'init variables
  substate = 0
  c_hd_ptr = 1
  c_tr_ptr = 1
  c_chr_ptr = 1
  __ar_offs = array_offset    'point to header/trailer description
end sub
  
  '----------------------------------------------------
function get_sync() as integer
  'get the header/trailer offset
  get_sync = __ar_offs    'pointer to header/trailer description
  
end function
  '----------------------------------------------------
  
function proc_in_record(inbuff as string, out_char as string, efc as integer, ef as integer) as integer
  'is usually called at every serial line interrupt to decode header, payload and trailer
  ' IMPORTANT NOTE: ***** the array name "ht_arry()" is fixed ***********
  'return values are:
  ' 0 ... data/payload is available and ef is set
  ' 1 ... base state, nothing in progress
  ' 2 ... header decode in progress
  ' 3 ... header is found/decoded now getting payload or trailer
  ' 4 ... trailer decode in progress
  
  'inbuff ... data input, is adjusted and trimmed off with header/trailer
  '           continuation is possible on next call with growing buffer
  ' out_char ... returns payload
  ' efc ... eventflag cluster
  ' ef ... eventflag
  
  proc_in_record = 1                     'indicate base state
  
  if substate = 0 or substate = 2 then                'sync on first char of header-string
    ' search for header
    do while c_chr_ptr <= len(inbuff)    'scan thru whatever is already received
      'try header
      if hex$(asc(mid$(inbuff,c_chr_ptr)),2) = mid$(ht_arry(__ar_offs),c_hd_ptr,2) then   'compare 2 sync chars with input
        if c_hd_ptr = len(ht_arry(__ar_offs))-1 then    'check if end of sync string
          inbuff = right$(inbuff, len(inbuff)-(c_chr_ptr)) 'remove header
          c_chr_ptr = 1
          c_hd_ptr = 1
          substate = 1          'mark header found
          proc_in_record = 3         'indicate header processed
          exit do
        else                    'char match, but not at end of sync
          if c_chr_ptr+1 < len(inbuff) then       'if there are still chars in inbuff
            inc c_chr_ptr,1           'bump pointer to input char
            inc c_hd_ptr,2            'bump pointer to sync chars
          else                  'no more chars in buffer
            proc_in_record = 2          'indicate header in progress
            exit function            'wait for more chars in inbuff to come
          end if
          
        end if
      else                      'no char match
        c_hd_ptr = 1            'no match, restore header pointer
        inc c_chr_ptr,1           'bump pointer to input char
      end if
    loop
    if proc_in_record = 1 then        'end of string without any match, throw away all but possible header begin
      if len(inbuff) > len(ht_arry(__ar_offs)) then
        mid$(inbuff,1,len(inbuff)-(len(ht_arry(__ar_offs))/2)) = ""   'delete all garbage chars
      end if
    end if
  end if
  
  if substate = 1 then                'sync on first char of header-string
    ' header found, now find trailer and exctract payload before start of trailer
    do while c_chr_ptr <= len(inbuff)    'scan thru whatever is already received
      'try trailer
      if hex$(asc(mid$(inbuff,c_chr_ptr)),2) = mid$(ht_arry(__ar_offs+1),c_hd_ptr,2) then   '2 sync chars match input
        if c_hd_ptr = len(ht_arry(__ar_offs+1))-1 then    'check if end of sync string
          mid$(inbuff, c_chr_ptr+1-(len(ht_arry(__ar_offs+1))/2), len(ht_arry(__ar_offs+1))/2) = ""   'remove trailer
          c_chr_ptr = c_chr_ptr - len(ht_arry(__ar_offs+1))/2   'point to end of payload
          out_char = left$(inbuff,c_chr_ptr )           'get payload for return
          mid$(inbuff,1, c_chr_ptr) = ""                ' remove payload from input buffer
          'if len(inbuff)>0 then print "resid ";len(inbuff)
          substate = 2          'mark trailer found
          c_chr_ptr = 1
          c_hd_ptr = 1
          proc_in_record = 0         'indicate payload available
          set_ef(efc, ef)
          exit function
        else                    'char match, but not at end of sync
          if c_chr_ptr < len(inbuff) then       'if there are still chars in inbuff
            inc c_chr_ptr,1           'bump pointer to input char
            inc c_hd_ptr,2            'bump pointer to sync chars
          else                  'no more chars in buffer
            proc_in_record = 4         'indicate trailer in progress
            exit function            'wait for more chars in inbuff to come
          end if
          
        end if
      else                      'no char match
        c_hd_ptr = 1            'no match, restore header pointer
        inc c_chr_ptr,1           'bump pointer to input char
        proc_in_record = 3         'indicate getting payload / searching for trailer
      end if
    loop
  end if
  
end function
  '----------------------------------------------------
  
sub proc_data
  local integer idx
  'processing of the radar data handed over in param_arry()
  
  for idx = 0 to 2
    if param_arry(idx,0)<> 0 then                                     'check if meaningful data are available
      alpha = 180 + atn(param_arry(idx,1)/param_arry(idx,0))*360/pi   'calc distance vector angle
      dist = sqr(param_arry(idx,0)^2 + param_arry(idx,1)^2)           'calc vector distance
    end if
  next idx
  
  if test_ef(gp_clu, prt_enab) then print_data    'print data on console
  if test_ef(gp_clu, dsp_enab) then draw_data     'draw position on display
end sub
  '----------------------------------------------------
  
sub draw_data
  local integer idx
  for idx = 0 to 3
    circle x_draw(idx), y_draw(idx), 5, 1, 1, rgb(black), rgb(black)      'delete old position
    if param_arry(idx,0) <> 0 then
      x_pos(idx) = param_arry(idx,0)*h_fact      'scale to diaplay
      y_pos(idx) = param_arry(idx,1)*v_fact
      x_draw(idx) = x_0 + x_pos(idx)                'get display position
      y_draw(idx) = y_0 - y_pos(idx)
      'circle x_draw(idx), y_draw(idx), 5, 1, 1, &hFF<<(8*idx), &hFF<<(8*idx)    'draw new position
      circle x_draw(idx), y_draw(idx), 5, 1, 1, col(idx), col(idx)    'draw new position
    end if
  next idx
end sub
  '----------------------------------------------------
  
sub print_data
  local integer idx
  for idx = 0 to 2
    if (test_ef(gp_clu, z_mode) = 1) or (param_arry(idx,0)<> 0) then    'check if printout possible
      print "Target: ";idx, param_arry(idx,0), param_arry(idx,1), param_arry(idx,2), param_arry(idx,3),    'print results
      print format$(alpha,"% 3.0f"), format$(dist,"% 4.0f")
    end if
  next idx
end sub
  '----------------------------------------------------
  
function send_cmd(s_chan as integer, cmd_head as string, cmd_str as string, cmd_trail as string) as integer
  'send command to sensor
  'command is as string of hex bytes as defined in the manual, commands are reformatted to low byte first
  'header and trailer are strings and left unchanged to comply with the manual
  local string outstr, in_str
  outstr = hex_to_bin_str(cmd_head)                                     'add header
  outstr = outstr + hex_to_bin_str(sw_byte(hex$(len(cmd_str)/2,4)))     'add command/data length
  outstr = outstr + hex_to_bin_str(sw_byte(cmd_str))                    'add command/data
  outstr = outstr + hex_to_bin_str(cmd_trail)                           'add trailer
  print #s_chan, outstr;                                                'send data to sensor
end function
  '----------------------------------------------------
  
sub calc_val (param_arry() as integer, tg as integer)
  local integer ofs
  'calculate values for target
  'param_arry:
  'first index is target number (0 ... 2) passed in tg in the call
  'second index is target value:
  '  0 ... x-distance
  '  1 ... y-distance
  '  2 ... velocity
  '  3 ... resolution
  set_ef(ef_clu, ser_lock)          'lock datastructure
  
  param_arry(tg,0) = d_arry(t1_x+(4*tg),3) : if (param_arry(tg,0) and &h8000) then param_arry(tg,0) = &h8000 - param_arry(tg,0)
  param_arry(tg,1) = d_arry(t1_y+(4*tg),3) - &h8000
  param_arry(tg,2) = d_arry(t1_speed+(4*tg),3) : if (param_arry(tg,2) and &h8000) then param_arry(tg,2) = &h8000 - param_arry(tg,2)
  param_arry(tg,3) = d_arry(t1_res+(4*tg),3)
  
  clr_ef(ef_clu, ser_lock)          'release datastructure
  
end sub
  '----------------------------------------------------
  
sub setup_field_data(fld_size as integer, fld_code as string, fld_endian as string, in_d_arry as string, in_s_arry as string)
'
' create array to store the field definition values and the integer result. Strig results are strored
' in a seperate string array with the same coloum dimension
' As there is no redim command until RC21 the arrays are created in 2 steps via 'execute' command
'
  local string ucmd
  
  on error skip
  dim integer __fldcnt=0
  
  on error skip
  static integer __d_arry(255,4)   'temporary table
  
  if fld_size > 0 then                    'save field definition into temporary table
    __d_arry(__fldcnt,fldsize) = fld_size
    
    select case ucase$(fld_code)
      case "BIN"
        __d_arry(__fldcnt,coding) = bin_code
      case "HEX"
        __d_arry(__fldcnt,coding) = hex_code
      case "STR"
        __d_arry(__fldcnt,coding) = str_code
      case else
        print "%W-setup_field-error in field type ";fld_code
    end select
    
    select case ucase$(fld_endian)
      case "B"
        __d_arry(__fldcnt,endian) = b_end
      case "L"
        __d_arry(__fldcnt,endian) = l_end
      case else
        print "%W-setup_field-error in fieled endian ";fld_endian
    end select
    
    if in_d_arry <> "" then       'define constant for field-name
      ucmd = "dim integer "+in_d_arry+"="+format$(__fldcnt,"%g")
      execute ucmd
    end if
    
    inc __fldcnt        'point to next field
    
  else                  'do array defintiton and copy part
    
    local integer r_idx, c_idx, idx
    
    'create 2 permanent tables
    'table 1
    ucmd="dim integer "+in_d_arry+"("+format$(__fldcnt,"%g")+",4)"
    execute ucmd
    
    'table 2
    ucmd="dim string "+in_s_arry+"("+format$(__fldcnt,"%g")+")"
    execute ucmd
    
    'copy data from temporary table to permanent table
    for r_idx = 0 to __fldcnt
      for c_idx = 0 to 3
        ucmd=in_d_arry+"("+format$(r_idx,"%g")+","+format$(c_idx,"%g")+")=__d_arry("+format$(r_idx,"%g")+","+format$(c_idx,"%g")+")
        execute ucmd
        '__d_arry(r_idx, c_idx) = 0                          'clear cell
      next c_idx
    next r_idx
    __fldcnt = 0
  end if
end sub
  
function decode_data(in_str as string, d_arry() as integer, s_arry() as string) as integer
  
  local integer ptr=1, idx, d_idx, i_idx, o_idx, item_len, i_work(255), i_tmp, no_shft, max_idx, lo_idx
  local string s_work(255) length 1
  
  '   const l_end=0, b_end=1, bin_code=0, hex_code=1, str_code=2
  '   const err_not_hex=1, err_strlen=2
  
  'd_arry(x,y)
  ' x ... fieldnumber
  ' (x,0) #of bytes in in_str
  '   0 ... end of table
  ' (x,1) little/big endian
  '   0 ... little endian, 1 ... big endian
  ' (x,2) coding
  '   0 ... binary, 1 ... hex, 2 ... string
  ' (x,3) return value
  
  do
    item_len = d_arry(d_idx,fldsize)            'get length of this data item
    
    'get each byte of the input string into array work()
    for idx = 0 to item_len-1               'loop for length of data item in input string
      s_work(idx) = mid$(in_str,ptr+idx,endian)  'get byte from input
    next idx
    
    'convert to binary into i_arry()
    select case d_arry(d_idx,coding)
      case bin_code                     'convert binary data byte for byte
        for o_idx = 0 to item_len-1
          i_work(o_idx) = asc(s_work(o_idx))
        next o_idx
        inc o_idx,-1                    'adjust to last value in for-loop
        
      case hex_code                   'convert 1 or 2 bytes hex to binary
        no_shft = 0                      'if 0 data have to be shiftes left by 4 bits
        if item_len = 1 then no_shft = 1 'if only one char do not shift up
        for idx = 0 to item_len-1     'loop thru all hex chars
          o_idx = int(idx/2)          'get output index
          select case asc(ucase$(s_work(idx)))  'get current char
            case 48 to 57, 65 to 70    'test for hex char
              i_work(o_idx) = i_work(o_idx) + val("&h"+(s_work(idx)))   'convert from hex
            case else                   'not hex, throw error
              'print "%decode_data-W-notHEX, not a HEX char, Field=";d_idx; " Pos=";idx+1; " char=";s_work(idx)
              decode_data = err_not_hex
          end select
          if no_shft = 0 then i_work(o_idx) = i_work(o_idx)<<4   'if 0 shift up 4 bits
          no_shft = not no_shft             'toggle shift for next pass
        next idx                      'next input char
        
      case str_code
        for o_idx = 0 to item_len-1
          s_arry(d_idx) = s_arry(d_idx) + s_work(o_idx)     'append char to output string
          i_work(o_idx) = asc(s_work(o_idx))                    'add binary representation as well
        next o_idx
        inc o_idx,-1                    'adjust to last value in for-loop
        
    end select
    
    'adjust for little_endian
    IF d_arry(d_idx,endian) = l_end then         'check if swap needed
      select case d_arry(d_idx,coding)
        case bin_code, str_code
          max_idx = item_len-1  'get maximum index for loop
        case hex_code
          max_idx = item_len-int(item_len/2)-1  'get maximum index for loop
      end select
      'print "max-idx, item_len ";max_idx, item_len
      for idx = 0 to max_idx                'loop thru binary values
        i_idx = idx                         'set input index
        lo_idx = max_idx - idx              'set output index
        if lo_idx < i_idx then              'do only for lower half of the table
          'print "<> ";i_idx, lo_idx
          i_tmp = i_work(i_idx)             'save lower value
          i_work(i_idx) = i_work(lo_idx)    'put higher value into it
          i_work(lo_idx) = i_tmp            'store lower value back
        end if
      next idx
    end if
    
    'all conversions done, save result for this field
    d_arry(d_idx,result) = i_work(0)         'save first element
    i_work(0) = 0                       'clear first element
    for idx = 1 to o_idx                  'loop thru the other elements for this field
      d_arry(d_idx,result) = (d_arry(d_idx,result)<<8) + i_work(idx)  'shift and add element
      i_work(idx) = 0                     'clear element
    next idx
    
    ptr = ptr + (item_len)              'point to next byte in input string
    inc d_idx                           'point to next data entry
  loop until d_arry(d_idx,fldsize) = 0        'exit if 0
  if item_len-1 <> o_idx then
    'print "%decode_data-W-length missmatch"
    decode_data = err_strlen
  end if
  
end function
  '----------------------------------------------------
  
function init_sensor(s_chan as integer, init_mode as string) as integer
  local integer stat, offs, errcnt
  'init sensor by sending begin-config, mode, end-config
  
  'send begin configuration
  offs = get_sync()           'save current offset
  set_sync(ACK_offs)          'setup header/trailer for ACK
  init_sensor = 0             'mak success
  
  'inchar = ""       'clear input buffer
  stat = send_cmd(s_chan, cmd_head, cmd_beg_conf, cmd_trail)
  errcnt = 0
  do
    stat = proc_in_record(inchar, out_str, ef_clu, data_ef)    'just wait for ACK message, payload not processed
    if stat=0 then exit do
    inc errcnt
    pause 50
    ser_int         'get response
    if errcnt > 10 then
      init_sensor = 1     'mark failure
      exit function
    end if
  loop
  
  'inchar = ""       'clear input buffer
  'send mode command
  stat =  send_cmd(s_chan, cmd_head, init_mode, cmd_trail)
  errcnt = 0
  do
    'print "in <- " : prt_str_hex(inchar)
    stat = proc_in_record(inchar, out_str, ef_clu, data_ef)    'just wait for ACK message, payload not processed
    if stat=0 then exit do
    inc errcnt
    pause 50
    ser_int         'get response
    if errcnt > 10 then
      init_sensor = 2     'mark failure
      exit function
    end if
  loop
  
  'inchar = ""       'clear input buffer
  stat = send_cmd(s_chan, cmd_head, cmd_end_conf, cmd_trail)
  errcnt = 0
  do
    'print "in <- " : prt_str_hex(inchar)
    stat = proc_in_record(inchar, out_str, ef_clu, data_ef)    'just wait for ACK message, payload not processed
    if stat=0 then exit do
    inc errcnt
    pause 50
    ser_int         'get responseinc errcnt
    if errcnt > 10 then
      init_sensor = 3     'mark failure
      exit function
    end if
  loop
  
  set_sync(offs)          'restore previous offset
end function
  '----------------------------------------------------
  '-------- end app specific functions and subs -------
  
  '++++++++ eventflag handlers ++++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++
sub set_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 1
end sub
  '----------------------------------------------------
  
sub clr_ef (ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = 0
end sub
  '----------------------------------------------------
  
sub toggle_ef(ef_clstr as integer, ef_no as integer)
  bit(ef_clstr, ef_no) = not bit(ef_clstr, ef_no)
end sub
  '----------------------------------------------------
  
function test_ef (ef_clstr as integer, ef_no as integer, bit_reset as integer) as integer
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  '   ef_no = -1 tests for any eventflag set
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  if ef_no = -1 then       'all ef specified
    if ef_clstr = 0 then   'test for any ef set
      test_ef = 0         'none set
    else                  'some set
      test_ef = 1
    end if
  else                    'specific ef
    test_ef = bit(ef_clstr, ef_no)      'get ef value
  end if
  if bit_reset = 1 then bit(ef_clstr, ef_no) = 0      'reset ef if requried
end function
  '----------------------------------------------------
  
sub waitfr_ef(byref ef_clstr as integer, ef_no as integer)
  'test if any ef specified in mask is set, otherwise do spinwait
  do while bit(ef_clstr, ef_no) = 0
  loop
end sub
  '----------------------------------------------------
  '---------- end eventflag handlers ------------------
  
  '++++++++++ Statistics timer functions ++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++
sub set_exec_timer (t_nbr as integer)
  'start timer 0 to 9 if read out before (timer-value = 0)
  on error skip 1
  dim float __exec_tmr(10)
  if __exec_tmr(t_nbr) = 0 then __exec_tmr(t_nbr) = timer
end sub
  '----------------------------------------------------
  
function get_exec_timer (t_nbr as integer) as float
  'get timer-value if non-zero and reset timer-value to 0
  if __exec_tmr(t_nbr) <> 0 then
    get_exec_timer  = timer  - __exec_tmr(t_nbr)
    __exec_tmr(t_nbr) = 0
  end if
end function
  '-----------------------------------------------------
  '---- end Statistics timer functions -----------------
  
  '+++++++++ helper functions and subs +++++++++++++++++
  '+++++++++++++++++++++++++++++++++++++++++++++++++++++
sub prt_str_hex(in_str as string)
  'conveniant sub to print out binary coded strings in hex format
  'used for debugging
  local integer idx, idx1, cpos
  for idx1 = 0 to 7         'number of lines
    for idx = 1 to 32         'number of cols
      cpos = idx1*32+idx
      if cpos > len(in_str) then exit for
      print hex$(asc(mid$(in_str, cpos, 1)),2);" ";
    next idx
    print
    if cpos > len(in_str) then exit for
  next idx1
  print
end sub
  '----------------------------------------------------
  
function sw_byte(in_str as string) as string
  'swap bytes in hex format "0102" to "0201"
  local integer idx
  sw_byte = ""
  for idx = 1 to len(in_str) step 4
    sw_byte = sw_byte + mid$(in_str,idx+2,2) + mid$(in_str,idx,2)
  next idx
end function
  '----------------------------------------------------
  
function bin_str_to_hex (bin_str as string) as string
  ' convert binary string into 2 chars/byte hex string
  local integer idx
  bin_str_to_hex = ""
  for idx = 1 to len(bin_str)
    bin_str_to_hex = bin_str_to_hex + hex$(asc(mid$(bin_str,idx,1)),2) + " "
  next idx
end function
  '----------------------------------------------------
  
function hex_to_bin_str(in_str as string) as string
  'convert each 2 chars of in_str in hex into single char
  local integer idx, t_int1, t_int2
  local string t_char1, t_char2
  
  hex_to_bin_str = ""
  for idx = 1 to len(in_str) step 2
    t_int1 = val("&h"+mid$(in_str,idx,1))
    t_int2 = val("&h"+mid$(in_str,idx+1,1))
    t_int1 = (t_int1<<4) + t_int2
    hex_to_bin_str = hex_to_bin_str + chr$(t_int1)
  next idx
end function
  '----------------------------------------------------
  '------- end helper functions and subs --------------
  
  '++++++++++ Interrupt service routines (ISR) +++++++++
  
  '+++ ISR for console character input ++++
sub char_isr
  'do
  cons_char = ucase$(inkey$)
  'loop until cons_char <> ""
  set_ef (ef_clu, cons_ef)        'signal console input available
end sub
  '----------------------------------------------------
  
  '++++++++ serial line interrupt ++++++++++++++++++++++
sub ser_int
  local string in_str
  in_str = input$(200,s_chan)         'get record
  if len(inchar) + len(in_str) <= 255 then
    inchar = inchar + in_str
  else
    print "%E-Overrun, inchar purged, length was ";len(inchar)
    inchar=""               'purge strings
    in_str=""
    exit sub
  end if
  ' it is possible to lock the datastructure with this semaphore
  if not test_ef(gp_clu,int_dis) then                   'if interrupts on serial are enabled
    if not test_ef(ef_clu, ser_lock) then               'if lock-flag set, datastructures are locked
      set_ef(ef_clu, ser_ef)                            'set eventflag only if lock-flag = 0
    else
      print "%Radar-W-Serial_ISR, Structure lock, data lost"
    end if
  end if
  
  if len(inchar) > 200 then 'emergency brake
    'prt_str_hex(inchar)
    print "%E-Overrun, inchar purged, length was ";len(inchar)
    inchar=""               'purge string
    set_ef(ef_clu, ovr_ef)        'signal overrun
  end if
end sub
  '----------------------------------------------------
  '------------ end ISR -------------------------------
