  '***********************************
  '*          radar target visualisation for LD2450 / RD-03D sensors
  '*
  '* partially from gerald at thebackshed-forum
  '***********************************
  
  'PicoMite MMBasic RP2040 Edition V6.00.03
  'OPTION SYSTEM SPI GP6,GP3,GP4
  'OPTION COLOURCODE ON
  'OPTION CPUSPEED (KHz) 200000
  'OPTION LCDPANEL ILI9341, LANDSCAPE,GP7,GP2,GP5
  '
  
  OPTION EXPLICIT
  OPTION DEFAULT NONE
  option escape
  
  dim string vers
  vers = "Radar-V2.0.8"
  
  '
  'Program to display the radar data in list form or on graphical display
  'Radar sensor used is LD2450
  'Note: zone filtering is currently not implemented
  '
  '
  '  PLEASE READ THE FOLLOWING SECTION CAREFULLY
  '
  '*****************************************************************
  '************                                     ****************
  '************ HW dependencies, change as required ****************
  '************                                     ****************
  '*****************************************************************
  '
  '++++++ definition of serial line: RX/TX pins, com-number ++++++
  '       baudrate, buffersize, ISR name, buffersize for interrupt
  '
  'the following configs are just examples, any valid RX/TX/com1/com2 combination can be used
  
  '+++++++++++++ Touch interrupt pin ++++++++++++++++++++++++++++++
  '!! NOTE: OPTION GUI CONTROLS must not be set !!!
  '!! in case disable with OPTION GUI CONTROLS DISABLE
  
  '+++++++++++ begin default configuration +++++++++++++++++++++++++++++++++++++
  ' this default config will be overwritten if the PLATFORM environment variable
  ' matches one of the configs below
  
  dim integer conf_done
  
  '+++++++++++ specific configurations +++++++++++++++++++++++++++++++++++++++++
  'config 2
  if mm.info(platform) = "480x320-res" then
    if conf_done = 0 then
      setpin gp20,gp21,com2
      const s_comspec = "com2:256000, 256, ser_int, 30"
      const touch_pin = "GP9"    'define the Touch Interrupt Pin
      inc conf_done              'mark configuration done
    end if
  end if
  
  'config 3
  if mm.info(platform) = "320x240-cap" then
    if conf_done = 0 then
      setpin gp20,gp21,com2
      const s_comspec = "com2:115200, 256, ser_int, 30"
      const touch_pin = "GP10"    'define the Touch Interrupt Pin
      inc conf_done               'mark configuration done
    end if
  end if
  '------------- end specific configurations -----------------------------------
  
  '++++++++++++++++ DEFAULT config +++++++++++++++++++++++++++++++++++++++++++++
  if conf_done = 0 then       'use default config
    setpin gp0,gp1,com1
    const s_comspec = "com1:256000, 256, ser_int, 30"
    const touch_pin = "GP9"    'define the Touch Interrupt Pin
    inc conf_done               'mark configuration done
  end if
  ' --------end default config -------------------------------------------------
  
  '+++++++++++++++++++++ font settings ++++++++++++++++++++++++++++
  'used for info and annotation on graphic screens and menues
  const fnt = 7              'font number
  const fnt_mul = 1          'font multiplier
  
  const lr_fnt = 1          'font for L/R annotation
  const lr_fnt_mul = 2      'font multiplier
  
  '++++++++++++++++++ console messages ++++++++++++++++++++++++++++
  'NOTE: if running on WebMite hardware, console messages should be disabled as
  'from experiance if there is no permanent Telnet connection it will lock the
  'program over time
  '
  'const cons_msg = 0        'disable console messages
  '
  'on other environments it can be enabled
  const cons_msg = 1        'enable console messages at startup
  '
  '*****************************************************************
  '****************** end dependencies *****************************
  '*****************************************************************
  
  
  '**********+ LD2450LD parameter definitions ***************
  '
  'definition of some length parameters for the sensor data exchange
  '+++++++++ record specs +++++++++++++++++++++++++++++++++++
  const data_hdr_len = 4       'offset for 1st data byte in record
  
  dim integer hdr_len     'size of actual header to process
  '--------- end record specs
  
  'defintiton of command and reply strings for the sensor
  'the definition conforms to the operations manual of the LD2450LD sensor
  'in the actual sent/received packets data value low/high bytes are corrected to high byte first
  'this is done in the send/receive routines
  
  '+++++++ definition of synch/ACK characters +++++++++++++++
  dim string bin_hdr_arr(1) = (hex_to_bin_str("AAFF0300"), hex_to_bin_str("FDFCFBFA"))            'header for data / ack
  dim string bin_trl_arr(1) = (hex_to_bin_str("55CC"), hex_to_bin_str("04030201"))                'trailer for data / ack
  
  dim integer hdr_idx
  '-------- end synch chars def -----------------------------
  
  '+++++++ definition of command characters +++++++++++++++++
  dim string cmd_head = "FDFCFBFA", cmd_trail = "04030201"      'cmd header/trailer
  dim string cmd_m_track = "0090", cmd_s_track = "0080"         'cmd code to track multi/single object
  dim string cmd_beg_conf = "00FF0001", cmd_end_conf = "00FE"   'cmd begin/end configuration
  '-------- end command chars def ---------------------------
  '**********+ end LD2450LD parameter definitions ************
  
  '+++++++++ variables section ++++++++++++++++++++++++++++++
  dim string sens_rec, sens_data, cons_char
  dim integer idx, stat, targ
  dim float t_tmr, alpha, dist
  dim integer param_arry(3,4)                 'contains the radar measurements for 3 targets
  '--------- end variables ----------------------------------
  
  '++++++++++ eventflag definitions +++++++++++++++++++++++++
  dim integer  ef_clu                             '64 bit eventflag cluster
  const ser_ef = 0, radar_ef = 1, ser_lock = 2, cons_ef = 3, sens_ef = 4    'eventflag numbers
  '---------- end eventflags --------------------------------
  
  '+++++++++ flags definitions ++++++++++++++++++++++++++++++
  dim integer fl_clu
  const int_dis = 1                                 'if set serial line interrupts are ignored
  const dsp_enab = 2, prt_enab = 3, stat_enab = 4   'if set display, print and statistics are enabled
  const z_mode = 6                                  '0 ... suppress / 1 ... eanble zero x-axis tuples
  const dis_msg = 7                                 '1 ... print messages on console
  
  '+++++++++ non volatile flags definition
  dim integer nv_clu
  const trk_mode = 0                                'clear ... single target, set ... multi target
  const swp_hor = 1                                 '0 ... no swap (exchange left/right), 1 ... swap
  const dim_flg = 2                                 'dimension, 0 ... metric, 1 ... inch
  
  '+++++++++ graphics definitions +++++++++++++++++++++++++++
  const d_max_mm = 8000           'define horizontal max range in mm
  const d_max_inch = 300*25.4     'define horizontal max range in Inch
  
  const grid_col = rgb(green)     'define grid color (arcs and lines)
  const grid_txt = rgb(yellow)    'define grid text color
  const grid_m = " m"             'define grid text unit meter
  const grid_i = " "+ chr$(34)             'define grid text unit inch
  const x_0 = mm.hres/2, y_0 = mm.vres    'origin for the radar grid relativ to display coordinate system
  
  dim integer x_pos, y_pos, x_draw(3), y_draw(3), h_mm, v_mm, d_max
  dim float h_fact, v_fact
  dim string grid_unit
  
  gosub set_inch_mm
  
  dim integer x_res, y_res
  dim float lin(5), bordr(4), f_idx
  'define border lines via x or y offset only
  bordr(0) = -mm.hres/2   'left border
  bordr(1) = mm.hres/2    'right border
  bordr(2) = mm.vres      'top border
  bordr(3) = 0            'bottom border
  '--------- end graphics -----------------------------------
  
  '+++++++++ PMG stuff ++++++++++++++++++++++++++++++++++++++
  'define number of GUI objects
  const max_pmg_obj = 10
  
  dim integer obj_stat(max_pmg_obj)   'status array for objects
  dim integer n_lin, n_obj_lin, x_sz, y_sz, x_spc, y_spc
  dim integer wid, hei, x_offs, y_offs, x_brd
  '------------ end PMG stuff --------------------------------
  
  '+++++++++++ serial channel definitions ++++++++++++++++++++
  const s_chan = 1                        'serial channel number
  '------------ end serial channel ---------------------------
  var restore                         'get flags cluster from nv memory
  font fnt, fnt_mul                   'set standard font
  
  'init menue display
  pmg_init(max_pmg_obj, touch_pin)    'init internal PMG data structures and the Touch Interrupt Pin
  pmg_set_font(1)                     'set pmg font
  
  pmg_get_size(6, 8, n_lin, n_obj_lin, x_sz, y_sz, x_spc, y_spc)  'compute size of objects
  
  wid = x_sz      'width
  hei = y_sz      'heigth
  
  x_offs = 0
  y_offs = mm.vres/2*0.8          'general y offset for menue display
  
  x_pos = 0             'start x position
  y_pos = 0        'start y position
  
  'setup pmg pages
  pmg_set_page(1)       'menue page
  
  'define objects for menue page
  'NOTE: objects have to have unique id's over all pages
  
  pmg_first_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)
  pmg_rbox(5, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"",1)   'BOX for dimensions
  
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)  'skip first box position
  pmg_rbox(1, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Single",1)   'BOX with rounded corners (-5)
  
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
  pmg_rbox(2, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Multi",1)   'BOX with rounded corners
  
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
  pmg_rbox(3, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Print",1)   'BOX with rounded corners
  
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
  pmg_rbox(4, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"noSwap",1)   'BOX with rounded corners
  
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'calc next position
  pmg_next_pos(x_pos, y_pos, x_offs, y_offs, x_brd, x_sz, y_sz, x_spc, y_spc, n_obj_lin, n_lin)    'skip next position
  pmg_rbox(9, x_pos, y_pos, wid, hei, -5, rgb(white), rgb(myrtle),"Exit",1)   'BOX with rounded corners
  
  pmg_set_page(0)     'radar display page
  
  'define menue button at 0,0
  pmg_rbox(0, 0, 0, 60, 20, -5, rgb(white), rgb(myrtle),"Menue",1)   'BOX with rounded corners
  
  'draw radar display (page 0)
  pmg_draw_page       'display current page
  draw_grid           'display distance grid
  gosub draw_anno     'draw annotation
  
  '--------- end variables/constants ------------------------
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++ Begin of executable code +++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  '++++++++ init section ++++++++++++++++++++++++++++++++++++
  '
  'set some configuration flags
  if cons_msg = 0 then      'enable/disable console messages on startup
    clr_ef(fl_clu,dis_msg)  'disable
  else
    set_ef(fl_clu,dis_msg)  'enable
  end if
  
  set_ef(fl_clu,int_dis)        'serial data are handled synchronous not by interrupt for setting up the radar
  set_ef(fl_clu,dsp_enab)       'enable graphical display
  
  'open serial line
  open s_comspec as s_chan
  
  'setup and initialize sensor
  stat = init_sensor (s_chan, cmd_s_track)
  'set_ef(nv_clu,trk_mode,1,"nv_clu")        'tracking mode is single, save in nv memory
  if stat <> 0 then
    print "%Radar-F-Init, init failed with error ";stat, bin_str_to_hex(sens_rec)
end           'end program if sensor init fails
  end if
  
  'now we enable interrupt processing on serial line to drive the processing
  clr_ef(fl_clu,int_dis)
  
  on key char_isr             'enable character interrupts from console
  
  'begin neasurement
  start_measure()
  
  'say hello to the outside world
  dsp_msg("*************", fl_clu, dis_msg)
  dsp_msg("%Radar-I-Start, measurement starts", fl_clu, dis_msg)
  dsp_msg("*************", fl_clu, dis_msg)
  
  'simulate console input to display help text
  cons_char = "H"
  cons_menue
  
  '----------------------------------------------------------
  '--------------- end init section -------------------------
  '----------------------------------------------------------
  
  'processing of sensor data is as follows:
  ' serial interrupt sets sens_ef, indicating some data are in the input-buffer (sens_rec)
  ' if sens_ef is set, the data are parsed/checked and computed into param_arry()
  ' if parsing is complete, sens_ef is reset and radar_ef is set to signal new data is available
  'if radar_ef is set, the data are displayed
  
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '+++++++++++++++ main code ++++++++++++++++++++++++++++++++
  '++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  
  do                            'main loop
    
    '++++++++++++++++ begin idle loop +++++++++++++++++++++++
    do while not test_ef(ef_clu, sens_ef)         'idle loop, wait for sensor serial eventflag to be set
      
      'idle time statstics
      set_exec_timer(0)       'start timer for idle loop
      
      'time statistics of radar data processing
      t_tmr = get_exec_timer(1)
      if t_tmr <> 0 and test_ef(fl_clu, stat_enab) = 1 then print "radar tmr "; t_tmr      'print timer for radar processing
      
      '++++++ process data from radar measurement +++++++++++++
      'currently we just calculate distance and angle for taget 0 and print them out
      
      if test_ef(ef_clu, radar_ef) then   'if radar flag is set, execute proc_data
        proc_data
        clr_ef(ef_clu, radar_ef)
      end if
      
      if test_ef(ef_clu, cons_ef) then    'if console ef is set execute cons_menue
        cons_menue
        clr_ef(ef_clu, cons_ef)
      end if
      
      '------ end radar data processing -----------------------
      
      '++++++++ menue/submenue handler via PMG routines +++++++
      menu_handler
      
      '+++++
      '----> execute here your other code but do not exceed about 86 mSec runtime
      '----> otherwise you may lose some measurements
      '-----
      
    loop
    '---------------- end idle loop -------------------------
    
    'arrive here if sensor data received and thus ser_ef was set
    
    'statistics for idle loop
    if test_ef(fl_clu, stat_enab) then print "idle tmr "; get_exec_timer(0)      'print timer for idle loop
    
    clr_ef(ef_clu, sens_ef)                  'reset sensor serial eventflag
    set_exec_timer(1)          'start timer for radar processing
    
    stat = get_payload(sens_rec, sens_data, bin_hdr_arr(), bin_trl_arr(), hdr_idx)  'get payload from radar message
    
    if hdr_idx = 0 and len(sens_data) > 0 then             'radar data packet found
      'calculate radar datasyn_hdr_arr
      for targ = 0 to 2
        calc_val(param_arry(), targ, sens_rec, hdr_len)      'compute values
      next targ
      set_ef (ef_clu, radar_ef)        'signal new radar data available
    end if
  loop
  '----------------------------------------------
  '-------------- end main code -----------------
  '----------------------------------------------
  
  'GOSUB area
draw_anno:
  'draw annotation text on radar page
  text 0,mm.vres-mm.fontheight,vers                         'shows version in left bottom corner
  if test_ef(nv_clu,trk_mode) then                          'test current tracking mode
    text mm.hres-mm.fontwidth,mm.vres-mm.fontheight,"M" 'shows mode M in right bottom corner
  else
    text mm.hres-mm.fontwidth,mm.vres-mm.fontheight,"S" 'shows mode S in right bottom corner
  end if
return
  '----------------------------------------------
  
set_inch_mm:
  'set value for d_max depending on mm or Inch to display beautiful numbers on scale
  if test_ef(nv_clu, dim_flg) then    'Inches
    d_max = d_max_inch         'maximum distance in Inch
  else                       'mm
    d_max = d_max_mm              'maximum distance in mm
  end if
  
  v_mm = d_max
  h_mm = d_max
  v_fact = mm.vres/v_mm     'hor scaling factor in pixel/mm
  h_fact = mm.hres/h_mm     'vert scaling factor in pixel/mm
  
return
  '----------------------------------------------
  
  '---------- End GOSUBs ------------------------
  
  '+++++++++++ DEV AREA +++++++++++++++++++
sub dsp_msg(msg as string, flg_clu as integer, flg as integer)
  if test_ef(flg_clu, flg) then
    print msg
  end if
end sub
  '---------- END DEV AREA ----------------
  
sub draw_grid
  local float txt_mark
  local string circ_text
  local integer font_save
  
  'draw lines
  for f_idx = -157.5 to -22.5 step 22.5   'select line angles
    lin(0) = 0                            'line origin is x/y 0,0
    lin(1) = 0
    lin(2) = mm.hres/2*cos(pi/180*f_idx)  'line end x, max x is mm.hre/2
    lin(3) = mm.vres*sin(pi/180*f_idx)    'line end y, may y is mm.vres
    
    'get intercept point of line with border of display
    get_intercept(lin(), bordr(), x_res, y_res)
    
    'draw line from origin to border
    'x origin is mm.hres/2
    'y origin is mm.vres
    'this gives an origin of bottom-middle
    line mm.hres/2,mm.vres, x_res+(mm.hres/2), mm.vres-y_res,,Rgb(green)
  next f_idx
  
  'draw arcs
  for f_idx = 1 to 4
    
    txt_mark = v_mm/4*(f_idx)           'now in mm
    if test_ef(nv_clu, dim_flg) then    'if Inch
      txt_mark = int(txt_mark/25.4)     'mm -> Inch, only integer part
      grid_unit = grid_i                '" for Inch
    else                                'if meter
      txt_mark = txt_mark/1000          'mm -> m
      grid_unit = grid_m                'm
    end if
    
    circle mm.hres/2,mm.vres,mm.vres*f_idx/4,,,grid_col
    text mm.hres/2,mm.vres*(4-f_idx)/4+(mm.fontheight*1.1),format$(txt_mark,"%g")+grid_unit,"CB",7,1,grid_txt
  next f_idx
  
  'annotate Left / Right side on the screen
  font_save = mm.info(font)         'save current font
  font lr_fnt, lr_fnt_mul           'set font for L/R annotation
  if test_ef(nv_clu, swp_hor) = 0 then    'check flag for set
    text 0,mm.vres/2-(mm.fontheight/2),"L"
    text mm.hres-mm.fontwidth,mm.vres/2-(mm.fontheight/2),"R"
  else                                    'flag cleared
    text 0,mm.vres/2-(mm.fontheight/2),"R"
    text mm.hres-mm.fontwidth,mm.vres/2-(mm.fontheight/2),"L"
  end if
  font font_save                    'restore to current font
end sub
  
sub get_intercept(lin() as float, bordr()as float, x_pos as integer, y_pos as integer)
  local float x1, y1, k1, dx1, dy1
  
  'common calculations
  dx1 = lin(2)-lin(0)   'delta x
  dy1 = lin(3)-lin(1)   'delta y
  if dx1 = 0 then dx1 = 1e-10   'prevent divide by zero error
  k1 = dy1/dx1    'slope
  if k1 = 0 then k1=1e-10       'prevent divide by zero error
  
  'specific calculations
  
  'x/y of intercept point with top border
  y1 = bordr(2)        'upper border y value
  x1 = (y1-lin(4))/k1 'intercept point x-coordinate
  y1 = k1*x1+lin(4)   'y-coordinate, check only
  x_pos = x1
  y_pos = y1
  if x_pos >= bordr(0) and x_pos <= bordr(1) then exit sub
  
  'x/y of intercept point with right border
  x1 = bordr(1)        'right border x value
  y1 = k1*x1+lin(4)   'intercept point, y-coordinate
  x1 = (y1-lin(4))/k1 'x-coordinate, check only
  x_pos = x1
  y_pos = y1
  if y_pos >= bordr(3) and y_pos <= bordr(2) then exit sub
  
  'x/y of intercept point with left border
  x1 = bordr(0)        'left border x value
  y1 = k1*x1+lin(4)   'intercept point, y-coordinate
  x1 = (y1-lin(4))/k1 'x-coordinate, check only
  x_pos = x1
  y_pos = y1
  if y_pos >= bordr(3) and y_pos <= bordr(2) then exit sub
  
  'x/y of intercept point with bottom border
  y1 = bordr(3)        'lower border y value
  x1 = (y1-lin(4))/k1 'intercept point x-coordinate
  y1 = k1*x1+lin(4)   'y-coordinate, check only
  x_pos = x1
  y_pos = y1
  
end sub
  
  '+++ ISR for console character input ++++
sub char_isr
  do
    cons_char = ucase$(inkey$)
  loop until cons_char <> ""
  set_ef (ef_clu, cons_ef)        'signal console input available
end sub
  
sub cons_menue
  if asc(cons_char) < 32 then exit sub    'discard control chars
  select case cons_char
    case "C"
      toggle_ef(fl_clu,dis_msg)       'toggle flag
      if test_ef(fl_clu, dis_msg) then
        print "C on"
      else
        print "C off"
      end if
    case "D"
      toggle_ef(fl_clu,dsp_enab)       'toggle flag
      if test_ef(fl_clu, dsp_enab) then
        print "D on"
      else
        print "D off"
      end if
    case "P"
      toggle_ef(fl_clu,prt_enab)       'toggle flag
      if test_ef(fl_clu, prt_enab) then
        print "P on"
      else
        print "P off"
      end if
    case "S"
      toggle_ef(fl_clu,stat_enab)       'toggle flag
      if test_ef(fl_clu, stat_enab) then
        print "s on"
      else
        print "S off"
      end if
    case "X"
end
    case "1"
      stat = init_sensor (s_chan, cmd_s_track)
      if stat = 0 then
        dsp_msg("%Radar-I-mode, mode set to single", fl_clu, dis_msg)
        clr_ef(nv_clu,trk_mode,1,"nv_clu")        'tracking mode is single, save in nv memory
        gosub draw_anno
      else
        dsp_msg("%Radar-E-modefail, set mode failed "+ str$(stat), fl_clu, dis_msg)
      end if
    case "M"
      stat = init_sensor (s_chan, cmd_m_track)
      if stat = 0 then
        dsp_msg("%Radar-I-mode, mode set to multi", fl_clu, dis_msg)
        set_ef(nv_clu,trk_mode,1,"nv_clu")        'tracking mode is multi, save in nv memory
        gosub draw_anno
      else
        dsp_msg("%Radar-E-modefail, set mode failed "+ str$(stat), fl_clu, dis_msg)
      end if
    case "Z"
      toggle_ef(fl_clu,z_mode)       'toggle flag
      if test_ef(fl_clu, z_mode) then
        print "Z on"
      else
        print "Z off"
      end if
    case "W"                          'swap (exchange left/right)
      toggle_ef(nv_clu,swp_hor,1,"nv_clu")       'toggle flag, save in nv memory
      if test_ef(nv_clu,swp_hor) then 'get current swap state
        pmg_rbox(4,,,,,,,,"Swap  ",1)     'redefine "swap" menue with changed annotation
      else
        pmg_rbox(4,,,,,,,,"noSwap",1)     'redefine "swap" menue with changed annotation
      end if
      if test_ef(nv_clu, swp_hor) then
        print "Swap"
      else
        print "noSwap"
      end if
      select case pmg_get_page() 'select current page
        case 0          'radar page
          pmg_draw_page   'redraw radar buttons
          draw_grid       'redraw grid
        case 1            'menue page
          pmg_draw_page   'redraw menue buttons
      end select
    case "R"              'redraw display
      select case pmg_get_page()
        case 0            'radar page
          pmg_draw_page   'redraw buttons
          draw_grid       'redraw grid
          gosub draw_anno
        case 1            'menue-page
          pmg_draw_page   'redraw buttons
      end select
    case "H"
      print "+++++++++++"
      print "  Version is ";vers
      print "C ... enable/disable console messages"
      print "D ... enable display"
      print "P ... print data on console"
      print "S ... print timer statistics on console"
      print "1 ... single target tracking"
      print "M ... multi-target tracking"
      print "Z ... enable/suppress zero x-axis values"
      print "R ... redraw grid-lines"
      print "W ... sWap horizontially"
      print "X ... Exit"
      print "H ... this help text"
      print "-----------"
    case else
      print cons_char;" is not a valid command ...."
  end select
end sub
  '----------------------------------------------------
  
  '++++++++++ SUBs and FUNCTIONs ++++++++++++++++
  
sub calc_val (param_arry() as integer, tg as integer, in_str as string, hdr_len as integer)
  
  local integer ofs
  'calculate values for target
  'param_arry:
  'first index is target number (0 ... 2) passed in tg in the call
  'second index is target value:
  '  0 ... x-distance
  '  1 ... y-distance
  '  2 ... velocity
  '  3 ... resolution
  'set_ef(ef_clu, )          'lock datastructure
  
  ofs = tg*8            'offset within record to each target data block
  
  '++++++++++++ get X-value
  param_arry(tg,0) = asc(mid$(in_str, hdr_len+ofs+1, 1)) + (asc(mid$(in_str, hdr_len+ofs+2, 1))<<8)
  'correction of x-value to pos/neg values
  if (asc(mid$(in_str, hdr_len+ofs+2, 1)) and &h80) then
    param_arry(tg,0) = &h8000 - param_arry(tg,0)
  end if
  
  if test_ef(nv_clu, swp_hor) then
    param_arry(tg,0) = param_arry(tg,0) * -1
  end if
  
  'compute y, velocity and resolution only if x-value is valid
  if param_arry(tg,0) <> 0 then       'if non-zero x-value, compute rest
    '++++++++++++ get Y-value
    param_arry(tg,1) = asc(mid$(in_str, hdr_len+ofs+3, 1)) + (asc(mid$(in_str, hdr_len+ofs+4, 1))<<8)
    'correction of y-value
    param_arry(tg,1) = param_arry(tg,1) - &h8000
    
    '++++++++++++ get velocity
    param_arry(tg,2) = asc(mid$(in_str, hdr_len+ofs+5, 1)) + (asc(mid$(in_str, hdr_len+ofs+6, 1))<<8)
    'correction of velocity value
    if (asc(mid$(in_str, hdr_len+ofs+6, 1)) and &h80) then
      param_arry(tg,2) = &h8000 - param_arry(tg,2)
    end if
    
    '++++++++++++ get resolution
    param_arry(tg,3) = asc(mid$(in_str, hdr_len+ofs+7, 1)) + (asc(mid$(in_str, hdr_len+ofs+8, 1))<<8)
  end if
  
end sub
  '----------------------------------------------------
  
sub proc_data
  local integer idx
  'processing of the radar data handed over in param_arry()
  
  for idx = 0 to 2
    if param_arry(idx,0)<> 0 then                                     'check if meaningful data are available
      alpha = 180 + atn(param_arry(idx,1)/param_arry(idx,0))*360/pi   'calc distance vector angle
      dist = sqr(param_arry(idx,0)^2 + param_arry(idx,1)^2)           'calc vector distance
    end if
  next idx
  
  if test_ef(fl_clu, prt_enab) then print_data    'print data on console
  if test_ef(fl_clu, dsp_enab) then draw_data     'draw position on display
end sub
  '----------------------------------------------------
  
sub draw_data
  for idx = 0 to 2        'restore screen areas
    on error skip 2
    blit write idx+1, x_draw(idx)-5, y_draw(idx)-5
    blit close idx+1
  next idx
  
  'all positions are calculated in mm
  for idx = 0 to 2      'calc new ball positions to save screen areas
    x_pos = param_arry(idx,0)*h_fact      'scale to diaplay
    y_pos = param_arry(idx,1)*v_fact
    x_draw(idx) = x_0 + x_pos                'get display position
    y_draw(idx) = y_0 - y_pos
    blit read idx+1, x_draw(idx)-5, y_draw(idx)-5, 12, 12   'save area to be overwritten
  next idx
  
  for idx = 0 to 2      'draw new ball positions
    if param_arry(idx,0) <> 0 then
      circle x_draw(idx), y_draw(idx), 5, 1, 1, rgb(white), rgb(white)    'draw new position
    end if
  next idx
  
end sub
  '----------------------------------------------------
  
sub print_data
  local integer idx
  for idx = 0 to 2
    if (test_ef(fl_clu, z_mode) = 1) or (param_arry(idx,0)<> 0) then    'check if printout possible
      print "TG: ";idx, param_arry(idx,0), param_arry(idx,1), param_arry(idx,2), param_arry(idx,3),    'print results
      print format$(alpha,"% 3.0f"), format$(dist,"% 4.0f")
    end if
  next idx
end sub
  '----------------------------------------------------
  
sub start_measure()
  'setup params for standard measurement records
  hdr_len = data_hdr_len
  
end sub
  '----------------------------------------------------
  
function init_sensor(s_chan as integer, init_mode as string) as integer
  'init sensor by sending begin-config, mode, end-config
  'function return codes are:
  ' 0 ... success
  ' 1 ... begin config failed
  ' 2 ... mode command failed
  ' 3 ... end config failed
  
  init_sensor = 0       'mark success
  
  'send begin configuration
  inc init_sensor       'mark first failure value
  stat = send_cmd(s_chan, cmd_head, cmd_beg_conf, cmd_trail)
  
  'send mode command
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, init_mode, cmd_trail)
  
  inc init_sensor       'mark next failure value
  stat = send_cmd(s_chan, cmd_head, cmd_end_conf, cmd_trail)
  
  init_sensor = 0       'mark success
  
end function
  '----------------------------------------------------
  
function send_cmd(s_chan as integer, cmd_head as string, cmd_str as string, cmd_trail as string) as integer
  'send command to sensor
  'command is as string of hex bytes as defined in the manual, commands are reformatted to low byte first
  'header and trailer are strings and left unchanged to comply with the manual
  local string outstr, in_str
  outstr = hex_to_bin_str(cmd_head)                                     'add header
  outstr = outstr + hex_to_bin_str(sw_byte(hex$(len(cmd_str)/2,4)))     'add command/data length
  outstr = outstr + hex_to_bin_str(sw_byte(cmd_str))                    'add command/data
  outstr = outstr + hex_to_bin_str(cmd_trail)                           'add trailer
  print #s_chan, outstr;                                                'send data to sensor
end function
  '----------------------------------------------------
  
function sw_byte(in_str as string) as string
  'swap bytes in hex format "0102" to "0201"
  local integer idx
  sw_byte = ""
  for idx = 1 to len(in_str) step 4
    sw_byte = sw_byte + mid$(in_str,idx+2,2) + mid$(in_str,idx,2)
  next idx
end function
  '----------------------------------------------------
  
function bin_str_to_hex (bin_str as string) as string
  local integer idx
  bin_str_to_hex = ""
  for idx = 1 to len(bin_str)
    bin_str_to_hex = bin_str_to_hex + hex$(asc(mid$(bin_str,idx,1)),2)
  next idx
end function
  '----------------------------------------------------
  
function hex_to_bin_str(in_str as string) as string
  'convert 2 chars of in_str in hex into single char in out_str
  local integer idx, t_int1, t_int2
  local string t_char1, t_char2
  
  hex_to_bin_str = ""
  for idx = 1 to len(in_str) step 2
    t_int1 = val("&h"+mid$(in_str,idx,1))
    t_int2 = val("&h"+mid$(in_str,idx+1,1))
    t_int1 = (t_int1<<4) + t_int2
    hex_to_bin_str = hex_to_bin_str + chr$(t_int1)
  next idx
end function
  '----------------------------------------------------
  
  'sub set_ef (ef_clstr as integer, ef_no as integer, ef_save as integer)
  'bit(ef_clstr, ef_no) = 1                'set flag
  'end sub
  
sub set_ef (ef_clstr as integer, ef_no as integer, ef_save as integer, ef_name as string)
  bit(ef_clstr, ef_no) = 1                'set flag
  if ef_save then execute "var save "+ef_name
end sub
  '----------------------------------------------------
  
  'sub clr_ef (ef_clstr as integer, ef_no as integer)
  'bit(ef_clstr, ef_no) = 0
  'end sub
  
sub clr_ef (ef_clstr as integer, ef_no as integer, ef_save as integer, ef_name as string)
  bit(ef_clstr, ef_no) = 0
  if ef_save then execute "var save "+ef_name
end sub
  '----------------------------------------------------
  
  'sub toggle_ef(ef_clstr as integer, ef_no as integer)
  'bit(ef_clstr, ef_no) = not bit(ef_clstr, ef_no)
  'end sub
  
sub toggle_ef(ef_clstr as integer, ef_no as integer, ef_save as integer, ef_name as string)
  bit(ef_clstr, ef_no) = not bit(ef_clstr, ef_no)
  if ef_save then execute "var save "+ef_name
end sub
  '----------------------------------------------------
  
function test_ef (ef_clstr as integer, ef_no as integer, bit_reset as integer) as integer
  'ef_clstr ... eventflag cluster as integer 64 bit
  'ef_no ... eventflag number 0 ... 63
  'bit_reset: 0 ... flag left unchanged, 1 ... reset flag after test
  
  test_ef = bit(ef_clstr, ef_no)      'get ef value
  if bit_reset = 1 then bit(ef_clstr, ef_no) = 0      'reset ef if requried
end function
  '----------------------------------------------------
  
sub waitfr_ef(byref ef_clstr as integer, ef_no as integer)
  'test if any ef specified in mask is set, otherwise do spinwait
  do while bit(ef_clstr, ef_no) = 0
  loop
end sub
  '----------------------------------------------------
  
sub set_exec_timer (t_nbr as integer)
  'start timer 0 to 9 if read out before (timer-value = 0)
  on error skip 1
  dim float __exec_tmr(10)
  if __exec_tmr(t_nbr) = 0 then __exec_tmr(t_nbr) = timer
end sub
  '----------------------------------------------------
  
function get_exec_timer (t_nbr as integer) as float
  'get timer-value if non-zero and reset timer-value to 0
  if __exec_tmr(t_nbr) <> 0 then
    get_exec_timer  = timer  - __exec_tmr(t_nbr)
    __exec_tmr(t_nbr) = 0
  end if
end function
  '----------------------------------------------------
  
  'conveniant sub to print out binary coded strings in hex format
  'used for debugging
sub prt_str_hex(in_str as string)
  local integer idx
  for idx = 1 to len(in_str)
    print hex$(asc(mid$(in_str, idx, 1)),2);
  next idx
  print
end sub
  '----------------------------------------------------
  
  '++++++++++ Interrupt service routines (ISR) +++++++++
  
  '++++++++ serial line interrupt ++++++++++++++++++++++
sub ser_int
  local in_len as integer, sens_in as string
  
  in_len = loc(s_chan)                              'get size of data pending
  sens_in = input$(in_len,s_chan)                   'get pending data
  
  'if recordbuffer exceeds 1 record print warning. This is not necessarily a problem
  if len(sens_rec) + len(sens_in) >= 255 then
    dsp_msg("%Radar-W-Buffsz, input buffer size reached "+format$(loc(s_chan)), fl_clu, dis_msg)
    sens_rec = ""                                   'clear buffer
  end if
  
  cat sens_rec, sens_in                             'add to record
  
  ' it is possible to lock the datastructure with this semaphore
  if not test_ef(fl_clu,int_dis) then                   'if interrupts on serial are enabled
    if not test_ef(ef_clu, ser_lock) then               'if lock-flag set, datastructures are locked
      set_ef(ef_clu, ser_ef)                            'set eventflag only if lock-flag = 0
    else
      dsp_msg("%Radar-W-Serial_ISR, Structure lock, data lost", fl_clu, dis_msg)
    end if
  end if
  set_ef(ef_clu,sens_ef)                                'set eventflag
end sub
  
  '----------------------------------------------------
  
  '+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
  '     PMG subs
  '+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
sub pmg_set_font(font_nr as integer)
  __pmg_font_save = mm.info(font)
  __pmg_font = font_nr
end sub
  '-----------------------------------------------
  
sub pmg_draw_page
  local integer idx
  
  cls
  __pmg_font_save = mm.info(font)   'save current font
  font __pmg_font                   'activate pmg font
  for idx = 0 to __pmg_obj_max      'loop through all objects
    if __pmg_obj(idx,8) = __pmg_cur_page then   'check for objects on current page
      select case __pmg_obj(idx,7)  'select object type
        case 1, 2                   'BOX, RBOX
          pmg_rbox(idx)             'draw BOXes
        case 3                      'CIRCLE
          pmg_circle(idx)           'draw CIRCLEs
      end select
    end if
  next idx                        'next object
  font __pmg_font_save            'restore previous font
end sub
  '-----------------------------------------------
  
sub pmg_first_pos(x_pos as integer, y_pos as integer, x_off as integer, y_off as integer, x_brd as integer, x_sz as integer, y_sz as integer, x_spc as integer, y_spc as integer, n_obj_lin as integer, n_lin as integer)
  x_pos = x_off
  y_pos = y_off
end sub
  '-----------------------------------------------
  
sub pmg_next_pos(x_pos as integer, y_pos as integer, x_off as integer, y_off as integer, x_brd as integer, x_sz as integer, y_sz as integer, x_spc as integer, y_spc as integer, n_obj_lin as integer, n_lin as integer)
  ' compute next position of BOX
  x_pos = x_pos+x_spc+x_sz        'next x-position
  if x_pos + x_spc + x_sz > mm.hres-x_brd then  'if overflow
    x_pos = x_offs                     'new line, x-pos = 0
    y_pos = y_pos + y_spc + y_sz 'new line y-pos
  end if
end sub
  '-----------------------------------------------
  
sub pmg_get_size (nbr_obj as integer, nbr_chars as integer, nbr_lines as integer, nbr_obj_line as integer, x_size as integer, y_size as integer, x_space as integer, y_space as integer)
  'input:
  ' nbr_obj ... number of objects to be placed
  ' nbr_chars ... max number of chars in annotation
  '
  'output:
  ' nbr_lines ... number of lines required
  ' nbr_obj_line ... number of objects per line
  ' x_size ... x size of an object
  ' y_size ... y size of an object
  ' x_space ... x spacing of objects
  ' y_space ... y spacing of lines
  
  local float x_txt, y_txt, t_x_nbr, t_y_nbr, t_x_spc, t_y_spc, t_y_lin
  x_txt = cint((mm.fontwidth*nbr_chars)*1.1)    'get needed x size + 10%
  y_txt = cint(mm.fontheight*2)               'get needed y size + 10%
  t_x_nbr = int(mm.hres/(x_txt+1))            'numer of objects per line, 1 pixel min spaceing
  t_y_nbr = int(mm.vres/(y_txt+1))            'numnber of available lines, 1 pixel min spaceing
  t_x_spc = cint((mm.hres-(t_x_nbr*x_txt))/t_x_nbr) 'x spaceing
  t_y_spc = cint((mm.vres-(t_y_nbr*y_txt))/t_y_nbr) 'y spaceing
  t_y_lin = int((nbr_obj/t_x_nbr)+1)
  
  nbr_lines = t_y_lin
  nbr_obj_line = t_x_nbr
  x_size = x_txt
  y_size = y_txt
  x_space = t_x_spc
  y_space = t_y_spc
end sub
  '-----------------------------------------------
  
sub pmg_init(pmg_obj_max as integer, t_pin as string)
  
  'setup global variables
  dim integer __pmg_obj_max = pmg_obj_max   'setup maximum objects
  dim integer __pmg_obj_t = -1    'object last touched
  dim integer __pmg_cur_page = 0  'current page for operation
  dim integer __pmg_font = mm.info(font)
  dim integer __pmg_font_save
  
  'setup touch box coordinate arrays
  dim integer __pmg_tch(pmg_obj_max,4)
  '0 ... x0
  '1 ... y0
  '2 ... x1
  '3 ... y1
  
  'setup object attribute array
  dim integer __pmg_obj(pmg_obj_max,9)
  '0 ... x0
  '1 ... y0
  '2 ... width / radius
  '3 ... heigth / line-width
  '4 ... corner radius / aspect ratio
  '5 ... color
  '6 ... fill-color
  '7 ... object type    0=invalid, 1=RBOX, 2=BOX, 3=CIRCLE
  '8 ... object page
  
  'setup annotation string store
  dim string __pmg_obj_ann(pmg_obj_max)
  
  'setup interrupt pin
  setpin mm.info(pinno t_pin), intl, __t_isr
end sub
  '-----------------------------------------------
  
sub pmg_set_page(page_no as integer)
  __pmg_cur_page = page_no
end sub
  '-----------------------------------------------
  
function pmg_get_page() as integer
  pmg_get_page = __pmg_cur_page
end function
  '-----------------------------------------------
  
sub pmg_del_obj(obj_id as integer)
  __pmg_obj(obj_id,7) = 0   'mark object invalid
end sub
  '-----------------------------------------------
  
sub pmg_circle(pmg_obj_id as integer,obj_x0 as integer, obj_y0 as integer, obj_r as integer, obj_lw as integer, obj_ar as float, obj_col as integer, obj_fill as integer, obj_ann as string, inh_drw as integer)
  
  '!!! the touch sensitive area around the circle is a square !!!
  'store touch area coordinates
  if obj_r <> 0 then
    __pmg_tch(pmg_obj_id,0) = obj_x0 - (obj_r*obj_ar)
    __pmg_tch(pmg_obj_id,1) = obj_y0 - (obj_r)
    __pmg_tch(pmg_obj_id,2) = obj_x0 + (obj_r*obj_ar)
    __pmg_tch(pmg_obj_id,3) = obj_y0 + (obj_r)
    
    'store circle attributes
    __pmg_obj(pmg_obj_id,0) = obj_x0
    __pmg_obj(pmg_obj_id,1) = obj_y0
    __pmg_obj(pmg_obj_id,2) = obj_r
    __pmg_obj(pmg_obj_id,3) = obj_lw
    __pmg_obj(pmg_obj_id,4) = obj_ar
    __pmg_obj(pmg_obj_id,5) = obj_col
    __pmg_obj(pmg_obj_id,6) = obj_fill
    __pmg_obj(pmg_obj_id,7) = 3     'object type CIRCLE
    __pmg_obj(pmg_obj_id,8) = __pmg_cur_page
  end if
  
  if obj_ann <> "" then                 'save annotation text
    __pmg_obj_ann(pmg_obj_id) = obj_ann
  end if
  
  if inh_drw then exit sub    'exit if inhibit draw is true
  
  if __pmg_obj(pmg_obj_id,7) <> 0 then    'draw if object is valid
    circle __pmg_obj(pmg_obj_id,0), __pmg_obj(pmg_obj_id,1), __pmg_obj(pmg_obj_id,2), __pmg_obj(pmg_obj_id,3), __pmg_obj(pmg_obj_id,4), __pmg_obj(pmg_obj_id,5), __pmg_obj(pmg_obj_id,6)
    
    text __pmg_obj(pmg_obj_id,0), __pmg_obj(pmg_obj_id,1),__pmg_obj_ann(pmg_obj_id),"CM",,,,__pmg_obj(pmg_obj_id,6)    'add annotation
  end if
end sub
  '-----------------------------------------------
  
sub pmg_rbox(pmg_obj_id as integer, obj_x0 as integer, obj_y0 as integer, obj_w as integer, obj_h as integer, obj_crad as integer, obj_col as integer, obj_fill as integer, obj_ann as string, inh_drw as integer)
  
  ' obj_crad controls if BOX or RBOX is used
  ' >= 0 is BOX and used as box line-width
  ' < 0 is RBOX and used as box corner radius
  
  if obj_w <> 0 and obj_h <> 0 then   'data provided, store it
    'store touch area coordinates
    __pmg_tch(pmg_obj_id,0) = obj_x0
    __pmg_tch(pmg_obj_id,1) = obj_y0
    __pmg_tch(pmg_obj_id,2) = obj_x0 + obj_w
    __pmg_tch(pmg_obj_id,3) = obj_y0 + obj_h
    
    'store box attributes
    __pmg_obj(pmg_obj_id,0) = obj_x0
    __pmg_obj(pmg_obj_id,1) = obj_y0
    __pmg_obj(pmg_obj_id,2) = obj_w
    __pmg_obj(pmg_obj_id,3) = obj_h
    __pmg_obj(pmg_obj_id,4) = obj_crad
    __pmg_obj(pmg_obj_id,5) = obj_col
    __pmg_obj(pmg_obj_id,6) = obj_fill
    __pmg_obj(pmg_obj_id,8) = __pmg_cur_page
    
    if __pmg_obj(pmg_obj_id,4) < 0 then
      __pmg_obj(pmg_obj_id,7) = 1     'object type RBOX
    else
      __pmg_obj(pmg_obj_id,7) = 2     'object type BOX
    end if
  end if
  
  if obj_ann <> "" then                 'save annotation text
    __pmg_obj_ann(pmg_obj_id) = obj_ann
  end if
  
  if inh_drw then exit sub    'exit if inhibit draw is true
  
  if __pmg_obj(pmg_obj_id,7) <> 0 then    'draw box if object is valid
    'draw box
    if __pmg_obj(pmg_obj_id,7) = 1 then    'test if RBOX
      rbox __pmg_obj(pmg_obj_id,0), __pmg_obj(pmg_obj_id,1), __pmg_obj(pmg_obj_id,2), __pmg_obj(pmg_obj_id,3), __pmg_obj(pmg_obj_id,4)*-1, __pmg_obj(pmg_obj_id,5), __pmg_obj(pmg_obj_id,6)
    end if
    
    if __pmg_obj(pmg_obj_id,7) = 2 then    'test if RBOX
      box __pmg_obj(pmg_obj_id,0), __pmg_obj(pmg_obj_id,1), __pmg_obj(pmg_obj_id,2), __pmg_obj(pmg_obj_id,3), __pmg_obj(pmg_obj_id,4), __pmg_obj(pmg_obj_id,5), __pmg_obj(pmg_obj_id,6)
    end if
    
    text __pmg_obj(pmg_obj_id,0)+(__pmg_obj(pmg_obj_id,2)/2), __pmg_obj(pmg_obj_id,1)+(__pmg_obj(pmg_obj_id,3)/2),__pmg_obj_ann(pmg_obj_id),"CM",,,,__pmg_obj(pmg_obj_id,6)    'add annotation
  end if
end sub
  
  '-----------------------------------------------
sub menu_handler
  if __pmg_obj_t <> -1 then     'check if touch is active
    select case __pmg_obj_t     'select the touched area
      case 0        'menue object, draw setup submenue
        clr_ef(fl_clu,dsp_enab)       'disable ball display
        'draw menue boxes
        if test_ef(nv_clu,swp_hor) then
          pmg_rbox(4,,,,,,,,"Swap  ")     'redraw "Swap" menue with changed annotation
        else
          pmg_rbox(4,,,,,,,,"noSwap")     'redraw "noSwap" menue with changed annotation
        end if
        if test_ef(nv_clu,dim_flg) then
          pmg_rbox(5,,,,,,,," Inch ")     'redraw "Inch" menue with changed annotation
        else
          pmg_rbox(5,,,,,,,,"Meter ")     'redraw "Meter" menue with changed annotation
        end if
        pmg_set_page(1)     'set menue page
        pmg_draw_page       'draw menue page
        gosub draw_anno
      case 1          'single target
        cons_char = "1"
        cons_menue
      case 2          'multi target
        cons_char = "M"
        cons_menue
      case 3          'print
        cons_char = "P"
        cons_menue
        if test_ef(fl_clu,prt_enab) then
          pmg_rbox(3,,,,,,,,"Pr on ")     'redraw "print" menue with changed annotation
        else
          pmg_rbox(3,,,,,,,,"Pr off")     'redraw "print" menue with changed annotation
        end if
      case 4              'swap horizontal view
        toggle_ef(nv_clu,swp_hor,1,"nv_clu")       'toggle flag, save in nv memory
        if test_ef(nv_clu,swp_hor) then
          pmg_rbox(4,,,,,,,,"Swap  ")     'redraw "Swap" menue with changed annotation
        else
          pmg_rbox(4,,,,,,,,"noSwap")     'redraw "noSwap" menue with changed annotation
        end if
      case 5               'dimension metric/inch
        toggle_ef(nv_clu,dim_flg,1,"nv_clu")       'toggle flag, save in nv memory
        gosub set_inch_mm                         'set parameters for Inch or mm
        if test_ef(nv_clu,dim_flg) then
          pmg_rbox(5,,,,,,,," Inch ")     'redraw "Inch" menue with changed annotation
        else
          pmg_rbox(5,,,,,,,,"Meter ")     'redraw "Meter" menue with changed annotation
        end if
      case 6
      case 7
      case 8
      case 9             'exit the settings submenue
        pmg_set_page(0)
        pmg_draw_page
        draw_grid
        gosub draw_anno
        set_ef(fl_clu,dsp_enab)   'enable ball display
    end select
    __pmg_obj_t = -1     'reset touch object id
  end if
end sub
  '-----------------------------------------------
  
sub __t_isr
  'touch interrupt service routine
  ' enterd on high-low transition of touch-IRQ pin
  ' search through object database for match and return object-id in __pmg_obj_t
  local integer idx
  for idx = 0 to __pmg_obj_max    'loop thru all objects
    if __pmg_obj(idx,8) = __pmg_cur_page and __pmg_obj(idx,7) <> 0 then 'if active page and object valid
      If (Touch(x) > __pmg_tch(idx,0)) and (Touch(x) < __pmg_tch(idx,2)) and (Touch(y) > __pmg_tch(idx,1)) and (Touch(y) < __pmg_tch(idx,3)) then 'if within touch box
        __pmg_obj_t = idx     'save object id
        exit sub              'and return
      EndIf
    end if
  next idx
  __pmg_obj_t = -1      'mark no touch
end sub
  '-----------------------------------------------
  
function get_payload(sens_rec as string, sens_data as string, bin_hdr_arr() as string, bin_trl_arr() as string, hdr_idx as integer) as integer
  'get_payload (O) 0=no valid data, 1=data available nothing pending, -1=data available and data pending
  'sens_rec (I) = raw data record from sensor with header/trailer
  'sens_data (O) = payload without header/Trailer
  'bin_hdr_arr() (I) = header strings as binary chars
  'bin_trl_arr() (I) = trailer srings as binary chars
  'hdr_idx (O) = index into bin_hdr_arr() and bin_trl_arr()
  
  local integer idx, hdr_pos, hdr_pos_2, trl_pos
  local integer hdr_size, hdr_size_2, trl_size, trl_idx, corr
  local integer max_hdr_len, max_trl_len
  
  'print "enter gpld len";len(sens_rec)
  
  'on first entry get the maximum header and trailer length
  if max_hdr_len = 0 or max_trl_len = 0 then       'get max header and trailer length
    for idx = 0 to bound(bin_hdr_arr())            'search thru arrays
      if max_hdr_len < len(bin_hdr_arr(idx)) then  'check header length
        max_hdr_len = len(bin_hdr_arr(idx))        'set header length
      end if
      
      if max_trl_len < len(bin_trl_arr(idx)) then  'check trailer length
        max_trl_len = len(bin_trl_arr(idx))        'set trailer length
      end if
    next idx
    'print "max hdr, trl ";max_hdr_len, max_trl_len
  end if
  
  sens_data = ""
  
  'look for headers and trailer each time
  hdr_pos = find_hdr(hdr_idx, 1, hdr_size, sens_rec, bin_hdr_arr())       'find first header
  trl_pos = find_trl(trl_idx, hdr_pos+hdr_size, trl_size, sens_rec, bin_trl_arr()) 'find trailer after first header
  hdr_pos_2 = find_hdr(hdr_idx, hdr_pos+hdr_size, hdr_size_2, sens_rec, bin_hdr_arr())       'find next header
  
  if hdr_pos > 0 then                   'header found
    if trl_pos > 0 then                 'trailer found
      IF hdr_pos_2 > 0 and hdr_pos_2 < trl_pos then       '2nd header found
        'print "2nd hdr"
        hdr_pos = hdr_pos_2             'skip first header
        'print "bef cut ";bin_str_to_hex(sens_rec)
        sens_rec = cut$(sens_rec,1,hdr_pos-1)   'remove all chars before 2nd header
        'print "aft cut ";bin_str_to_hex(sens_rec)
        hdr_pos = 1                     'use new header on pos 1
        trl_pos = find_trl(trl_idx, hdr_pos, trl_size, sens_rec, bin_trl_arr()) 'find trailer again
      end if
      sens_data = get_data(sens_rec, hdr_pos, trl_pos, hdr_size, trl_size)    'get payload
      'sens_rec = cut$(sens_rec, hdr_pos, trl_pos+trl_size)                   'remove header, payload and trailer
      corr = trim_trailer(sens_rec, trl_pos, trl_size)                        'remove header, payload and trailer
      if len(sens_rec) = 0 then        'no data pending
        get_payload = 1
      else
        get_payload = -1               'data still pending
      end if
      exit function
    else                                'no trailer found
      get_payload = 0                   'no trailer, no usable data yet
      exit function
    end if
  else
    get_payload = 0                     'no header thus no usable data yet
    exit function
  end if
  
end function
  '-----------------------------------------------
  
  
function find_hdr(hdr_idx as integer, hdr_start as integer, hdr_size as integer, sens_rec as string, bin_hdr_arr() as string) as integer
  'debug ok
  'return: position of next header from hdr_start
  
  local integer tmp_pos, idx
  
  for idx = 0 to bound(bin_hdr_arr(idx))  'now search for a matching header
    tmp_pos = instr(hdr_start,sens_rec,bin_hdr_arr(idx))    'look for header
    if tmp_pos > 0 then                   'header found
      hdr_idx = idx                       'set header index
      hdr_size = len(bin_hdr_arr(idx))
      find_hdr = tmp_pos                  'set header start position
      'print "hdr found pos/idx/len ";tmp_pos, hdr_idx, hdr_size
      exit for
    end if
  next idx
end function
  
function find_trl(trl_idx as integer, trl_start as integer, trl_size as integer, sens_rec as string, bin_trl_arr() as string) as integer
  'debug ok
  'return: position of next header from hdr_start
  
  local integer tmp_pos, idx
  
  for idx = 0 to bound(bin_trl_arr(idx))  'now search for a matching header
    tmp_pos = instr(trl_start,sens_rec,bin_trl_arr(idx))    'look for header
    if tmp_pos > 0 then                   'header found
      trl_idx = idx                       'set header index
      trl_size = len(bin_trl_arr(idx))
      find_trl = tmp_pos                  'set header start position
      'print "len sens_rec ";len(sens_rec)
      'print "trl found pos/idx/len ";tmp_pos, trl_idx, trl_size
      exit for
    end if
  next idx
end function
  
function get_data(sens_rec as string, hdr_pos as integer, trl_pos as integer, hdr_size as integer, trl_size as integer) as string
  'debug ok
  'extract data from input string between end-of-header and begin-of-trailer
  'print "get data start, end ";bin_str_to_hex(sens_rec);" ";hdr_pos+hdr_size, trl_pos-1
  get_data = seg$(sens_rec, hdr_pos+hdr_size, trl_pos-1)      'get payload data
  'print "get_data ";bin_str_to_hex(get_data)
end function
  
function seg$(in_str as string, start_pos as integer, end_pos as integer) as string
  'debug ok
  'get string between start_pos and end_pos
  'print "seg$"
  'print len(in_str),start_pos, end_pos
  seg$ = mid$(in_str, start_pos, end_pos-start_pos+1)
end function
  
function cut$(in_str as string, start_pos as integer, end_pos as integer) as string
  'debug ok
  'remove string between and including start_pos and end_pos
  cut$ = left$(in_str, start_pos-1)
  cut$ = cut$ + right$(in_str$, len(in_str)-end_pos)
end function
  
function trim_header(sens_rec as string, hdr_pos as integer, hdr_size as integer) as integer
  'debug ok
  'remove all chars from beginning of the string up to and including the header
  trim_header = -(hdr_pos+hdr_size)+1       'negative number of chars removed
  'print "trim_header"
  'print hdr_pos, hdr_size, trim_header
  sens_rec = right$(sens_rec, len(sens_rec)+1-(hdr_pos+hdr_size))   'remove all chars left to header and header
  'print sens_rec
end function
  
function trim_trailer(sens_rec as string, trl_pos as integer, trl_size as integer) as integer
  'debug ok
  'remove all chars from beginning of the string up to and including the trailer
  trim_trailer = -trl_size       'negative number of chars removed
  'print "trim_trailer"
  'print trl_pos, trl_size, trim_trailer
  sens_rec = right$(sens_rec, len(sens_rec)-(trl_pos+trl_size-1))   'remoce all chars left to trailer and trailer
  'print sens_rec
end function
  
