'-------------------------------------------------------------------------
' Wiznet W5100 Micromite Examples
'
' Hardware Connections
'
' +--------+--------------------------------------------+
' | Wiznet | Micromite                                  |
' +--------+--------------------------------------------+
' | Ground |  Pin 19                                    |
' | +5V    |  +5V                                       |
' | RST    |  Pin 23 (Wiznet W5100 Reset)               |
' | SS     |  Pin 15 (SPI Slave Select)                 |
' | CLK    |  Pin 25 (SPI Clock)                        |
' | MO     |  Pin 3 (SPI MOSI - Master Out, Slave In)   |
' | MI     |  Pin 14 (SPI MISO - Master In, Slave Out)  |
' | Gnd    |  not used                                  |
' | PoE+   |  not used                                  |
' | PoE-   |  not used                                  |
' +--------+--------------------------------------------+

option explicit
Option base 1

CONST optionBase=1       ' Set to match option base above (if changed)
CONST wiznetResetPin%=23 ' Set to pin used for SPI Slave Select of Wiznet

dim ip1%, ip2%, ip3%, ip4%
dim example$
dim txstring$

wiznetReset wiznetResetPin%

ip1%=192
ip2%=168
ip3%=1
ip4%=65

' Set static MAC address/IP address/subnet mask/gateway here

SetMAC &H00,&HFE,ip1%,ip2%,ip3%,ip4%
SetIP ip1%,ip2%,ip3%,ip4%
SetMask 255,255,255,0
SetGateway ip1%,ip2%,ip3%,1

Print GetIP$()

wiznetInitialize    ' Sets various internal registers for Wiznet

' Set to "server" or "client" to choose example to run:

example$="client"

if example$="client" then
' Client example
  pause 500
  do
    if (SockTCPInit(0, 64000)=1) then
      print:print:print "Socket initialized on local port 64000"
    else
      print:print:error "Failed to initialize socket"
    end if
    pause 500
    if SockTCPConnect(0,216,58,220,100,80)=1 then ' http://www.google.com
'    if SockTCPConnect(0,192,168,1,14,80)=1 then ' http://192.168.1.14
      print "Successful connection to webserver on port 80"
    else
      error "Failed to connect to webserver on port 80"
    end if
    
    SockWrite 0, "GET / HTTP/1.1"+chr$(&H0D)+CHR$(&H0A)
    SockWrite 0, "Host: 216.58.220.100"+chr$(&H0D)+CHR$(&H0A)
    SockWrite 0, "User-Agent: Micromite!"+chr$(&H0D)+CHR$(&H0A)
    SockWrite 0, "Connection: close"+chr$(&H0D)+CHR$(&H0A)
    SockWrite 0, chr$(&H0D)+CHR$(&H0A)

    pause 1000

    do while sockData(0)
      print chr$(sockRead(0));
    loop
    
    do while sockEstablished(0)=1 'wait for server to close connection
    loop

    'Get any remaining incoming data waiting in buffer
    do while sockData(0)
      print chr$(sockRead(0));
    loop
    pause 9500
  loop
else
' Server example
  pause 500
  do
    'If socket closed, start listening for incoming connections
    if (SockClosed(0)=1) then
      if (SockTCPInit(0, 23)=1) then
        print:print:print "Socket initialized on local port 23"
      else
        print:print:error "Failed to initialize socket"
      end if
      pause 500
      if (SockTCPListen(0)=1) then
        print "Socket listening for incoming connections - telnet to IP address"
      else
        error "Failed to listen on socket"
      end if
    end if
    
    'Get any incoming data waiting in buffer
    do while sockData(0)
      print chr$(sockRead(0));
    loop
    
    'Send some data to the client
    if SockEstablished(0)=1 then
      txstring$="0123456789"
      SockWrite 0, txstring$
    end if
  loop
end if
end

'-------------------------------------------------------------------------
' WIZNET W5100 Subs and Functions

sub wiznetReset(resetPin%)
  SetPin resetPin%, dout
  Pin(resetPin%)=0
  ' Wiznet docs: assert low for 2us to reset Wiznet
  Pause 10 ' wait 10 milliseconds
  Pin(resetPin%)=1
end sub
  
sub wiznetInitialize()
  wiznetWrite(0, 0, &B00000000) ' Mode Register, enable ping
  wiznetWrite(0, &H16, &B00000000) ' Interrupt Mask Register
  wiznetWrite(0, &H17, &H0F) ' Retry time value high (400ms)
  wiznetWrite(0, &H18, &HA0) ' Retry time value low (400ms)
  wiznetWrite(0, &H19, &H08) ' retry count register
  wiznetWrite(0, &H1A, &B01010101) ' RX Memory Size (2KB/socket)
  wiznetWrite(0, &H1B, &B01010101) ' TX Memory Size (2KB/socket)
end sub
  
Function wiznetRead(addrHi%, addrLo%)
  Local read%(2)
  SPI open 4000000, 0, 8, 15
  SPI write 1, &H0F
  SPI write 1, addrHi%
  SPI write 1, addrLo%
  SPI read 1, read%()
  wiznetRead=read%(optionBase)
  SPI close
End Function
  
Sub wiznetWrite(addrHi%, addrLo%, writeData%)
  SPI open 4000000, 0, 8, 15
  SPI write 1, &HF0
  SPI write 1, addrHi%
  SPI write 1, addrLo%
  SPI write 1, writeData%
  SPI close
End Sub
  
Sub SetGateway (oct1%, oct2%, oct3%, oct4%)
  wiznetWrite(&H00, &H01, oct1%)
  wiznetWrite(&H00, &H02, oct2%)
  wiznetWrite(&H00, &H03, oct3%)
  wiznetWrite(&H00, &H04, oct4%)
End Sub
  
Sub SetMask (oct1%, oct2%, oct3%, oct4%)
  wiznetWrite(&H00, &H05, oct1%)
  wiznetWrite(&H00, &H06, oct2%)
  wiznetWrite(&H00, &H07, oct3%)
  wiznetWrite(&H00, &H08, oct4%)
End Sub
  
Sub SetMAC (oct1%, oct2%, oct3%, oct4%, oct5%, oct6%)
  wiznetWrite(&H00, &H09, oct1%)
  wiznetWrite(&H00, &H0A, oct2%)
  wiznetWrite(&H00, &H0B, oct3%)
  wiznetWrite(&H00, &H0C, oct4%)
  wiznetWrite(&H00, &H0D, oct5%)
  wiznetWrite(&H00, &H0E, oct6%)
End Sub
  
Sub SetIP (oct1%, oct2%, oct3%, oct4%)
  wiznetWrite(&H00, &H0F, oct1%)
  wiznetWrite(&H00, &H10, oct2%)
  wiznetWrite(&H00, &H11, oct3%)
  wiznetWrite(&H00, &H12, oct4%)
End Sub
  
Function GetIP$()
  Local oct1%, oct2%, oct3%, oct4%
  oct1%=wiznetRead(&H00, &H0F)
  oct2%=wiznetRead(&H00, &H10)
  oct3%=wiznetRead(&H00, &H11)
  oct4%=wiznetRead(&H00, &H12)
  GetIP$=Str$(oct1%)+"."+Str$(oct2%)+"."+Str$(oct3%)+"."+Str$(oct4%)
End Function
  
function SockTCPInit(socket%, srcport%)
  wiznetWrite(socket%+4, &H00, &H01) 'Sn_MR Mode Register
  wiznetWrite(socket%+4, &H04, (srcport% and &HFF00) / 256) 'Sn_PORT Source Port Hi
  wiznetWrite(socket%+4, &H05, srcport% AND &HFF) 'Sn_PORT Source Port Lo
  wiznetWrite(socket%+4, &H01, &H01) 'Sn_CR Command Register, Open
  'Sn_SR Status Register
  if wiznetRead(socket%+4, &H03)<>&H13 then
    'Failed to init socket
    SockTCPInit=-1
    wiznetWrite(socket%+4, &H01, &H10) 'Sn_CR Command Register, Close
  else
    SockTCPInit=1
  end if
end function
  
function SockTCPListen(socket%)
  wiznetWrite(socket%+4, &H01, &H02) 'Sn_CR Command Register, Listen
  'Sn_SR Status Register
  if wiznetRead(socket%+4, &H03)<>&H14 then
    'Failed to listen
    SockTCPListen=-1
    wiznetWrite(socket%+4, &H01, &H10) 'Sn_CR Command Register, Close
  else
    SockTCPListen=1
  end if
end function
  
function SockEstablished(socket%)
  local SnCR%
  SnCR%=wiznetRead(socket%+4, &H03)
  if SnCR%=&H1C then
    wiznetWrite(socket%+4, &H01, &H08) 'Sn_CR Command Register, DISCON
  end if
  if SnCR%=&H17 then
    SockEstablished=1
  else
    SockEstablished=-1
  end if
end function
  
function SockClosed(socket%)
  local SnCR%
  SnCR%=wiznetRead(socket%+4, &H03)
  if SnCR%=&H00 then
    SockClosed=1
  else
    SockClosed=-1
  end if
end function
  
function SockData(socket%)
  SockData=wiznetRead(socket%+4, &H26)*256+wiznetRead(socket%+4, &H27)
end function
  
function SockRead(socket%)
  LOCAL SnRXRead%, offset%, address%
  SnRXRead%=wiznetRead(socket%+4, &H28)*256+wiznetRead(socket%+4, &H29)
  offset%=SnRXRead% AND &H07FF
  address%=&H6000+(socket%*&H800)+offset%
  SockRead=wiznetRead((address% and &HFF00) / 256, address% and &HFF) 'read incoming byte
  SnRXRead%=SnRXRead%+1
  if SnRXRead%>65535 then SnRXRead%=0
  wiznetWrite(socket%+4, &H28, (SnRXRead% and &HFF00)/256) 'increment counter
  wiznetWrite(socket%+4, &H29, SnRXRead% AND &HFF)
  wiznetWrite(socket%+4, &H01, &H40) 'Sn_CR Command Register, RECV
end function
  
sub SockWrite(socket%, sockTX$)
  LOCAL TXchars%, SnTXFSR%, SnTXWR%, offset%, address%, i%
  TXchars%=LEN(sockTX$)
  waitforprocessing:
    SnTXFSR%=wiznetRead(socket%+4, &H20)*256+wiznetRead(socket%+4, &H21)
  if SnTXFSR%<TXchars% then goto waitforprocessing
  SnTXWR%=wiznetRead(socket%+4, &H24)*256+wiznetRead(socket%+4, &H25)
  offset%=SnTXWR% AND &H07FF
  address%=&H4000+(socket%*&H800)+offset%
  for i%=1 to TXchars%
    if address%>&H4000+(socket%*&H800)+&H7FF then address%=address%-&H800
    wiznetWrite((address% and &HFF00) / 256, address% and &HFF, asc(mid$(sockTX$, i%, 1)))
    address%=address%+1
  next i%
  SnTXWR%=SnTXWR%+TXchars%
  if SnTXWR%>65535 then SnTXWR%=SnTXWR%-65536
  wiznetWrite(socket%+4, &H24, (SnTXWR% and &HFF00)/256)
  wiznetWrite(socket%+4, &H25, SnTXWR% and &HFF)
  wiznetWrite(socket%+4, &H01, &H20) 'Sn_CR Command Register, SEND
end sub
  
function SockTCPConnect(socket%, oct1%, oct2%, oct3%, oct4%, destport%)
  LOCAL timerstart%, SnCR%
  wiznetWrite(socket%+4, &H0C, oct1%) 'Destination IP Address
  wiznetWrite(socket%+4, &H0D, oct2%)
  wiznetWrite(socket%+4, &H0E, oct3%)
  wiznetWrite(socket%+4, &H0F, oct4%)
  wiznetWrite(socket%+4, &H10, (destport% and &HFF00) / 256) 'Sn_DPORT Hi
  wiznetWrite(socket%+4, &H11, destport% AND &HFF) ' Lo
  wiznetWrite(socket%+4, &H01, &H04) 'Sn_CR Command Register, Connect
  SockTCPConnect=-1
  timerstart% = timer
  do while (timer - timerstart%)<4000 ' Wait at least 4 seconds for connection
    SnCR%=wiznetRead(socket%+4, &H03)
    if SnCR%=&H17 then ' Established
      SockTCPConnect=1
      exit do
    else if SnCR%=&H00 then ' Closed (Timeout connecting)
      exit do
    end if
  loop
end function
